/**
 * Copyright 2010 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package zkclient;

import java.util.List;
import java.util.Map;

import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.Op;
import org.apache.zookeeper.OpResult;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper.States;
import org.apache.zookeeper.data.ACL;
import org.apache.zookeeper.data.Stat;

/**
 * 
 * @author mxsm
 *
 */

public interface IZookeeperConnection {


	/**
	 * 
	 * @param watcher
	 */
    public void connect(Watcher watcher);

    
    /**
     * 
     * @throws InterruptedException
     */
    void close() throws InterruptedException;

    /**
     * 
     * @param path
     * @param data
     * @param mode
     * @return
     * @throws KeeperException
     * @throws InterruptedException
     */
    public String create(String path, byte[] data, CreateMode mode) throws KeeperException, InterruptedException;

    /**
     * 
     * @param path
     * @param data
     * @param acl
     * @param mode
     * @return
     * @throws KeeperException
     * @throws InterruptedException
     */
    public String create(String path, byte[] data, List<ACL> acl, CreateMode mode) throws KeeperException, InterruptedException;

    /**
     * 
     * @param path
     * @throws InterruptedException
     * @throws KeeperException
     */
    public void delete(String path) throws InterruptedException, KeeperException;

    /**
     * 
     * @param path
     * @param version
     * @throws InterruptedException
     * @throws KeeperException
     */
    public void delete(String path, int version) throws InterruptedException, KeeperException;

    /**
     * 
     * @param path
     * @param watch
     * @return
     * @throws KeeperException
     * @throws InterruptedException
     */
    boolean exists(final String path, final boolean watch) throws KeeperException, InterruptedException;

    /**
     * 
     * @param path
     * @param watch
     * @return
     * @throws KeeperException
     * @throws InterruptedException
     */
    List<String> getChildren(final String path, final boolean watch) throws KeeperException, InterruptedException;

    /**
     * 
     * @param path
     * @param stat
     * @param watch
     * @return
     * @throws KeeperException
     * @throws InterruptedException
     */
    public byte[] readData(String path, Stat stat, boolean watch) throws KeeperException, InterruptedException;

    /**
     * 
     * @param path
     * @param data
     * @param expectedVersion
     * @throws KeeperException
     * @throws InterruptedException
     */
    public void writeData(String path, byte[] data, int expectedVersion) throws KeeperException, InterruptedException;

    /**
     * 
     * @param path
     * @param data
     * @param expectedVersion
     * @return
     * @throws KeeperException
     * @throws InterruptedException
     */
    public Stat writeDataReturnStat(String path, byte[] data, int expectedVersion) throws KeeperException, InterruptedException;

    /**
     * 
     * @return
     */
    public States getZookeeperState();

    /**
     * 
     * @param path
     * @return
     * @throws KeeperException
     * @throws InterruptedException
     */
    public long getCreateTime(String path) throws KeeperException, InterruptedException;

    /**
     * 
     * @return
     */
    public String getServers();

    /**
     * 
     * @return
     */
    public String getResolvedServers();

    /**
     * 
     * @param ops
     * @return
     * @throws KeeperException
     * @throws InterruptedException
     */
    public List<OpResult> multi(Iterable<Op> ops) throws KeeperException, InterruptedException;

    /**
     * 
     * @param scheme
     * @param auth
     */
    public void addAuthInfo(String scheme, byte[] auth);

    /**
     * 
     * @param path
     * @param acl
     * @param version
     * @throws KeeperException
     * @throws InterruptedException
     */
    public void setAcl(final String path, List<ACL> acl, int version) throws KeeperException, InterruptedException;

    /**
     * 
     * @param path
     * @return
     * @throws KeeperException
     * @throws InterruptedException
     */
    public Map.Entry<List<ACL>, Stat> getAcl(final String path) throws KeeperException, InterruptedException;
}
