/**
 * Copyright 2010 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package zkclient;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import zkclient.exception.ZkNoNodeException;
import zkclient.listener.IZookeeperDataListener;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * @param <T>
 *            The data type that is being watched.
 */
public final class ContentWatcher<T extends Object> implements IZookeeperDataListener {

    private static final Logger LOG = LoggerFactory.getLogger(ContentWatcher.class);

    private Lock contentLock = new ReentrantLock(true);
    
    private Condition contentAvailable = contentLock.newCondition();

    private Holder<T> content;
    private String fileName;
    private ZookeeperClient zkClient;

    public ContentWatcher(ZookeeperClient zkClient, String fileName) {
        this.fileName = fileName;
        this.zkClient = zkClient;
    }

    public void start() {
        zkClient.subscribeDataChanges(fileName, this);
        readData();
        LOG.debug("Started ContentWatcher");
    }

    @SuppressWarnings("unchecked")
    private void readData() {
        try {
            setContent((T) zkClient.readData(fileName));
        } catch (ZkNoNodeException e) {
            // ignore if the node has not yet been created
        }
    }

    public void stop() {
        zkClient.unsubscribeDataChanges(fileName, this);
    }

    public void setContent(T data) {
        LOG.debug("Received new data: " + data);
        contentLock.lock();
        try {
            content = new Holder<T>(data);
            contentAvailable.signalAll();
        } finally {
            contentLock.unlock();
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public void handleDataChange(String dataPath, Object data) {
        setContent((T) data);
    }

    @Override
    public void handleDataDeleted(String dataPath) {
        // ignore
    }

    public T getContent() throws InterruptedException {
        contentLock.lock();
        try {
            while (content == null) {
                contentAvailable.await();
            }
            return content.get();
        } finally {
            contentLock.unlock();
        }
    }
}
