package cn.dolphin.wechat.api.mp;

import cn.dolphin.core.exception.ApiException;
import cn.dolphin.core.map.StrHashMap;
import cn.dolphin.https.client.Http;
import cn.dolphin.https.model.HttpResult;
import cn.dolphin.wechat.api.commom.WxBaseAPI;
import cn.dolphin.wechat.mp.consts.MpOAuth2Constant;
import cn.dolphin.wechat.mp.enums.LangType;
import cn.dolphin.wechat.core.enums.WechatEnum;
import cn.dolphin.wechat.mp.model.oauth2.AuthAccessToken;
import cn.dolphin.wechat.mp.enums.AuthType;
import cn.dolphin.wechat.mp.model.oauth2.ValidAccessToken;
import cn.dolphin.wechat.mp.model.user.UserInfo;
import cn.dolphin.wechat.core.util.JsonMapper;
import cn.dolphin.wechat.core.util.Wechatutils;

/**
 * 公众号网页授权
 */
public class Oauth2Api extends WxBaseAPI {


    /**
     * 生成OAuth重定向URI（用户同意授权，获取code）
     *
     * @param appid 公众号的唯一标识
     * @param redirectURI 授权后重定向的回调链接地址， 请使用 urlEncode 对链接进行处理
     * @param scope 应用授权作用域,snsapi_base,snsapi_userinfo
     * @param state 重定向后会带上state参数，开发者可以填写a-zA-Z0-9的参数值，最多128字节
     * @return
     */
    public static String generateRedirectURI(String appid, String redirectURI, AuthType scope, String state) {
        StringBuffer url = new StringBuffer();
        url.append(MpOAuth2Constant.OAUTH2_AUTHORIZE);
        url.append("?appid=").append(Wechatutils.urlEncode(appid));
        url.append("&redirect_uri=").append(Wechatutils.urlEncode(redirectURI));
        url.append("&response_type=code");
        url.append("&scope=").append(Wechatutils.urlEncode(scope.scope()));
        url.append("&state=").append(Wechatutils.urlEncode(state));
        url.append("#wechat_redirect");
        return url.toString();
    }



    /**
     * 通过code换取网页授权access_token
     * @param appid 公众号的唯一标识
     * @param secret 公众号的appsecret
     * @param code 填写第一步获取的code参数
     * @return
     */
    public static AuthAccessToken getAccessToken(String appid,String secret,String code){

        try {
            StrHashMap params = new StrHashMap();
            params.put("appid",Wechatutils.urlEncode(appid));
            params.put("secret",Wechatutils.urlEncode(secret));
            params.put("code",Wechatutils.urlEncode(code));
            params.put("grant_type","authorization_code");

            HttpResult httpResult = Http.getInstance().get(MpOAuth2Constant.OAUTH2_ACCESS_TOKEN, params);
            if(httpResult.getCode()==200){
                // 检查
                Wechatutils.check(httpResult.getContent());

                //return JSONObject.parseObject(httpResult.getContent(), AuthAccessToken.class);
                return JsonMapper.defaultMapper().fromJson(httpResult.getContent(), AuthAccessToken.class);
            }else {
                throw new ApiException(WechatEnum.ERROR.code(), WechatEnum.ERROR.msg());
            }
        } catch (Exception e) {
            throw new ApiException(e.getMessage(),e);
        }
    }


    /**
     * 刷新access_token（如果需要）
     * @param refreshToken 填写通过access_token获取到的refresh_token参数
     * @param appid 公众号的唯一标识
     * @return
     */
    public static AuthAccessToken refreshAccessToken(String refreshToken,String appid){

        try {

            StrHashMap params = new StrHashMap();
            params.put("refresh_token",Wechatutils.urlEncode(refreshToken));
            params.put("appid",Wechatutils.urlEncode(appid));
            params.put("grant_type","refresh_token");

            HttpResult httpResult = Http.getInstance().get(MpOAuth2Constant.OAUTH2_REFRESH_TOKEN,params);
            if(httpResult.getCode()==200){
                // 检查
                Wechatutils.check(httpResult.getContent());
                //return JSONObject.parseObject(httpResult.getContent(), AuthAccessToken.class);
                return JsonMapper.defaultMapper().fromJson(httpResult.getContent(), AuthAccessToken.class);
            }else {
                throw new ApiException(WechatEnum.ERROR.code(), WechatEnum.ERROR.msg());
            }
        } catch (Exception e) {
            throw new ApiException(e.getMessage(),e);
        }
    }



    /**
     * 拉取用户信息(需scope为 snsapi_userinfo)
     * @param accessToken 网页授权接口调用凭证
     * @param openid 用户的唯一标识
     * @param langType 返回国家地区语言版本，zh_CN 简体，zh_TW 繁体，en 英语
     * @return
     */
    public static UserInfo getUserinfo(String accessToken,String openid,LangType langType)  {
        try {
            StrHashMap params = new StrHashMap();
            params.put("access_token",Wechatutils.urlEncode(accessToken));
            params.put("openid",Wechatutils.urlEncode(openid));
            params.put("lang",Wechatutils.urlEncode(langType.code()));

            HttpResult httpResult = Http.getInstance().get(MpOAuth2Constant.USERINFO,params);
            if(httpResult.getCode()==200){
                // 检查
                Wechatutils.check(httpResult.getContent());
                //return JSONObject.parseObject(httpResult.getContent(), UserInfo.class);
                return JsonMapper.defaultMapper().fromJson(httpResult.getContent(), UserInfo.class);
            }else {
                throw new ApiException(WechatEnum.ERROR.code(), WechatEnum.ERROR.msg());
            }
        } catch (Exception e) {
            throw new ApiException(e.getMessage(),e);
        }
    }

    /**
     * 检验授权凭证（access_token）是否有效
     * @param accessToken 网页授权接口调用凭证
     * @param openid 用户的唯一标识
     * @return
     */
    public static ValidAccessToken validAccessToken(String accessToken, String openid){

        try {
            StrHashMap params = new StrHashMap();
            params.put("access_token",Wechatutils.urlEncode(accessToken));
            params.put("openid",Wechatutils.urlEncode(openid));
            HttpResult httpResult = Http.getInstance().get(MpOAuth2Constant.AUTH,params);
            if(httpResult.getCode()==200){
                // 检查
                Wechatutils.check(httpResult.getContent());
                //return JSONObject.parseObject(httpResult.getContent(), ValidAccess.class);
                return JsonMapper.defaultMapper().fromJson(httpResult.getContent(), ValidAccessToken.class);
            }else {
                throw new ApiException(WechatEnum.ERROR.code(), WechatEnum.ERROR.msg());
            }
        } catch (Exception e) {
            throw new ApiException(e.getMessage(),e);
        }
    }

}
