package cn.dolphin.https.common;

import cn.dolphin.core.io.IOUtil;
import cn.dolphin.core.util.CastUtil;
import cn.dolphin.core.xml.Dom4jUtil;
import cn.dolphin.https.base.ConnectPool;
import cn.dolphin.https.json.HttpJson;
import cn.dolphin.https.model.HttpResult;
import cn.dolphin.https.proxy.CloseableHttpClientBuilder;
import cn.dolphin.https.util.Qs;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import org.apache.http.*;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.*;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicHeader;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.CharArrayBuffer;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.ObjectUtils;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.*;

/**
 * 构建基类
 * @param <T>
 */
@SuppressWarnings("all")
public abstract class Request<T extends Request>{

    protected static Logger logger = LoggerFactory.getLogger(Request.class);

    protected static Header jsonHeader = new BasicHeader(HttpHeaders.CONTENT_TYPE, ContentType.APPLICATION_JSON.toString());
    protected static Header xmlHeader = new BasicHeader(HttpHeaders.CONTENT_TYPE,ContentType.APPLICATION_XML.toString());
    protected static Header htmlHeader = new BasicHeader(HttpHeaders.CONTENT_TYPE,ContentType.TEXT_HTML.toString());
    protected static Header streamHeader = new BasicHeader(HttpHeaders.CONTENT_TYPE,ContentType.APPLICATION_OCTET_STREAM.toString());
    protected static Header formHeader = new BasicHeader(HttpHeaders.CONTENT_TYPE,ContentType.APPLICATION_FORM_URLENCODED.toString());

    //private volatile CloseableHttpClient httpClient;
    protected static CloseableHttpClient httpClient;

    //private ConnectPool connectPool;
    protected static ConnectPool connectPool;

    // 编码格式。发送编码格式统一用UTF-8
    protected static final String ENCODING = "UTF-8";

    // 设置连接超时时间，单位毫秒。
    protected static final int CONNECT_TIMEOUT = 5000;

    // 请求获取数据的超时时间(即响应时间)，单位毫秒。
    protected static final int SOCKET_TIMEOUT = 10000;


    /**
     * 构造一个使用连接池的http请求工具
     *
     * @param connectPool 连接池配置对象,当为null时,使用默认配置
     */
    public Request(ConnectPool connectPool) {
        if (connectPool == null) {
            connectPool = new ConnectPool();
        }
        this.connectPool = connectPool;
    }

    /**
     * 构造一个不使用连接池的http请求工具
     */
    public Request(){}


    /**
     * 构建连接池
     * @return
     */
    public synchronized CloseableHttpClient getHttpClient() {
        if (this.connectPool == null) {
            return HttpClients.createDefault();
        } else {
            if (this.httpClient == null) {
                synchronized (this) {
                    if (this.httpClient == null) {
                        this.httpClient = CloseableHttpClientBuilder.builder(this.connectPool);
                    }
                }
            }
            return this.httpClient;
        }
    }

    /**
     * 关闭连接池
     */
    public void close() {
        try {
            if (this.httpClient != null) {
                this.httpClient.close();
                this.httpClient = null;
            }
        } catch (IOException e) {
            //e.printStackTrace();
            logger.error("http工具关闭连接池异常",e);
        }
    }


    /**
     * 关闭连接池
     */
    public void close2() {
        try {
            if (this.httpClient != null) {
                this.httpClient.close();
            }
        } catch (IOException e) {
            //e.printStackTrace();
            logger.error("http工具关闭连接池异常",e);
        }
    }

    /**
     * 释放资源
     *
     * @param httpResponse
     * @param httpClient
     * @throws IOException
     */
    protected static void release(CloseableHttpResponse httpResponse, CloseableHttpClient httpClient){
        try {
            // 释放资源
            if (httpResponse != null) {
                httpResponse.close();
            }
//            if (httpClient != null) {
//                httpClient.close();
//            }
        } catch (IOException e) {
            //e.printStackTrace();
            logger.error("http工具关闭连接池异常",e);
        }

    }



//    /**
//     * 封装请求头
//     * @param params
//     * @param httpMethod
//     */
//    protected static void packageHeader(Map<String, String> params, HttpRequestBase httpMethod) {
//        // 封装请求头
//        if (params != null) {
//            Set<Map.Entry<String, String>> entrySet = params.entrySet();
//            for (Map.Entry<String, String> entry : entrySet) {
//                // 设置到请求头到HttpRequestBase对象中
//                httpMethod.setHeader(entry.getKey(), entry.getValue());
//            }
//        }
//    }

    /**
     * 设置请求头
     */
    protected static void packageHeader(Map<String, String> headers, HttpRequestBase httpMethod) {
        // 封装请求头
        if (!ObjectUtils.isEmpty(headers)) {
            headers.forEach(httpMethod::setHeader);
        }
    }



    /**
     * 封装请求参数
     *
     * @param params
     * @param httpMethod
     * @throws UnsupportedEncodingException
     */
    protected static void packageParam(Map<String, String> params, HttpEntityEnclosingRequestBase httpMethod) {
        // 封装请求参数
        if (params != null) {
            List<NameValuePair> nvps = new ArrayList<NameValuePair>();
            Set<Map.Entry<String, String>> entrySet = params.entrySet();
            for (Map.Entry<String, String> entry : entrySet) {
                nvps.add(new BasicNameValuePair(CastUtil.castString(entry.getKey()), CastUtil.castString(entry.getValue())));
            }

            UrlEncodedFormEntity formEntity = new UrlEncodedFormEntity(nvps, Charset.forName("UTF-8"));
            //formEntity.setContentType("application/x-www-form-urlencoded; charset=UTF-8");
            formEntity.setContentType(formHeader);
            formEntity.setContentEncoding(ENCODING);

            // 设置到请求的http对象中
            httpMethod.setEntity(formEntity);
        }
    }


    /**
     * 封装请求参数
     *
     * @param params
     * @param httpMethod
     * @throws UnsupportedEncodingException
     */
    protected static void packageJsonParam(Map<String, String> params, HttpEntityEnclosingRequestBase httpMethod) {
        // 封装请求参数
        if (params != null) {

            StringEntity stringEntity = new StringEntity(HttpJson.toJsonString(params), "UTF-8");//解决中文乱码问题
            stringEntity.setContentType(jsonHeader);
            stringEntity.setContentEncoding(ENCODING);
            // 设置到请求的http对象中
            httpMethod.setEntity(stringEntity);
        }
    }


    /**
     * 封装请求参数
     *
     * @param params
     * @param httpMethod
     * @throws UnsupportedEncodingException
     */
    protected static void packageJsonParam(String json, HttpEntityEnclosingRequestBase httpMethod) {
        // 封装请求参数
        if (json != null) {
            StringEntity stringEntity = new StringEntity(json, "UTF-8");//解决中文乱码问题
            stringEntity.setContentType(jsonHeader);
            stringEntity.setContentEncoding(ENCODING);
            // 设置到请求的http对象中
            httpMethod.setEntity(stringEntity);
        }
    }


    /**
     * 获得响应结果
     *
     * @param httpClient
     * @param httpMethod
     * @return
     * @throws Exception
     */
    protected static HttpResult getHttpClientResult(CloseableHttpClient httpClient, HttpRequestBase httpMethod) throws Exception {
        // 创建httpResponse对象
        CloseableHttpResponse httpResponse = null;
        try {
            // 执行请求
            httpResponse = httpClient.execute(httpMethod);

            // 获取返回结果
            if (httpResponse != null && httpResponse.getStatusLine() != null) {
                String content = "";
                if (httpResponse.getEntity() != null) {
//                    content = EntityUtils.toString(httpResponse.getEntity(), ENCODING);
//                    //解决Unicode编码转换Utf8,解决微null丢字段
//                    content = JSON.toJSONString(JSON.parseObject(content), SerializerFeature.WriteMapNullValue);
                    content = entryToString(httpResponse.getEntity());
                    //关闭
                    EntityUtils.consume(httpResponse.getEntity());
                }
                return new HttpResult(httpResponse.getStatusLine().getStatusCode(),content);
            }else {
                return new HttpResult(HttpStatus.SC_INTERNAL_SERVER_ERROR);
            }
        }catch (Exception e){
            return new HttpResult(HttpStatus.SC_INTERNAL_SERVER_ERROR);
        }finally {
            // 释放资源
            release(httpResponse, httpClient);
        }
    }

    /**
     * 获得响应结果
     *
     * @param httpClient
     * @param httpMethod
     * @return
     * @throws Exception
     */
    protected static HttpResult getHttpClientResultXml(CloseableHttpClient httpClient, HttpRequestBase httpMethod) throws Exception {
        // 创建httpResponse对象
        CloseableHttpResponse httpResponse = null;
        try {
            // 执行请求
            httpResponse = httpClient.execute(httpMethod);

            // 获取返回结果
            if (httpResponse != null && httpResponse.getStatusLine() != null) {
                String content = "";
                if (httpResponse.getEntity() != null) {
                    content = EntityUtils.toString(httpResponse.getEntity(), ENCODING);
                    Map result = Dom4jUtil.toMap(content,ENCODING);
                    content = HttpJson.toJsonString(result);
                    //关闭
                    EntityUtils.consume(httpResponse.getEntity());
                }
                return new HttpResult(httpResponse.getStatusLine().getStatusCode(),content);
            }else {
                return new HttpResult(HttpStatus.SC_INTERNAL_SERVER_ERROR);
            }
        }catch (Exception e){
            return new HttpResult(HttpStatus.SC_INTERNAL_SERVER_ERROR);
        }finally {
            // 释放资源
            release(httpResponse, httpClient);
        }
    }




    /**
     * head请求
     *
     * @param url     请求地址
     * @param headers 请求头
     */
    public CloseableHttpResponse headRequest(String url, Header... headers) {
        if (logger.isTraceEnabled()) {
            logger.trace("本次请求地址:" + url);
        }
        CloseableHttpClient httpclient = this.getHttpClient();
        HttpHead httpHead = new HttpHead(url);
        if (headers != null && headers.length != 0) {
            httpHead.setHeaders(headers);
        }
        CloseableHttpResponse response = null;
        try {
            response = httpclient.execute(httpHead);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return response;
    }

    /**
     * head请求
     *
     * @param url 请求地址
     */
    public CloseableHttpResponse headRequest(String url) {
        return this.headRequest(url, (Header[]) null);
    }

    /**
     * head请求
     *
     * @param url       请求地址
     * @param urlParams url参数
     * @param headers   请求头
     */
    public CloseableHttpResponse headRequest(String url, Object urlParams, Header... headers) {
        if (urlParams != null) {
            String urlParam = Qs.stringify(urlParams);
            if (url.contains("?")) {
                url += "&" + urlParam;
            } else {
                url += "?" + urlParam;
            }
        }
        return this.headRequest(url, headers);
    }

    /**
     * head请求
     *
     * @param url       请求地址
     * @param urlParams url参数
     */
    public CloseableHttpResponse headRequest(String url, Object urlParams) {
        return this.headRequest(url, urlParams, (Header[]) null);
    }

    /**
     * options请求
     *
     * @param url     请求地址
     * @param headers 请求头
     */
    public CloseableHttpResponse optionsRequest(String url, Header... headers) {
        if (logger.isTraceEnabled()) {
            logger.trace("本次请求地址:" + url);
        }
        CloseableHttpClient httpclient = this.getHttpClient();
        HttpOptions httpOptions = new HttpOptions(url);
        if (headers != null && headers.length != 0) {
            httpOptions.setHeaders(headers);
        }
        CloseableHttpResponse response = null;
        try {
            response = httpclient.execute(httpOptions);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return response;
    }

    /**
     * options请求
     *
     * @param url 请求地址
     */
    public CloseableHttpResponse optionsRequest(String url) {
        return this.optionsRequest(url, (Header[]) null);
    }

    /**
     * options请求
     *
     * @param url       请求地址
     * @param urlParams url参数
     * @param headers   请求头
     */
    public CloseableHttpResponse optionsRequest(String url, Object urlParams, Header... headers) {
        if (urlParams != null) {
            String urlParam = Qs.stringify(urlParams);
            if (url.contains("?")) {
                url += "&" + urlParam;
            } else {
                url += "?" + urlParam;
            }
        }
        return this.optionsRequest(url, headers);
    }

    /**
     * options请求
     *
     * @param url       请求地址
     * @param urlParams url参数
     */
    public CloseableHttpResponse optionsRequest(String url, Object urlParams) {
        return this.optionsRequest(url, urlParams, (Header[]) null);
    }

    /**
     * 把返回数据转换为指定对象
     */
    public <T> T responseDataConversion(HttpResponse response, Class<T> type) {
        if (response == null) {
            return null;
        }
        HttpEntity entity = response.getEntity();
        String entityString = null;
        try {
            //entityString = EntityUtils.toString(entity);
            entityString = entryToString(entity);
        } catch (ParseException | IOException e1) {
            throw new RuntimeException(e1);
        }
        T result = null;
        result = HttpJson.conversion(entityString, type);
        try {
            EntityUtils.consume(entity);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return result;
    }


    /**
     * 处理json和xml合适数据
     * @param entry
     * @return
     * @throws IOException
     */
    protected static String entryToString(HttpEntity entry) throws IOException {
        // TODO Auto-generated method stub
        String result = null;
        if(entry != null) {
            long length = entry.getContentLength();
            if(length != -1  && length <2048) {
                result = EntityUtils.toString(entry, ENCODING);
                //解决Unicode编码转换Utf8,解决微null丢字段
                result = JSON.toJSONString(JSON.parseObject(result), SerializerFeature.WriteMapNullValue);
            }else {
                InputStreamReader reader = new InputStreamReader(entry.getContent(),"UTF-8");
                CharArrayBuffer buffer = new CharArrayBuffer(2048);
                char [] temp = new char[1024];
                int l;
                while((l = reader.read(temp))!= -1) {
                    buffer.append(temp,0,l);
                }
                IOUtil.close(reader);
                result = buffer.toString();
            }
        }
        return result;
    }


    /**
     * map转成string
     * @param url 地址
     * @param params 参数
     * @param first 拼接符号，？
     * @return
     */
    protected static String mapToString(String url, Map<String, String> params, String first) {
        StringBuilder sb;
        if (url != null) {
            sb = new StringBuilder(url);
        } else {
            sb = new StringBuilder();
        }
        if (params != null) {
            boolean isFirst = true;
            Iterator<String> iterator = params.keySet().iterator();
            while (iterator.hasNext()) {
                String key = iterator.next();
                if (isFirst) {
                    if (first != null) {
                        sb.append(first);
                    }
                    isFirst = false;
                } else {
                    sb.append("&");
                }
                sb.append(key);
                sb.append("=");
                sb.append(params.get(key));
            }
        }
        return sb.toString();
    }


}
