package cn.dolphin.email.mail;

import cn.dolphin.core.date.TimeUtil;
import cn.dolphin.core.util.CastUtil;
import cn.dolphin.core.util.EmptyUtil;
import cn.dolphin.core.util.ListUtil;
import cn.dolphin.core.util.StrUtil;
import cn.dolphin.email.entity.AttachmentFile;
import cn.dolphin.email.entity.EmailAuthenticator;
import cn.dolphin.email.entity.EmailResult;
import cn.dolphin.core.freemarker.FreemarkerUtil;
import cn.dolphin.email.util.MailUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.activation.DataHandler;
import javax.activation.FileDataSource;
import javax.mail.*;
import javax.mail.Message.RecipientType;
import javax.mail.internet.*;
import java.io.File;
import java.util.*;

/**
 * 邮件基础封装类
 */
public class EmailMessage {

    private static Logger logger = LoggerFactory.getLogger(EmailMessage.class);

    /**
     * 连接到邮箱服务器session
     */
    private static Session session;
    /**
     * 邮件服务器登录验证
     * 生命周期仅存于调用者的内存中
     */
    private static EmailAuthenticator authenticator;

    /**
     * 默认编码
     */
    private static String defaultEncoding = "UTF-8";

    /**
     * 设置抄送人
     */
    private List<InternetAddress> cc = new ArrayList<InternetAddress>();
    /**
     * 设置密送人
     */
    private List<InternetAddress> bcc = new ArrayList<InternetAddress>();

    /**
     * 邮件附件
     */
    private List<AttachmentFile> attachments = new ArrayList<AttachmentFile>();


    /**
     * 当前对象实例
     */
    private static volatile EmailMessage getInstance = null;

    /**
     * 静态工厂方法 获取当前对象实例 多线程安全单例模式(使用双重同步锁)
     */

    public static EmailMessage getInstance() {
        if (getInstance == null) {
            synchronized (EmailMessage.class) {
                if (getInstance == null)
                    getInstance = new EmailMessage();
            }
        }
        return getInstance;
    }

    private EmailMessage(){
//        super();
    }


    /**
     * 默认配置
     * @return
     */
    private static Properties defaultConfig() {
        //1.创建连接对象，连接到邮箱服务器
        Properties props=new Properties();
        props.put("mail.transport.protocol", "smtp"); //define protocolo de envio como SMTP
        props.put("mail.smtp.auth", "true");// 打开认证
        props.put("mail.smtp.ssl.enable", "true");
        props.put("mail.debug", "false");
        props.put("mail.smtp.timeout", "10000");
        //props.put("mail.smtp.port", "465");
        props.put("mail.smtp.port", 25); //port
        return props;
    }

    /**
     * 企业QQ邮箱
     * smtp entnterprise qq
     * @return properties
     */
    public static Properties SMTP_ENT_QQ() {
        Properties props = defaultConfig();
        props.put("mail.smtp.host", "smtp.exmail.qq.com");//设置服务器主机名
        return props;
    }

    /**
     * QQ邮箱
     * smtp qq
     * @return properties
     */
    public static Properties SMTP_QQ()  {
        Properties props = defaultConfig();
        props.put("mail.smtp.host", "smtp.qq.com");
        return props;
    }

    /**
     * 163邮箱
     * smtp 163
     * @return properties
     */
    public static Properties SMTP_163() {
        Properties props = defaultConfig();
        props.put("mail.smtp.host", "smtp.163.com");
        return props;
    }


//    /**
//     * 创建session，连接到邮箱服务器
//     * config username and password
//     *
//     * @param props    email property config
//     * @param username email auth username
//     * @param password email auth password
//     */
//    public static void config(Properties props, final String username, final String password) {
//        session = Session.getDefaultInstance(props, new Authenticator() {
//            public PasswordAuthentication getPasswordAuthentication() {
//                return new PasswordAuthentication(username, password); // 发件人邮箱账号、授权码
//            }
//        });
//        //session.setDebug(false);
//    }



    /**
     * 创建session，连接到邮箱服务器
     * config username and password
     *
     * @param props    email property config
     * @param username email auth username
     * @param password email auth password
     */
    public static void config(Properties props, final String username, final String password) {
        config(props,username,password, null,null);
    }

    /**
     * 创建session，连接到邮箱服务器
     * 创建freemarker配置-项目路径
     * config username and password
     *
     * @param props    email property config
     * @param username email auth username
     * @param password email auth password
     * @param templatePath email ftl
     */
    public static void config(Properties props, final String username, final String password,final String templatePath) {
        config(props,username,password, templatePath,2);
    }
    /**
     * 创建session，连接到邮箱服务器
     * 创建freemarker配置
     * config username and password
     *
     * @param props    email property config
     * @param username email auth username
     * @param password email auth password
     * @param templatePath email ftl
     * @param type 模板文件夹路径类型，1：文件系统路径；2：项目路径
     */
    public static void config(Properties props, final String username, final String password,final String templatePath,Integer type) {
        if(StrUtil.isNotBlank(templatePath)){
            FreemarkerUtil.initFreeMarker(templatePath,type);
            // 验证
            authenticator = new EmailAuthenticator(username, password,props.getProperty("mail.smtp.host"),templatePath);
        }else {
            // 验证
            authenticator = new EmailAuthenticator(username, password,props.getProperty("mail.smtp.host"));
        }
        session = Session.getDefaultInstance(props,authenticator);// 发件人邮箱账号、授权码
        //session.setDebug(false);

    }

    /**
     * 发送简单文本
     * @param subject 邮件主题名
     * @param content 内容
     * @param from 发件人邮箱地址
     * @param receivers 收件人邮箱地址
     * @return
     */
    public static synchronized EmailResult sendTextEmail(String subject,String content,String from,String[] receivers){
        return sendTextEmail(subject,content,from,null,receivers);
    }

    /**
     * 发送简单文本
     * @param subject 邮件主题名
     * @param content 内容
     * @param from 发件人邮箱地址
     * @param fromName 发件人姓名
     * @param receivers 收件人邮箱地址
     * @return
     */
    public static synchronized EmailResult sendTextEmail(String subject,String content,String from,String fromName,String[] receivers){
        return sendTextEmail(subject,content,false,from,fromName,receivers);
    }

    /**
     * 发送简单文本
     * @param subject 邮件主题名
     * @param content 内容
     * @param isDebug 是否开启debug模式：开启true，关闭false,开启后输出邮件日志和错误异常打印
     * @param from 发件人邮箱地址
     * @param fromName 发件人姓名
     * @param receivers 收件人邮箱地址
     * @return
     */
    public static synchronized EmailResult sendTextEmail(String subject,String content,boolean isDebug,String from,String fromName,String[] receivers){
        EmailResult result = new EmailResult();
        // 发送邮件
        Transport transport = null;
        try {
            session.setDebug(isDebug);
            //创建MimeMessage实例对象
            Message message = new MimeMessage(session);
            // 设置发件人
            message.setFrom(new InternetAddress(from));

            if(StrUtil.isBlank(fromName)){
                message.setFrom(new InternetAddress(from));//设置发件人
            }else {
                message.setFrom(new InternetAddress(from, fromName, defaultEncoding));
            }
            // 设置邮件主题
            message.setSubject(subject);
            // 设置收件人
//            message.setRecipient(RecipientType.TO, new InternetAddress(addressee));
            // 设置收件人
            for (InternetAddress to : MailUtil.getReceivers(receivers)) {
                message.addRecipient(RecipientType.TO, to);
            }
            // 抄送
            if(ListUtil.isNotEmpty(EmailMessage.getInstance().cc)){
                message.setRecipients(RecipientType.CC, EmailMessage.getInstance().cc.stream().toArray(Address[]::new)); //设置收件人,支持多个
            }

            // 密送 (不会在邮件收件人名单中显示出来)
            if(ListUtil.isNotEmpty(EmailMessage.getInstance().bcc)){
                message.setRecipients(RecipientType.BCC, EmailMessage.getInstance().bcc.stream().toArray(Address[]::new)); //设置收件人,支持多个
            }

            // 设置邮件的文本内容
            // 向multipart对象中添加各个部分内容，包含文本和附件
            Multipart multipart = new MimeMultipart();
            if (EmptyUtil.isNotEmpty(content)) {
                BodyPart contentPart = new MimeBodyPart();
                contentPart.setText(CastUtil.castString(content));
                multipart.addBodyPart(contentPart);
            }

            // 附件操作
            if (EmailMessage.getInstance().attachments != null && EmailMessage.getInstance().attachments.size() > 0) {
                for (AttachmentFile file : EmailMessage.getInstance().attachments) {
                    MimeBodyPart filesPart = new MimeBodyPart();
                    // 得到附件本身并至入BodyPart
                    filesPart.setDataHandler(file.getFile());
                    // 得到文件名同样至入BodyPart
                    filesPart.setFileName(file.getFileName());
                    multipart.addBodyPart(filesPart);
                }
            }

            // 设置发送时间
            message.setSentDate(new Date());
            // 设置纯文本内容为邮件正文
            //message.setContent(content, "text/html;charset=utf-8");
            // Multipart加入到信件
            message.setContent(multipart);
            // 保存并生成最终的邮件内容
            message.saveChanges();
            // 获得Transport实例对象
            transport = session.getTransport();
            // 打开连接
            transport.connect(authenticator.getHost(),authenticator.getUsername(), authenticator.getPassword());
            // 将message对象传递给transport对象，将邮件发送出去
            transport.sendMessage(message, message.getAllRecipients());
            // 发送
            // Transport.send(message);
            // 关闭连接
            transport.close();
            result.setStatus("success");
            result.setMsg("邮件发送成功！");
        } catch (AuthenticationFailedException e) {
            result.setStatus("failed");
            result.setMsg("邮件发送失败！错误原因:身份验证错误！");
            if(isDebug){
                e.printStackTrace();
            }
        } catch (MessagingException e) {
            result.setStatus("failed");
            result.setMsg("邮件发送失败！错误原因：" + e.getMessage());
            if(isDebug){
                Exception ex = null;
                if ((ex = e.getNextException()) != null) {
                    ex.printStackTrace();
                }
            }

        } catch (Exception e) {
            result.setStatus("failed");
            result.setMsg("邮件发送失败！错误原因：未知");
            if(isDebug){
                e.printStackTrace();
            }
        } finally {
            if(null !=transport){
                try {
                    transport.close();
                }catch (MessagingException e){
                    if(isDebug){
                        e.printStackTrace();
                    }
                }
            }
        }
        return result;
    }

    /**
     * 发送HTML文本
     * @param subject 邮件主题名
     * @param templateName 模板文件名
     * @param params 数据Map
     * @param from 发件人邮箱地址
     * @param receivers 收件人邮箱地址
     * @return
     */
    public static synchronized EmailResult sendHtmlEmail(String subject, String templateName, Map<String, Object> params,String from,String[] receivers){
        return sendHtmlEmail(subject,templateName,params,false,from,null,receivers);
    }

    /**
     * 发送HTML文本
     * @param subject 邮件主题名
     * @param templateName 模板文件名
     * @param params 数据Map
     * @param from 发件人邮箱地址
     * @param fromName 发件人姓名
     * @param receivers 收件人邮箱地址
     * @return
     */
    public static synchronized EmailResult sendHtmlEmail(String subject, String templateName, Map<String, Object> params,String from, String fromName, String[] receivers){
        return sendHtmlEmail(subject,templateName,params,false,from,fromName,receivers);
    }

    /**
     * 发送HTML文本
     * @param subject 邮件主题名
     * @param templateName 模板文件名
     * @param params 数据Map
     * @param isDebug 是否开启debug模式：开启true，关闭false,开启后输出邮件日志和错误异常打印
     * @param from 发件人邮箱地址
     * @param fromName 发件人姓名
     * @param receivers 收件人邮箱地址
     * @return
     */
    public static synchronized EmailResult sendHtmlEmail(String subject, String templateName, Map<String, Object> params, boolean isDebug, String from, String fromName, String[] receivers){
        EmailResult result = new EmailResult();
        // 发送邮件
        Transport transport = null;
        try {
            session.setDebug(isDebug);
            //创建MimeMessage实例对象
            Message message = new MimeMessage(session);
            // 设置发件人
            message.setFrom(new InternetAddress(from));

            if(StrUtil.isBlank(fromName)){
                message.setFrom(new InternetAddress(from));//设置发件人
            }else {
                message.setFrom(new InternetAddress(from, fromName, defaultEncoding));
            }
            // 设置邮件主题
            message.setSubject(subject);
            // 设置收件人
//            message.setRecipient(RecipientType.TO, new InternetAddress(addressee));
            // 设置收件人
            for (InternetAddress to : MailUtil.getReceivers(receivers)) {
                message.addRecipient(RecipientType.TO, to);
            }

            // 抄送
            if(ListUtil.isNotEmpty(EmailMessage.getInstance().cc)){
                message.setRecipients(RecipientType.CC, EmailMessage.getInstance().cc.stream().toArray(Address[]::new)); //设置收件人,支持多个
            }

            // 密送 (不会在邮件收件人名单中显示出来)
            if(ListUtil.isNotEmpty(EmailMessage.getInstance().bcc)){
                message.setRecipients(RecipientType.BCC, EmailMessage.getInstance().bcc.stream().toArray(Address[]::new)); //设置收件人,支持多个
            }

            // 设置发送时间
            message.setSentDate(new Date());

            // 向multipart对象中添加各个部分内容
            Multipart multipart = new MimeMultipart();
            //给消息对象设置内容
            BodyPart bodyPart=new MimeBodyPart();//新建一个存放信件内容的BodyPart对象
            bodyPart.setContent(FreemarkerUtil.crateHTML(templateName,params),"text/html;charset=UTF-8");//给BodyPart对象设置内容和格式/编码方式
            multipart.addBodyPart(bodyPart);//将BodyPart加入到MimeMultipart对象中(可以加入多个BodyPart)

            // 附件操作
            if (EmailMessage.getInstance().attachments != null && EmailMessage.getInstance().attachments.size() > 0) {
                for (AttachmentFile file : EmailMessage.getInstance().attachments) {
                    MimeBodyPart filesPart = new MimeBodyPart();
                    // 得到附件本身并至入BodyPart
                    filesPart.setDataHandler(file.getFile());
                    // 得到文件名同样至入BodyPart
                    filesPart.setFileName(file.getFileName());
                    multipart.addBodyPart(filesPart);
                }
            }

            // 设置纯文本内容为邮件正文
            message.setContent(multipart);
            // 保存并生成最终的邮件内容
            message.saveChanges();
            // 获得Transport实例对象
            transport = session.getTransport();
            // 打开连接
            transport.connect(authenticator.getHost(),authenticator.getUsername(), authenticator.getPassword());
            // 将message对象传递给transport对象，将邮件发送出去
            transport.sendMessage(message, message.getAllRecipients());
            // 发送
            // Transport.send(message);
            // 关闭连接
            transport.close();
            result.setStatus("success");
            result.setMsg("邮件发送成功！");
        } catch (AuthenticationFailedException e) {
            result.setStatus("failed");
            result.setMsg("邮件发送失败！错误原因:身份验证错误！");
            if(isDebug){
                e.printStackTrace();
            }
        } catch (MessagingException e) {
            result.setStatus("failed");
            result.setMsg("邮件发送失败！错误原因：" + e.getMessage());
            if(isDebug){
                Exception ex = null;
                if ((ex = e.getNextException()) != null) {
                    ex.printStackTrace();
                }
            }

        } catch (Exception e) {
            result.setStatus("failed");
            result.setMsg("邮件发送失败！错误原因：未知");
            if(isDebug){
                e.printStackTrace();
            }
        } finally {
            if(null !=transport){
                try {
                    transport.close();
                }catch (MessagingException e){
                    if(isDebug){
                        e.printStackTrace();
                    }
                }
            }
        }
        return result;

    }

    /**
     * 设置抄送人
     * @param cc
     * @throws AddressException
     */
    public void sendCC(List<String> cc){
        try {
            if(cc!=null&& cc.size()>0){
                List<InternetAddress> ccList=new ArrayList<InternetAddress>();
                for(int i=0;i<cc.size();i++){
                    String ccTmp=cc.get(i);
                    ccList.add(new InternetAddress(ccTmp));
                }
                this.cc=ccList;
            }
        }catch (Exception e){
            logger.error("发送邮件设置抄送人异常");
            //e.printStackTrace();
        }
    }

    /**
     * 设置密送人
     * @param bcc
     * @throws AddressException
     */
    public void sendBCC(List<String> bcc){
        try {
            if(bcc!=null&& bcc.size()>0){
                List<InternetAddress> bccList=new ArrayList<InternetAddress>();
                for(int i=0;i<bcc.size();i++){
                    String bccTmp=bcc.get(i);
                    bccList.add(new InternetAddress(bccTmp));
                }
                this.bcc=bccList;
            }
        }catch (Exception e){
            logger.error("发送邮件设置密送人异常");
            //e.printStackTrace();
        }
    }

    /**
     * 设置附件
     * @param filepath
     */
    public void sendFilesPath(String filepath){
        try {
            // 附件操作
            if(StrUtil.isNotBlank(filepath)){
                // 得到数据源
                FileDataSource fds = new FileDataSource(filepath);
                // 处理附件名称中文（附带文件路径）乱码问题
                attachments.add(new AttachmentFile(MimeUtility.encodeText(fds.getName()),new DataHandler(fds)));
            }
        }catch (Exception e){
            logger.error("发送邮件设置附件异常");
            //e.printStackTrace();
        }
    }

    /**
     * 设置附件
     * @param filepath
     */
    public void sendFilesPath(List<String> filepath){
        try {
            // 附件操作
            if (filepath != null && filepath.size() > 0) {
                for (String filename : filepath) {
                    // 得到数据源
                    FileDataSource fds = new FileDataSource(filename);
                    // 处理附件名称中文（附带文件路径）乱码问题
                    attachments.add(new AttachmentFile(MimeUtility.encodeText(fds.getName()),new DataHandler(fds)));
                }
            }
        }catch (Exception e){
            logger.error("发送邮件设置附件异常");
            //e.printStackTrace();
        }
    }

    /**
     * 设置附件
     * @param filepath
     */
    public void sendFiles(File filepath){
        try {
            // 附件操作
            // 得到数据源
            FileDataSource fds = new FileDataSource(filepath);
            // 处理附件名称中文（附带文件路径）乱码问题
            attachments.add(new AttachmentFile(MimeUtility.encodeText(fds.getName()),new DataHandler(fds)));
        }catch (Exception e){
            logger.error("发送邮件设置附件异常");
            //e.printStackTrace();
        }
    }

    /**
     * 设置附件
     * @param filepath
     */
    public void sendFiles(List<File> filepath){
        try {
            // 附件操作
            if (filepath != null && filepath.size() > 0) {
                for (File filename : filepath) {
                    // 得到数据源
                    FileDataSource fds = new FileDataSource(filename);
                    // 处理附件名称中文（附带文件路径）乱码问题
                    attachments.add(new AttachmentFile(MimeUtility.encodeText(fds.getName()),new DataHandler(fds)));
                }
            }
        }catch (Exception e){
            logger.error("发送邮件设置附件异常");
            //e.printStackTrace();
        }
    }



    public static String createContent(String username, Date startTime, String validtime){
        String content="<style type='text/css'>"
                +"*{font:16px 'microsoft yahie';color:#333;}"
                +"#mailcontent{border:1px solid #ccc;width:900px; height:auto;margin:0 auto;background:#fff;padding:20px 50px 35px 50px;}"
                +"#mailcontent a{text-align:center;display:block;width:900px;height:auto;padding:20px 0;color:#000;}"
                +"#mailcontent .resetpwd{display:block;width:100px;height:20px;padding:5px;font:16px 'microsoft yahei';background:#f2f2f2;border:1px solid #bbb;"
                +"text-align:center;margin-top:24px;border-radius:2px;color:#666;text-decoration:none;margin-left:30px;}"
                +"#mailcontent .resetpwd:hover{text-decoration:none;color:#fff;box-shadow:0 0 5px rgba(20,20,20,0.3);background:#19b4ea;}"
                +"#mailcontent .from{text-align:right;margin-top:40px;}"
                +"</style>"
                +"<div id='mailcontent'>"
                +"<a href='" +  "测试" + "'><img alt='logo' src='" +  "测试" + "/resources/images/biglogo.png'/></a>"
                +"<p>亲爱的" + username + "用户：</p>"
                +"<p>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;这是一封来自<a style='display:inline' href='" + "测试"+ "'>JAVA技术论坛</a>的邮件，"
                +"本邮件用于找回密码，请先确定这是否是您本人操作，如果不是，请忽略此邮件，如果是您本人操作，那么请注意以下事项："
                +"</p>"
                +"<ol>"
                +"<li>本邮件有效期为" + validtime + "分钟，请在有效时间内点击下方按钮进行密码重置</li>"
                +"<li>如果按钮无法点击，请复制下方链接粘贴至浏览器地址栏</li>"
                +"<li>此邮件为系统自动发送，请勿回复</li>"
                +"</ol>"
                +"<a class='resetpwd' href='" +  "测试" + "'>点击重置密码</a>"
                +"<p>密码重置链接：" +  "测试" + "</p>"
                +"<p>如果有任何疑问，请联系网站管理员，邮箱地址：<a style='display:inline' href='mailto:bbshelp@163.com'>bbshelp@163.com</a></p>"
                +"<div class='from'>"
                +"<span>来自BBS</span>"
                +"<p>" + TimeUtil.format(startTime, TimeUtil.YYYYMMDDHH) + "</p>"
                +"</div>"
                +"</div>";
        return content;
    }

    public static void main(String[] args) {
        EmailMessage.config(EmailMessage.SMTP_ENT_QQ(),"ncai@netconcepts.cn","rP6S45YWrk5A4kff");
        List<String> cc = new ArrayList<String>();
        cc.add("wangzhiqiang@netconcepts.cn");
        EmailMessage.getInstance().sendCC(cc);
        String content=createContent("先生",new Date(),"1");
        List<String> file = new ArrayList<String>();
        file.add("/Users/cloud/Desktop/新旧表关系.txt");
        EmailMessage.getInstance().sendFilesPath(file);
        System.out.println(EmailMessage.sendTextEmail("NCAI邮件测试","测试","ncai@netconcepts.cn","测试工程师","yangheqing@netconcepts.cn".split(",")));

        EmailMessage.config(EmailMessage.SMTP_ENT_QQ(),"ncai@netconcepts.cn","rP6S45YWrk5A4kff","/template");
        Map<String,Object> params = new HashMap<String, Object>();
        params.put("userName","测试");
        params.put("age",12);
        // 发送附件
        List<String> file1 = new ArrayList<String>();
        file.add("/Users/cloud/Desktop/新旧表关系.txt");
        EmailMessage.getInstance().sendFilesPath(file1);
        //FreemarkerUtil.printConsole("user.ftl",params);
        System.out.println(EmailMessage.sendHtmlEmail("NCAI邮件测试","user.ftl",params,"ncai@netconcepts.cn","测试工程师","yangheqing@netconcepts.cn".split(",")));
    }




}
