package cn.dolphin.core.web.util;

import cn.dolphin.core.consts.CharsetConstant;
import cn.dolphin.core.consts.StringConstant;
import cn.dolphin.core.exception.ParamRuntimeException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.UnsupportedEncodingException;
import java.net.*;
import java.nio.charset.Charset;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;


/**
 * URL处理工具
 *
 */
@SuppressWarnings("all")
@Slf4j
public class UrlUtil extends org.springframework.web.util.UriUtils {
    /**
     * url 编码，同js decodeURIComponent
     *
     * @param source  url
     * @param charset 字符集
     * @return 编码后的url
     */
    public static String encodeURL(String source, Charset charset) {
        return UrlUtil.encode(source, charset.name());
    }

    /**
     * url 解码
     *
     * @param source  url
     * @param charset 字符集
     * @return 解码url
     */
    public static String decodeURL(String source, Charset charset) {
        return UrlUtil.decode(source, charset.name());
    }

    /**
     * 获取url路径
     *
     * @param uriStr 路径
     * @return url路径
     */
    public static String getPath(String uriStr) {
        URI uri;

        try {
            uri = new URI(uriStr);
        } catch (URISyntaxException var3) {
            throw new RuntimeException(var3);
        }

        return uri.getPath();
    }


    /**
     *
     * URL编码
     *
     * @param url
     * @return
     *
     */
    public static final String encode(String url){
        if (StringUtils.isBlank(url)) {
            return StringConstant.EMPTY;
        }
        try {
            url = URLEncoder.encode(url, CharsetConstant.UTF8);
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return url;
    }
    /**
     *
     * URL解码
     *
     * @param url
     * @return
     *
     */
    public static final String decode(String url){
        if (StringUtils.isBlank(url)) {
            return StringConstant.EMPTY;
        }
        try {
            url = URLDecoder.decode(url,  CharsetConstant.UTF8);
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return url;
    }

    /**
     * 将URL查询参数规范的字符串转换为Map
     *  转换时会进行URL解码参数，所以如果参数中可能包含&时应使用URL编码参数
     * @param urlParams 如 k1=v1&k2=v2
     * @return LinkedHashMap，保证字符串的键值对顺序
     * @throw  参数非URL查询参数规范的字符串时抛出IllegalArgumentException
     */
    public static final Map<String,String> parseUrlParamsToMap(String urlParams){
        Map<String,String> map = new LinkedHashMap<String,String>();
        //参数为空时返回空Map
        if (StringUtils.isBlank(urlParams)) {
            return map;
        }
        //URL解码
        urlParams = UrlUtil.decode(urlParams);
        //切割
        String[] split = urlParams.split(StringConstant.PARAM_SEPARATOR);
        //Put
        for (String s : split) {
            String[] split2 = s.split(StringConstant.EQUAL);
            if (split2.length == 2) {
                map.put(split2[0], split2[1]);
            }else{
                map.put(split2[0], StringConstant.EMPTY);
            }
        }
        return map;
    }

    /**
     * 将Map转换为URL查询参数规范的字符串
     *
     * @param  map
     * @return urlParams 如 k1=v1&k2=v2
     */
    public static final String parseMapToUrlParams(Map<String,String> map){
        //Map为空时返回""
        if (map == null ) {
            return StringConstant.EMPTY;
        }
        int size =  map.size();
        if (size == 0 ) {
            return StringConstant.EMPTY;
        }
        //遍历拼接
        int i = 0;
        StringBuilder sd = new StringBuilder();
        for (Map.Entry<String, String> entry : map.entrySet()) {
            sd.append(entry.getKey()+StringConstant.EQUAL+entry.getValue());
            if (i != size -1) {
                sd.append(StringConstant.PARAM_SEPARATOR);
            }
        }
        return sd.toString();
    }

    /**
     * 将Url地址转换为URL对象
     *
     * @param url
     *
     * @return URL，为空或格式错误抛异常
     */
    public static final URL getUrl(String url) {
        if (StringUtils.isBlank(url)) {
            throw ParamRuntimeException.NULL;
        }
        try {
            return new URL(url);
        } catch (MalformedURLException e) {
            throw ParamRuntimeException.ERROR;
        }
    }

    /**
     * 获取URL地址的协议部分
     *
     * @param url
     * @return URL的协议，如 http , https
     */
    public static final String getUrlProtocol(String url) {
        URL u = getUrl(url);
        return u == null ? StringConstant.EMPTY : u.getProtocol();
    }

    /**
     * 获取URL地址的域名(主机名)部分
     *
     * @param url
     * @return URL的主机名，如http://localhost:8320/login?id=10获取localhost， http://www.demoncat.cn/login?id=10获取www.demoncat.cn
     */
    public static final String getUrlHost(String url) {
        URL u = getUrl(url);
        return u == null ? StringConstant.EMPTY : u.getHost();
    }

    /**
     * 获取URL地址的Domain(一级域名)
     * @param url
     * @return URL的一级域名，如 http://www.demoncat.cn和http://user.demoncat.cn获取demoncat.cn
     */
    public static final String getDomain(String url){
        url = getUrlHost(url);
        if(url.equals(StringConstant.EMPTY)) return StringConstant.EMPTY;
        String[] split = url.split("\\.");
        int len = split.length;
        //localhost或一级域名
        if ( len == 1 || len == 2) {
            return url;
        }else {
            return split[len-2]+StringConstant.POINT+split[len-1];
        }
    }

    /**
     * 获取URL地址的路径部分和参数部分
     *
     * @param url
     * @return URL的路径和参数，如 /query/index.do?name=111&value=222
     */
    public static final String getUrlUP(String url) {
        URL u = getUrl(url);
        return u == null ? StringConstant.EMPTY : u.getFile();
    }

    /**
     * 获取URL地址的URI(路径)部分
     *
     * @param url
     * @return URL的URI，如 /query/index.do
     */
    public static final String getUrlURI(String url) {
        URL u = getUrl(url);
        return u == null ? StringConstant.EMPTY : u.getPath();
    }

    /**
     * 获取URL地址的Param(查询参数)部分
     *
     * @param url
     * @return URL的Param，如 name=111&value=222
     */
    public static final String getUrlParam(String url) {
        URL u = getUrl(url);
        return u == null ? StringConstant.EMPTY : u.getQuery();
    }

    /**
     * 设置参数
     * @param path 需要赋值的path
     * @param map 参数map
     * @param charset 编码格式 默认编码为utf-8(取消默认)
     * @return 已经赋值好的url 只需要访问即可
     */
    public static String setParmas(String path,Map<String, String> map,String charset) throws Exception {
        String result = "";
        boolean hasParams = false;
        if (path != null && !"".equals(path)) {
            if (MapUtils.isNotEmpty(map)) {
                StringBuilder builder = new StringBuilder();
                Set<Map.Entry<String, String>> params = map.entrySet();
                for (Map.Entry<String, String> entry : params) {
                    String key = entry.getKey().trim();
                    String value = entry.getValue().trim();
                    if (hasParams) {
                        builder.append("&");
                    } else {
                        hasParams = true;
                    }
                    if (charset != null && !"".equals(charset)) {
                        // builder.append(key).append("=").append(URLDecoder.(value,charset));
                        builder.append(key).append("=").append(URLEncoder.encode(value, charset));
                    } else {
                        builder.append(key).append("=").append(value);
                    }
                }
                result = builder.toString();
            }
        }
        return doUrlPath(path, result).toString();
    }

    /**
     * 设置连接参数
     * @param path 路径
     * @return
     */
    private static URL doUrlPath(String path, String query) throws Exception {
        URL url = new URL(path);
        if (StringUtils.isEmpty(path)) {
            return url;
        }
        if (StringUtils.isEmpty(url.getQuery())) {
            if (path.endsWith("?")) {
                path += query;
            } else {
                path = path + "?" + query;
            }
        } else {
            if (path.endsWith("&")) {
                path += query;
            } else {
                path = path + "&" + query;
            }
        }
        return new URL(path);
    }





    public static void main(String[] args)throws Exception {
        System.out.println(URLEncoder.encode("疑问2(1).gif", "UTF-8"));
        //System.out.println(UrlUtil.toUTF8("https://api.java.tiennet.com/files/20190904/d68e8c55013c4f4fa8a8388ced9b3943/thumb/疑问2(1).gif"));
    }
}
