package cn.dolphin.core.web.util;

import com.alibaba.fastjson.JSON;
import cn.dolphin.core.consts.CharsetConstant;
import cn.dolphin.core.consts.StringConstant;
import cn.dolphin.core.exception.IoRuntimeException;
import cn.dolphin.core.result.ApiResult;
import cn.dolphin.core.util.Utf8Util;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.PrintWriter;

/**
 * 扩展HttpServletResponse的功能，返回工具类
 */
@SuppressWarnings("all")
public class ResponseUtil {


    /**
     * 下载的ContentType
     */
    private static final String DOWN_TYPE = "applicatoin/octet-stream";

    /**
     * 下载的Header1
     */
    private static final String DOWN_HEADER1 = "Content-Disposition";
    /**
     * 下载的Header2
     */
    private static final String DOWN_HEADER2 = "attachment; filename=";

    /**
     * 导出Excel的ContentType
     */
    private static final String EXCEL_TYPE = "application/vnd.ms-excel;charset=utf-8";

    /**
     * 页面不缓存
     */
    public static final String CACHE_CONTROL = "Cache-Control";
    public static final String NO_CACHE= "no-cache";
    public static final String PRAGMA= "Pragma";
    public static final String EXPIRESS= "Expires";

    /**
     * 转发前缀
     */
    public static final String FORWARD = "forward:";

    /**
     * 重定向前缀
     */
    public static final String REDIRECT = "redirect:";

    private ResponseUtil() {
    }

    /**
     * 获取Response
     *
     * @return
     *
     */
    public static HttpServletResponse getResponse() {
        return ((ServletRequestAttributes)(RequestContextHolder.getRequestAttributes())).getResponse();
    }


    /**
     * 设置文件下载的响应头
     *
     * @param response
     * @param fileName
     */
    public static final void setDownloadHeader(HttpServletResponse response, String fileName) {
        //重置响应头
        response.reset();
        //设置文件类型：通用的二进制文件下载类型
        response.setContentType(DOWN_TYPE);
        //设置附件名称
        response.setHeader(DOWN_HEADER1, DOWN_HEADER2 + Utf8Util.utf2iso(fileName));
    }

    /**
     * 设置不缓存页面的响应头（如读取验证码图片）
     *
     * @param response
     *
     */
    public static final void setNocachHeader(HttpServletResponse response) {
        //重置响应头
        response.reset();
        //设置页面不缓存
        response.setHeader(PRAGMA, NO_CACHE);
        response.setHeader(CACHE_CONTROL, NO_CACHE);
        response.setDateHeader(EXPIRESS, 0);
    }

    /**
     *
     * 设置导出Excel的响应头
     *
     * @param response
     * @param fileName
     *
     */
    public static final void setExcelHeader(HttpServletResponse response, String fileName) {
        //重置响应头
        response.reset();
        //设置文件类型：Excel类型
        response.setContentType(EXCEL_TYPE);
        //设置附件名称
        response.setHeader(DOWN_HEADER1, DOWN_HEADER2 + Utf8Util.utf2iso(fileName));
    }


    /**
     *
     * Response发送JSON文字信息
     *
     * @param text
     *
     */
    public static final void response(String text){
        response(getResponse(),text);
    }

    /**
     *
     * Response发送JSON文字信息
     *
     * @param response
     * @param text
     *
     */
    public static final void response(HttpServletResponse response,String text){
        //设置响应头
        cors(response, null);
        response.setStatus(HttpStatus.OK.value());
        response.setContentType(MediaType.APPLICATION_JSON_VALUE);
        response.setCharacterEncoding(CharsetConstant.UTF8);
        //发送信息
        PrintWriter writer = null;
        try {
            writer = response.getWriter();
            writer.write(text);
        } catch (IOException e) {
            throw new IoRuntimeException("发送JSON响应失败", e);
        }finally {
            try {
                writer.close();
            } catch (Exception e2) {
                e2.printStackTrace();
            }
        }
    }

    /**
     * Response发送文字响应
     *
     */
    public static final void responseText(String text){
        responseText(getResponse(),text);
    }

    /**
     * Response发送文字响应
     *
     */
    public static final void responseText(HttpServletResponse response,String text){
        try {
            response.getWriter().write(text);
            response.getWriter().flush();
        } catch (IOException e) {
            throw new IoRuntimeException("发送文字响应失败", e);
        }finally {
            try {
                response.getWriter().close();
            } catch (Exception e2) {
                e2.printStackTrace();
            }
        }
    }

    /**
     * 获取spring-mvc转发返回值
     *
     * @param url
     * @return
     */
    public static final String forward(String url){
        return FORWARD+url;
    }

    /**
     * 转发
     * @param url
     */
    public static final void sendForward(HttpServletRequest request, HttpServletResponse response, String url){
        try {
            request.getRequestDispatcher(url).forward(request,response);
        } catch (ServletException | IOException e) {
            throw new IoRuntimeException("转发失败：url = "+url, e);
        }
    }

    /**
     * 转发
     * @param url
     */
    public static final void sendForward(String url){
        sendForward(RequestUtil.getRequest(),getResponse(), url);
    }


    /**
     *
     * 获取spring-mvc重定向返回值
     *
     * @param url
     * @return
     *
     */
    public static final String redirect(String url){
        return REDIRECT+url;
    }

    /**
     * 重定向
     *
     * @param url
     *
     */
    public static final void sendRedirect(HttpServletResponse response,String url){
        try {
            response.sendRedirect(url);
        } catch (IOException e) {
            throw new IoRuntimeException("重定向失败：url = "+url, e);
        }
    }

    /**
     * 重定向
     *
     * @param url
     *
     */
    public static final void sendRedirect(String url){
        sendRedirect(getResponse(), url);
    }


    /**
     * 下载文件
     *
     * @param bs
     * @param fileName
     *
     */
    public static final void responseFile(byte[] bs,String fileName){
        responseFile(bs, fileName, getResponse());
    }

    /**
     * 下载文件
     *
     * @param bs
     * @param fileName
     * @param response
     *
     */
    public static final void responseFile(byte[] bs,String fileName,HttpServletResponse response){
        // 响应头
        setDownloadHeader(response, fileName);
        ServletOutputStream output = null;
        try {
            // 输出
            output = response.getOutputStream();
            output.write(bs);
            output.flush();
        } catch (IOException e) {
            throw new IoRuntimeException("发送文件响应失败", e);
        }finally {
            try {
                output.close();
            } catch (Exception e2) {
                e2.printStackTrace();
            }
        }
    }

    /**
     * 跨域设置：如果已设置则忽略
     *
     * @param origin	放行的域，* 或 null 表示全部
     * @param response
     *
     */
    public static final void cors(HttpServletResponse response, String origin) {
        if(StringUtils.isBlank(response.getHeader(HttpHeaders.ACCESS_CONTROL_ALLOW_ORIGIN)) ) {
            response.setHeader(HttpHeaders.ACCESS_CONTROL_ALLOW_ORIGIN, origin == null ? StringConstant.STAR : origin);
        }
    }


    /**
     * SSE响应头设置：用于Nginx反向代理时设置不缓存数据，以便长连接请求可以及时推送
     */
    public static final void sse() {
        getResponse().setHeader("X-Accel-Buffering", "no");
    }


    /**
     * token鉴权 返回错误信息
     *
     * @param httpResponse
     * @throws IOException
     */
    public static void errorToken(HttpServletResponse httpResponse, String msg) throws IOException {
        ApiResult result = ApiResult.error(msg);
        String json = JSON.toJSONString(result);
        httpResponse.setCharacterEncoding("UTF-8");
        httpResponse.setContentType("application/json;charset=UTF-8");
        httpResponse.setHeader("Content-type", "application/json;charset=UTF-8");
        httpResponse.setHeader("Access-Control-Allow-Origin", "*");
        httpResponse.setStatus(HttpServletResponse.SC_UNAUTHORIZED);
        httpResponse.getWriter().write(json);
    }


    /**
     * token鉴权 返回错误信息
     *
     * @param httpResponse
     * @throws IOException
     */
    public static void errorToken(HttpServletResponse httpResponse, String code, String msg) throws IOException {
        ApiResult result = ApiResult.error(code,msg);
        String json = JSON.toJSONString(result);
        httpResponse.setCharacterEncoding("UTF-8");
        httpResponse.setContentType("application/json;charset=UTF-8");
        httpResponse.setHeader("Content-type", "application/json;charset=UTF-8");
        httpResponse.setHeader("Access-Control-Allow-Origin", "*");
        httpResponse.setStatus(HttpServletResponse.SC_UNAUTHORIZED);
        httpResponse.getWriter().write(json);
    }


    /**
     * 无权访问 返回错误信息
     *
     * @param httpResponse
     * @throws IOException
     */
    public static void errorAuth(HttpServletResponse httpResponse, String code, String msg) throws IOException {
        ApiResult result = ApiResult.error(code,msg);
        String json = JSON.toJSONString(result);
        httpResponse.setCharacterEncoding("UTF-8");
        httpResponse.setContentType("application/json;charset=UTF-8");
        httpResponse.setHeader("Content-type", "application/json;charset=UTF-8");
        httpResponse.setHeader("Access-Control-Allow-Origin", "*");
        httpResponse.setStatus(HttpServletResponse.SC_UNAUTHORIZED);
        httpResponse.getWriter().write(json);
    }

    /**
     * 响应Http请求
     * @param response
     * @param resData
     * @throws JsonProcessingException
     */
    public static void responseData(HttpServletResponse response, Object resData) throws JsonProcessingException {
        //将实体对象转换为jackson Object转换
        ObjectMapper objectMapper = new ObjectMapper();
        String jsonString = objectMapper.writeValueAsString(resData);
        response.setCharacterEncoding("UTF-8");
        response.setContentType("application/json; charset=utf-8");
        PrintWriter out = null;
        try {
            out = response.getWriter();
            out.append(jsonString);
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (out != null) {
                out.close();
            }
        }
    }




}
