package cn.dolphin.core.web.util;

import cn.dolphin.core.date.DateFormatUtil;
import cn.dolphin.core.ip.IPUtil;
import cn.dolphin.core.util.BeanUtil;
import cn.dolphin.core.util.StrUtil;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.servlet.ServletRequest;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.text.ParseException;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


/**
 * 扩展HttpServletRequest的功能，所有请求参数获取都通过该类方法来获取
 */
@SuppressWarnings("all")
public class RequestUtil {


    private static Logger logger = LoggerFactory.getLogger(RequestUtil.class);

    /**
     * 请求类型的Header键
     */
    public static final String RT_HEADER = "X-Requested-With";

    /**
     * Ajax请求类型
     */
    public static final String AJAX_RT = "XMLHttpRequest";

    public static final String RETURNURL="returnUrl";


    /**
     * 自定义token名称
     */
    public static final String TOKEN = "token";

    /**
     * 默认token名称
     */
    public static final String AUTHORIZATION = "Authorization";


    private RequestUtil() {
    }


    /**
     * 获取Request
     *
     * @return
     *
     */
    public static HttpServletRequest getRequest() {
        return ((ServletRequestAttributes)(RequestContextHolder.getRequestAttributes())).getRequest();
    }

    /**
     * 设置header
     * @param request
     * @param response
     */
    public static void setHeader(HttpServletRequest request, HttpServletResponse response){
        //跨域的header设置
        response.setHeader("Access-control-Allow-Origin", request.getHeader("Origin"));
        response.setHeader("Access-Control-Allow-Methods", "POST,GET,OPTIONS,DELETE,PUT");
        response.setHeader("Access-Control-Allow-Credentials", "false");
        response.setHeader("Access-Control-Allow-Headers", "*");
        response.setHeader("Access-Control-Max-Age", "3600");
        response.setHeader("Content-Type","application/json;charset=UTF-8");
        response.setStatus(HttpStatus.OK.value());
    }

    /**
     * 获取user-agent
     * @param request
     * @return
     */
    public static String getUserAgent(HttpServletRequest request) {
        return request.getHeader("user-agent");
    }

    /**
     * 获取domain
     * @return
     */
    public static String getDomain(){
        HttpServletRequest request = getRequest();
        StringBuffer url = request.getRequestURL();
        return url.delete(url.length() - request.getRequestURI().length(), url.length()).toString();
    }

    /**
     * 获取origin
     * @return
     */
    public static String getOrigin(){
        HttpServletRequest request = getRequest();
        return request.getHeader("Origin");
    }

    /**
     * 判断是否为AJAX请求
     *
     * @param request
     * @return
     *
     */
    public static final boolean isAjax(HttpServletRequest request){
        //获取请求类型
        String requestType = request.getHeader(RT_HEADER);
        if (org.apache.commons.lang3.StringUtils.isBlank(requestType)) {
            return false;
        }else{
            return requestType.equalsIgnoreCase(AJAX_RT);
        }
    }


    /**
     * 取字符串类型的参数。 如果取得的值为null，则返回默认字符串
     * @param request
     * @param key 字段名
     * @param defaultValue
     * @param b
     * @return
     */
    public static String getString(HttpServletRequest request, String key,
                                   String defaultValue, boolean b) {
        String value = request.getParameter(key);
        if (StrUtil.isEmpty(value))
            return defaultValue;
        if(b){
            return value.replace("'", "''").trim();
        }else{
            return value.trim();
        }
    }

    /**
     * 取字符串类型的参数。 如果取得的值为null，则返回默认字符串
     * @param key 字段名
     * @param defaultValue
     * @param b
     * @return
     */
    public static String getString(String key,
                                   String defaultValue, boolean b) {
        String value = getRequest().getParameter(key);
        if (StrUtil.isEmpty(value))
            return defaultValue;
        if(b){
            return value.replace("'", "''").trim();
        }else{
            return value.trim();
        }
    }



    /**
     * 取字符串类型的参数。 如果取得的值为null，则返回空字符串
     * @param request
     * @param key 字段名
     * @return
     */
    public static String getString(HttpServletRequest request, String key) {
        return getString(request, key, "",true);
    }



    /**
     * 取字符串类型的参数。 如果取得的值为null，则返回空字符串
     * @param request
     * @param key
     * @param b
     * @return
     */
    public static String getString(HttpServletRequest request, String key,
                                   boolean b) {
        return getString(request, key, "", b);
    }


    /**
     * 取字符串类型的参数。 如果取得的值为null，则返回空字符串
     * @param request
     * @param key
     * @param defaultValue
     * @return
     */
    public static String getString(HttpServletRequest request, String key,
                                   String defaultValue) {
        return getString(request, key, defaultValue, true);
    }


    public static String getStringAry(HttpServletRequest request, String key){
        String[] aryValue = request.getParameterValues(key);
        if(aryValue==null || aryValue.length==0){
            return "";
        }
        String tmp="";
        for(String v:aryValue){
            if("".equals(tmp)){
                tmp+=v;
            }
            else{
                tmp+="," + v;
            }
        }
        return tmp;
    }


    /**
     * 取得安全字符串
     * @param request
     * @param key
     * @param defaultValue
     * @return
     */
    public static String getSecureString(HttpServletRequest request,
                                         String key, String defaultValue) {
        String value = request.getParameter(key);
        if (StrUtil.isEmpty(value))
            return defaultValue;
        return filterInject(value);

    }


    /**
     * 取得安全字符串，防止程序sql注入，脚本攻击
     * @param request
     * @param key
     * @return
     */
    public static String getSecureString(HttpServletRequest request, String key) {
        return getSecureString(request, key, "");
    }



    /**
     * 过滤script|iframe|\\||;|exec|insert|select|delete|update|count|chr|truncate|char字符串 防止SQL注入
     * @param str
     * @return
     */
    public static String filterInject(String str) {
        String injectstr = "\\||;|exec|insert|select|delete|update|count|chr|truncate|char";
        Pattern regex = Pattern.compile(injectstr, Pattern.CANON_EQ
                | Pattern.DOTALL | Pattern.CASE_INSENSITIVE
                | Pattern.UNICODE_CASE);
        Matcher matcher = regex.matcher(str);
        str = matcher.replaceAll("");
        str = str.replace("'", "''");
        str = str.replace("-", "—");
        str = str.replace("(", "（");
        str = str.replace(")", "）");
        str = str.replace("%", "％");

        return str;
    }


    /**
     * 从Request中取得指定的小写值
     * @param request
     * @param key
     * @return
     */
    public static String getLowercaseString(HttpServletRequest request,
                                            String key) {
        return getString(request, key).toLowerCase();
    }


    /**
     * 从request中取得int值
     * @param request
     * @param key
     * @return
     */
    public static int getInt(HttpServletRequest request, String key) {
        return getInt(request, key, 0);
    }


    /**
     * 从request中取得int值,如果无值则返回缺省值
     * @param request
     * @param key
     * @param defaultValue
     * @return
     */
    public static int getInt(HttpServletRequest request, String key,
                             int defaultValue) {
        String str = request.getParameter(key);
        if (StrUtil.isEmpty(str))
            return defaultValue;
        return Integer.parseInt(str);

    }

    /**
     * 从request中取得int值,如果无值则返回缺省值
     * @param key
     * @param defaultValue
     * @return
     */
    public static int getInt(String key, int defaultValue) {
        String str = getRequest().getParameter(key);
        if (StrUtil.isEmpty(str))
            return defaultValue;
        return Integer.parseInt(str);

    }

    /**
     * 从Request中取得long值
     * @param request
     * @param key
     * @return
     */
    public static long getLong(HttpServletRequest request, String key) {
        return getLong(request, key, 0);
    }


    /**
     * 取得长整形数组
     * @param request
     * @param key
     * @return
     */
    public static Long[] getLongAry(HttpServletRequest request, String key) {
        String[] aryKey = request.getParameterValues(key);
        if (BeanUtil.isEmpty(aryKey))
            return null;
        Long[] aryLong = new Long[aryKey.length];
        for (int i = 0; i < aryKey.length; i++) {
            aryLong[i] = Long.parseLong(aryKey[i]);
        }
        return aryLong;
    }


    /**
     * 根据一串逗号分隔的长整型字符串取得长整形数组
     * @param request
     * @param key
     * @return
     */
    public static Long[] getLongAryByStr(HttpServletRequest request, String key) {
        String str = request.getParameter(key);
        if(StrUtil.isEmpty(str)) return null;
        String[] aryId = str.split(",");
        Long[] lAryId = new Long[aryId.length];
        for (int i = 0; i < aryId.length; i++) {
            lAryId[i] = Long.parseLong(aryId[i]);
        }
        return lAryId;
    }


    /**
     * 根据一串逗号分隔的字符串取得字符串形数组
     * @param request
     * @param key
     * @return
     */
    public static String[] getStringAryByStr(HttpServletRequest request,
                                             String key) {
        String str = request.getParameter(key);
        if(StrUtil.isEmpty(str)) return null;
        String[] aryId = str.split(",");
        String[] lAryId = new String[aryId.length];
        for (int i = 0; i < aryId.length; i++) {
            lAryId[i] = (aryId[i]);
        }
        return lAryId;
    }


    /**
     * 根据键值取得整形数组
     * @param request
     * @param key
     * @return
     */
    public static Integer[] getIntAry(HttpServletRequest request, String key) {
        String[] aryKey = request.getParameterValues(key);
        if(BeanUtil.isEmpty(aryKey)) return null;
        Integer[] aryInt = new Integer[aryKey.length];
        for (int i = 0; i < aryKey.length; i++) {
            aryInt[i] = Integer.parseInt(aryKey[i]);
        }
        return aryInt;
    }

    public static Float[] getFloatAry(HttpServletRequest request, String key) {
        String[] aryKey = request.getParameterValues(key);
        if(BeanUtil.isEmpty(aryKey)) return null;
        Float[] fAryId = new Float[aryKey.length];
        for (int i = 0; i < aryKey.length; i++) {
            fAryId[i] = Float.parseFloat(aryKey[i]);
        }
        return fAryId;
    }


    /**
     * 从Request中取得long值,如果无值则返回缺省值
     * @param request
     * @param key
     * @param defaultValue
     * @return
     */
    public static long getLong(HttpServletRequest request, String key,
                               long defaultValue) {
        String str = request.getParameter(key);
        if (StrUtil.isEmpty(str))
            return defaultValue;
        return Long.parseLong(str.trim());
    }


    /**
     * 从Request中取得long值,如果无值则返回缺省值
     * @param request
     * @param key
     * @param defaultValue
     * @return
     */
    public static Long getLong(HttpServletRequest request, String key,
                               Long defaultValue) {
        String str = request.getParameter(key);
        if (StrUtil.isEmpty(str))
            return defaultValue;
        return Long.parseLong(str.trim());
    }


    /**
     * 从Request中取得float值
     * @param request
     * @param key
     * @return
     */
    public static float getFloat(HttpServletRequest request, String key) {
        return getFloat(request, key, 0);
    }


    /**
     * 从Request中取得float值,如无值则返回缺省值
     * @param request
     * @param key
     * @param defaultValue
     * @return
     */
    public static float getFloat(HttpServletRequest request, String key,
                                 float defaultValue) {
        String str = request.getParameter(key);
        if (StrUtil.isEmpty(str))
            return defaultValue;
        return Float.parseFloat(request.getParameter(key));
    }

    /**
     * 从Request中取得boolean值,如无值则返回缺省值 false, 如值为数字1，则返回true
     * @param request
     * @param key
     * @return
     */
    public static boolean getBoolean(HttpServletRequest request, String key) {
        return getBoolean(request, key, false);
    }


    /**
     * 从Request中取得boolean值 对字符串,如无值则返回缺省值, 如值为数字1，则返回true
     * @param request
     * @param key
     * @param defaultValue
     * @return
     */
    public static boolean getBoolean(HttpServletRequest request, String key,
                                     boolean defaultValue) {
        String str = request.getParameter(key);
        if (StrUtil.isEmpty(str))
            return defaultValue;
        if (StringUtils.isNumeric(str))
            return (Integer.parseInt(str) == 1 ? true : false);
        return Boolean.parseBoolean(str);
    }



    /**
     * 从Request中取得boolean值,如无值则返回缺省值 0
     * @param request
     * @param key
     * @return
     */
    public static Short getShort(HttpServletRequest request, String key) {
        return getShort(request, key, (short) 0);
    }


    /**
     * 从Request中取得Short值 对字符串,如无值则返回缺省值
     * @param request
     * @param key
     * @param defaultValue
     * @return
     */
    public static Short getShort(HttpServletRequest request, String key,
                                 Short defaultValue) {
        String str = request.getParameter(key);
        if (StrUtil.isEmpty(str))
            return defaultValue;
        return Short.parseShort(str);
    }


    /**
     * 从Request中取得Date值,如无值则返回缺省值,如有值则返回 yyyy-MM-dd HH:mm:ss 格式的日期,或者自定义格式的日期
     * @param request
     * @param key
     * @param style
     * @return
     * @throws ParseException
     */
    public static Date getDate(HttpServletRequest request, String key,
                               String style) throws ParseException {
        String str = request.getParameter(key);
        if (StrUtil.isEmpty(str))
            return null;
        if (StrUtil.isEmpty(style))
            style = "yyyy-MM-dd HH:mm:ss";
        return (Date) DateFormatUtil.parse(str, style);
    }


    /**
     * 从Request中取得Date值,如无值则返回缺省值null, 如果有值则返回 yyyy-MM-dd 格式的日期
     * @param request
     * @param key
     * @return
     * @throws ParseException
     */
    public static Date getDate(HttpServletRequest request, String key)
            throws ParseException {
        String str = request.getParameter(key);
        if (StrUtil.isEmpty(str))
            return null;
        return (Date) DateFormatUtil.parseDate(str);

    }


    /**
     * 从Request中取得Date值,如无值则返回缺省值 如有值则返回 yyyy-MM-dd HH:mm:ss 格式的日期
     * @param request
     * @param key
     * @return
     * @throws ParseException
     */
    public static Date getTimestamp(HttpServletRequest request, String key)
            throws ParseException {
        String str = request.getParameter(key);
        if (StrUtil.isEmpty(str))
            return null;
        return (Date) DateFormatUtil.parseDateTime(str);
    }


    /**
     * 取得当前页URL,如有参数则带参数
     * @param request
     * @return
     */
    public static String getUrl(HttpServletRequest request) {
        StringBuffer urlThisPage = new StringBuffer();
        urlThisPage.append(request.getRequestURI());
        Enumeration<?> e = request.getParameterNames();
        String para = "";
        String values = "";
        urlThisPage.append("?");
        while (e.hasMoreElements()) {
            para = (String) e.nextElement();
            values = request.getParameter(para);
            urlThisPage.append(para);
            urlThisPage.append("=");
            urlThisPage.append(values);
            urlThisPage.append("&");
        }
        return urlThisPage.substring(0, urlThisPage.length() - 1);
    }

    /**
     * 得到访问请求根域名
     * @param request
     * @return
     */
    public static String getRootHost(HttpServletRequest request){
        return request.getRequestURL().toString().split(request.getRequestURI())[0];
    }

    /**
     * 得到访问请求项目根域名
     * @param request
     * @return
     */
    public static String getSysHost(HttpServletRequest request){
        return request.getRequestURL().toString().split(request.getRequestURI())[0] + request.getContextPath();
    }


    /**
     * 功能：把当前上下文的请求封装在map中<br>
     * 详细：<br>
     * 作者：by heqing.y<br>
     * 参数：<br>
     * @param request
     * @param remainArray
     *            保持为数组
     * @param isSecure
     *            过滤不安全字符
     * @return
     */
    public static Map<String,Object> getParameterValueMap(HttpServletRequest request,
                                                          boolean remainArray, boolean isSecure) {
        Map<String,Object> map = new HashMap<String,Object>();
        Enumeration<?> params = request.getParameterNames();
        while (params.hasMoreElements()) {
            String key = params.nextElement().toString();
            String[] values = request.getParameterValues(key);
            if (values == null)
                continue;
            if (values.length == 1) {
                String tmpValue = values[0];
                if (tmpValue == null)
                    continue;
                tmpValue = tmpValue.trim();
                if (tmpValue.equals(""))
                    continue;
                if (isSecure)
                    tmpValue = filterInject(tmpValue);
                if (tmpValue.equals(""))
                    continue;
                map.put(key, tmpValue);
            } else {
                String rtn = getByAry(values, isSecure);
                if (rtn.length() > 0) {
                    if (remainArray)
                        map.put(key, rtn.split(","));
                    else
                        map.put(key, rtn);
                }
            }
        }
        return map;
    }


    private static String getByAry(String[] aryTmp, boolean isSecure) {
        String rtn = "";
        for (int i = 0; i < aryTmp.length; i++) {
            String str = aryTmp[i].trim();
            if (!str.equals("")) {
                if (isSecure) {
                    str = filterInject(str);
                    if (!str.equals(""))
                        rtn += str + ",";
                } else {
                    rtn += str + ",";
                }
            }
        }
        if (rtn.length() > 0)
            rtn = rtn.substring(0, rtn.length() - 1);
        return rtn;
    }



    /**
     * 根据参数名称获取参数值
     * 1.根据参数名称取得参数值的数组。
     * 2.将数组使用逗号分隔字符串。
     * @param request
     * @param paramName
     * @return
     */
    public static String getStringValues(HttpServletRequest request,
                                         String paramName) {
        String[] values = request.getParameterValues(paramName);
        if (BeanUtil.isEmpty(values))
            return "";
        String tmp = "";
        for (int i = 0; i < values.length; i++) {
            if (i == 0) {
                tmp += values[i];
            } else {
                tmp += "," + values[i];
            }
        }
        return tmp;
    }



    /**
     * 取得local
     * @param request
     * @return
     */
    public static Locale getLocal(HttpServletRequest request) {
        Locale local = request.getLocale();
        if (local == null)
            local = Locale.CHINA;
        return local;
    }


    /**
     * 得到GET请求网站URL
     * @param url
     * @param postParams
     * @return
     */
    public static String getGetFullUrl(String url,String... postParams){
        if(postParams!=null){
            for (int i = 0; i < postParams.length; i=i+2) {
                if(!url.contains("?") && i==0){
                    url += "?"+postParams[i]+"="+postParams[i+1];
                }else{
                    url += "&"+postParams[i]+"="+postParams[i+1];
                }
            }
        }
        return url;
    }

    /**
     * 得到GET请求网站URL
     * @param url
     * @param postParams
     * @return
     */
    public static String getGetFullUrl(String url,Map<String, ?> postParams){
        if(postParams!=null){
            int i = 0;
            for (String key : postParams.keySet()) {
                if(!url.contains("?") && i==0){
                    url += "?"+key+"="+String.valueOf(postParams.get(key));
                }else{
                    url += "&"+key+"="+String.valueOf(postParams.get(key));
                }
                i++;
            }
        }
        return url;
    }


    /**
     * 获取出错的url
     * @param request
     * @return
     */
    public final static String getErrorUrl(HttpServletRequest request) {
        String errorUrl = (String) request
                .getAttribute("javax.servlet.error.request_uri");
        if (errorUrl == null) {
            errorUrl = (String) request
                    .getAttribute("javax.servlet.forward.request_uri");
        }
        if (errorUrl == null) {
            errorUrl = (String) request
                    .getAttribute("javax.servlet.include.request_uri");
        }
        if (errorUrl == null) {
            errorUrl = request.getRequestURL().toString();
        }
        return errorUrl;
    }


    /**
     * 将请求中的参数封装成Map（转码utf-8）
     * @param request Http请求对象
     * @return 将所有请求参数封装为Map集合并返回
     */
    public static Map<String, String> getParamsUtf8(HttpServletRequest request){
        Map<String, String> params = new HashMap<String, String>();
        for (String key : request.getParameterMap().keySet()) {
            params.put(key, RequestUtil.urlDecode(request.getParameter(key), "UTF-8"));
        }
        return params;
    }

    /**
     * 将请求中的参数封装成Map(不进行转码)
     * @param request Http请求对象
     * @return 将所有请求参数封装为Map集合并返回
     */
    public static Map<String, Object> getParams(HttpServletRequest request){
        Map<String, Object> params = new HashMap<String, Object>();
        for (String key : request.getParameterMap().keySet()) {
            params.put(key, request.getParameter(key));
        }
        return params;
    }

    /**
     * 请求中参数转Map<String, String>,for支付宝异步回调,平时建议直接使用request.getParameterMap(),返回Map<String, String[]>
     * @param request
     * @return
     */
    @SuppressWarnings("rawtypes")
    public static Map<String, String> getParameterMap(HttpServletRequest request) {
        Map<String, String> result = new HashMap<String, String>();
        Enumeration parameterNames = request.getParameterNames();
        while (parameterNames.hasMoreElements()) {
            String parameterName = (String) parameterNames.nextElement();
            result.put(parameterName, request.getParameter(parameterName));
        }
        return result;
    }

    /**
     * 将请求中的参数封装成Map（转码utf-8）
     * @param request Http请求对象
     * @return 将所有请求参数封装为Map集合并返回
     */
    public static Map<String, String> getParamsUtf8(ServletRequest request){
        Map<String, String> params = new HashMap<String, String>();
        for (String key : request.getParameterMap().keySet()) {
            params.put(key, RequestUtil.urlDecode(request.getParameter(key), "UTF-8"));
        }
        return params;
    }

    /**
     * 将请求中的参数封装成Map，并去掉前后空格
     * @param request Http请求对象
     * @return 将所有请求参数封装为Map集合并返回
     */
    public static Map<String, String> getTrimedParamsUtf8(HttpServletRequest request){
        Map<String, String> params = new HashMap<String, String>();
        for (String key : request.getParameterMap().keySet()) {
            params.put(key, RequestUtil.urlDecode(request.getParameter(key), "UTF-8").trim());
        }
        return params;
    }




    /**
     * 将Session域中的参数封装成Map
     * @param session HttpSession请求对象
     * @return
     */
    public static Map<String, Object> getParams(HttpSession session) {
        Map<String, Object> params = new HashMap<String, Object>();
        Enumeration<String> enumeration = session.getAttributeNames();
        while(enumeration.hasMoreElements()){
            String key = enumeration.nextElement();
            params.put(key, session.getAttribute(key));
        }
        return params;
    }

    /**
     * isAjaxRequest:判断请求是否为Ajax请求. <br/>
     * @param request 请求对象
     * @return boolean
     */
    public static boolean isAjaxRequest(HttpServletRequest request){
        String header = request.getHeader("X-Requested-With");
        boolean isAjax = "XMLHttpRequest".equals(header) ? true:false;
        return isAjax;
    }

    /**
     * 进行UrlDecode解码
     * @param text 字符串
     * @param charset 编码
     * @return
     */
    public static String urlDecode(String text, String charset) {
        try {
            return URLDecoder.decode(text,charset);
        } catch (Exception e) {
            return text;
        }
    }

    /**
     * 进行urlEncode加码
     * @param text 字符串
     * @param charset 编码
     * @return
     */
    public static String urlEncode(String text, String charset) {
        try {
            return URLEncoder.encode(text,charset);
        } catch (Exception e) {
            return text;
        }
    }


    /**
     * 获取token
     *
     * @param request
     * @return
     */
    public static String getAuthHeader(HttpServletRequest request) {

        // 先从Header里面获取
        String accessToken = request.getHeader(TOKEN);
        if (!StrUtil.isBlank(accessToken)) {
            return accessToken;
        }

        accessToken = request.getHeader(AUTHORIZATION);
        if (!StrUtil.isBlank(accessToken)) {
            return accessToken;
        }

        // 获取不到再从Parameter中拿
        accessToken = request.getParameter(TOKEN);
        if (!StrUtil.isBlank(accessToken)) {
            return accessToken;
        }
        // 还是获取不到再从Cookie中拿
        Cookie[] cookies = request.getCookies();
        if (null == cookies || cookies.length == 0) {
            return null;
        }
        for (Cookie cookie : cookies) {
            if (cookie.getName().equals(TOKEN)) {
                accessToken = cookie.getValue();
                break;
            }
        }
        return accessToken;

    }

    /**
     * 获取请求basePath
     * @param request
     * @return
     */
    public static String getBasePath(HttpServletRequest request) {
        StringBuffer basePath = new StringBuffer();
        String scheme = request.getScheme();
        String domain = request.getServerName();
        int port = request.getServerPort();
        basePath.append(scheme);
        basePath.append("://");
        basePath.append(domain);
        if("http".equalsIgnoreCase(scheme) && 80 != port) {
            basePath.append(":").append(String.valueOf(port));
        } else if("https".equalsIgnoreCase(scheme) && port != 443) {
            basePath.append(":").append(String.valueOf(port));
        }
        return basePath.toString();
    }


    /**
     * 移除request指定参数
     *
     * @param request
     * @param paramName
     * @return
     */
    public static String removeParam(HttpServletRequest request, String paramName) {
        String queryString = "";
        Enumeration keys = request.getParameterNames();
        while (keys.hasMoreElements()) {
            String key = (String) keys.nextElement();
            if (key.equals(paramName)) {
                continue;
            }
            if (queryString.equals("")) {
                queryString = key + "=" + request.getParameter(key);
            } else {
                queryString += "&" + key + "=" + request.getParameter(key);
            }
        }
        return queryString;
    }

    /**
     * 获取ip工具类，除了getRemoteAddr，其他ip均可伪造
     * 如果配置了nginx
     * 需要在location中加上
     *       proxy_set_header  Host               $host;
     *       proxy_set_header  X-Real-IP          $remote_addr;#提交请求的真实ip，为了目标服务也能获取到请求的真实ip
     *       proxy_set_header  X-Forwarded-For    $proxy_add_x_forwarded_for;
     *       proxy_set_header  X-Forwarded-Proto  http;
     * @param request
     * @return
     */
    public static String getIpAddr(HttpServletRequest request) {
        return IPUtil.getIpAddress(request);
    }

    /**
     * 获取上传文件
     *
     * @param request
     * @return
     */
    public static Collection<MultipartFile> getUploadFiles(HttpServletRequest request) {
        Map<String, MultipartFile> fileMap = null;
        //检查form中是否有enctype="multipart/form-data"
        if (ServletFileUpload.isMultipartContent(request)) {
            //将request变成多部分request
            MultipartHttpServletRequest multiRequest = (MultipartHttpServletRequest) request;
            fileMap = multiRequest.getFileMap();
        }
        return Optional.ofNullable(fileMap)
                .map(map -> map.values())
                .orElse(Collections.emptyList());
    }


}
