package cn.dolphin.core.web.util;

import cn.dolphin.core.exception.ParamRuntimeException;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.Arrays;
import java.util.Collection;
import java.util.Map;


/**
 * 参数非空检测工具
 *
 */
@SuppressWarnings("all")
public final class ParamEmptyUtil {

    /**
     * 判断参数是否为空，只要有一个返回true就抛NULL异常
     *
     * @param checks	参数空校验表达式
     *
     */
    public static final void check(boolean ...checks){
        for (boolean check : checks) {
            if (check) {
                throw ParamRuntimeException.NULL;
            }
        }
    }

    /**
     * 判断参数是否为空，只要有一个为空就抛异常
     *
     * @param params
     *
     */
    public static final void checkBlank(String ...params){
        if (StringUtils.isAnyBlank(params)) {
            throw ParamRuntimeException.NULL;
        }
    }


    /**
     * 判断参数是否为Null，只要有一个为Null则抛异常
     *
     * @param params
     *
     */
    public static final void checkNull(Object ...params){
        if (params == null || params.length ==0) {
            throw ParamRuntimeException.NULL;
        }
        for (Object obj : params) {
            if (obj == null) {
                throw ParamRuntimeException.NULL;
            }
        }
    }
    /**
     * 判断参数是否为Null，如果都为Null则抛异常
     *
     * @param params
     *
     */
    public static final void checkAllNull(Object ...params){
        if (params == null || params.length ==0) {
            throw ParamRuntimeException.NULL;
        }
        for (Object obj : params) {
            if (obj != null) {
                return;
            }
        }
        //都为空
        throw ParamRuntimeException.NULL;
    }
    /**
     * 判断参数是否为Null或Blank，只要一个为空或null就抛异常
     *
     * @param params
     *
     */
    public static final void checkEmpty(Object ...params){
        if (params == null || params.length ==0) {
            throw ParamRuntimeException.NULL;
        }
        for (Object obj : params) {
            //对象
            if (obj == null) {
                throw ParamRuntimeException.NULL;
            }
            //字符串
            if (obj instanceof String) {
                if (((String)obj).length() ==0){
                    throw ParamRuntimeException.NULL;
                }
            }
        }
    }

    /**
     * 功能描述：只要有一个参数为null，返回true
     *
     * @param src
     * @return
     *
     */
    public final static boolean isNull(Object... src) {
        if (src == null || src.length == 0) {
            return true;
        }
        for (Object o : src) {
            if (o == null) {
                return true;
            }
        }
        return false;
    }

    /**
     *
     * 只要有一个参数非null，返回true
     *
     * @param src
     * @return
     *
     */
    public final static boolean isNotNull(Object... src) {
        if (src == null || src.length == 0) {
            return false;
        }
        for (Object o : src) {
            if (o != null) {
                return true;
            }
        }
        return false;
    }

    /**
     *
     * 只要有一个字符串为空，返回true
     *
     * @param src
     * @return
     *
     */
    public final static boolean isBlank(String... src) {
        if (src == null || src.length == 0) {
            return true;
        }
        for (String s : src) {
            if (StringUtils.isBlank(s)) {
                return true;
            }
        }
        return false;
    }

    /**
     *
     * 只要有一个字符串非空，返回true
     *
     * @param src
     * @return
     *
     */
    public final static boolean isNotBlank(String... src) {
        if (src == null || src.length == 0) {
            return false;
        }
        for (String s : src) {
            if (StringUtils.isNotBlank(s)) {
                return true;
            }
        }
        return false;
    }
    /**
     *
     * 只要有一个集合为空，返回true
     *
     * @param src
     * @return
     *
     */
    public final static boolean isEmpty(Object[] ...src) {
        if (src == null || src.length == 0) {
            return true;
        }
        //遍历
        for ( Object[] a : src) {
            if (a == null || a.length == 0 ) {
                return true;
            }
        }
        return false;
    }
    /**
     *
     * 只要有一个数组不为空，返回true
     *
     * @param src
     * @return
     *
     */
    public final static boolean isNotEmpty(Object[] ...src) {
        if (src == null || src.length == 0) {
            return false;
        }
        //遍历
        for (Object[] a : src) {
            if (a != null && a.length > 0) {
                return true;
            }
        }
        return false;
    }
    /**
     *
     * 只要有一个集合为空，返回true
     *
     * @param src
     * @return
     *
     */
    public final static boolean isEmpty(Collection<?> ...src) {
        if (src == null || src.length == 0) {
            return true;
        }
        //遍历
        for (Collection<?> c : src) {
            if (CollectionUtils.isEmpty(c)) {
                return true;
            }
        }
        return false;
    }
    /**
     *
     * 只要有一个集合不为空，返回true
     *
     * @param src
     * @return
     *
     */
    public final static boolean isNotEmpty(Collection<?> ...src) {
        if (src == null || src.length == 0) {
            return false;
        }
        //遍历
        for (Collection<?> c : src) {
            if (CollectionUtils.isNotEmpty(c)) {
                return true;
            }
        }
        return false;
    }
    /**
     *
     * 只要有一个Map为空，返回true
     *
     * @param src
     * @return
     *
     */
    public final static boolean isEmpty(Map<?,?> ...src) {
        if (src == null || src.length == 0) {
            return true;
        }
        //遍历
        for (Map<?,?> m : src) {
            if (m == null || m.size() == 0) {
                return true;
            }
        }
        return false;
    }
    /**
     *
     * 只要有一个Map不为空，返回true
     *
     * @param src
     * @return
     *
     */
    public final static boolean isNotEmpty(Map<?,?> ...src) {
        if (src == null || src.length == 0) {
            return false;
        }
        //遍历
        for (Map<?,?> m : src) {
            if (m != null && m.size() > 0) {
                return true;
            }
        }
        return false;
    }
    /**
     *
     * 只要有一个为空，返回true
     *
     * @param src
     * @return
     *
     */
    public final static boolean isEmpty(Object ...src) {
        if (src == null || src.length == 0) {
            return true;
        }
        //遍历
        for (Object o : src) {
            if (o == null) {
                return true;
            }
            //字符串类型
            if (o instanceof String) {
                if (((String)o).length() ==0){
                    return true;
                }
            }
            //集合类型
            else if (o instanceof Collection) {
                if (((Collection<?>)o).size() ==0) {
                    return true;
                }
            }
            //Map类型
            else if (o instanceof Map<?,?>) {
                if (((Map<?,?>)o).size() ==0) {
                    return true;
                }
            }
            //数组类型
            else if (o instanceof Object[]){
                if (((Object[])o).length == 0) {
                    return true;
                }
            }
        }
        return false;
    }
    /**
     *
     * 只要有一个不为空，返回true
     *
     * @param src
     * @return
     *
     */
    public final static boolean isNotEmpty(Object ...src) {
        if (src == null || src.length == 0) {
            return false;
        }
        //遍历
        for (Object o : src) {
            if (o == null) {
                continue;
            }
            //字符串类型
            if (o instanceof String) {
                if (((String)o).length() >0){
                    return true;
                }
            }
            //集合类型
            else if (o instanceof Collection) {
                if (((Collection<?>)o).size() >0) {
                    return true;
                }
            }
            //Map类型
            else if (o instanceof Map<?,?>) {
                if (((Map<?,?>)o).size() >0) {
                    return true;
                }
            }
            //数组类型
            else if (o instanceof Object[]){
                if (((Object[])o).length > 0) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * 判断多个对象都不为空
     *
     * @param os
     */
    public static final boolean allNotEmpty(Object ... os){
        for (Object o : os) {
            if (isEmpty(o)) {
                return false;
            }
        }
        return true;
    }
    /**
     * 判断多个对象都为空
     *
     * @param os
     */
    public static final boolean allEmpty(Object ... os){
        for (Object o : os) {
            if (isNotEmpty(o)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 判断两个对象中，首个非空，末个为空
     *
     * @param o1
     * @param o2
     */
    public static final boolean firstNotEmpty(Object o1,Object o2){
        return (isNotEmpty(o1) && isEmpty(o2));
    }

    /**
     * 判断集合里是否含有空值
     *
     * @param list 参数集合
     * @param <T>  泛型
     * @return true为包含
     */
    public static <T> boolean ListHasEmpty(Collection<? extends T> list) {
        return list.stream().anyMatch(ParamEmptyUtil::isEmpty);
    }

    /**
     * 检查多参数里面是否有空值
     *
     * @param params 多个参数集合
     * @return true为包括, false不包括
     */
    public static boolean multiEmpty(Object... params) {
        return Arrays.stream(params).anyMatch(ParamEmptyUtil::isEmpty);
    }

}
