package cn.dolphin.core.web.context;

import cn.dolphin.core.json.FastJsonUtil;
import cn.dolphin.core.util.Bytes;
import cn.dolphin.core.util.EmptyUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.text.StringEscapeUtils;
import org.json.XML;
import org.springframework.http.HttpMethod;

import javax.servlet.ReadListener;
import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.io.*;
import java.util.HashMap;
import java.util.Map;

/**
 * 解决json请求读取问题
 */
@Slf4j
@SuppressWarnings("all")
public class BodyCacheHttpServletRequestWrapper extends HttpServletRequestWrapper {

    private final static byte[] DEFAULT_BYTE = new byte[0];
    private byte[] body;
    private ServletInputStreamWrapper inputStreamWrapper;

    public BodyCacheHttpServletRequestWrapper(HttpServletRequest request) throws IOException {
        super(request);
        this.body = IOUtils.toByteArray(request.getInputStream());
        ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(this.body != null ? this.body : DEFAULT_BYTE);
        // 初始 ServletInputStreamWrapper
        this.inputStreamWrapper = new ServletInputStreamWrapper(byteArrayInputStream);
        // 设置 InputStream 到我们自己的包装类中
        this.inputStreamWrapper.setInputStream(byteArrayInputStream);
    }


    @Override
    public String[] getParameterValues(String parameter) {
        String[] values = super.getParameterValues(parameter);
        if (values == null || values.length <= 0) {
            return null;
        }
        int count = values.length;
        String[] encodedValues = new String[count];
        for (int i = 0; i < count; i++) {
            encodedValues[i] = StringEscapeUtils.escapeHtml4(values[i]);
        }
        return encodedValues;
    }

    @Override
    public String getParameter(String parameter) {
        String value = super.getParameter(parameter);
        return StringEscapeUtils.escapeHtml4(value);
    }

    @Override
    public String getHeader(String name) {
        String value = super.getHeader(name);
        return StringEscapeUtils.escapeHtml4(value);
    }

    public byte[] getBody() {
        return body;
    }


    public static Map<String, Object> getParamsMap( HttpServletRequest request){
        Map<String, Object> paramMap = new HashMap<String, Object>();// 转化参数集合
        final String httpmethod = request.getMethod();
//        log.info("请求头部："+httpmethod);
        // 如果不是 POST PATCH PUT 等有流的接口则无需进行类型转换，提高性能
        if (HttpMethod.POST.matches(httpmethod) || HttpMethod.PATCH.matches(httpmethod) || HttpMethod.PUT.matches(httpmethod)) {
            if ((request instanceof BodyCacheHttpServletRequestWrapper)) {
                String body = Bytes.toString(((BodyCacheHttpServletRequestWrapper) request).getBody());
                try {
                    if (StringUtils.isNotBlank(body)) {
                        String json = JSON.toJSONString(JSON.parseObject(body), SerializerFeature.WriteMapNullValue);//解决Unicode编码转换Utf8
                        if(EmptyUtil.isNotEmpty(json)){
                            JSONObject jsonObject = JSONObject.parseObject(json);
                            if(EmptyUtil.isNotEmpty(jsonObject)){
                                paramMap.putAll(JSONObject.toJavaObject(jsonObject, Map.class));
                            }
                        }
                    }
                } catch (Exception e) { //解析xml格式参数
                    org.json.JSONObject json = XML.toJSONObject(body);
                    JSONObject jsonObject = null;
                    if(!json.isNull("xml")){
                         jsonObject = FastJsonUtil.parseObject(json.get("xml").toString());
                    }else {
                        jsonObject = FastJsonUtil.parseObject(json.toString());
                    }
                    if(EmptyUtil.isNotEmpty(jsonObject)){
                        paramMap.putAll(JSONObject.toJavaObject(jsonObject, Map.class));
                    }
                }
            }else {
                Map<String, String[]> ParameterMap = request.getParameterMap();// 请求体中所有的参数集合

                for (String name : ParameterMap.keySet()) {
                    // 拼接参数值字符串并进行utf-8解码，防止中文乱码
                    //paramMap.put(name, URLDecoder.decode(ParameterMap.get(name)[0], "UTF-8"));
                    paramMap.put(name, ParameterMap.get(name)[0]);
                }
            }
        }else {
            Map<String, String[]> ParameterMap = request.getParameterMap();// 请求体中所有的参数集合

            for (String name : ParameterMap.keySet()) {
                // 拼接参数值字符串并进行utf-8解码，防止中文乱码
                //paramMap.put(name, URLDecoder.decode(ParameterMap.get(name)[0], "UTF-8"));
                paramMap.put(name, ParameterMap.get(name)[0]);

//            log.debug("requestBody中的请求数据，KEY = [" + name + "]");
//            log.debug("requestBody中的请求数据，VALUE = [" + ParameterMap.get(name)[0] + "]");
//            log.debug("requestBody中的请求数据，VALUE = [" + URLDecoder.decode(ParameterMap.get(name)[0], "UTF-8") + "]");
            }
        }

        return paramMap;
    }

    @Override
    public ServletInputStream getInputStream() throws IOException {
        return this.inputStreamWrapper;
    }

    @Override
    public BufferedReader getReader() throws IOException {
        return new BufferedReader(new InputStreamReader(this.inputStreamWrapper));
    }
//    @Override
//    public ServletInputStream getInputStream() throws IOException {
//        final ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(body);
//        ServletInputStream servletInputStream = new ServletInputStream() {
//            public boolean isFinished() {
//                return false;
//            }
//            public boolean isReady() {
//                return false;
//            }
//            public void setReadListener(ReadListener readListener) {}
//            public int read() throws IOException {
//                return byteArrayInputStream.read();
//            }
//        };
//        return servletInputStream;
//
//    }



//    @Override
//    public BufferedReader getReader() throws IOException {
//        return new BufferedReader(new InputStreamReader(this.getInputStream()));
//    }


    @EqualsAndHashCode(callSuper = true)
    @Data
    @AllArgsConstructor
    private static class ServletInputStreamWrapper extends ServletInputStream {

        private InputStream inputStream;

        @Override
        public boolean isFinished() {
            return true;
        }

        @Override
        public boolean isReady() {
            return false;
        }

        @Override
        public void setReadListener(ReadListener readListener) {

        }

        @Override
        public int read() throws IOException {
            return this.inputStream.read();
        }
    }
}
