package cn.dolphin.core.util;

import cn.dolphin.core.consts.CharsetConstant;
import cn.dolphin.core.consts.StringConstant;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

/**
 * 编码转换工具类
 */
public class Utf8Util {

    private static final Logger logger = LoggerFactory.getLogger(Utf8Util.class);

    private Utf8Util(){
        // 私有类构造方法
    }

    /**
     * 获取UTF8格式值
     * @param value
     * @return
     */
    public static String getUtf8(String value){
        if (StrUtil.isEmpty(value)) {
            return "";
        }
        try {
            if(java.nio.charset.Charset.forName("ISO-8859-1").newEncoder().canEncode(value))
                value = new String(value.getBytes("ISO-8859-1"),"UTF-8");
            if(java.nio.charset.Charset.forName("GBK").newEncoder().canEncode(value))
                value = new String(value.getBytes("GBK"),"UTF-8");
            if(java.nio.charset.Charset.forName("GB2312").newEncoder().canEncode(value))
                value = new String(value.getBytes("GB2312"),"UTF-8");
        } catch (UnsupportedEncodingException e) {
        }
        return value;
    }

    /**
     *
     * 将iso-8859-1字符串转换为UTF-8，用于解决网络输入乱码
     *
     * @param str
     * @return
     *
     */
    public static final String iso2Utf(String str){
        if (StringUtils.isBlank(str)) {
            return StringConstant.EMPTY;
        }
        try {

            return new String(str.getBytes(CharsetConstant.ISO8859), CharsetConstant.UTF8);
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
            return StringConstant.EMPTY;
        }
    }

    /**
     *
     * 将字符串转换为iso-8859-1，用于解决网络输出乱码
     *
     * @param str
     * @return
     *
     */
    public static final String utf2iso(String str){
        if (StringUtils.isBlank(str)) {
            return StringConstant.EMPTY;
        }
        try {

            return new String(str.getBytes(CharsetConstant.UTF8), CharsetConstant.ISO8859);
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
            return "";
        }
    }

    /**
     * 转换字符串的编码格式,将str字符的编码格式转换成utf-8的格式
     *
     * @param str 需要编码的格式
     * @return 返回utf-8字符编码的str
     */
    public static String parseToUTF8(String str) {
        try {
            return java.net.URLEncoder.encode(str, CharsetConstant.UTF8);
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * ISO-8859-1 to UTF-8
     *
     * @param convertString
     * @return
     */
    public static String convertEncoding(String convertString) {
        String str = convertString;
        if (convertString != null) {
            try {
                // 判断当前字符串的编码格式
                if (convertString.equals(new String(convertString
                        .getBytes("ISO-8859-1"), "ISO-8859-1"))) {
                    str = new String(convertString.getBytes("ISO-8859-1"),
                            "UTF-8");
                }
            } catch (UnsupportedEncodingException e) {
                logger.error(e.toString(), e);
            }
        }
        return str;
    }

    /**
     * 转换编码 ISO-8859-1到GB2312
     *
     * @param text
     * @return
     */
    public static final String ISO2GB(String text) {
        String result = "";
        try {
            result = new String(text.getBytes("ISO-8859-1"), "GB2312");
        } catch (UnsupportedEncodingException ex) {
            result = ex.toString();
        }
        return result;
    }

    /**
     * 转换编码 GB2312到ISO-8859-1
     *
     * @param text
     * @return
     */
    public static final String GB2ISO(String text) {
        String result = "";
        try {
            result = new String(text.getBytes("GB2312"), "ISO-8859-1");
        } catch (UnsupportedEncodingException ex) {
            ex.printStackTrace();
        }
        return result;
    }

    /**
     * Utf8URL编码
     *
     * @param text
     * @return
     */
    public static final String Utf8URLencode(String text) {
        StringBuffer result = new StringBuffer();

        for (int i = 0; i < text.length(); i++) {

            char c = text.charAt(i);
            if (c >= 0 && c <= 255) {
                result.append(c);
            } else {

                byte[] b = new byte[0];
                try {
                    b = Character.toString(c).getBytes("UTF-8");
                } catch (Exception ex) {
                }

                for (int j = 0; j < b.length; j++) {
                    int k = b[j];
                    if (k < 0)
                        k += 256;
                    result.append("%" + Integer.toHexString(k).toUpperCase());
                }

            }
        }

        return result.toString();
    }

    /**
     * Utf8URL解码
     *
     * @param text
     * @return
     */
    public static final String Utf8URLdecode(String text) {
        String result = "";
        int p = 0;

        if (text != null && text.length() > 0) {
            text = text.toLowerCase();
            p = text.indexOf("%e");
            if (p == -1)
                return text;

            while (p != -1) {
                result += text.substring(0, p);
                text = text.substring(p, text.length());
                if (text == "" || text.length() < 9)
                    return result;

                result += CodeToWord(text.substring(0, 9));
                text = text.substring(9, text.length());
                p = text.indexOf("%e");
            }

        }

        return result + text;
    }

    /**
     * utf8URL编码转字符
     *
     * @param text
     * @return
     */
    private static final String CodeToWord(String text) {
        String result;

        if (Utf8codeCheck(text)) {
            byte[] code = new byte[3];
            code[0] = (byte) (Integer.parseInt(text.substring(1, 3), 16) - 256);
            code[1] = (byte) (Integer.parseInt(text.substring(4, 6), 16) - 256);
            code[2] = (byte) (Integer.parseInt(text.substring(7, 9), 16) - 256);
            try {
                result = new String(code, "UTF-8");
            } catch (UnsupportedEncodingException ex) {
                result = null;
            }
        } else {
            result = text;
        }

        return result;
    }

    /**
     * 编码是否有效
     *
     * @param text
     * @return
     */
    private static final boolean Utf8codeCheck(String text) {
        String sign = "";
        if (text.startsWith("%e"))
            for (int i = 0, p = 0; p != -1; i++) {
                p = text.indexOf("%", p);
                if (p != -1)
                    p++;
                sign += p;
            }
        return sign.equals("147-1");
    }

    /**
     * 判断是否Utf8Url编码
     *
     * @param text
     * @return
     */
    public static final boolean isUtf8Url(String text) {
        text = text.toLowerCase();
        int p = text.indexOf("%");
        if (p != -1 && text.length() - p > 9) {
            text = text.substring(p, p + 9);
        }
        return Utf8codeCheck(text);
    }

    /**
     * 编码
     *
     * @param source
     * @param encode
     * @return
     */
    public static String urlEncode(String source, String encode) {
        String result = source;
        try {
            result = URLEncoder.encode(source, encode);
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return result;
    }

    /**
     * 测试
     *
     * @param args
     */
    public static void main(String[] args) {

        String url;

        url = "http://www.google.com/search?hl=zh-CN&newwindow=1&q=%E4%B8%AD%E5%9B%BD%E5%A4%A7%E7%99%BE%E7%A7%91%E5%9C%A8%E7%BA%BF%E5%85%A8%E6%96%87%E6%A3%80%E7%B4%A2&btnG=%E6%90%9C%E7%B4%A2&lr=";
        if (isUtf8Url(url)) {
            System.out.println(Utf8URLdecode(url));
        } else {
            // System.out.println(URLDecoder.decode(url));
        }

        url = "http://www.baidu.com/baidu?word=%D6%D0%B9%FA%B4%F3%B0%D9%BF%C6%D4%DA%CF%DF%C8%AB%CE%C4%BC%EC%CB%F7&tn=myie2dg";
        if (isUtf8Url(url)) {
            System.out.println(Utf8URLdecode(url));
        } else {
            // System.out.println(URLDecoder.decode(url));
        }

    }
}
