package cn.dolphin.core.util;


import java.sql.Timestamp;
import java.util.Date;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

/**
 * 获取系统时间
 * MybatisPlus 自带SystemClock优化System.currentTimeMillis()性能
 * 高并发场景下System.currentTimeMillis()的性能问题的优化
 * System.currentTimeMillis()的调用比new一个普通对象要耗时的多（具体耗时高出多少我还没测试过，有人说是100倍左右）<p>
 * System.currentTimeMillis()之所以慢是因为去跟系统打了一次交道<p>
 * 后台定时更新时钟，JVM退出时，线程自动回收<p>
 * 10亿：43410,206,210.72815533980582%<p>
 * 1亿：4699,29,162.0344827586207%<p>
 * 1000万：480,12,40.0%<p>
 * 100万：50,10,5.0%<p>
 */
@SuppressWarnings("all")
public class SystemClock {

    //频率
    private final long period;

    private final AtomicLong now;

    private SystemClock(long period) {
        this.period = period;
        this.now = new AtomicLong(System.currentTimeMillis());
        this.scheduleClockUpdating();
    }

    private static SystemClock instance() {
        return InstanceHolder.INSTANCE;
    }

    /**
     * 获取当前系统时间
     * @return
     */
    public static long now() {
        return instance().currentTimeMillis();
    }

    /**
     * 获取当前系统时间
     * @return
     */
    public static String nowDateString() {
        return (new Timestamp(instance().currentTimeMillis())).toString();
    }

    /**
     * 获取当前系统时间
     * @return
     */
    public static Timestamp nowDate(){
        return (new Timestamp(instance().currentTimeMillis()));
    }

    /**
     * 获取当前系统时间
     * @param l 累加时间
     * @return
     */
    public static Timestamp nowDate(int l){
        return (new Timestamp(instance().currentTimeMillis()+l));
    }

    private void scheduleClockUpdating() {
        ScheduledExecutorService scheduler = Executors.newSingleThreadScheduledExecutor(new ThreadFactory() {
            public Thread newThread(Runnable runnable) {
                Thread thread = new Thread(runnable, "System Clock");
                thread.setDaemon(true);
                return thread;
            }
        });
        scheduler.scheduleAtFixedRate(new Runnable() {
            public void run() {
                SystemClock.this.now.set(System.currentTimeMillis());
            }
        }, this.period, this.period, TimeUnit.MILLISECONDS);
    }

    private long currentTimeMillis() {
        return this.now.get();
    }

    private static class InstanceHolder {
        public static final SystemClock INSTANCE = new SystemClock(1L);

        private InstanceHolder() {
        }
    }

    public static void main(String[] args) {
        Date date = SystemClock.nowDate();
        System.out.println(date);
        System.out.println(SystemClock.now());
        System.out.println(SystemClock.nowDate());
        System.out.println(SystemClock.nowDate(60 * 60 * 24 * 1000));

    }
}
