package cn.dolphin.core.util;


import cn.dolphin.core.web.util.UrlUtil;
import org.springframework.lang.Nullable;
import org.springframework.util.StringUtils;


import java.io.File;
import java.net.URL;

/**
 * 用来获取各种目录
 */
@SuppressWarnings("all")
public class PathUtil {
    public static final String FILE_PROTOCOL = "file";
    public static final String JAR_PROTOCOL = "jar";
    public static final String ZIP_PROTOCOL = "zip";
    public static final String FILE_PROTOCOL_PREFIX = "file:";
    public static final String JAR_FILE_SEPARATOR = "!/";
    public static final String PATH_SEPERATOR = "/";
    public static final char PATH_SEPERATOR_CHAR = '/';

    public static final String WINDOWS_PATH_SEPERATOR = "\\";

    public static final String SCHEMA_SEPERATOR = ":";

    /**
     * 标准化指定的路径；
     * <p>
     * 标准化的过程包括：
     * 1、清理字符串中的 ".";
     * 2、清除字符串中的 ".."以及路径中相应的上一级路径；(例如：ab/../cd 处理后变为 cd )
     * 2、将 windows 的分隔符"\"替换为标准分隔符"/"；
     * 3、将连续重复的分隔符替换为单个分隔符；
     * 4、去除结尾的分隔符；
     * 5、去除其中的空白字符；
     * <p>
     * 注：以冒号":"分隔的 schema 头将被保留；
     *
     * @param path
     * @return
     */
    public static String standardize(String path){
        path = StringUtils.trimAllWhitespace(path);
        path = StringUtils.cleanPath(path);
        path = cleanRepeatlySeperator(path);
        if (path.endsWith(PATH_SEPERATOR)) {
            return path.substring(0, path.length() - 1);
        }
        return path;
    }

    /**
     * 将指定的路径转换为绝对路径；
     * <p>
     * 方法将检测指定的路径如果既没有以路径分隔符"/"开头，也没有冒号":"分隔的 schema 开头(例如 file:)，
     * <p>
     * 则在开头加上路径分隔符"/"返回；
     * <p>
     * 注：方法不会检测路径是否标准，也不会自动将其标准化；
     *
     * @param path
     * @return
     */
    public static String absolute(String path){
        if (path.startsWith(PATH_SEPERATOR)) {
            return path;
        }
        if (path.indexOf(SCHEMA_SEPERATOR) >= 0) {
            return path;
        }
        return PATH_SEPERATOR + path;
    }

    /**
     * 清除路径中的重复分隔符；
     *
     * @param path
     * @return
     */
    public static String cleanRepeatlySeperator(String path){
        // 去除重复的分隔符；
        String schema = "";
        String pathToProcess = path;
        int index = path.indexOf("://");
        if (index >= 0) {
            schema = path.substring(0, index + 3);
            for (index = index + 3; index < path.length(); index++) {
                if (path.charAt(index) != PATH_SEPERATOR_CHAR) {
                    break;
                }
            }
            pathToProcess = path.substring(index);
        }
        StringBuilder pathToUse = new StringBuilder();
        boolean hit = false;
        char ch;
        for (int i = 0; i < pathToProcess.length(); i++) {
            ch = pathToProcess.charAt(i);
            if (ch == PATH_SEPERATOR_CHAR) {
                if (hit) {
                    continue;
                } else {
                    hit = true;
                }
            } else {
                hit = false;
            }
            pathToUse.append(ch);
        }
        return schema + pathToUse;
    }

    /**
     * @param paths
     */
    public static String concatPaths(String... paths){
        if (paths == null || paths.length == 0) {
            return "";
        }
        StringBuilder path = new StringBuilder();
        for (String p : paths) {
            path.append(p);
            path.append(PATH_SEPERATOR_CHAR);
        }
        return standardize(path.toString());
    }


    /**
     * 获取jar包运行时的当前目录
     *
     * @return {String}
     */
    @Nullable
    public static String getJarPath() {
        try {
            URL url = PathUtil.class.getResource("/").toURI().toURL();
            return PathUtil.toFilePath(url);
        } catch (Exception e) {
            String path = PathUtil.class.getResource("").getPath();
            return new File(path).getParentFile().getParentFile().getAbsolutePath();
        }
    }

    @Nullable
    public static String toFilePath(@Nullable URL url) {
        if (url == null) {
            return null;
        }
        String protocol = url.getProtocol();
        String file = UrlUtil.decodeURL(url.getPath(), Charsets.UTF_8);
        if (FILE_PROTOCOL.equals(protocol)) {
            return new File(file).getParentFile().getParentFile().getAbsolutePath();
        } else if (JAR_PROTOCOL.equals(protocol) || ZIP_PROTOCOL.equals(protocol)) {
            int ipos = file.indexOf(JAR_FILE_SEPARATOR);
            if (ipos > 0) {
                file = file.substring(0, ipos);
            }
            if (file.startsWith(FILE_PROTOCOL_PREFIX)) {
                file = file.substring(FILE_PROTOCOL_PREFIX.length());
            }
            return new File(file).getParentFile().getAbsolutePath();
        }
        return file;
    }

}