package cn.dolphin.core.util;

import cn.dolphin.core.consts.P;
import cn.dolphin.core.consts.StringConstant;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.util.Assert;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.ParseException;


/**
 * 数值处理工具
 */
@SuppressWarnings("all")
public final class NumberUtil extends NumberUtils{

    /**
     * 私有化构造器
     */
    private NumberUtil(){}

    /**
     * 两位小数点的价格
     */
    public final static String PRICE_FORMAT_1 = "#,##0.00";
    /**
     * 两位小数点
     */
    public final static String PRICE_FORMAT_2 = "0.00";
    /**
     * 整数
     */
    public final static String INT = "0";
    /**
     * 百分比 0%：不同于val+%，数值将*100再格式化
     */
    public final static String PERCENT_FORMAT_1 = "0%";
    /**
     * 百分比0.00%：不同于val+%，数值将*100再格式化
     */
    public final static String PERCENT_FORMAT_2 = "0.00%";


    /**
     * 判断类型是否为数字型(byte, short, int, long, float, double)
     *
     * @param clazz 类型
     * @return true为数字型
     */
    public static boolean isNumber(Class<?> clazz)
    {
        return clazz == byte.class || clazz == short.class || clazz == int.class || clazz == long.class || clazz == Byte.class
                || clazz == Short.class || clazz == Integer.class || clazz == Long.class || clazz == float.class || clazz == double.class
                || clazz == Float.class || clazz == Double.class;
    }

    /**
     * 判断类型是否为整型(byte, short, int, long)
     *
     * @param clazz 类型
     * @return true为整型
     */
    public static boolean isInteger(Class<?> clazz)
    {
        return clazz == byte.class || clazz == short.class || clazz == int.class || clazz == long.class || clazz == Byte.class
                || clazz == Short.class || clazz == Integer.class || clazz == Long.class;
    }

    /**
     * 判断类型是否为浮点型(float, double)
     *
     * @param clazz 类型
     * @return true为浮点型
     */
    public static boolean isFloat(Class<?> clazz)
    {
        return clazz == float.class || clazz == double.class || clazz == Float.class || clazz == Double.class;
    }

    /**
     * 把 byte 数组的前4个字节转成 int，不足4个字节按实际的长度算
     *
     * @param bytes byte数组
     * @return byte数组前4个字节所代表的 int 值
     */
    public static int toInt(byte[] bytes)
    {
        int length = bytes.length > 4 ? 4 : bytes.length;
        int result = 0;
        for (int i = 0; i < length; i++)
        {
            result = (result << 8) - Byte.MIN_VALUE + (int)bytes[i];
        }
        return result;
    }

    /**
     * 把 byte 转换为十六进制字符串
     *
     * @param val
     * @return 十六进制字符串，范围是00~ff
     */
    public static String toHexString(byte val)
    {
        return Integer.toHexString(val & 0xff | 0x100).substring(1);
    }

    /**
     * 把 byte 转换为十六进制字符串
     *
     * @param val 十六进制字符串
     * @return byte
     */
    public static byte toByte(String val)
    {
        try
        {
            int i = Integer.parseInt(val, 16);
            return (byte)(i % 0xff);
        }
        catch (NumberFormatException e)
        {
            return 0;
        }
    }

    /**
     * 把 byte 数组转换为十六进制字符串
     *
     * @param bytes byte 数组
     * @return 十六进制字符串，范围是00~ff，每个字节占用两个字符
     */
    public static String toHexString(byte[] bytes)
    {
        StringBuilder sb = new StringBuilder(bytes.length * 2);
        for (byte b : bytes)
        {
            sb.append(Integer.toHexString(b & 0xff | 0x100).substring(1));
        }
        return sb.toString();
    }

    /**
     * 把十六进制字符串转化为 byte 数组
     *
     * @param val 十六进制字符串
     * @return byte 数组，如果有非法字符，用 0 代替
     */
    public static byte[] toBytes(String val)
    {
        int length = val.length();
        boolean mod2 = (length & 1) == 0; // 模2
        int size = mod2 ? length / 2 : length / 2 + 1;
        int j = mod2 ? 2 : 1;
        int k = j;
        byte[] bytes = new byte[size];
        String s = val.substring(0, j);
        bytes[0] = toByte(s);

        j += k;
        for (int i = 1; i < bytes.length; i++, j += 2)
        {
            s = val.substring(j - 2, j);
            bytes[i] = toByte(s);
        }
        return bytes;
    }

//    /**
//     * 把 long 转换化成 length 长度的字符串
//     *
//     * @param val long值
//     * @param length 字符串的长度
//     * @return 字符串，不足前面补 0，超出则是原 long 的长度
//     */
//    public static String toString(long val, int length)
//    {
//        String strVal = String.valueOf(val);
//        int zeroLen = length - strVal.length() < 0 ? 0 : length - strVal.length();
//        StringBuilder sb = new StringBuilder();
//        for (int i = 0; i < zeroLen; i++)
//        {
//            sb.append('0');
//        }
//        sb.append(strVal);
//        return sb.toString();
//    }

    /**
     * 把 int 转换为十六进制字符串
     *
     * @param val int值
     * @return 十六进制字符串，范围是0000~ffff
     */
    public static String toHexString(short val)
    {
        return Integer.toHexString(val & 0xffff | 0x10000).substring(1);
    }

    /**
     * 把 int 转换为十六进制字符串
     *
     * @param val int值
     * @return 十六进制字符串，范围是00000000~ffffffff
     */
    public static String toHexString(int val)
    {
        String tmp = Integer.toHexString(val);
        StringBuilder sb = new StringBuilder("00000000");
        sb.replace(8 - tmp.length(), 8, tmp);
        return sb.toString();
    }

    /**
     * 把 long 转换为十六进制字符串
     *
     * @param val long值
     * @return 十六进制字符串，范围是0000000000000000~ffffffffffffffff
     */
    public static String toHexString(long val)
    {
        String tmp = Long.toHexString(val);
        StringBuilder sb = new StringBuilder("0000000000000000");
        sb.replace(16 - tmp.length(), 16, tmp);
        return sb.toString();
    }

    /**
     * 格式化字节，把字节转化为 B, KB, MB, GB 等表示法
     *
     * @param size 字节数
     * @return 格式后字符串
     */
    public static String byteFormat(long size)
    {
        String[] a = { "B", "KB", "MB", "GB", "TB", "PB" };
        double val = size;
        int pos = 0;
        while (val >= 1024.0d)
        {
            val /= 1024;
            pos++;
        }
        String s = String.format("%.2f%s", val, a[pos]);
        return s;
    }

    private static char[] cnArr = { '一', '二', '三', '四', '五', '六', '七', '八', '九' };
    private static char[] chArr = { '十', '百', '千', '万', '亿' };
    /**
     * 中文数字转化为阿拉伯数字
     *
     * @param chn 中文数字
     * @return 阿拉伯数字
     */
    public static int chnToInt(String chn)
    {
        Assert.hasText(chn, "中文数字不能为空");
        if (NumberUtils.isDigits(chn))
        {
//			String n = StringUtils.stripStart(chn, "0");
//			System.out.println(n);
            return Integer.parseInt(chn);
        }

        int result = 0;
        int temp = 1;// 存放一个单位的数字如：十万
        int count = 0;// 判断是否有chArr

        for (int i = 0; i < chn.length(); i++)
        {
            boolean b = true; // 判断是否是chArr
            char c = chn.charAt(i);
            for (int j = 0; j < cnArr.length; j++) // 非单位，即数字
            {
                if (c == cnArr[j])
                {
                    if (0 != count) // 添加下一个单位之前，先把上一个单位值添加到结果中
                    {
                        result += temp;
                        temp = 1;
                        count = 0;
                    }
                    // 下标+1，就是对应的值
                    temp = j + 1;
                    b = false;
                    break;
                }
            }
            if (b) // 单位{'十','百','千','万','亿'}
            {
                for (int j = 0; j < chArr.length; j++)
                {
                    if (c == chArr[j])
                    {
                        switch (j)
                        {
                            case 0:
                                temp *= 10;
                                break;
                            case 1:
                                temp *= 100;
                                break;
                            case 2:
                                temp *= 1000;
                                break;
                            case 3:
                                temp *= 10000;
                                break;
                            case 4:
                                temp *= 100000000;
                                break;
                            default:
                                break;
                        }
                        count++;
                    }
                }
            }
            if (i == chn.length() - 1) // 遍历到最后一个字符
            {
                result += temp;
            }
        }
        return result;

    }


    /**
     * 判断数值为null或为0
     *
     * @param num
     * @return
     */
    public static final boolean isBlank(Number num){
        if (num == null || num.intValue() == 0) {
            return true;
        }
        return false;
    }

    /**
     *
     * 格式化数值
     *
     * @param src 源数值如果为空，按0计算
     * @param format
     * @return 格式化字符串
     *
     */
    public final static String format(Number src, String format){
        if (src == null) {
            src = 0;
        }
        return new DecimalFormat(format).format(src);
    }

    /**
     *
     * 格式化数值
     *
     * @param src 源数值如果为空，按0计算
     * @param format
     * @return 格式化字符串
     *
     */
    public final static String format(String src, String format){
        return format(DoubleUtil.valueOf(src),format);
    }

    /**
     * 转换数据为字符串（解决科学计数法问题）
     *
     * @param num
     * @return
     *
     */
    public final static String toString(String num) {
        if (StringUtils.isBlank(num)) {
            num = StringConstant.N0;
        }
        return new BigDecimal(num).toPlainString();
    }
    /**
     * 转换数据为字符串（解决科学计数法问题）
     *
     * @param num
     * @return
     *
     */
    public final static String toString(double num) {
        return new BigDecimal(num).toPlainString();
    }

    /**
     * 转换数据为字符串（解决科学计数法问题）
     *
     * @param num
     * @return
     *
     */
    public final static String toString(long num) {
        return new BigDecimal(num).toPlainString();
    }
    /**
     * 转换数据为字符串（解决科学计数法问题）
     *
     * @param num
     * @param scale	小数位
     * @return
     *
     */
    public final static String toString(String num,int scale) {
        if (StringUtils.isBlank(num)) {
            num = StringConstant.N0;
        }
        return new BigDecimal(num).setScale(scale, BigDecimal.ROUND_HALF_UP).toPlainString();
    }
    /**
     * 转换数据为字符串（解决科学计数法问题）
     *
     * @param num
     * @param scale	小数位
     * @return
     *
     */
    public final static String toString(double num,int scale) {
        return new BigDecimal(num).setScale(scale, BigDecimal.ROUND_HALF_UP).toPlainString();
    }
    /**
     * 转换数据为字符串（解决科学计数法问题）
     *
     * @param num
     * @param scale	小数位
     * @return
     *
     */
    public final static String toString(long num,int scale) {
        return new BigDecimal(num).setScale(scale, BigDecimal.ROUND_HALF_UP).toPlainString();
    }

    /**
     * 转换数据为float
     * @param f
     * @return
     */
    public static float numFormat2(float f) {
        return new BigDecimal(f).setScale(2, BigDecimal.ROUND_HALF_UP).floatValue();
    }

    /**
     * 转换数据为double
     * @param d
     * @return
     */
    public static double numFormat2(double d) {
        return new BigDecimal(d).setScale(2, BigDecimal.ROUND_HALF_UP).doubleValue();
    }

    /**
     * 转换数据为long
     * @param l
     * @return
     */
    public static long numFormat2(long l) {
        return new BigDecimal(l).setScale(0, BigDecimal.ROUND_HALF_UP).longValue();
    }

    /**
     * 转换数据为int
     * @param i
     * @return
     */
    public static int numFormat2(int i) {
        return new BigDecimal(i).setScale(0, BigDecimal.ROUND_HALF_UP).intValue();
    }

    public static boolean isNumber(char ch) {
        return ch > 47 && ch < 58;
    }

    /**
     * 百分号
     */
    public static String percent(Number number) {
        return new DecimalFormat(P.NUMBER_FORMAT_PERCENT).format(number);
    }

    /**
     * 四舍五入，保留2位小数
     */
    public static BigDecimal roundHalfUp2(Number number) {
        return roundHalfUp(2, number);
    }

    /**
     * 四舍五入
     */
    public static BigDecimal roundHalfUp(int scale, Number number) {
        return ConvertUtil.toBigDecimal(number).setScale(scale, BigDecimal.ROUND_HALF_UP);
    }

    /**
     * 千分号
     */
    public static String permil(Number number) {
        return new DecimalFormat(P.NUMBER_FORMAT_PERMIL).format(number);
    }

    /**
     * 格式化货币，小数点保留分，四舍五入
     */
    public static String money(Number number) {
        return new DecimalFormat(P.NUMBER_FORMAT_MONEY).format(number);
    }

    /**
     * 转换为带逗号的英国数字
     */
    public static String ukNumber(Number number){
        return new DecimalFormat(P.NUMBER_FORMAT_UK).format(number);
    }

    /**
     * 转换为带逗号的英国数字
     */
    public static Number parseChnNumber(String number) throws ParseException{
        return new DecimalFormat(P.NUMBER_FORMAT_UK).parse(number);
    }
}
