package cn.dolphin.core.util;

import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.RoundingMode;
import java.text.DecimalFormat;

import static java.math.RoundingMode.HALF_UP;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;


/**
 * 是所有数值格式的抽象基类,此类提供格式化和解析数值的接口
 */
public final class NumberFormatUtil{

    /** The Constant LOGGER. */
    private static final Logger LOGGER = LoggerFactory.getLogger(NumberFormatUtil.class);

    /** Don't let anyone instantiate this class. */
    private NumberFormatUtil(){
        //AssertionError不是必须的. 但它可以避免不小心在类的内部调用构造器. 保证该类在任何情况下都不会被实例化.
        //see 《Effective Java》 2nd
        throw new AssertionError("No " + getClass().getName() + " instances for you!");
    }

    //---------------------------------------------------------------

    /**
     * 将 {@link Number} 使用 {@link RoundingMode} numberPattern格式化.
     *
     * 示例:
     * NumberFormatUtil.format(toBigDecimal(1.15), "#####.#",null)     =   1.2
     * NumberFormatUtil.format(toBigDecimal(1.25), "#####.#",null)     =   1.3
     * NumberFormatUtil.format(toBigDecimal(1.251), "#####.#",null)    =   1.3
     *
     * NumberFormatUtil.format(toBigDecimal(-1.15), "#####.#",null)    =   -1.2
     * NumberFormatUtil.format(toBigDecimal(-1.25), "#####.#",null)    =   -1.3
     * NumberFormatUtil.format(toBigDecimal(-1.251), "#####.#",null)   =   -1.3
     *
     *
     * NumberFormatUtil.format(toBigDecimal(1.15), "#####.#", RoundingMode.HALF_EVEN)     =   1.2
     * NumberFormatUtil.format(toBigDecimal(1.25), "#####.#", RoundingMode.HALF_EVEN)     =   1.2
     * NumberFormatUtil.format(toBigDecimal(1.251), "#####.#", RoundingMode.HALF_EVEN)    =   1.3
     *
     * NumberFormatUtil.format(toBigDecimal(-1.15), "#####.#", RoundingMode.HALF_EVEN)    =   -1.2
     * NumberFormatUtil.format(toBigDecimal(-1.25), "#####.#", RoundingMode.HALF_EVEN)    =   -1.2
     * NumberFormatUtil.format(toBigDecimal(-1.251), "#####.#", RoundingMode.HALF_EVEN)   =   -1.3
     *
     * 关于参数 :
     * 请尽量传递Integer,Long,BigDecimal,而不要使用 float,double等浮点类型,否则可能结果不准确,特别是jdk8以下的版本
     *
     * @param value
     *            the value
     * @param numberPattern
     *            格式化数字格式,可以参见或者使用NumberPattern
     * @param roundingMode
     *            舍入模式{@link RoundingMode},如果 为null,使用常用的 {@link RoundingMode#HALF_UP}
     * @return 如果 value 是null,抛出 {@link NullPointerException}
     *         如果 numberPattern 是null,抛出 {@link NullPointerException}
     *         如果 numberPattern 是blank,抛出 {@link IllegalArgumentException}
     */
    public static String format(Number value,String numberPattern,RoundingMode roundingMode){
        Validate.notNull(value, "value can't be null!");
        Validate.notBlank(numberPattern, "numberPattern can't be null!");

        //该构造方法内部 调用了applyPattern(pattern, false)
        DecimalFormat decimalFormat = new DecimalFormat(numberPattern);
        decimalFormat.setRoundingMode(defaultIfNull(roundingMode, HALF_UP));

        String result = decimalFormat.format(value);

        if (LOGGER.isTraceEnabled()){
            String message = "input:[{}],with:[{}]=[{}],localizedPattern:[{}]";
            LOGGER.trace(message, value, numberPattern, result, decimalFormat.toLocalizedPattern());
        }
        return result;
    }
}