package cn.dolphin.core.util;

import cn.dolphin.core.consts.StringConstant;
import cn.dolphin.core.exception.ParamRuntimeException;
import cn.dolphin.core.regex.NumberRegex;
import cn.dolphin.core.regex.RegexUtil;
import org.apache.commons.lang3.StringUtils;

/**
 * 金额相关工具
 */
@SuppressWarnings("all")
public final class MoneyUtil {

    /**
     * 数字型金额转换为汉字金额(最多处理千万亿级)
     *
     * @param money 数字型金额：单位为分
     * @param isBig 是否为大写金额汉字：true 壹,false 一
     * @return
     *
     */
    public static final String toChinese(String money,boolean isBig) {
        //特殊值处理
        if (StringUtils.isBlank(money) || StringConstant.N0.equals(money)) {
            return "零元";
        }
        //校验
        boolean matcher = RegexUtil.matcher(money, NumberRegex.POSITIVE_INTEGER);
        if (!matcher) {
            throw new ParamRuntimeException("金额必须为正整数");
        }else if (money.length() > 18) {
            throw new ParamRuntimeException("最大转换千万亿级的金额");
        }
        //汉字
        char[] hunit = null;
        char[] vunit = null;
        char[] digit = null;
        if (isBig) {
            hunit = new char[]{ '拾', '佰', '仟' };
            vunit = new char[]{ '万', '亿', '万' };
            digit = new char[]{ '零', '壹', '贰', '叁', '肆', '伍', '陆', '柒', '捌', '玖' };
        }else{
            hunit = new char[]{ '十', '百', '千' };
            vunit = new char[]{ '万', '亿', '万' };
            digit = new char[]{ '零', '一', '二', '三', '四', '五', '六', '七', '八', '九' };
        }
        //小额处理
        if (money.length() == 2) {
            return digit[money.charAt(0)-'0'] + "角" + digit[money.charAt(1)-'0'] + "分";
        }else if (money.length() == 1) {
            return digit[money.charAt(0)-'0'] + "分";
        }
        // 分割元和分
        String head = money.substring(0, money.length() - 2);
        String rail = money.substring(money.length() - 2);
        String prefix = StringConstant.EMPTY;
        String suffix = StringConstant.EMPTY;
        // 处理分
        if (rail.equals("00")) {
            suffix = "整";
        } else {
            suffix = digit[rail.charAt(0) - '0'] + "角" + digit[rail.charAt(1) - '0'] + "分";
        }
        // 处理元
        char[] chDig = head.toCharArray();
        // 标志当前位的上一位是否为有效0位
        boolean preZero = false;
        // 连续出现0的次数
        byte zeroSerNum = 0;
        // 循环处理每个数字
        for (int i = 0; i < chDig.length; i++) {
            // 取段内位置
            int idx = (chDig.length - i - 1) % 4;
            // 取段位置
            int vidx = (chDig.length - i - 1) / 4;
            // 如果当前字符是0
            if (chDig[i] == '0') {
                preZero = true;
                // 连续0次数递增
                zeroSerNum++;
                if (idx == 0 && vidx > 0 && zeroSerNum < 4) {
                    prefix += vunit[vidx - 1];
                    preZero = false;
                }
            } else {
                // 连续0次数清零
                zeroSerNum = 0;
                // 上一位为有效0位
                if (preZero) {
                    prefix += digit[0];
                    preZero = false;
                }
                // 转化该数字表示
                prefix += digit[chDig[i] - '0'];
                if (idx > 0)
                    prefix += hunit[idx - 1];
                if (idx == 0 && vidx > 0) {
                    // 段结束位置应该加上段名如万,亿
                    prefix += vunit[vidx - 1];

                }
            }
        }
        return prefix+"元"+suffix;
    }

    /**
     *
     * 数字型金额转换为汉字金额（最多处理千万亿级）
     *
     * @param money 数字型金额：单位为分
     * @param isBig 是否为大写金额汉字：true 壹,false 一
     * @return
     *
     */
    public static final String toChinese(long money,boolean isBig) {
        return toChinese(String.valueOf(money), isBig);
    }

    /**
     *
     * 数字型金额转换为汉字金额（最多处理千万亿级；因为大金额会用科学计数法导致偏移，所以万亿级以上小数位有可能出现差错，应慎用）
     *
     * @param money 数字型金额：单位为元
     * @param isBig 是否为大写金额汉字：true 壹,false 一
     * @return
     *
     */
    public static final String toChinese(double money,boolean isBig) {
        return toChinese((long)(money * 100), isBig);
    }

    /**
     * 分转元
     *
     * @param fen
     * @return
     *
     */
    public static final String toYuan(String fen){
        return NumberUtil.format(fen,NumberUtil.PRICE_FORMAT_2);
    }

    /**
     * 元转分
     *
     * @param yuan
     * @return
     *
     */
    public static final Long toFen(String yuan){
        // 直接转换方式（处理大额时会报错） --> (long)(DoubleUtil.valueOf(yuan) * 100);
        // 字符串切割方式
        String[] arr = yuan.split("\\.");
        if (arr.length == 1) {
            yuan = arr[0] +StringConstant.N00;
        }else{
            String fen = arr[1];
            if (fen.length() == 1) {
                yuan = arr[0] + fen +StringConstant.N0;
            } else if (fen.length() == 2) {
                yuan = arr[0] + fen;
            } else{
                yuan = arr[0] + fen.substring(0,2);
            }
        }
        return Long.valueOf(yuan);
    }


}
