package cn.dolphin.core.util;


import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Map;

/**
 * 加密工具类
 */
public class MD5Util {


    private final static Logger logger = LoggerFactory.getLogger(MD5Util.class);

    /** * 16进制字符集 */
    private static final String hexDigits[] = { "0", "1", "2", "3", "4", "5", "6", "7", "8", "9", "a", "b", "c", "d", "e", "f" };
    private static final char HEX_DIGITS[] = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F' };
    /** * 指定算法为MD5的MessageDigest */
    private static MessageDigest messageDigest = null;

    private static byte[] lock = new byte[1];

    /** * 初始化messageDigest的加密算法为MD5 */
    static {
        try {
            messageDigest = MessageDigest.getInstance("MD5");
        } catch (NoSuchAlgorithmException ne) {
//            e.printStackTrace();
            logger.error("NoSuchAlgorithmException: md5", ne);
        }
    }

    /**
     * * 获取文件的MD5值
     *
     * @param file
     *            目标文件
     *
     * @return MD5字符串
     */
    public static String getFileMD5String(File file) {
        String ret = "";
        synchronized (lock) {
            FileInputStream in = null;
            FileChannel ch = null;
            try {
                in = new FileInputStream(file);
                ch = in.getChannel();
                ByteBuffer byteBuffer = ch.map(FileChannel.MapMode.READ_ONLY, 0, file.length());
                messageDigest.update(byteBuffer);
                ret = bytesToHex(messageDigest.digest());
            } catch (IOException e) {
                e.printStackTrace();

            } finally {
                if (in != null) {
                    try {
                        in.close();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }
                if (ch != null) {
                    try {
                        ch.close();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }
            }
        }
        return ret;
    }

    /**
     * ** * 获取文件的MD5值
     *
     * @param fileName
     *            目标文件的完整名称
     *
     * @return MD5字符串
     */
    public static String getFileMD5String(String fileName) {
        return getFileMD5String(new File(fileName));
    }

    /**
     * * MD5加密字符串
     *
     * @param str
     *            目标字符串
     *
     * @return MD5加密后的字符串
     */
    public static String getMD5String(String str) {
        return getMD5String(str.getBytes());
    }

    /**
     * 求一个字符串的md5值
     * @param target 字符串
     * @return md5 value
     */
    public static String md5(String target) {
        return DigestUtils.md5Hex(target);
    }

    /**
     * * MD5加密以byte数组表示的字符串
     *
     * @param bytes
     *            目标byte数组
     *
     * @return MD5加密后的字符串
     */
    public static String getMD5String(byte[] bytes) {
        String result = null;
        synchronized (lock) {
            messageDigest.update(bytes);
            result = bytesToHex(messageDigest.digest());
        }
        return result;
    }

    /**
     * MD5 签名
     * @param data 源数据（String）
     * @return 数据签名(32位16进制字符串)
     */
    public static String digest(byte[] data) {
        return DigestUtils.md5Hex(data);
    }

    /**
     * MD5 签名
     * @param data 源数据（String）
     * @return 数据签名(32位16进制字符串)
     */
    public static String digest(String data) {
        return DigestUtils.md5Hex(data);
    }

    /**
     * MD5 签名
     * @param data 源数据（InputStream）
     * @return 数据签名(32位16进制字符串)
     */
    public static String digest(InputStream data) {
        try {
            return DigestUtils.md5Hex(data);
        } catch (IOException e) {
            throw new RuntimeException("MD5签名失败",e);
        }
    }



    /**
     * ** * 校验密码与其MD5是否一致
     *
     * @param pwd
     *            密码字符串
     *
     * @param md5
     *            基准MD5值
     *
     * @return 检验结果
     */
    public static boolean checkPassword(String pwd, String md5) {
        return getMD5String(pwd).equalsIgnoreCase(md5);
    }

    /**
     * * 校验密码与其MD5是否一致
     *
     * @param pwd
     *            以字符数组表示的密码
     *
     * @param md5
     *            基准MD5值
     *
     * @return 检验结果
     */
    public static boolean checkPassword(char[] pwd, String md5) {
        return checkPassword(new String(pwd), md5);

    }

    /**
     * * 检验文件的MD5值
     *
     * @param file
     *            目标文件
     *
     * @param md5
     *            基准MD5值
     *
     * @return 检验结果
     */
    public static boolean checkFileMD5(File file, String md5) {
        return getFileMD5String(file).equalsIgnoreCase(md5);

    }

    /**
     * * 检验文件的MD5值
     *
     * @param fileName
     *            目标文件的完整名称
     *
     * @param md5
     *            基准MD5值
     *
     * @return 检验结果
     */
    public static boolean checkFileMD5(String fileName, String md5) {
        return checkFileMD5(new File(fileName), md5);

    }

    /**
     * * 将字节数组转换成16进制字符串
     *
     * @param bytes
     *            目标字节数组
     *
     * @return 转换结果
     */
    public static String bytesToHex(byte bytes[]) {
        return bytesToHex(bytes, 0, bytes.length);

    }

    /**
     * * 将字节数组中指定区间的子数组转换成16进制字符串
     *
     * @param bytes
     *            目标字节数组
     *
     * @param start
     *            起始位置（包括该位置）
     *
     * @param end
     *            结束位置（不包括该位置）
     *
     * @return 转换结果
     */
    public static String bytesToHex(byte bytes[], int start, int end) {
        StringBuilder sb = new StringBuilder();
        for (int i = start; i < start + end; i++) {
            sb.append(byteToHex(bytes[i]));
        }
        return sb.toString();

    }

    /**
     * * 将单个字节码转换成16进制字符串
     *
     * @param bt
     *            目标字节
     *
     * @return 转换结果
     */
    public static String byteToHex(byte bt) {
        return HEX_DIGITS[(bt & 0xf0) >> 4] + "" + HEX_DIGITS[bt & 0xf];

    }

    /**
     * 获取该输入流的MD5值
     *
     * @param is
     * @return
     * @throws NoSuchAlgorithmException
     * @throws IOException
     */
    public static String md5(InputStream is) {
        try {
            StringBuffer md5 = new StringBuffer();
            MessageDigest md = MessageDigest.getInstance("MD5");
            byte[] dataBytes = new byte[1024];

            int nread = 0;
            while ((nread = is.read(dataBytes)) != -1) {
                md.update(dataBytes, 0, nread);
            };
            byte[] mdbytes = md.digest();

            // convert the byte to hex format
            for (int i = 0; i < mdbytes.length; i++) {
                md5.append(Integer.toString((mdbytes[i] & 0xff) + 0x100, 16).substring(1));
            }
            return md5.toString();
        }catch (Exception e){
            logger.error("md5 InputStream " + " failed:" + e.getMessage());
            return null;
        }

    }

    /**
     * 将url参数转换成map
     * @param param aa=11&bb=22&cc=33
     * @return
     */
    public static Map<String, Object> getUrlParams(String param) {
        Map<String, Object> map = new HashMap<String, Object>(0);
        if (StringUtils.isBlank(param)) {
            return map;
        }
        String[] params = param.split("&");
        for (int i = 0; i < params.length; i++) {
            String[] p = params[i].split("=");
            if (p.length == 2) {
                map.put(p[0], p[1]);
            }
        }
        return map;
    }

    /**
     * 将map转换成url
     * @param map
     * @return
     */
    public static String getUrlParamsByMap(Map<String, Object> map) {
        if (map == null) {
            return "";
        }
        StringBuffer sb = new StringBuffer();
        for (Map.Entry<String, Object> entry : map.entrySet()) {
            sb.append(entry.getKey() + "=" + entry.getValue());
            sb.append("&");
        }
        String s = sb.toString();
        if (s.endsWith("&")) {
            s = org.apache.commons.lang3.StringUtils.substringBeforeLast(s, "&");
        }
        return s;
    }


    private static String byteArrayToHexString(byte b[]) {
        StringBuffer resultSb = new StringBuffer();
        for (int i = 0; i < b.length; i++)
            resultSb.append(byteToHexString(b[i]));

        return resultSb.toString();
    }

    private static String byteToHexString(byte b) {
        int n = b;
        if (n < 0)
            n += 256;
        int d1 = n / 16;
        int d2 = n % 16;
        return hexDigits[d1] + hexDigits[d2];
    }

    public static String MD5Encode(String origin, String charsetname) {
        String resultString = null;
        try {
            resultString = new String(origin);
            MessageDigest md = MessageDigest.getInstance("MD5");
            if (charsetname == null || "".equals(charsetname))
                resultString = byteArrayToHexString(md.digest(resultString.getBytes()));
            else
                resultString = byteArrayToHexString(md.digest(resultString.getBytes(charsetname)));
        } catch (Exception exception) {
        }
        return resultString;
    }


    /**
     * MD5密码加密（一次）
     * @param pasw
     * @return
     */
    public static String encryptPasswordOnce(String pasw){
        try {
            // 得到一个信息摘要器
            MessageDigest digest = MessageDigest.getInstance("md5");
            byte[] result = digest.digest(pasw.getBytes());
            StringBuffer buffer = new StringBuffer();
            // 把没一个byte 做一个与运算 0xff;
            for (byte b : result) {
                // 与运算
                int number = b & 0xff;// 加盐
                String str = Integer.toHexString(number);
                if (str.length() == 1) {
                    buffer.append("0");
                }
                buffer.append(str);
            }

            // 标准的md5加密后的结果
            return buffer.toString();
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
            return "";
        }
    }

    /**
     * MD5加密密码两次
     * @param pasw
     * @return
     */
    public static String encryptPasswordTwice(String pasw){
        return encryptPasswordOnce(encryptPasswordOnce(pasw));
    }



    /**
     * 加密
     * @param code
     * @return
     */
    public static String encode(String code) {
        if (code == null || "".equals(code.trim())) {
            throw new RuntimeException("the code which will be encoded with 'MD5' must not be null or empty!");
        }
        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            md.update(code.getBytes("utf-8")); //MD5加密算法只是对字符数组而不是字符串进行加密计算，得到要加密的对象
            byte[] bs = md.digest();   //进行加密运算并返回字符数组
            StringBuilder sb = new StringBuilder();
            for (byte b : bs) {    //字节数组转换成十六进制字符串，形成最终的密文
                int v = b & 0xff;
                if (v < 16) {
                    sb.append(0);
                }
                sb.append(Integer.toHexString(v));
            }
            return sb.toString();
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return null;
    }

}
