package cn.dolphin.core.util;


import java.util.Collection;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;


/**
 * 线程池工具类
 */
@SuppressWarnings("all")
public class ExecutorUtil {

    private ExecutorUtil() {
        throw new UnsupportedOperationException("EmptyUtil can not be instantiated");
    }

    /**
     * 基于单线程处理任务
     *
     * @param data     需要处理的数据
     * @param consumer 数据处理函数
     * @param <T>      数据类型
     */
    public static <T extends Object> void executeBySingleThread(Collection<T> data, Consumer<T> consumer) throws InterruptedException {
        if (EmptyUtil.isEmpty(data)) return;

        ExecutorService service = Executors.newSingleThreadExecutor();
        data.forEach(t -> service.execute(() -> consumer.accept(t)));
        service.shutdown();
        service.awaitTermination(1, TimeUnit.DAYS); //等待一天，若线程还未停止，则停止继续加入任务，然后等执行中的任务执行结束后关闭线程池
    }

    /**
     * 基于缓存线程处理任务
     *
     * @param data     需要处理的数据
     * @param consumer 数据处理函数
     * @param <T>      数据类型
     */
    public static <T extends Object> void executeByCacheThread(Collection<T> data, Consumer<T> consumer) throws InterruptedException {
        if (EmptyUtil.isEmpty(data)) return;

        ExecutorService service = Executors.newCachedThreadPool();
        data.forEach(t -> service.execute(() -> consumer.accept(t)));
        service.shutdown();
        service.awaitTermination(1, TimeUnit.DAYS);
    }

    /**
     * 基于指定线程数线程处理任务
     *
     * @param data     需要处理的数据
     * @param consumer 数据处理函数
     * @param <T>      数据类型
     */
    public static <T extends Object> void executeByFixedThread(Collection<T> data, Consumer<T> consumer) throws InterruptedException {
        executeByFixedThread(Runtime.getRuntime().availableProcessors(), data, consumer);
    }

    /**
     * 基于指定线程数线程处理任务
     *
     * @param data     需要处理的数据
     * @param consumer 数据处理函数
     * @param <T>      数据类型
     */
    public static <T extends Object> void executeByFixedThread(int threadCount, Collection<T> data, Consumer<T> consumer) throws InterruptedException {
        if (EmptyUtil.isEmpty(data)) return;

        ExecutorService service = Executors.newFixedThreadPool(threadCount == 0 ? Runtime.getRuntime().availableProcessors() : threadCount);
        data.forEach(t -> service.execute(() -> consumer.accept(t)));
        service.shutdown();
        service.awaitTermination(1, TimeUnit.DAYS);
    }
}
