package cn.dolphin.core.util;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigDecimal;

/**
 * Double工具
 */
public final class DoubleUtil {

    private static final Logger logger = LoggerFactory.getLogger(DoubleUtil.class);

    //默认除法运算精度
    private static final int DEF_DIV_SCALE = 10;
    //这个类不能实例化
    private DoubleUtil(){
    }

    /**
     * 提供精确的加法运算。
     * @param v1 被加数
     * @param v2 加数
     * @return 两个参数的和
     */
    public static double add(double v1,double v2){
        BigDecimal b1 = new BigDecimal(Double.toString(v1));
        BigDecimal b2 = new BigDecimal(Double.toString(v2));
        return b1.add(b2).doubleValue();
    }

    /**
     * 提供精确的减法运算。
     * @param v1 被减数
     * @param v2 减数
     * @return 两个参数的差
     */
    public static double sub(double v1,double v2){
        BigDecimal b1 = new BigDecimal(Double.toString(v1));
        BigDecimal b2 = new BigDecimal(Double.toString(v2));
        return b1.subtract(b2).doubleValue();
    }

    /**
     * 提供精确的乘法运算。
     * @param v1 被乘数
     * @param v2 乘数
     * @return 两个参数的积
     */
    public static double mul(double v1,double v2){
        BigDecimal b1 = new BigDecimal(Double.toString(v1));
        BigDecimal b2 = new BigDecimal(Double.toString(v2));
        return b1.multiply(b2).doubleValue();
    }

    /**
     * 提供（相对）精确的除法运算，当发生除不尽的情况时，精确到
     * 小数点以后10位，以后的数字四舍五入。
     * @param v1 被除数
     * @param v2 除数
     * @return 两个参数的商
     */
    public static double div(double v1,double v2){
        return div(v1,v2,DEF_DIV_SCALE);
    }

    /**
     * 提供（相对）精确的除法运算。当发生除不尽的情况时，由scale参数指
     * 定精度，以后的数字四舍五入。
     * @param v1 被除数
     * @param v2 除数
     * @param scale 表示表示需要精确到小数点以后几位。
     * @return 两个参数的商
     */
    public static double div(double v1,double v2,int scale){
        if(scale<0){
            throw new IllegalArgumentException(
                    "The scale must be a positive integer or zero");
        }
        BigDecimal b1 = new BigDecimal(Double.toString(v1));
        BigDecimal b2 = new BigDecimal(Double.toString(v2));
        return b1.divide(b2,scale,BigDecimal.ROUND_HALF_UP).doubleValue();
    }

    /**
     * 提供精确的小数位四舍五入处理。
     * @param v 需要四舍五入的数字
     * @param scale 小数点后保留几位
     * @return 四舍五入后的结果
     */
    public static double round(double v,int scale){
        if(scale<0){
            throw new IllegalArgumentException(
                    "The scale must be a positive integer or zero");
        }
        BigDecimal b = new BigDecimal(Double.toString(v));
        BigDecimal one = new BigDecimal("1");
        return b.divide(one,scale,BigDecimal.ROUND_HALF_UP).doubleValue();
    }


    /**
     * null转0
     *
     * @param o
     * @return
     *
     */
    public static final Double toEmpty(Double o) {
        return o == null ? 0 : o;
    }
    /**
     * 转换字符串为Double
     * @param num
     * @return 如果字符串为空则返回null，如果字符串非数值则异常
     *
     */
    public static final Double valueOf(String num){
        //如果为空，返回null
        if (StrUtil.isBlank(num)) {
            return null;
        }
        //非空转换，格式错误抛异常
        return Double.valueOf(num.trim());
    }
    /**
     * 转换字符串为Double
     * @param num
     * @return 如果字符串为空则返回0，如果字符串非数值则异常
     *
     */
    public static final Double valueOfZero(String num){
        //如果为空，返回0
        if (StrUtil.isBlank(num)) {
            return 0D;
        }
        //非空转换，格式错误抛异常
        return Double.valueOf(num.trim());
    }
    /**
     * 格式化double类型数据，保留2位小数
     * @param num
     * @return
     */
    public static Double fmtDouble(Double num) {
        if (num == null || 0 == num) {
            logger.warn("num is null == {}",num);
            return 0.00D;
        }
        try {
            return new BigDecimal(num).setScale(2,BigDecimal.ROUND_HALF_UP).doubleValue();
        } catch (Exception e) {
            logger.warn("fmt double error : num = {} , error = {}",num,e.getMessage());
            return 0.00D;
        }
    }

    /**
     * 高精度比较Double
     * @param x
     * @param y
     * @return
     */
    public static boolean compareDouble(Double x,Double y) {

        if (null == x || null == y)
        {
            return false;
        }
        BigDecimal data1 = new BigDecimal(x);
        BigDecimal data2 = new BigDecimal(y);
        if (data1.compareTo(data2) == 0)
        {
            return true;
        }
        return false;
    }

}
