package cn.dolphin.core.util;

import cn.dolphin.core.bean.BeanFieldCache;
import cn.dolphin.core.bean.IntrospectionCache;
import cn.dolphin.core.date.DateUtil;
import cn.dolphin.core.enums.DateEnum;
import cn.dolphin.core.exception.UtilsRuntimeException;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.springframework.core.BridgeMethodResolver;
import org.springframework.core.DefaultParameterNameDiscoverer;
import org.springframework.core.MethodParameter;
import org.springframework.core.ParameterNameDiscoverer;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.core.annotation.SynthesizingMethodParameter;
import org.springframework.web.method.HandlerMethod;

import java.beans.PropertyDescriptor;
import java.io.File;
import java.io.FileFilter;
import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.net.JarURLConnection;
import java.net.URL;
import java.util.*;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * 反射加载类目录和jar下的class
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public class ClassUtil extends org.springframework.util.ClassUtils{

    private static final String OBJECT_NAME = "java.lang.Object";

    /**基本数据类型**/
    private static final List<Class> BASE_CLASS = Lists.newArrayList(
            Integer.class, int.class,
            Long.class, long.class,
            Short.class, short.class,
            Byte.class, short.class,
            Float.class, float.class,
            Double.class, double.class,
            BigDecimal.class,
            String.class,
            Date.class,
            Integer[].class, int[].class,
            Long[].class, long[].class,
            Short[].class, short[].class,
            Byte[].class, short[].class,
            Float[].class, float[].class,
            Double[].class, double[].class,
            BigDecimal[].class,
            String[].class,
            Map.class, Collection.class
    );

    private static final List<Class> NUMBER_CLASS = Lists.newArrayList(
            Integer.class, int.class,
            Long.class, long.class,
            Short.class, short.class,
            Byte.class, short.class,
            Float.class, float.class,
            Double.class, double.class,
            BigDecimal.class
    );

    private static final List<Class> DECIMAL_CLASS = Lists.newArrayList(
            Float.class, float.class,
            Double.class, double.class,
            BigDecimal.class
    );

    private static final List<Class> INTEGER_CLASS = Lists.newArrayList(
            Integer.class, int.class,
            Long.class, long.class,
            Short.class, short.class,
            Byte.class, short.class
    );

    private static final Map<String,Class> KEY_CLASS = Maps.newHashMap();

    static{
        for (Class clazz : BASE_CLASS) {
            KEY_CLASS.put(clazz.getSimpleName(),clazz);
            KEY_CLASS.put(clazz.getSimpleName().toLowerCase(),clazz);
        }
    }

    private static final ParameterNameDiscoverer PARAMETER_NAME_DISCOVERER = new DefaultParameterNameDiscoverer();

    /**
     * 获取方法参数信息
     * @param constructor 构造器
     * @param parameterIndex 参数序号
     * @return {MethodParameter}
     */
    public static MethodParameter getMethodParameter(Constructor<?> constructor, int parameterIndex) {
        MethodParameter methodParameter = new SynthesizingMethodParameter(constructor, parameterIndex);
        methodParameter.initParameterNameDiscovery(PARAMETER_NAME_DISCOVERER);
        return methodParameter;
    }

    /**
     * 获取方法参数信息
     * @param method 方法
     * @param parameterIndex 参数序号
     * @return {MethodParameter}
     */
    public static MethodParameter getMethodParameter(Method method, int parameterIndex) {
        MethodParameter methodParameter = new SynthesizingMethodParameter(method, parameterIndex);
        methodParameter.initParameterNameDiscovery(PARAMETER_NAME_DISCOVERER);
        return methodParameter;
    }

    /**
     * 获取Annotation
     * @param method Method
     * @param annotationType 注解类
     * @param <A> 泛型标记
     * @return {Annotation}
     */
    public static <A extends Annotation> A getAnnotation(Method method, Class<A> annotationType) {
        Class<?> targetClass = method.getDeclaringClass();
        // The method may be on an interface, but we need attributes from the target class.
        // If the target class is null, the method will be unchanged.
        Method specificMethod = ClassUtil.getMostSpecificMethod(method, targetClass);
        // If we are dealing with method with generic parameters, find the original method.
        specificMethod = BridgeMethodResolver.findBridgedMethod(specificMethod);
        // 先找方法，再找方法上的类
        A annotation = AnnotatedElementUtils.findMergedAnnotation(specificMethod, annotationType);
        if (null != annotation) {
            return annotation;
        }
        // 获取类上面的Annotation，可能包含组合注解，故采用spring的工具类
        return AnnotatedElementUtils.findMergedAnnotation(specificMethod.getDeclaringClass(), annotationType);
    }

    /**
     * 获取Annotation
     * @param handlerMethod HandlerMethod
     * @param annotationType 注解类
     * @param <A> 泛型标记
     * @return {Annotation}
     */
    public static <A extends Annotation> A getAnnotation(HandlerMethod handlerMethod, Class<A> annotationType) {
        // 先找方法，再找方法上的类
        A annotation = handlerMethod.getMethodAnnotation(annotationType);
        if (null != annotation) {
            return annotation;
        }
        // 获取类上面的Annotation，可能包含组合注解，故采用spring的工具类
        Class<?> beanType = handlerMethod.getBeanType();
        return AnnotatedElementUtils.findMergedAnnotation(beanType, annotationType);
    }

    // 获取包路径
    public static String getPackageName(Class<?> clazz) {
        return clazz.getPackage().getName();
    }

    // 获取类名
    public static String getClassName(Class<?> clazz) {
        return clazz.getSimpleName();
    }

    // 获取类路劲
    public static String getClass(Class<?> clazz) {
        return clazz.getName();
    }


    /**是否是Java基础数据类型**/
    public static boolean isBaseDataType(Class<?> clazz){
        return BASE_CLASS.contains(clazz);
    }

    /**是否是Java基础数据类型**/
    public static boolean isExtendDataType(Object data){
        for (Class clazz : BASE_CLASS) {
            if( clazz.isInstance(data) ){
                return true;
            }
        }
        return false;
    }

    public static boolean isNumberType(Class<?> clazz){
        return NUMBER_CLASS.contains(clazz);
    }

    /**是否是Java基础数据类型**/
    public static boolean isExtendNumberType(Object data){
        for (Class clazz : NUMBER_CLASS) {
            if( clazz.isInstance(data) ){
                return true;
            }
        }
        return false;
    }

    public static boolean isDecimalType(Class<?> clazz){
        return DECIMAL_CLASS.contains(clazz);
    }

    /**是否是Java基础数据类型**/
    public static boolean isExtendDecimalType(Object data){
        for (Class clazz : DECIMAL_CLASS) {
            if( clazz.isInstance(data) ){
                return true;
            }
        }
        return false;
    }

    public static boolean isIntegerType(Class<?> clazz){
        return INTEGER_CLASS.contains(clazz);
    }

    /**是否是Java基础数据类型**/
    public static boolean isExtendIntegerType(Object data){
        for (Class clazz : INTEGER_CLASS) {
            if( clazz.isInstance(data) ){
                return true;
            }
        }
        return false;
    }

    /**获取表单对象属性(包含父类属性)**/
    public static List<Field> getAllFields(Class clazz) {
        // todo 这里增加本地缓存,提高速度
        List<Field> fields = new ArrayList<>();
        fields.addAll(Arrays.asList(clazz.getDeclaredFields()));

        Class<?> superClass = clazz.getSuperclass();
        while( null != superClass ){
            if( OBJECT_NAME.equalsIgnoreCase(superClass.getName()) ){
                break;
            }
            getFields(superClass, fields);
            superClass = superClass.getSuperclass();
        }

        return fields;
    }

    public static Method getMethod(Class clazz,String methodName){
        Method[] methods = clazz.getDeclaredMethods();
        for (Method method : methods) {
            if(method.getName().equals(methodName)){
                return method;
            }
        }
        return null;
    }

    /**获取单元格值类型**/
    public static Class<?> getFieldType(Class clazz, String fieldName) {
        Field field = getField(clazz,fieldName);
        if( Objects.nonNull(field) ){
            return field.getType();
        }
        return null;
    }

    public static Field getField(Class clazz, String fieldName){
        Field[] fields = clazz.getDeclaredFields();
        for (Field field : fields) {
            if( field.getName().equals(fieldName) ){
                return field;
            }
        }
        return null;
    }

    /**是否包含指定字段属性**/
    public static boolean hasField(Class clazz,String fieldName){
        Field field = getField(clazz,fieldName);
        return Objects.nonNull(field);
    }

    /**是否包含至少一个字段属性**/
    public static boolean hasField(Object data){
        Field[] fields = data.getClass().getDeclaredFields();
        return fields.length > 0;
    }

    /**设置类属性值**/
    public static boolean setValue(Object entity, String fieldName, Object fieldValue){
        try {
            PropertyDescriptor propertyDescriptor = PropertyUtils.getPropertyDescriptor(entity, fieldName);
            Class<?> fieldType = propertyDescriptor.getPropertyType();
            fieldValue = transfer(fieldType,fieldValue);
            PropertyUtils.setSimpleProperty(entity, fieldName, fieldValue);
            return true;
        } catch (Exception e) {
            ExceptionUtil.printStackTrace(e);
        }
        return false;
    }

    /**根据key获取value**/
    public static <E,V> V getValue(E entity,String fieldName) {
        Object id = null;
        try {
            if( Map.class.isInstance(entity) ){
                id = ((Map)entity).get(fieldName);
            }else{
                id = PropertyUtils.getSimpleProperty(entity, fieldName);
            }
        } catch (Exception e) {
            ExceptionUtil.printStackTrace(e);
        }
        return (V)id;
    }

    /**
     * 根据key获取value
     */
    @Deprecated
    public static <E,V> V getValue(String fieldName, E entity) {
        return getValue(entity,fieldName);
    }

    /**根据类名,获取类**/
    public static Class<?> getBaseClass(String simpleClassName){
        return KEY_CLASS.get(simpleClassName);
    }

    /*
     ****************************************私有方法区*******************************************
                   _               _                           _    _                 _
                  (_)             | |                         | |  | |               | |
      _ __   _ __  _ __   __ __ _ | |_  ___   _ __ ___    ___ | |_ | |__    ___    __| |
     | '_ \ | '__|| |\ \ / // _` || __|/ _ \ | '_ ` _ \  / _ \| __|| '_ \  / _ \  / _` |
     | |_) || |   | | \ V /| (_| || |_|  __/ | | | | | ||  __/| |_ | | | || (_) || (_| |
     | .__/ |_|   |_|  \_/  \__,_| \__|\___| |_| |_| |_| \___| \__||_| |_| \___/  \__,_|
     | |
     |_|
     ****************************************私有方法区*******************************************
     */


    /**获取类字段**/
    private static void getFields(Class clazz, List<Field> fields) {

        if( null == clazz ){ return; }

        if( OBJECT_NAME.equalsIgnoreCase(clazz.getSimpleName()) ){
            return;
        }

        fields.addAll(Arrays.asList(clazz.getDeclaredFields()));
    }

    /**根据值类型,转换对应的字段值**/
    private static Object transfer(Class<?> fieldType, Object value) {
        if( Objects.isNull(value) ){
            return null;
        }
        if (fieldType.equals(Integer.class)) {
            return (Integer.valueOf(value.toString()));
        } else if (fieldType.equals(Long.class)) {
            return (Long.valueOf(value.toString()));
        } else if (fieldType.equals(Float.class)) {
            return Float.valueOf(value.toString());
        } else if (fieldType.equals(Double.class)) {
            return Double.valueOf(value.toString());
        } else if (fieldType.equals(Byte.class)) {
            return Byte.valueOf(value.toString());
        } else if (fieldType.equals(Boolean.class)) {
            return Boolean.valueOf(value.toString());
        } else if (fieldType.equals(String.class)) {
            return value;
        } else if (fieldType.equals(BigDecimal.class)) {
            return new BigDecimal(value.toString());
        } else if (fieldType.equals(Date.class)) {
            return DateUtil.format(DateUtil.create(((Date)value).getTime()), DateEnum.FORMAT_YYMMDDHMS_MID);
        }
        return value;
    }


    /**
     * 获取指定包名下的所有类
     * @param packageName
     * @param isRecursive
     * @return
     */
    public static List<Class<?>> getClassList(String packageName, boolean isRecursive) {
        List<Class<?>> classList = new ArrayList<Class<?>>();
        try {
            Enumeration<URL> urls = Thread.currentThread().getContextClassLoader().getResources(packageName.replaceAll("\\.", "/"));
            while (urls.hasMoreElements()) {
                URL url = urls.nextElement();
                if (url != null) {
                    String protocol = url.getProtocol();
                    if (protocol.equals("file")) {
                        String packagePath = url.getPath();
                        addClass(classList, packagePath, packageName, isRecursive);
                    } else if (protocol.equals("jar")) {
                        JarURLConnection jarURLConnection = (JarURLConnection) url.openConnection();
                        JarFile jarFile = jarURLConnection.getJarFile();
                        Enumeration<JarEntry> jarEntries = jarFile.entries();
                        while (jarEntries.hasMoreElements()) {
                            JarEntry jarEntry = jarEntries.nextElement();
                            String jarEntryName = jarEntry.getName();
                            if (jarEntryName.endsWith(".class")) {
                                String className = jarEntryName.substring(0, jarEntryName.lastIndexOf(".")).replaceAll("/", ".");
                                if (isRecursive || className.substring(0, className.lastIndexOf(".")).equals(packageName)) {
                                    classList.add(Class.forName(className));
                                }
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return classList;
    }

    /**
     * 获取指定包名下指定注解的所有类
     * @param packageName
     * @param annotationClass
     * @return
     */
    public static List<Class<?>> getClassListByAnnotation(String packageName, Class<? extends Annotation> annotationClass) {
        List<Class<?>> classList = new ArrayList<Class<?>>();
        try {
            Enumeration<URL> urls = Thread.currentThread().getContextClassLoader().getResources(packageName.replaceAll("\\.", "/"));
            while (urls.hasMoreElements()) {
                URL url = urls.nextElement();
                if (url != null) {
                    String protocol = url.getProtocol();
                    if (protocol.equals("file")) {
                        String packagePath = url.getPath();
                        addClassByAnnotation(classList, packagePath, packageName, annotationClass);
                    } else if (protocol.equals("jar")) {
                        JarURLConnection jarURLConnection = (JarURLConnection) url.openConnection();
                        JarFile jarFile = jarURLConnection.getJarFile();
                        Enumeration<JarEntry> jarEntries = jarFile.entries();
                        while (jarEntries.hasMoreElements()) {
                            JarEntry jarEntry = jarEntries.nextElement();
                            String jarEntryName = jarEntry.getName();
                            if (jarEntryName.endsWith(".class")) {
                                String className = jarEntryName.substring(0, jarEntryName.lastIndexOf(".")).replaceAll("/", ".");
                                Class<?> cls = Class.forName(className);
                                if (cls.isAnnotationPresent(annotationClass)) {
                                    classList.add(cls);
                                }
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return classList;
    }

    /**
     * 获取指定包名下指定父类的所有类
     * @param packageName
     * @param superClass
     * @return
     */
    public static List<Class<?>> getClassListBySuper(String packageName, Class<?> superClass) {
        List<Class<?>> classList = new ArrayList<Class<?>>();
        try {
            Enumeration<URL> urls = Thread.currentThread().getContextClassLoader().getResources(packageName.replaceAll("\\.", "/"));
            while (urls.hasMoreElements()) {
                URL url = urls.nextElement();
                if (url != null) {
                    String protocol = url.getProtocol();
                    if (protocol.equals("file")) {
                        String packagePath = url.getPath();
                        addClassBySuper(classList, packagePath, packageName, superClass);
                    } else if (protocol.equals("jar")) {
                        JarURLConnection jarURLConnection = (JarURLConnection) url.openConnection();
                        JarFile jarFile = jarURLConnection.getJarFile();
                        Enumeration<JarEntry> jarEntries = jarFile.entries();
                        while (jarEntries.hasMoreElements()) {
                            JarEntry jarEntry = jarEntries.nextElement();
                            String jarEntryName = jarEntry.getName();
                            if (jarEntryName.endsWith(".class")) {
                                String className = jarEntryName.substring(0, jarEntryName.lastIndexOf(".")).replaceAll("/", ".");
                                Class<?> cls = Class.forName(className);
                                if (superClass.isAssignableFrom(cls) && !superClass.equals(cls)) {
                                    classList.add(cls);
                                }
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return classList;
    }

    /**
     * 获取指定包名下指定接口的所有实现类
     * @param packageName
     * @param interfaceClass
     * @return
     */
    public static List<Class<?>> getClassListByInterface(String packageName, Class<?> interfaceClass) {
        List<Class<?>> classList = new ArrayList<Class<?>>();
        try {
            Enumeration<URL> urls = Thread.currentThread().getContextClassLoader().getResources(packageName.replaceAll("\\.", "/"));
            while (urls.hasMoreElements()) {
                URL url = urls.nextElement();
                if (url != null) {
                    String protocol = url.getProtocol();
                    if (protocol.equals("file")) {
                        String packagePath = url.getPath();
                        addClassByInterface(classList, packagePath, packageName, interfaceClass);
                    } else if (protocol.equals("jar")) {
                        JarURLConnection jarURLConnection = (JarURLConnection) url.openConnection();
                        JarFile jarFile = jarURLConnection.getJarFile();
                        Enumeration<JarEntry> jarEntries = jarFile.entries();
                        while (jarEntries.hasMoreElements()) {
                            JarEntry jarEntry = jarEntries.nextElement();
                            String jarEntryName = jarEntry.getName();
                            if (jarEntryName.endsWith(".class")) {
                                String className = jarEntryName.substring(0, jarEntryName.lastIndexOf(".")).replaceAll("/", ".");
                                Class<?> cls = Class.forName(className);
                                if (interfaceClass.isAssignableFrom(cls) && !interfaceClass.equals(cls)) {
                                    classList.add(cls);
                                }
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return classList;
    }

    private static void addClass(List<Class<?>> classList, String packagePath, String packageName, boolean isRecursive) {
        try {
            File[] files = getClassFiles(packagePath);
            if (files != null) {
                for (File file : files) {
                    String fileName = file.getName();
                    if (file.isFile()) {
                        String className = getClassName(packageName, fileName);
                        classList.add(Class.forName(className));
                    } else {
                        if (isRecursive) {
                            String subPackagePath = getSubPackagePath(packagePath, fileName);
                            String subPackageName = getSubPackageName(packageName, fileName);
                            addClass(classList, subPackagePath, subPackageName, isRecursive);
                        }
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private static File[] getClassFiles(String packagePath) {
        return new File(packagePath).listFiles(new FileFilter() {
            @Override
            public boolean accept(File file) {
                return (file.isFile() && file.getName().endsWith(".class")) || file.isDirectory();
            }
        });
    }

    private static String getClassName(String packageName, String fileName) {
        String className = fileName.substring(0, fileName.lastIndexOf("."));
        if (packageName!=null) {
            className = packageName + "." + className;
        }
        return className;
    }

    private static String getSubPackagePath(String packagePath, String filePath) {
        String subPackagePath = filePath;
        if (packagePath!=null) {
            subPackagePath = packagePath + "/" + subPackagePath;
        }
        return subPackagePath;
    }

    private static String getSubPackageName(String packageName, String filePath) {
        String subPackageName = filePath;
        if (packageName!=null) {
            subPackageName = packageName + "." + subPackageName;
        }
        return subPackageName;
    }

    private static void addClassByAnnotation(List<Class<?>> classList, String packagePath, String packageName, Class<? extends Annotation> annotationClass) {
        try {
            File[] files = getClassFiles(packagePath);
            if (files != null) {
                for (File file : files) {
                    String fileName = file.getName();
                    if (file.isFile()) {
                        String className = getClassName(packageName, fileName);
                        Class<?> cls = Class.forName(className);
                        if (cls.isAnnotationPresent(annotationClass)) {
                            classList.add(cls);
                        }
                    } else {
                        String subPackagePath = getSubPackagePath(packagePath, fileName);
                        String subPackageName = getSubPackageName(packageName, fileName);
                        addClassByAnnotation(classList, subPackagePath, subPackageName, annotationClass);
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private static void addClassBySuper(List<Class<?>> classList, String packagePath, String packageName, Class<?> superClass) {
        try {
            File[] files = getClassFiles(packagePath);
            if (files != null) {
                for (File file : files) {
                    String fileName = file.getName();
                    if (file.isFile()) {
                        String className = getClassName(packageName, fileName);
                        Class<?> cls = Class.forName(className);
                        if (superClass.isAssignableFrom(cls) && !superClass.equals(cls)) {
                            classList.add(cls);
                        }
                    } else {
                        String subPackagePath = getSubPackagePath(packagePath, fileName);
                        String subPackageName = getSubPackageName(packageName, fileName);
                        addClassBySuper(classList, subPackagePath, subPackageName, superClass);
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private static void addClassByInterface(List<Class<?>> classList, String packagePath, String packageName, Class<?> interfaceClass) {
        try {
            File[] files = getClassFiles(packagePath);
            if (files != null) {
                for (File file : files) {
                    String fileName = file.getName();
                    if (file.isFile()) {
                        String className = getClassName(packageName, fileName);
                        Class<?> cls = Class.forName(className);
                        if (interfaceClass.isAssignableFrom(cls) && !interfaceClass.equals(cls)) {
                            classList.add(cls);
                        }
                    } else {
                        String subPackagePath = getSubPackagePath(packagePath, fileName);
                        String subPackageName = getSubPackageName(packageName, fileName);
                        addClassByInterface(classList, subPackagePath, subPackageName, interfaceClass);
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    /**
     * 返回JavaBean所有属性的<code>PropertyDescriptor</code>
     *
     * @param beanClass the bean class
     * @return the property descriptor [ ]
     */
    public static PropertyDescriptor[] getSelfPropertyDescriptors(Class<?> beanClass) {

        return getPropertyDescriptors(beanClass, beanClass.getSuperclass());
    }

    /**
     * 返回JavaBean所有属性的<code>PropertyDescriptor</code>
     *
     * @param beanClass the bean class
     * @return the property descriptor [ ]
     */
    public static PropertyDescriptor[] getPropertyDescriptors(Class<?> beanClass) {

        return getPropertyDescriptors(beanClass, null);
    }


    /**
     * 返回JavaBean所有属性的<code>PropertyDescriptor</code>
     *
     * @param beanClass the bean class
     * @param stopClass the stop class
     * @return the property descriptor [ ]
     */
    public static PropertyDescriptor[] getPropertyDescriptors(Class<?> beanClass, Class<?> stopClass) {

        IntrospectionCache introspectionCache = IntrospectionCache.forClass(beanClass, stopClass);
        return introspectionCache.getPropertyDescriptors();
    }

    /**
     * 返回JavaBean给定JavaBean给定属性的 <code>PropertyDescriptors</code>
     *
     * @param beanClass    the bean class
     * @param propertyName the name of the property
     * @return the corresponding PropertyDescriptor, or <code>null</code> if none
     */
    public static PropertyDescriptor getPropertyDescriptor(Class<?> beanClass, String propertyName) {

        IntrospectionCache introspectionCache = IntrospectionCache.forClass(beanClass);
        return introspectionCache.getPropertyDescriptor(propertyName);
    }


    /**
     * 返回JavaBean的所有field
     *
     * @param beanClass the bean class
     * @return the fields
     */
    public static Field[] getSelfFields(Class<?> beanClass) {

        return getBeanFields(beanClass, beanClass.getSuperclass());
    }

    /**
     * 返回JavaBean的所有field
     *
     * @param beanClass the bean class
     * @return the fields
     */
    public static Field[] getBeanFields(Class<?> beanClass) {

        return getBeanFields(beanClass, null);
    }


    /**
     * 返回JavaBean的所有field
     *
     * @param beanClass the bean class
     * @param stopClass the stop class
     * @return the fields
     */
    public static Field[] getBeanFields(Class<?> beanClass, Class<?> stopClass) {
        BeanFieldCache beanFieldCache = BeanFieldCache.forClass(beanClass, stopClass);
        return beanFieldCache.getBeanFields();
    }

    /**
     * 返回JavaBean给定名称的field
     *
     * @param beanClass the bean class
     * @param fieldName the name
     * @return the field, or <code>null</code> if none
     */
    public static Field getBeanField(Class<?> beanClass, String fieldName) {

        BeanFieldCache beanFieldCache = BeanFieldCache.forClass(beanClass);
        return beanFieldCache.getBeanField(fieldName);
    }


//    /**
//     * Map keyed by class containing CachedIntrospectionResults.
//     * Needs to be a WeakHashMap with WeakReferences as values to allow
//     * for proper garbage collection in case of multiple class loaders.
//     */
//    private static final Map<Class<?>, BeanInfo> CLASS_CACHE = Collections
//            .synchronizedMap(new WeakHashMap<Class<?>, BeanInfo>());

//    /**
//     * 获取类本身的BeanInfo，不包含父类属性
//     *
//     * @param clazz
//     * @return
//     */
//    public static BeanInfo getBeanInfo(Class<?> clazz, Class<?> stopClazz) {
//        try {
//            BeanInfo beanInfo;
//            if (CLASS_CACHE.get(clazz) == null) {
//                beanInfo = Introspector.getBeanInfo(clazz, stopClazz);
//                CLASS_CACHE.put(clazz, beanInfo);
//                // Immediately remove class from Introspector cache, to allow for proper
//                // garbage collection on class loader shutdown - we cache it here anyway,
//                // in a GC-friendly manner. In contrast to CachedIntrospectionResults,
//                // Introspector does not use WeakReferences as values of its WeakHashMap!
//                Class<?> classToFlush = clazz;
//                do {
//                    Introspector.flushFromCaches(classToFlush);
//                    classToFlush = classToFlush.getSuperclass();
//                } while (classToFlush != null);
//            } else {
//                beanInfo = CLASS_CACHE.get(clazz);
//            }
//            return beanInfo;
//        } catch (IntrospectionException e) {
//            throw new SonsureException("获取BeanInfo失败", e);
//        }
//    }

//    /**
//     * 获取类的BeanInfo,包含父类属性
//     *
//     * @param clazz
//     * @return
//     */
//    public static BeanInfo getBeanInfo(Class<?> clazz) {
//
//        return getBeanInfo(clazz, Object.class);
//    }
//
//    /**
//     * 获取类本身的BeanInfo，不包含父类属性
//     *
//     * @param clazz
//     * @return
//     */
//    public static BeanInfo getSelfBeanInfo(Class<?> clazz) {
//
//        return getBeanInfo(clazz, clazz.getSuperclass());
//    }

//    /**
//     * 获取类属性的PropertyDescriptors
//     *
//     * @param clazz
//     * @return
//     */
//    public static PropertyDescriptor[] getPropertyDescriptors(Class<?> clazz) {
//        BeanInfo beanInfo = getBeanInfo(clazz);
//        return beanInfo.getPropertyDescriptors();
//    }

//    /**
//     * 获取类属性的PropertyDescriptor
//     *
//     * @param clazz
//     * @param name
//     * @return
//     */
//    public static PropertyDescriptor getPropertyDescriptor(Class<?> clazz, String name) {
//        PropertyDescriptor[] propertyDescriptors = getPropertyDescriptors(clazz);
//        if (propertyDescriptors == null) {
//            return null;
//        }
//        for (PropertyDescriptor pd : propertyDescriptors) {
//            if (StringUtils.equals(pd.getName(), name)) {
//                return pd;
//            }
//        }
//        return null;
//    }

    /**
     * 获取对象指定属性值
     *
     * @param clazz
     * @param obj
     * @param fieldName
     * @return
     */
    public static Object getFieldValue(Class<?> clazz, Object obj, String fieldName) {
        PropertyDescriptor propertyDescriptor = getPropertyDescriptor(clazz, fieldName);
        Method readMethod = propertyDescriptor.getReadMethod();
        return invokeMethod(readMethod, obj);
    }

    /**
     * bean属性转换为map
     *
     * @param object
     * @return
     */
    public static Map<String, Object> getSelfBeanPropMap(Object object, Class<? extends Annotation> ignoreAnnotation) {

        Map<String, Object> propMap = new HashMap<String, Object>();
        PropertyDescriptor[] propertyDescriptors = getSelfPropertyDescriptors(object.getClass());
        if (propertyDescriptors == null) {
            return propMap;
        }
        for (PropertyDescriptor pd : propertyDescriptors) {
            Method readMethod = pd.getReadMethod();
            if (readMethod == null || readMethod.getAnnotation(ignoreAnnotation) != null) {
                continue;
            }

            Object value = invokeMethod(readMethod, object);
            propMap.put(pd.getName(), value);
        }
        return propMap;
    }

    public static Method getMethod(Class<?> beanClazz, String methodName, Class<?>[] paramTypes) {
        try {
            return beanClazz.getMethod(methodName, paramTypes);
        } catch (NoSuchMethodException e) {
            throw new UtilsRuntimeException("获取Method失败:" + methodName, e);
        }
    }

    /**
     * invokeMethod
     *
     * @param method
     * @param bean
     * @param value
     */
    public static Object invokeMethod(Method method, Object bean, Object... value) {
        try {
            methodAccessible(method);
            Object[] parameters = new Object[value.length];
            Class<?>[] parameterTypes = method.getParameterTypes();
            for (int i = 0; i < value.length; i++) {
                if (parameterTypes[i] == value[i].getClass()) {
                    parameters[i] = value[i];
                } else if (parameterTypes[i] == Boolean.class || parameterTypes[i] == boolean.class) {
                    parameters[i] = BooleanUtils.toBoolean(String.valueOf(value[i]));
                } else if (parameterTypes[i] == Integer.class || parameterTypes[i] == int.class) {
                    parameters[i] = Integer.valueOf(String.valueOf(value[i]));
                } else if (parameterTypes[i] == Long.class || parameterTypes[i] == long.class) {
                    parameters[i] = Long.valueOf(String.valueOf(value[i]));
                } else {
                    parameters[i] = value[i];
                }
            }
            return method.invoke(bean, parameters);
        } catch (Exception e) {
            throw new UtilsRuntimeException("执行invokeMethod失败:" + (method == null ? "null" : method.getName()), e);
        }
    }

    /**
     * invokeMethod
     *
     * @param method
     * @param bean
     */
    public static Object invokeMethod(Method method, Object bean) {
        try {
            methodAccessible(method);
            return method.invoke(bean);
        } catch (Exception e) {
            throw new UtilsRuntimeException("执行invokeMethod失败:" + (method == null ? "null" : method.getName()), e);
        }
    }

    /**
     * 设置method访问权限
     *
     * @param method
     */
    public static void methodAccessible(Method method) {
        if (!Modifier.isPublic(method.getDeclaringClass().getModifiers())) {
            method.setAccessible(true);
        }
    }

    /**
     * 初始化实例
     *
     * @param clazz
     * @return
     */
    public static Object newInstance(Class<?> clazz) {
        try {
            return clazz.newInstance();
        } catch (Exception e) {
            throw new UtilsRuntimeException("根据class创建实例失败:" + (clazz == null ? "null" : clazz.getName()), e);
        }
    }

    /**
     * 初始化实例
     *
     * @param clazz
     * @return
     */
    public static Object newInstance(String clazz) {

        try {
            Class<?> loadClass = getDefaultClassLoader().loadClass(clazz);
            return loadClass.newInstance();
        } catch (Exception e) {
            throw new UtilsRuntimeException("根据class创建实例失败:" + clazz, e);
        }
    }

    /**
     * 加载类
     *
     * @param clazz
     * @return
     */
    public static Class<?> loadClass(String clazz) {
        try {
            Class<?> loadClass = getDefaultClassLoader().loadClass(clazz);
            return loadClass;
        } catch (Exception e) {
            throw new UtilsRuntimeException("根据class名称加载class失败:" + clazz, e);
        }
    }

    /**
     * 将value的数据类型转换到实际目标类型
     *
     * @param value
     * @return
     */
    public static Object toTargetTypeValue(Object value, Class<?> targetType) {
        String typeName = targetType.getName();
        if (StrUtil.equals(typeName, boolean.class.getName())
                || StrUtil.equals(typeName, Boolean.class.getName())) {
            return Boolean.valueOf(value.toString());
        }
        if (StrUtil.equals(typeName, int.class.getName()) || StrUtil.equals(typeName, Integer.class.getName())) {
            return Integer.valueOf(value.toString());
        }
        if (StrUtil.equals(typeName, long.class.getName()) || StrUtil.equals(typeName, Long.class.getName())) {
            return Long.valueOf(value.toString());
        }
        if (StrUtil.equals(typeName, short.class.getName()) || StrUtil.equals(typeName, Short.class.getName())) {
            return Short.valueOf(value.toString());
        }
        if (StrUtil.equals(typeName, float.class.getName()) || StrUtil.equals(typeName, Float.class.getName())) {
            return Float.valueOf(value.toString());
        }
        if (StrUtil.equals(typeName, double.class.getName())
                || StrUtil.equals(typeName, Double.class.getName())) {
            return Double.valueOf(value.toString());
        }
        if (StrUtil.equals(typeName, byte.class.getName()) || StrUtil.equals(typeName, Byte.class.getName())) {
            return Byte.valueOf(value.toString());
        }
        return value;
    }

    /**
     * 当前线程的classLoader
     *
     * @return
     */
    public static ClassLoader getDefaultClassLoader() {
        return Thread.currentThread().getContextClassLoader();
    }
}
