package cn.dolphin.core.util;

import cn.dolphin.core.consts.CharsetConstant;
import cn.dolphin.core.consts.StringConstant;
import cn.dolphin.core.exception.UtilsRuntimeException;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.text.DateFormat;
import java.text.NumberFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Locale;
import java.util.Set;

/**
 * 类型及格式转换工具
 */
@SuppressWarnings("all")
public final class CastUtil {

    private CastUtil() {
        // 静态类不可实例化
        throw new IllegalAccessError("Utility class");
    }

    /**
     * 类型转换实现
     * @param obj
     * @param type
     * @param format
     * @return
     * @throws UtilsRuntimeException
     */
    @SuppressWarnings("unused")
    public static Object convert(Object obj, String type, String format) throws UtilsRuntimeException {
        Locale locale = new Locale("zh", "CN", "");
        if (obj == null)
            return null;
        if (obj.getClass().getName().equals(type))
            return obj;
        if ("Object".equals(type) || "java.lang.Object".equals(type))
            return obj;
        String fromType = null;
        if (obj instanceof String) {
            fromType = "String";
            String str = (String) obj;
            if ("String".equals(type) || "java.lang.String".equals(type))
                return obj;
            if (str.length() == 0)
                return null;
            if ("Boolean".equals(type) || "java.lang.Boolean".equals(type)) {
                Boolean value = null;
                if (str.equalsIgnoreCase("TRUE"))
                    value = new Boolean(true);
                else
                    value = new Boolean(false);
                return value;
            }
            if ("Double".equals(type) || "java.lang.Double".equals(type))
                try {
                    Number tempNum = getNf(locale).parse(str);
                    return new Double(tempNum.doubleValue());
                } catch (ParseException e) {
                    throw new UtilsRuntimeException("Could not convert " + str + " to " + type + ". ", e);
                }
            if ("BigDecimal".equals(type) || "java.math.BigDecimal".equals(type))
                try {
                    BigDecimal retBig = new BigDecimal(str);
                    int iscale = str.indexOf(".");
                    int keylen = str.length();
                    if (iscale > -1) {
                        iscale = keylen - (iscale + 1);
                        return retBig.setScale(iscale, 5);
                    } else {
                        return retBig.setScale(0, 5);
                    }
                } catch (Exception e) {
                    throw new UtilsRuntimeException("Could not convert " + str + " to " + type + ". ", e);
                }
            if ("Float".equals(type) || "java.lang.Float".equals(type))
                try {
                    Number tempNum = getNf(locale).parse(str);
                    return new Float(tempNum.floatValue());
                } catch (ParseException e) {
                    throw new UtilsRuntimeException("Could not convert " + str + " to " + type + ". ", e);
                }
            if ("Long".equals(type) || "java.lang.Long".equals(type))
                try {
                    NumberFormat nf = getNf(locale);
                    nf.setMaximumFractionDigits(0);
                    Number tempNum = nf.parse(str);
                    return new Long(tempNum.longValue());
                } catch (ParseException e) {
                    throw new UtilsRuntimeException("Could not convert " + str + " to " + type + ". ", e);
                }
            if ("Integer".equals(type) || "java.lang.Integer".equals(type))
                try {
                    NumberFormat nf = getNf(locale);
                    nf.setMaximumFractionDigits(0);
                    Number tempNum = nf.parse(str);
                    return new Integer(tempNum.intValue());
                } catch (ParseException e) {
                    throw new UtilsRuntimeException("Could not convert " + str + " to " + type + ". ", e);
                }
            if ("Date".equals(type) || "java.sql.Date".equals(type)) {
                if (format == null || format.length() == 0)
                    try {
                        return Date.valueOf(str);
                    } catch (Exception e) {
                        try {
                            DateFormat df = null;
                            if (locale != null)
                                df = DateFormat.getDateInstance(3, locale);
                            else
                                df = DateFormat.getDateInstance(3);
                            java.util.Date fieldDate = df.parse(str);
                            return new Date(fieldDate.getTime());
                        } catch (ParseException e1) {
                            throw new UtilsRuntimeException("Could not convert " + str + " to " + type + ". ", e);
                        }
                    }
                try {
                    SimpleDateFormat sdf = new SimpleDateFormat(format);
                    java.util.Date fieldDate = sdf.parse(str);
                    return new Date(fieldDate.getTime());
                } catch (ParseException e) {
                    throw new UtilsRuntimeException("Could not convert " + str + " to " + type + ". ", e);
                }
            }
            if ("Timestamp".equals(type) || "java.sql.Timestamp".equals(type)) {
                if (str.length() == 10)
                    str = str + " 00:00:00";
                if (format == null || format.length() == 0)
                    try {
                        return Timestamp.valueOf(str);
                    } catch (Exception e) {
                        try {
                            DateFormat df = null;
                            if (locale != null)
                                df = DateFormat.getDateTimeInstance(3, 3, locale);
                            else
                                df = DateFormat.getDateTimeInstance(3, 3);
                            java.util.Date fieldDate = df.parse(str);
                            return new Timestamp(fieldDate.getTime());
                        } catch (ParseException e1) {
                            throw new UtilsRuntimeException("Could not convert " + str + " to " + type + ". ", e);
                        }
                    }
                try {
                    SimpleDateFormat sdf = new SimpleDateFormat(format);
                    java.util.Date fieldDate = sdf.parse(str);
                    return new Timestamp(fieldDate.getTime());
                } catch (ParseException e) {
                    throw new UtilsRuntimeException("Could not convert " + str + " to " + type + ". ", e);
                }
            } else {
                throw new UtilsRuntimeException("Conversion from " + fromType + " to " + type + " not currently supported");
            }
        }
        if (obj instanceof BigDecimal) {
            fromType = "BigDecimal";
            BigDecimal bigD = (BigDecimal) obj;
            if ("String".equals(type))
                return getNf(locale).format(bigD.doubleValue());
            if ("BigDecimal".equals(type) || "java.math.BigDecimal".equals(type))
                return obj;
            if ("Double".equals(type))
                return new Double(bigD.doubleValue());
            if ("Float".equals(type))
                return new Float(bigD.floatValue());
            if ("Long".equals(type))
                return new Long(Math.round(bigD.doubleValue()));
            if ("Integer".equals(type))
                return new Integer((int) Math.round(bigD.doubleValue()));
            else
                throw new UtilsRuntimeException("Conversion from " + fromType + " to " + type + " not currently supported");
        }
        if (obj instanceof Double) {
            fromType = "Double";
            Double dbl = (Double) obj;
            if ("String".equals(type) || "java.lang.String".equals(type))
                return getNf(locale).format(dbl.doubleValue());
            if ("Double".equals(type) || "java.lang.Double".equals(type))
                return obj;
            if ("Float".equals(type) || "java.lang.Float".equals(type))
                return new Float(dbl.floatValue());
            if ("Long".equals(type) || "java.lang.Long".equals(type))
                return new Long(Math.round(dbl.doubleValue()));
            if ("Integer".equals(type) || "java.lang.Integer".equals(type))
                return new Integer((int) Math.round(dbl.doubleValue()));
            if ("BigDecimal".equals(type) || "java.math.BigDecimal".equals(type))
                return new BigDecimal(dbl.toString());
            else
                throw new UtilsRuntimeException("Conversion from " + fromType + " to " + type + " not currently supported");
        }
        if (obj instanceof Float) {
            fromType = "Float";
            Float flt = (Float) obj;
            if ("String".equals(type))
                return getNf(locale).format(flt.doubleValue());
            if ("BigDecimal".equals(type) || "java.math.BigDecimal".equals(type))
                return new BigDecimal(flt.doubleValue());
            if ("Double".equals(type))
                return new Double(flt.doubleValue());
            if ("Float".equals(type))
                return obj;
            if ("Long".equals(type))
                return new Long(Math.round(flt.doubleValue()));
            if ("Integer".equals(type))
                return new Integer((int) Math.round(flt.doubleValue()));
            else
                throw new UtilsRuntimeException("Conversion from " + fromType + " to " + type + " not currently supported");
        }
        if (obj instanceof Long) {
            fromType = "Long";
            Long lng = (Long) obj;
            if ("String".equals(type) || "java.lang.String".equals(type))
                return getNf(locale).format(lng.longValue());
            if ("Double".equals(type) || "java.lang.Double".equals(type))
                return new Double(lng.doubleValue());
            if ("Float".equals(type) || "java.lang.Float".equals(type))
                return new Float(lng.floatValue());
            if ("BigDecimal".equals(type) || "java.math.BigDecimal".equals(type))
                return new BigDecimal(lng.toString());
            if ("Long".equals(type) || "java.lang.Long".equals(type))
                return obj;
            if ("Integer".equals(type) || "java.lang.Integer".equals(type))
                return new Integer(lng.intValue());
            else
                throw new UtilsRuntimeException("Conversion from " + fromType + " to " + type + " not currently supported");
        }
        if (obj instanceof Integer) {
            fromType = "Integer";
            Integer intgr = (Integer) obj;
            if ("String".equals(type) || "java.lang.String".equals(type))
                return getNf(locale).format(intgr.longValue());
            if ("Double".equals(type) || "java.lang.Double".equals(type))
                return new Double(intgr.doubleValue());
            if ("Float".equals(type) || "java.lang.Float".equals(type))
                return new Float(intgr.floatValue());
            if ("BigDecimal".equals(type) || "java.math.BigDecimal".equals(type)) {
                String str = intgr.toString();
                BigDecimal retBig = new BigDecimal(intgr.doubleValue());
                int iscale = str.indexOf(".");
                int keylen = str.length();
                if (iscale > -1) {
                    iscale = keylen - (iscale + 1);
                    return retBig.setScale(iscale, 5);
                } else {
                    return retBig.setScale(0, 5);
                }
            }
            if ("Long".equals(type) || "java.lang.Long".equals(type))
                return new Long(intgr.longValue());
            if ("Integer".equals(type) || "java.lang.Integer".equals(type))
                return obj;
            else
                throw new UtilsRuntimeException("Conversion from " + fromType + " to " + type + " not currently supported");
        }
        if (obj instanceof Date) {
            fromType = "Date";
            Date dte = (Date) obj;
            if ("String".equals(type) || "java.lang.String".equals(type))
                if (format == null || format.length() == 0) {
                    return dte.toString();
                } else {
                    SimpleDateFormat sdf = new SimpleDateFormat(format);
                    return sdf.format(new java.util.Date(dte.getTime()));
                }
            if ("Date".equals(type) || "java.sql.Date".equals(type))
                return obj;
            if ("Time".equals(type) || "java.sql.Time".equals(type))
                throw new UtilsRuntimeException("Conversion from " + fromType + " to " + type + " not currently supported");
            if ("Timestamp".equals(type) || "java.sql.Timestamp".equals(type))
                return new Timestamp(dte.getTime());
            else
                throw new UtilsRuntimeException("Conversion from " + fromType + " to " + type + " not currently supported");
        }
        if (obj instanceof Timestamp) {
            fromType = "Timestamp";
            Timestamp tme = (Timestamp) obj;
            if ("String".equals(type) || "java.lang.String".equals(type))
                if (format == null || format.length() == 0) {
                    return (new SimpleDateFormat("yyyy-MM-dd HH:mm:ss")).format(tme).toString();
                } else {
                    SimpleDateFormat sdf = new SimpleDateFormat(format);
                    return sdf.format(new java.util.Date(tme.getTime()));
                }
            if ("Date".equals(type) || "java.sql.Date".equals(type))
                return new Date(tme.getTime());
            if ("Time".equals(type) || "java.sql.Time".equals(type))
                return new Time(tme.getTime());
            if ("Timestamp".equals(type) || "java.sql.Timestamp".equals(type))
                return obj;
            else
                throw new UtilsRuntimeException("Conversion from " + fromType + " to " + type + " not currently supported");
        }
        if (obj instanceof Boolean) {
            fromType = "Boolean";
            Boolean bol = (Boolean) obj;
            if ("Boolean".equals(type) || "java.lang.Boolean".equals(type))
                return bol;
            if ("String".equals(type) || "java.lang.String".equals(type))
                return bol.toString();
            if ("Integer".equals(type) || "java.lang.Integer".equals(type)) {
                if (bol.booleanValue())
                    return new Integer(1);
                else
                    return new Integer(0);
            } else {
                throw new UtilsRuntimeException("Conversion from " + fromType + " to " + type + " not currently supported");
            }
        }
        if ("String".equals(type) || "java.lang.String".equals(type))
            return obj.toString();
        else
            throw new UtilsRuntimeException(
                    "Conversion from " + obj.getClass().getName() + " to " + type + " not currently supported");
    }

    /**
     * 转换为String
     * @param obj
     * @return
     */
    public static String castString(Object obj){
        return CastUtil.castString(obj,"");
    }

    /**
     * 转换为String （提供默认值）
     * @param obj
     * @param defaultValue
     * @return
     */
    public static String castString(Object obj,String defaultValue){
        return obj !=null ? String.valueOf(obj.toString().trim()) : defaultValue;
    }



    /**
     * 转换为int
     * 如果给定的值为<code>null</code>，或者转换失败，返回默认值<code>null</code><br>
     * 转换失败不会报错
     *
     * @param value 被转换的值
     * @return 结果
     */
    public static Integer castInt(Object obj) {
        //return CastUtil.castInt(obj,0);
        return castInt(obj, null);
    }



    /**
     * 转换成int (提供默认值)
     * 如果给定的值为空，或者转换失败，返回默认值
     * 转换失败不会报错
     *
     * @param obj 被转换的值
     * @param defaultValue 转换错误时的默认值
     * @return 结果
     */
    public static Integer castInt(Object obj, Integer defaultValue) {
        if (obj == null){
            return defaultValue;
        }
        if(obj instanceof Integer) {
            return (Integer)obj;
        }
        if(obj instanceof Number){
            return ((Number) obj).intValue();
        }
        final String valueStr = castString(obj, null);
        if (StrUtil.isBlank(valueStr)){
            return defaultValue;
        }
        try {
            return Integer.parseInt(valueStr.trim());
        } catch (NumberFormatException e) {
            return defaultValue;
        }
    }

    /**
     * 转换为Integer数组<br>
     * @param isIgnoreConvertError 是否忽略转换错误，忽略则给值null
     * @param values 被转换的值
     * @return 结果
     * @throws Exception
     */
    public static Integer[] castIntToArray(boolean isIgnoreConvertError, Object... values) throws Exception {
        if(CollectionUtil.isEmpty(values)) {
            return new Integer[]{};
        }
        final Integer[] ints = new Integer[values.length];
        for(int i = 0; i < values.length; i++) {
            final Integer v = castInt(values[i], null);
            if(null == v && isIgnoreConvertError == false) {
                throw new Exception(StrFormatter.format("Convert [{}] to Integer error!", values[i]));
            }
            ints[i] = v;
        }
        return ints;
    }


    /**
     * 将Long转换为Integer
     * @param l
     * @return
     */
    public static int castLongToInt(Long l){
        long l1 = l;
        int i1 = (int) l1;
        return Integer.valueOf(i1);
    }



    /**
     * 转换成Long (提供默认值)
     * 如果给定的值为空，或者转换失败，返回默认值<br>
     * 转换失败不会报错
     *
     * @param obj 被转换的值
     * @param defaultValue 转换错误时的默认值
     * @return 结果
     */
    public static Long castLong(Object obj, Long defaultValue) {
        if (obj == null){
            return defaultValue;
        }
        if(obj instanceof Long) {
            return (Long)obj;
        }
        if(obj instanceof Number){
            return ((Number) obj).longValue();
        }
        final String valueStr = castString(obj, null);
        if (StrUtil.isBlank(valueStr)){
            return defaultValue;
        }
        try {
            //支持科学计数法
            return new BigDecimal(valueStr.trim()).longValue();
        } catch (Exception e) {
            return defaultValue;
        }
    }

    /**
     * 转换为long<br>
     * 如果给定的值为<code>null</code>，或者转换失败，返回默认值<code>null</code><br>
     * 转换失败不会报错
     *
     * @param obj 被转换的值
     * @return 结果
     */
    public static Long castLong(Object obj) {
//        return CastUtil.castLong(obj,0);
        return castLong(obj, null);
    }

    /**
     * 转换为Long数组<br>
     * @param isIgnoreConvertError 是否忽略转换错误，忽略则给值null
     * @param values 被转换的值
     * @return 结果
     * @throws Exception
     */
    public static Long[] castLongToArray(boolean isIgnoreConvertError, Object... values) throws Exception {
        if(CollectionUtil.isEmpty(values)) {
            return new Long[]{};
        }
        final Long[] longs = new Long[values.length];
        for(int i = 0; i < values.length; i++) {
            final Long v = castLong(values[i], null);
            if(null == v && isIgnoreConvertError == false) {
                throw new Exception(StrFormatter.format("Convert [{}] to Long error!", values[i]));
            }
            longs[i] = v;
        }
        return longs;
    }


    /**
     * 转换为Double
     * @param obj
     * @return
     */
    public static double castDouble(Object obj){
        return CastUtil.castDouble(obj,0);
    }

    /**
     * 转换Double精确的小数位四舍五入处理
     * @param obj
     * @param scale 小数点后保留几位
     * @return
     */
    public static double castDoubleScale(Object obj, int scale){

        double dou = CastUtil.castDouble(obj);
        return  CastUtil.castRound(dou,scale);
    }

    /**
     *
     * 将Double转换为Integer
     * @param 	d
     * @return	Intger
     */
    public static Integer castDoubleToInt(Double d){
        double db = d;
        int i = (int) db;
        return i;
    }

    /**
     *
     * 将一个Double类型的字符串，转换为Integer类型
     * @param 	sd 数字的字符串
     * @return	Integer
     */
    public static Integer castStringDoubleToInt(String sd){
        Double d = Double.valueOf(sd);
        return castDoubleToInt(d);
    }



    /**
     * 转换为Double (提供默认值)
     * @param obj
     * @param defaultValue
     * @return
     */
    private static double castDouble(Object obj, double defaultValue) {
        double value = defaultValue;
        if( obj != null){
            String strValue = castString(obj);
            if(StringUtils.isNotBlank(strValue)) {
                try {
                    value = Double.parseDouble(strValue);
                }catch (NumberFormatException e){
                    value = defaultValue;
                }
            }
        }
        return value;
    }

    /**
     * 提供精确的小数位四舍五入处理。
     *
     * @param v
     *            需要四舍五入的数字
     * @param scale
     *            小数点后保留几位
     * @return 四舍五入后的结果
     */
    public static double castRound(double v, int scale) {
        if (scale < 0) {
            throw new IllegalArgumentException("The scale must be a positive integer or zero");
        }
        BigDecimal b = new BigDecimal(Double.toString(v));
        BigDecimal one = new BigDecimal("1");
        return b.divide(one, scale, BigDecimal.ROUND_HALF_UP).doubleValue();
    }

    /**
     * 将String数组转换为Long类型数组
     * @param strs
     * @return
     */
    public static Long[] castStringToLong(String[] strs){// 将String数组转换为Long类型数组
        Long[] longs = new Long[strs.length]; //声明long类型的数组
        for(int i = 0;i<strs.length;i++) {
            String str = strs[i]; //将strs字符串数组中的第i个值赋值给str
            long thelong = Long.valueOf(str);//将str转换为long类型，并赋值给thelong
            longs[i] = thelong;//将thelong赋值给 longs数组中对应的地方
        }
        return longs; //返回long数组
    }
    /**
     * 读取InputStream为UTF-8格式字符串
     * @param is
     * @return String
     */
    public static String convertStreamToString(InputStream is) {
        StringBuilder sb1 = new StringBuilder();
        byte[] bytes = new byte[4096];
        int size = 0;
        try {
            while ((size = is.read(bytes)) > 0) {
                String str = new String(bytes, 0, size, CharsetConstant.UTF8);
                sb1.append(str);
            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            try {
                is.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return sb1.toString();
    }

    /**
     * 字符串转换为BigDecimal
     */
    public static BigDecimal castStringToBigDecimal(String str){
        return new BigDecimal(str.trim().replace(StringConstant.COMMA, StringConstant.EMPTY).replace(StringConstant.SPACING, StringConstant.EMPTY));
    }

    /**
     * 转换为字符<br>
     * 如果给定的值为null，或者转换失败，返回默认值<br>
     * 转换失败不会报错
     *
     * @param value 被转换的值
     * @param defaultValue 转换错误时的默认值
     * @return 结果
     */
    public static Character castChar(Object value, Character defaultValue) {
        if(null == value) {
            return defaultValue;
        }
        if(value instanceof Character) {
            return (Character)value;
        }

        final String valueStr = castString(value, null);
        return StrUtil.isEmpty(valueStr) ? defaultValue : valueStr.charAt(0);
    }

    /**
     * 转换为字符<br>
     * 如果给定的值为<code>null</code>，或者转换失败，返回默认值<code>null</code><br>
     * 转换失败不会报错
     *
     * @param value 被转换的值
     * @return 结果
     */
    public static Character castChar(Object value) {
        return castChar(value, null);
    }

    /**
     * 转换为byte<br>
     * 如果给定的值为<code>null</code>，或者转换失败，返回默认值<br>
     * 转换失败不会报错
     *
     * @param value 被转换的值
     * @param defaultValue 转换错误时的默认值
     * @return 结果
     */
    public static Byte castByte(Object value, Byte defaultValue) {
        if (value == null){
            return defaultValue;
        }
        if(value instanceof Byte) {
            return (Byte)value;
        }
        if(value instanceof Number){
            return ((Number) value).byteValue();
        }
        final String valueStr = castString(value, null);
        if (StrUtil.isBlank(valueStr)){
            return defaultValue;
        }
        try {
            return Byte.parseByte(valueStr);
        } catch (Exception e) {
            return defaultValue;
        }
    }

    /**
     * 转换为byte<br>
     * 如果给定的值为<code>null</code>，或者转换失败，返回默认值<code>null</code><br>
     * 转换失败不会报错
     *
     * @param value 被转换的值
     * @return 结果
     */
    public static Byte castByte(Object value) {
        return castByte(value, null);
    }

    /**
     * 转换为Short<br>
     * 如果给定的值为<code>null</code>，或者转换失败，返回默认值<br>
     * 转换失败不会报错
     *
     * @param value 被转换的值
     * @param defaultValue 转换错误时的默认值
     * @return 结果
     */
    public static Short castShort(Object value, Short defaultValue) {
        if (value == null){
            return defaultValue;
        }
        if(value instanceof Short) {
            return (Short)value;
        }
        if(value instanceof Number){
            return ((Number) value).shortValue();
        }
        final String valueStr = castString(value, null);
        if (StrUtil.isBlank(valueStr)){
            return defaultValue;
        }
        try {
            return Short.parseShort(valueStr.trim());
        } catch (Exception e) {
            return defaultValue;
        }
    }

    /**
     * 转换为Short<br>
     * 如果给定的值为<code>null</code>，或者转换失败，返回默认值<code>null</code><br>
     * 转换失败不会报错
     *
     * @param value 被转换的值
     * @return 结果
     */
    public static Short castShort(Object value) {
        return castShort(value, null);
    }

    /**
     * 转换为Number<br>
     * 如果给定的值为空，或者转换失败，返回默认值<br>
     * 转换失败不会报错
     *
     * @param value 被转换的值
     * @param defaultValue 转换错误时的默认值
     * @return 结果
     */
    public static Number castNumber(Object value, Number defaultValue) {
        if (value == null){
            return defaultValue;
        }
        if(value instanceof Number) {
            return (Number)value;
        }
        final String valueStr = castString(value, null);
        if (StrUtil.isBlank(valueStr)){
            return defaultValue;
        }
        try {
            return NumberFormat.getInstance().parse(valueStr);
        } catch (Exception e) {
            return defaultValue;
        }
    }

    /**
     * 转换为Number<br>
     * 如果给定的值为空，或者转换失败，返回默认值<code>null</code><br>
     * 转换失败不会报错
     *
     * @param value 被转换的值
     * @return 结果
     */
    public static Number castNumber(Object value) {
        return castNumber(value, null);
    }

    /**
     * 转换为Float<br>
     * 如果给定的值为空，或者转换失败，返回默认值<br>
     * 转换失败不会报错
     *
     * @param value 被转换的值
     * @param defaultValue 转换错误时的默认值
     * @return 结果
     */
    public static Float castFloat(Object value, Float defaultValue) {
        if (value == null){
            return defaultValue;
        }
        if(value instanceof Float) {
            return (Float)value;
        }
        if(value instanceof Number){
            return ((Number) value).floatValue();
        }
        final String valueStr = castString(value, null);
        if (StrUtil.isBlank(valueStr)){
            return defaultValue;
        }
        try {
            return Float.parseFloat(valueStr.trim());
        } catch (Exception e) {
            return defaultValue;
        }
    }

    /**
     * 转换为Float<br>
     * 如果给定的值为空，或者转换失败，返回默认值<code>null</code><br>
     * 转换失败不会报错
     *
     * @param value 被转换的值
     * @return 结果
     */
    public static Float castFloat(Object value) {
        return castFloat(value, null);
    }

    /**
     * 转换为Float数组<br>
     * @param isIgnoreConvertError 是否忽略转换错误，忽略则给值null
     * @param values 被转换的值
     * @return 结果
     * @throws Exception
     */
    public static <T> Float[] castFloatToArray(boolean isIgnoreConvertError, Object... values) throws Exception {
        if(CollectionUtil.isEmpty(values)) {
            return new Float[]{};
        }
        final Float[] floats = new Float[values.length];
        for(int i = 0; i < values.length; i++) {
            final Float v = castFloat(values[i], null);
            if(null == v && isIgnoreConvertError == false) {
                throw new Exception(StrFormatter.format("Convert [{}] to Float error!", values[i]));
            }
            floats[i] = v;
        }
        return floats;
    }


    /**
     * 转换为boolean<br>
     * String支持的值为：true、false、yes、ok、no，1,0
     * 如果给定的值为空，或者转换失败，返回默认值<br>
     * 转换失败不会报错
     *
     * @param value 被转换的值
     * @param defaultValue 转换错误时的默认值
     * @return 结果
     */
    public static Boolean castBool(Object value, Boolean defaultValue) {
        if (value == null){
            return defaultValue;
        }
        if(value instanceof Boolean) {
            return (Boolean)value;
        }
        String valueStr = castString(value, null);
        if (StrUtil.isBlank(valueStr)){
            return defaultValue;
        }
        valueStr = valueStr.trim().toLowerCase();
        switch (valueStr) {
            case "true":
                return true;
            case "false":
                return false;
            case "yes":
                return true;
            case "ok":
                return true;
            case "no":
                return false;
            case "1":
                return true;
            case "0":
                return false;
            default:
                return defaultValue;
        }
    }

    /**
     * 转换为boolean<br>
     * 如果给定的值为空，或者转换失败，返回默认值<code>null</code><br>
     * 转换失败不会报错
     *
     * @param value 被转换的值
     * @return 结果
     */
    public static Boolean castBool(Object value) {
        return castBool(value, null);
    }

    /**
     * 转换为Boolean数组<br>
     * @param isIgnoreConvertError 是否忽略转换错误，忽略则给值null
     * @param values 被转换的值
     * @return 结果
     * @throws Exception
     */
    public static Boolean[] castBooleanToArray(boolean isIgnoreConvertError, Object... values) throws Exception {
        if(CollectionUtil.isEmpty(values)) {
            return new Boolean[]{};
        }
        final Boolean[] bools = new Boolean[values.length];
        for(int i = 0; i < values.length; i++) {
            final Boolean v = castBool(values[i], null);
            if(null == v && isIgnoreConvertError == false) {
                throw new Exception(StrFormatter.format("Convert [{}] to Boolean error!", values[i]));
            }
            bools[i] = v;
        }
        return bools;
    }

    /**
     * 转换为Enum对象<br>
     * 如果给定的值为空，或者转换失败，返回默认值<br>
     * @param clazz Enum的Class
     * @param value 值
     * @param defaultValue 默认值
     * @return Enum
     */
    public static <E extends Enum<E>> E castEnum(Class<E> clazz, Object value, E defaultValue) {
        if (value == null){
            return defaultValue;
        }
        if (clazz.isAssignableFrom(value.getClass())) {
            @SuppressWarnings("unchecked")
            E myE = (E) value;
            return myE;
        }
        final String valueStr = castString(value, null);
        if (StrUtil.isBlank(valueStr)){
            return defaultValue;
        }
        try {
            return Enum.valueOf(clazz,valueStr);
        } catch (Exception e) {
            return defaultValue;
        }
    }

    /**
     * 转换为Enum对象<br>
     * 如果给定的值为空，或者转换失败，返回默认值<code>null</code><br>
     * @param clazz Enum的Class
     * @param value 值
     * @return Enum
     */
    public static <E extends Enum<E>> E castEnum(Class<E> clazz, Object value) {
        return castEnum(clazz, value, null);
    }


    /**
     * 转换为BigInteger<br>
     * 如果给定的值为空，或者转换失败，返回默认值<br>
     * 转换失败不会报错
     *
     * @param value 被转换的值
     * @param defaultValue 转换错误时的默认值
     * @return 结果
     */
    public static BigInteger castBigInteger(Object value, BigInteger defaultValue) {
        if (value == null){
            return defaultValue;
        }
        if(value instanceof BigInteger) {
            return (BigInteger)value;
        }
        if(value instanceof Long) {
            return BigInteger.valueOf((Long)value);
        }
        final String valueStr = castString(value, null);
        if (StrUtil.isBlank(valueStr)){
            return defaultValue;
        }
        try {
            return new BigInteger(valueStr);
        } catch (Exception e) {
            return defaultValue;
        }
    }

    /**
     * 转换为BigInteger<br>
     * 如果给定的值为空，或者转换失败，返回默认值<code>null</code><br>
     * 转换失败不会报错
     *
     * @param value 被转换的值
     * @return 结果
     */
    public static BigInteger castBigInteger(Object value) {
        return castBigInteger(value, null);
    }

    /**
     * 转换为BigDecimal<br>
     * 如果给定的值为空，或者转换失败，返回默认值<br>
     * 转换失败不会报错
     *
     * @param value 被转换的值
     * @param defaultValue 转换错误时的默认值
     * @return 结果
     */
    public static BigDecimal castBigDecimal(Object value, BigDecimal defaultValue) {
        if (value == null){
            return defaultValue;
        }
        if(value instanceof BigDecimal) {
            return (BigDecimal)value;
        }
        if(value instanceof Long) {
            return new BigDecimal((Long)value);
        }
        if(value instanceof Double) {
            return new BigDecimal((Double)value);
        }
        if(value instanceof Integer) {
            return new BigDecimal((Integer)value);
        }
        final String valueStr = castString(value, null);
        if (StrUtil.isBlank(valueStr)){
            return defaultValue;
        }
        try {
            return new BigDecimal(valueStr);
        } catch (Exception e) {
            return defaultValue;
        }
    }

    /**
     * 转换为BigDecimal<br>
     * 如果给定的值为空，或者转换失败，返回默认值<br>
     * 转换失败不会报错
     *
     * @param value 被转换的值
     * @return 结果
     */
    public static BigDecimal castBigDecimal(Object value) {
        return castBigDecimal(value, null);
    }

    // ----------------------------------------------------------------------- 全角半角转换
    /**
     * 半角转全角
     *
     * @param input String.
     * @return 全角字符串.
     */
    public static String castSBC(String input) {
        return castSBC(input, null);
    }

    /**
     * 半角转全角
     *
     * @param input String
     * @param notConvertSet 不替换的字符集合
     * @return 全角字符串.
     */
    public static String castSBC(String input, Set<Character> notConvertSet) {
        char c[] = input.toCharArray();
        for (int i = 0; i < c.length; i++) {
            if(null != notConvertSet && notConvertSet.contains(c[i])) {
                //跳过不替换的字符
                continue;
            }

            if (c[i] == ' ') {
                c[i] = '\u3000';
            } else if (c[i] < '\177') {
                c[i] = (char) (c[i] + 65248);

            }
        }
        return new String(c);
    }

    /**
     * 全角转半角
     *
     * @param input String.
     * @return 半角字符串
     */
    public static String castDBC(String input) {
        return castDBC(input, null);
    }

    /**
     * 替换全角为半角
     * @param text 文本
     * @param notConvertSet 不替换的字符集合
     * @return 替换后的字符
     */
    public static String castDBC(String text, Set<Character> notConvertSet) {
        char c[] = text.toCharArray();
        for (int i = 0; i < c.length; i++) {
            if(null != notConvertSet && notConvertSet.contains(c[i])) {
                //跳过不替换的字符
                continue;
            }

            if (c[i] == '\u3000') {
                c[i] = ' ';
            } else if (c[i] > '\uFF00' && c[i] < '\uFF5F') {
                c[i] = (char) (c[i] - 65248);
            }
        }
        String returnString = new String(c);

        return returnString;
    }

    /**
     * byte数组转16进制串
     * @param bytes 被转换的byte数组
     * @return 转换后的值
     */
    public static String castHex(byte[] bytes) {
        final StringBuilder des = new StringBuilder();
        String tmp = null;
        for (int i = 0; i < bytes.length; i++) {
            tmp = (Integer.toHexString(bytes[i] & 0xFF));
            if (tmp.length() == 1) {
                des.append("0");
            }
            des.append(tmp);
        }
        return des.toString();
    }

    /**
     * obj转date
     * @param obj 日期
     * @param format 日期格式
     * @return
     * @throws UtilsRuntimeException
     */
    public static Date castSqlDate(Object obj, String format) throws UtilsRuntimeException {
        return (Date) convert(obj, "Date", format);
    }


    /**
     * obj转Timestamp
     * @param obj 日期
     * @param format 日期格式
     * @return
     * @throws UtilsRuntimeException
     */
    public static Timestamp castTimestamp(Object obj, String format) throws UtilsRuntimeException {
        return (Timestamp) convert(obj, "Timestamp", format);
    }

    private static NumberFormat getNf(Locale locale) {
        NumberFormat nf = null;
        if (locale == null)
            nf = NumberFormat.getNumberInstance();
        else
            nf = NumberFormat.getNumberInstance(locale);
        nf.setGroupingUsed(false);
        return nf;
    }



}
