package cn.dolphin.core.util;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigDecimal;
import java.math.BigInteger;


/**
 * 基于精度数据类型的相关操作
 * 提供基于精度类型的基本计算
 */
public class BigDecimalUtil {

    private static final Logger logger = LoggerFactory.getLogger(BigDecimalUtil.class);

    /**
     * 由于Java的简单类型不能够精确的对浮点数进行运算，这个工具类提供精 确的浮点数运算，包括加减乘除和四舍五入。
     */
    // 默认除法运算精度
    private static final int DEF_DIV_SCALE = 10;

    // 这个类不能实例化
    private BigDecimalUtil() {
    }


    /**
     * 判断num1是否小于num2
     *
     * @param num1
     * @param num2
     * @return num1小于num2返回true
     */
    public static boolean lessThan(BigDecimal num1, BigDecimal num2) {
        return num1.compareTo(num2) == -1;
    }

    /**
     * 判断num1是否小于等于num2
     *
     * @param num1
     * @param num2
     * @return num1小于或者等于num2返回true
     */
    public static boolean lessEqual(BigDecimal num1, BigDecimal num2) {
        return (num1.compareTo(num2) == -1) || (num1.compareTo(num2) == 0);
    }

    /**
     * 判断num1是否大于num2
     *
     * @param num1
     * @param num2
     * @return num1大于num2返回true
     */
    public static boolean greaterThan(BigDecimal num1, BigDecimal num2) {
        return num1.compareTo(num2) == 1;
    }

    /**
     * 判断num1是否大于等于num2
     *
     * @param num1
     * @param num2
     * @return num1大于或者等于num2返回true
     */
    public static boolean greaterEqual(BigDecimal num1, BigDecimal num2) {
        return (num1.compareTo(num2) == 1) || (num1.compareTo(num2) == 0);
    }

    /**
     * 判断num1是否等于num2
     *
     * @param num1
     * @param num2
     * @return num1等于num2返回true
     */
    public static boolean equal(BigDecimal num1, BigDecimal num2) {
        return num1.compareTo(num2) == 0;
    }


    /**
     * 将Object转换为BigDecimal
     */
    public static BigDecimal toBigDecimal(Object obj) {
        if (obj == null) {
            throw new NullPointerException("Parameter is null");
        }
        if (obj instanceof BigDecimal) {
            return (BigDecimal) obj;
        }
        if (obj instanceof BigInteger) {
            return new BigDecimal((BigInteger) obj);
        }
        if (obj instanceof Float) {
            return new BigDecimal((Float) obj);
        }
        if (obj instanceof Double) {
            return new BigDecimal((Double) obj);
        }
        if (obj instanceof Byte) {
            return new BigDecimal((Byte) obj);
        }
        if (obj instanceof Short) {
            return new BigDecimal((Short) obj);
        }
        if (obj instanceof Integer) {
            return new BigDecimal((Integer) obj);
        }
        if (obj instanceof Long) {
            return new BigDecimal((Long) obj);
        }
        if (obj instanceof String) {
            return new BigDecimal((String) obj);
        }
        throw new RuntimeException("Unknown type of parameter");
    }

//    public static BigDecimal toBigDecimal(double d){
//        return new BigDecimal(Double.toString(d));
//    }
//
//    public static BigDecimal toBigDecimal(float f){
//        return new BigDecimal(Double.toString(f));
//    }


    /**
     * add 正常计算
     */
    public static BigDecimal add(Object o1, Object o2) {
        return toBigDecimal(o1).add(toBigDecimal(o2));
    }

    /**
     * add 四舍五入
     */
    public static BigDecimal addRounding(Object o1, Object o2, int scale) {
        if (scale < 0) {
            throw new IllegalArgumentException("Passed an invalid or incorrect argument");
        }
        BigDecimal db = toBigDecimal(o1).add(toBigDecimal(o2));
        return db.setScale(scale, BigDecimal.ROUND_HALF_UP);
    }

    /**
     * add 截断
     */
    public static BigDecimal addTrunc(Object o1, Object o2, int scale) {
        if (scale < 0) {
            throw new IllegalArgumentException("Passed an invalid or incorrect argument");
        }
        BigDecimal db = toBigDecimal(o1).add(toBigDecimal(o2));
        return db.setScale(scale, BigDecimal.ROUND_DOWN);
    }

    /**
     * subtract 正常计算
     */
    public static BigDecimal sub(Object o1, Object o2) {
        return toBigDecimal(o1).subtract(toBigDecimal(o2));
    }

    /**
     * subtract 四舍五入
     */
    public static BigDecimal subRounding(Object o1, Object o2, int scale) {
        if (scale < 0) {
            throw new IllegalArgumentException("Passed an invalid or incorrect argument");
        }
        BigDecimal db = toBigDecimal(o1).subtract(toBigDecimal(o2));
        return db.setScale(scale, BigDecimal.ROUND_HALF_UP);
    }

    /**
     * subtract 截断
     */
    public static BigDecimal subTrunc(Object o1, Object o2, int scale) {
        if (scale < 0) {
            throw new IllegalArgumentException("Passed an invalid or incorrect argument");
        }
        BigDecimal db = toBigDecimal(o1).subtract(toBigDecimal(o2));
        return db.setScale(scale, BigDecimal.ROUND_DOWN);
    }

    /**
     * multiply 正常计算
     */
    public static BigDecimal mul(Object o1, Object o2) {
        return toBigDecimal(o1).multiply(toBigDecimal(o2));
    }

    /**
     * multiply 四舍五入
     */
    public static BigDecimal mulRounding(Object o1, Object o2, int scale) {
        if (scale < 0) {
            throw new IllegalArgumentException("Passed an invalid or incorrect argument");
        }
        BigDecimal db = toBigDecimal(o1).multiply(toBigDecimal(o2));
        return db.setScale(scale, BigDecimal.ROUND_HALF_UP);
    }

    /**
     * multiply 截断
     */
    public static BigDecimal mulTrunc(Object o1, Object o2, int scale) {
        if (scale < 0) {
            throw new IllegalArgumentException("Passed an invalid or incorrect argument");
        }
        BigDecimal db = toBigDecimal(o1).multiply(toBigDecimal(o2));
        return db.setScale(scale, BigDecimal.ROUND_DOWN);
    }

    /**
     * divide 四舍五入
     */
    public static BigDecimal divRounding(Object o1, Object o2, int scale) {
        if (scale < 0) {
            throw new IllegalArgumentException("Passed an invalid or incorrect argument");
        }
        return toBigDecimal(o1).divide(toBigDecimal(o2), scale, BigDecimal.ROUND_HALF_UP);
    }

    /**
     * divide 截断
     */
    public static BigDecimal divTrunc(Object o1, Object o2, int scale) {
        if (scale < 0) {
            throw new IllegalArgumentException("Passed an invalid or incorrect argument");
        }
        return toBigDecimal(o1).divide(toBigDecimal(o2), scale, BigDecimal.ROUND_DOWN);
    }

    /**
     * 保留小数点，四舍五入
     */
    public static BigDecimal toRounding(Object obj, int scale) {
        if (scale < 0) {
            throw new IllegalArgumentException("Passed an invalid or incorrect argument");
        }
        return toBigDecimal(obj).setScale(scale, BigDecimal.ROUND_HALF_UP);
    }

    /**
     * 保留小数点，截断
     */
    public static BigDecimal toTrunc(Object obj, int scale) {
        if (scale < 0) {
            throw new IllegalArgumentException("Passed an invalid or incorrect argument");
        }
        return toBigDecimal(obj).setScale(scale, BigDecimal.ROUND_DOWN);
    }

    /**
     * 精确对比两个数字
     *
     * @param v1
     *            需要被对比的第一个数
     * @param v2
     *            需要被对比的第二个数
     * @return 如果两个数一样则返回0，如果第一个数比第二个数大则返回1，反之返回-1
     */
    public static int compareTo(BigDecimal v1, BigDecimal v2) {
        if(v1 == null)
            return -1;
        else if(v2 == null)
            return 1;
        else
            return v1.compareTo(v2);
    }

    /**
     * 计算百分比
     *
     * @param child
     * @param mother
     */
    public static double score(double child, double mother) {
        return Math.floor(child * 100 / mother);
    }

    /**
     * 返回金钱格式
     * @param num 单位为分
     * @return
     */
    public static String fmtMoney(Long num) {
        if (num == null || 0 == num) {
            return "0.00";
        }
        return new BigDecimal(num).divide(new BigDecimal(100)).setScale(2,BigDecimal.ROUND_HALF_UP).toString();
    }

    /**
     * 格式化double类型数据，保留2位小数
     * @param num
     * @return
     */
    public static Double fmtDouble(Double num) {
        if (num == null || 0 == num) {
            logger.info("num is null == {}",num);
            return 0.00D;
        }
        try {
            return new BigDecimal(num).setScale(2,BigDecimal.ROUND_HALF_UP).doubleValue();
        } catch (Exception e) {
            logger.info("fmt double error : num = {} , error = {}",num,e.getMessage());
            return 0.00D;
        }
    }

    /**
     * 返回金钱格式，保留2位小数
     * @param num
     * @return
     */
    public static Double fmtDouble(BigDecimal num) {
        if (num == null || 0 == num.doubleValue()) {
            logger.info("num is null == {}",num);
            return 0.00D;
        }
        try {
            return num.setScale(2,BigDecimal.ROUND_HALF_UP).doubleValue();
        } catch (Exception e) {
            logger.info("fmt double error : num = {} , error = {}",num,e.getMessage());
            return 0.00D;
        }
    }

//    /**
//     * 提供精确的加法运算。
//     *
//     * @param v1
//     *            被加数
//     * @param v2
//     *            加数
//     * @return 两个参数的和
//     */
//    public static double add(double v1, double v2) {
//        BigDecimal b1 = new BigDecimal(Double.toString(v1));
//        BigDecimal b2 = new BigDecimal(Double.toString(v2));
//        return b1.add(b2).doubleValue();
//    }
//
//    /**
//     * 提供精确的减法运算。
//     *
//     * @param v1
//     *            被减数
//     * @param v2
//     *            减数
//     * @return 两个参数的差
//     */
//    public static double sub(double v1, double v2) {
//        BigDecimal b1 = new BigDecimal(Double.toString(v1));
//        BigDecimal b2 = new BigDecimal(Double.toString(v2));
//        return b1.subtract(b2).doubleValue();
//    }
//
//    /**
//     * 提供精确的乘法运算。
//     *
//     * @param v1
//     *            被乘数
//     * @param v2
//     *            乘数
//     * @return 两个参数的积
//     */
//    public static double mul(double v1, double v2) {
//        BigDecimal b1 = new BigDecimal(Double.toString(v1));
//        BigDecimal b2 = new BigDecimal(Double.toString(v2));
//        return b1.multiply(b2).doubleValue();
//    }
//
//    /**
//     * 提供（相对）精确的除法运算，当发生除不尽的情况时，精确到 小数点以后10位，以后的数字四舍五入。
//     *
//     * @param v1
//     *            被除数
//     * @param v2
//     *            除数
//     * @return 两个参数的商
//     */
//    public static double div(double v1, double v2) {
//        return div(v1, v2, DEF_DIV_SCALE);
//    }
//
//    /**
//     * 提供（相对）精确的除法运算。当发生除不尽的情况时，由scale参数指 定精度，以后的数字四舍五入。
//     *
//     * @param v1
//     *            被除数
//     * @param v2
//     *            除数
//     * @param scale
//     *            表示表示需要精确到小数点以后几位。
//     * @return 两个参数的商
//     */
//    public static double div(double v1, double v2, int scale) {
//        if (scale < 0) {
//            throw new IllegalArgumentException(
//                    "The scale must be a positive integer or zero");
//        }
//        BigDecimal b1 = new BigDecimal(Double.toString(v1));
//        BigDecimal b2 = new BigDecimal(Double.toString(v2));
//        return b1.divide(b2, scale, BigDecimal.ROUND_HALF_UP).doubleValue();
//    }
//
//    /**
//     * 提供精确的小数位四舍五入处理。
//     *
//     * @param v
//     *            需要四舍五入的数字
//     * @param scale
//     *            小数点后保留几位
//     * @return 四舍五入后的结果
//     */
//    public static double round(double v, int scale) {
//        if (scale < 0) {
//            throw new IllegalArgumentException(
//                    "The scale must be a positive integer or zero");
//        }
//        BigDecimal b = new BigDecimal(Double.toString(v));
//        BigDecimal one = new BigDecimal("1");
//        return b.divide(one, scale, BigDecimal.ROUND_HALF_UP).doubleValue();
//    }
//
//    public static double roundDown(double v, int scale) {
//        if (scale < 0) {
//            throw new IllegalArgumentException(
//                    "The scale must be a positive integer or zero");
//        }
//        BigDecimal b = new BigDecimal(Double.toString(v));
//        BigDecimal one = new BigDecimal("1");
//        return b.divide(one, scale, BigDecimal.ROUND_HALF_DOWN).doubleValue();
//    }
//
//    /**
//     * 通过字串解析以免丢失精度
//     * @param fvar
//     * @return
//     */
//    public static double convertFloatToDouble(float fvar){
//        return new BigDecimal(String.valueOf(fvar)).doubleValue();
//    }
//
//    /**
//     * 提供精确的类型转换(Float)
//     * 高精度Double到低进度Float不会丢失精度
//     * @param v
//     *            需要被转换的数字
//     * @return 返回转换结果
//     */
//    public static float convertsToFloat(double v) {
//        BigDecimal b = new BigDecimal(v);
//        return b.floatValue();
//    }
//
//    /**
//     * 提供精确的类型转换(Int)不进行四舍五入
//     *
//     * @param v
//     *            需要被转换的数字
//     * @return 返回转换结果
//     */
//    public static int convertsToInt(double v) {
//        BigDecimal b = new BigDecimal(v);
//        return b.intValue();
//    }
//
//    /**
//     * 提供精确的类型转换(Long)
//     *
//     * @param v
//     *            需要被转换的数字
//     * @return 返回转换结果
//     */
//    public static long convertsToLong(double v) {
//        BigDecimal b = new BigDecimal(v);
//        return b.longValue();
//    }
//
//    /**
//     * 返回两个数中大的一个值
//     *
//     * @param v1
//     *            需要被对比的第一个数
//     * @param v2
//     *            需要被对比的第二个数
//     * @return 返回两个数中大的一个值
//     */
//    public static double returnMax(double v1, double v2) {
//        BigDecimal b1 = new BigDecimal(v1);
//        BigDecimal b2 = new BigDecimal(v2);
//        return b1.max(b2).doubleValue();
//    }
//
//    /**
//     * 返回两个数中小的一个值
//     *
//     * @param v1
//     *            需要被对比的第一个数
//     * @param v2
//     *            需要被对比的第二个数
//     * @return 返回两个数中小的一个值
//     */
//    public static double returnMin(double v1, double v2) {
//        BigDecimal b1 = new BigDecimal(v1);
//        BigDecimal b2 = new BigDecimal(v2);
//        return b1.min(b2).doubleValue();
//    }
//
//
//    public static void main(String[] args) {
//        String s="1.234567890123456789";
//        Double d =  Double.valueOf(s);
//        System.out.println(d);
//        System.out.println(BigDecimalUtil.div(10, 3));
//        System.out.println(BigDecimalUtil.div(10, 3, 2));
//        System.out.println(BigDecimalUtil.round(1.191, 2));
//        System.out.println(BigDecimalUtil.round(1.195, 2));
//        System.out.println(BigDecimalUtil.roundDown(1.191, 2));
//        System.out.println(BigDecimalUtil.roundDown(1.195, 2));
//        System.out.println(BigDecimalUtil.compareTo(new BigDecimal("0"), new BigDecimal("0.00")));
//    }
}
