package cn.dolphin.core.util;

import cn.dolphin.core.consts.StringConstant;
import cn.dolphin.core.exception.ReflexRuntimeException;
import cn.dolphin.core.exception.UtilsRuntimeException;
import com.google.common.collect.Maps;
import net.sf.cglib.beans.BeanCopier;
import org.apache.commons.beanutils.*;
import org.apache.commons.beanutils.converters.DateConverter;
import org.apache.commons.beanutils.converters.LongConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternResolver;
import org.springframework.core.type.classreading.CachingMetadataReaderFactory;
import org.springframework.core.type.classreading.MetadataReader;
import org.springframework.core.type.classreading.MetadataReaderFactory;
import org.springframework.util.ReflectionUtils;
import org.springframework.web.context.ContextLoader;

import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;


/**
 * 实体类处理工具
 */
@SuppressWarnings("all")
public final class BeanUtil extends org.springframework.beans.BeanUtils{

    private static final String CLASS = "class";

    private static Logger logger= LoggerFactory.getLogger(BeanUtil.class);


    /**
     * BeanUtil类型转换器
     */
    public static ConvertUtilsBean convertUtilsBean = new ConvertUtilsBean();

    private static BeanUtilsBean beanUtilsBean = new BeanUtilsBean(convertUtilsBean,new PropertyUtilsBean());

    static{
        convertUtilsBean.register(new DateConverter(), Date.class);
        convertUtilsBean.register(new LongConverter(null),Long.class);
    }


    /**
     * 实例化对象
     *
     * @param clazz 类
     * @param <T>   泛型标记
     * @return 对象
     */
    @SuppressWarnings("unchecked")
    public static <T> T newInstance(Class<?> clazz) {
        return (T) instantiateClass(clazz);
    }

    /**
     * 实例化对象
     *
     * @param clazzStr 类名
     * @param <T>      泛型标记
     * @return 对象
     */
    public static <T> T newInstance(String clazzStr) {
        try {
            Class<?> clazz = ClassUtil.forName(clazzStr, null);
            return newInstance(clazz);
        } catch (ClassNotFoundException e) {
            throw new UtilsRuntimeException(e);
        }
    }



    /**
     * 拷贝属性
     * 动态代理
     * @param source
     * @param target
     */
    public static void copyProperties2(Object source, Object target) {
        BeanCopier copier = BeanCopier.create(source.getClass(), target.getClass(), false);
        copier.copy(source, target, null);
    }


    /**
     * 可以用于判断 Map,Collection,String,Array,Long是否为空
     * @param o
     * @return
     */
    public static boolean isEmpty(Object o){
        if (o == null) return true;
        if (o instanceof String){
            if (((String) o).trim().length() == 0){
                return true;
            }
        }
        else if (o instanceof Collection){
            if (((Collection) o).isEmpty()){
                return true;
            }
        }
        else if (o.getClass().isArray()){
            if (((Object[]) o).length == 0){
                return true;
            }
        }
        else if (o instanceof Map){
            if (((Map) o).isEmpty()){
                return true;
            }
        }


        return false;

    }



    /**
     * 判断对象是否为空
     * o:为null 返回true；
     * 如果为集合，判断集合大小是否为0 为0则返回true；
     * Double，Float,Long,Short,Integer 为0也表示为空。
     * @param o
     * @return
     */
    public static boolean isZeroEmpty(Object o)
    {
        if (o == null) return true;
        if (o instanceof String){
            if (((String) o).trim().length() == 0){
                return true;
            }
        }
        else if (o instanceof Collection){
            if (((Collection) o).isEmpty()){
                return true;
            }
        }
        else if (o.getClass().isArray()){
            if (((Object[]) o).length == 0){
                return true;
            }
        }
        else if (o instanceof Map){
            if (((Map) o).isEmpty()){
                return true;
            }
        }
        else if (o instanceof Double){
            Double lEmpty=0.0;
            if(o==lEmpty){
                return true;
            }
        }
        else if (o instanceof Float){
            Float lEmpty=0f;
            if(o==lEmpty){
                return true;
            }
        }
        else if (o instanceof Long){
            Long lEmpty=0L;
            if(o==lEmpty){
                return true;
            }
        }
        else if (o instanceof Short){
            Short sEmpty=0;
            if(o==sEmpty ){
                return true;
            }
        }
        else if (o instanceof Integer){
            Integer sEmpty=0;
            if(o==sEmpty ){
                return true;
            }
        }

        return false;

    }



    /**
     * 可以用于判断 Map,Collection,String,Array是否不为空
     * @param o
     * @return
     */
    public static boolean isNotEmpty(Object o)
    {
        return !isEmpty(o);
    }

    public static boolean isNotEmpty(Long o)
    {
        return !isEmpty(o);
    }



    /**
     * 判断对象不为空
     * 如果为0表示为空
     * @param o
     * @return
     */
    public static boolean isNotIncZeroEmpty(Object o)
    {
        return !isZeroEmpty(o);
    }

    /**
     * 判断是否为数字
     * @param o
     * @return
     */
    public static boolean isNumber(Object o) {
        if (o == null) 	return false;

        if (o instanceof Number){
            return true;
        }
        if (o instanceof String){
            try{
                Double.parseDouble((String) o);
                return true;
            }
            catch (NumberFormatException e){
                return false;
            }
        }
        return false;
    }


    /**
     * 封装
     * @param map
     * @param entity
     * @return
     * @throws IllegalAccessException
     * @throws InvocationTargetException
     */
    public static Object populateEntity(Map map,Object entity)throws IllegalAccessException, InvocationTargetException{
        beanUtilsBean.populate(entity, map);
        return entity;
    }


    /**
     * 根据指定的类名判定指定的类是否存在
     * @param className
     * @return
     */
    public static boolean validClass(String className){
        try{
            Class.forName(className);
            return true;
        } catch (ClassNotFoundException e) {
            return false;
        }
    }


    /**
     * 判定类是否继承自父类
     * @param cls 子类
     * @param parentClass 父类
     * @return
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    public static boolean isInherit(Class cls,Class parentClass)
    {
        return parentClass.isAssignableFrom(cls);
    }


    /**
     * 克隆对象
     * @param bean
     * @return
     */
    public static Object cloneBean(Object bean){
        try{
            return beanUtilsBean.cloneBean(bean);
        }catch (Exception e){
            handleReflectionException(e);
            return null;
        }
    }


    /**
     * 获取bean
     * @param cls
     * @return
     */
    public  static Object getBean(Class cls) {
        ApplicationContext ctx = ContextLoader.getCurrentWebApplicationContext();
        return ctx.getBean(cls);
    }



    /**
     * 输入基类包名，扫描其下的类，返回类的全路径
     * @param basePackages 如：com.webgate
     * @return
     * @throws IllegalArgumentException
     */
    @SuppressWarnings("all")
    public static List<String> scanPackages(String basePackages) throws IllegalArgumentException{

        ResourcePatternResolver rl = new PathMatchingResourcePatternResolver();
        MetadataReaderFactory metadataReaderFactory = new CachingMetadataReaderFactory(rl);
        List result = new ArrayList();
        String[] arrayPackages = basePackages.split(",");
        try {
            for(int j = 0; j < arrayPackages.length; j++) {
                String packageToScan = arrayPackages[j];
                String packagePart = packageToScan.replace('.', '/');
                String classPattern = "classpath*:/" + packagePart + "/**/*.class";
                Resource[] resources = rl.getResources(classPattern);
                for (int i = 0; i < resources.length; i++) {
                    Resource resource = resources[i];
                    MetadataReader metadataReader = metadataReaderFactory.getMetadataReader(resource);
                    String className = metadataReader.getClassMetadata().getClassName();
                    result.add(className);
                }
            }
        }catch(Exception e) {
            new IllegalArgumentException("scan pakcage class error,pakcages:"+basePackages);
        }

        return result;
    }



    /**
     * 拷贝一个bean中的非空属性于另一个bean中
     * @param dest
     * @param orig
     */
    public static void copyNotNullProperties(Object dest, Object orig){

        // Validate existence of the specified beans
        if (dest == null) {
            logger.error("No destination bean specified");
            return;
        }
        if (orig == null) {
            logger.error("No origin bean specified");
            return;
        }

        try{
            // Copy the properties, converting as necessary
            if (orig instanceof DynaBean) {
                DynaProperty[] origDescriptors = ((DynaBean) orig).getDynaClass()
                        .getDynaProperties();
                for (int i = 0; i < origDescriptors.length; i++) {
                    String name = origDescriptors[i].getName();
                    if (beanUtilsBean.getPropertyUtils().isReadable(orig, name)
                            && beanUtilsBean.getPropertyUtils().isWriteable(dest, name)) {
                        Object value = ((DynaBean) orig).get(name);
                        beanUtilsBean.copyProperty(dest, name, value);
                    }
                }
            } else if (orig instanceof Map) {
                Iterator entries = ((Map) orig).entrySet().iterator();
                while (entries.hasNext()) {
                    Map.Entry entry = (Map.Entry) entries.next();
                    String name = (String) entry.getKey();
                    if (beanUtilsBean.getPropertyUtils().isWriteable(dest, name)) {
                        beanUtilsBean.copyProperty(dest, name, entry.getValue());
                    }
                }
            } else /* if (orig is a standard JavaBean) */{
                PropertyDescriptor[] origDescriptors = beanUtilsBean.getPropertyUtils()
                        .getPropertyDescriptors(orig);
                for (int i = 0; i < origDescriptors.length; i++) {
                    String name = origDescriptors[i].getName();
                    if ("class".equals(name)) {
                        continue; // No point in trying to set an object's class
                    }
                    if (beanUtilsBean.getPropertyUtils().isReadable(orig, name)
                            && beanUtilsBean.getPropertyUtils().isWriteable(dest, name)) {
                        try {
                            Object value = beanUtilsBean.getPropertyUtils().getSimpleProperty(orig, name);
                            if (value != null) {
                                beanUtilsBean.copyProperty(dest, name, value);
                            }
                        } catch (NoSuchMethodException e) {
                            e.printStackTrace();
                        }
                    }
                }
            }
        }catch(Exception ex){
            handleReflectionException(ex);
        }

    }

    @SuppressWarnings("unchecked")
    public static <T> T copyProperties(Class<T> destClass, Object orig){
        Object target = null;
        try {
            target = destClass.newInstance();
            copyProperties((Object) target, orig);
            return (T) target;
        }
        catch (Exception e) {
            handleReflectionException(e);
            return null;
        }
    }

    public static void copyProperties(Object dest, Object orig){
        try {
            beanUtilsBean.copyProperties(dest, orig);
        }catch (Exception e) {
            handleReflectionException(e);
        }
    }

    public static void copyProperty(Object bean, String name, Object value){
        try {
            beanUtilsBean.copyProperty(bean, name, value);
        }catch (Exception e){
            handleReflectionException(e);
        }
    }

    public static Map describe(Object bean){
        try{
            return beanUtilsBean.describe(bean);
        }catch (Exception e) {
            handleReflectionException(e);
            return null;
        }
    }

    public static String[] getArrayProperty(Object bean, String name){
        try{
            return beanUtilsBean.getArrayProperty(bean, name);
        }catch (Exception e){
            handleReflectionException(e);
            return null;
        }
    }

    public static ConvertUtilsBean getConvertUtils()
    {
        return beanUtilsBean.getConvertUtils();
    }

    public static String getIndexedProperty(Object bean, String name, int index){
        try{
            return beanUtilsBean.getIndexedProperty(bean, name, index);
        }catch (Exception e){
            handleReflectionException(e);
            return null;
        }
    }

    public static String getIndexedProperty(Object bean, String name){
        try{
            return beanUtilsBean.getIndexedProperty(bean, name);
        }catch (Exception e){
            handleReflectionException(e);
            return null;
        }
    }

    public static String getMappedProperty(Object bean, String name, String key){
        try{
            return beanUtilsBean.getMappedProperty(bean, name, key);
        }catch (Exception e){
            handleReflectionException(e);
            return null;
        }
    }

    public static String getMappedProperty(Object bean, String name){
        try{
            return beanUtilsBean.getMappedProperty(bean, name);
        }catch (Exception e){
            handleReflectionException(e);
            return null;
        }
    }

    public static String getNestedProperty(Object bean, String name){
        try{
            return beanUtilsBean.getNestedProperty(bean, name);
        }catch (Exception e){
            handleReflectionException(e);
            return null;
        }
    }

    public static String getProperty(Object bean, String name){
        try{
            return beanUtilsBean.getProperty(bean, name);
        }catch (Exception e){
            handleReflectionException(e);
            return null;
        }
    }

    public static PropertyUtilsBean getPropertyUtils(){
        try{
            return beanUtilsBean.getPropertyUtils();
        }catch (Exception e){
            handleReflectionException(e);
            return null;
        }
    }

    public static String getSimpleProperty(Object bean, String name){
        try{
            return beanUtilsBean.getSimpleProperty(bean, name);
        }catch (Exception e){
            handleReflectionException(e);
            return null;
        }
    }

    public static void populate(Object bean, Map properties){
        try{
            beanUtilsBean.populate(bean, properties);
        }catch (Exception e){
            handleReflectionException(e);
        }
    }

    public static void setProperty(Object bean, String name, Object value){
        try{
            beanUtilsBean.setProperty(bean, name, value);
        }catch (Exception e){
            handleReflectionException(e);
        }
    }

    private static void handleReflectionException(Exception e)
    {
        ReflectionUtils.handleReflectionException(e);
    }




    /**
     * 将字符串数据按照指定的类型进行转换
     * @param typeName 实际的数据类型
     * @param value 字符串值
     * @return
     */
    public static Object convertByActType(String typeName,String value){
        Object o = null;
        if (typeName.equals("int")) {
            o = Integer.parseInt(value);
        } else if (typeName.equals("short")) {
            o = Short.parseShort(value);
        } else if (typeName.equals("long")) {
            o = Long.parseLong(value);
        } else if (typeName.equals("float")) {
            o = Float.parseFloat(value);
        } else if (typeName.equals("double")) {
            o = Double.parseDouble(value);
        } else if (typeName.equals("boolean")) {
            o = Boolean.parseBoolean(value);
        } else if (typeName.equals("java.lang.String")) {
            o = value;
        } else{
            o=value;
        }
        return o;
    }


    /**
     *
     * 设置属性
     *
     * @param obj	对象
     * @param field	属性名
     * @param value	属性值
     * @return
     *
     */
    public static final void setField(Object obj,String field, Object value) {
        try {
            BeanUtils.setProperty(obj, field, value);
        } catch (IllegalAccessException | InvocationTargetException e) {
            throw new ReflexRuntimeException(e.getMessage(), e);
        }
    }
    /**
     *
     * 获取对象的属性
     *
     * @param obj	对象
     * @param field	属性名
     * @return	属性值（toString）
     *
     */
    public static final String getField(Object obj,String field){
        try {
            return BeanUtils.getProperty(obj, field);
        } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
            throw new ReflexRuntimeException(e.getMessage(), e);
        }
    }
    /**
     *
     * Map转换为Bean
     *
     * @param fields	属性-值
     * @param beanClass	对象Class
     * @return
     *
     */
    public static final Object mapToObj(Map<String,Object> fields, Class<?> beanClass){
        try {
            Object obj = beanClass.newInstance();
            BeanUtils.populate(obj, fields);
            return obj;
        } catch (InstantiationException | IllegalAccessException | InvocationTargetException e) {
            throw new ReflexRuntimeException(e.getMessage(), e);
        }
    }
    /**
     *
     * 将Bean转为Map
     *
     * @param obj
     * @return
     *
     */
    public static final Map<String,Object> objToMap(Object obj){
        try {
            Map<String, Object> map = PropertyUtils.describe(obj);
            //去除class键
            map.remove(CLASS);
            return map;
        } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
            throw new ReflexRuntimeException(e.getMessage(), e);
        }
    }

    /**
     *
     * 将Bean转为Map<String,String>
     *
     * @param obj
     * @return
     *
     */
    public static final Map<String,String> objToStrMap(Object obj){
        try {
            //转换为Map，转换为Map<String,String>
            Map<String, String> map = new HashMap<String,String>();

            Map<String, String> beanLog = new HashMap<String, String>();
            beanLog = BeanUtils.describe(obj);
            //PropertyUtils.describe(obj).entrySet()//springboot2.x 这个写法不支持强转
            for (Map.Entry<String, String> entry : beanLog.entrySet()) {
                map.put(entry.getKey(), entry.getValue()+ StringConstant.EMPTY);
            }
            //去除class键
            map.remove(CLASS);
            return map;
        } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
            throw new ReflexRuntimeException(e.getMessage(), e);
        }
    }


    /**
     *
     * 计算对象的非空字段数
     *
     * @param ent
     * @return
     *
     */
    public static int getFnum(Object ent){
        Collection<Object> values = null;
        try {
            //转换对象为Map
            Map<String, Object> map = PropertyUtils.describe(ent);
            //去除class键
            map.remove(CLASS);
            //获取所有Map值
            values = map.values();
        } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
            throw new ReflexRuntimeException(e.getMessage(), e);
        }
        //查询非空的属性值数量
        int i = 0;
        for (Object obj : values) {
            if (obj == null) {
                continue;
            }
            if (obj instanceof String) {
                if(((String) obj).length()==0){
                    continue;
                }
            }
            i ++;
        }
        return i;
    }


    /**
     * 如果bean是List接口的一个实例,就直接返回
     * 如果bean不是List接口的一个实例,将其使用List对象将其包装
     * @param <T> -- 多态类型
     * @param bean -- 被包装的bean
     * @return -- 包装了bean的List
     */
    public static <T> List<T> wrapToList(Object bean) {

        if (List.class.isInstance(bean)) {
            return (List<T>) bean;
        } else {
            List<T> target = new ArrayList<T>(1);
            if (null != bean) {
                target.add((T) bean);
            }
            return target;
        }
    }

    /**
     * 如果bean是Map接口的一个实例,就直接返回
     * 如果bean不是Map接口的一个实例,将返回一个空元素(map.size()等于0)的map对象
     * @param <K> -- 多态map的key类型
     * @param <T> -- 多态map的value类型
     * @param bean -- 被包装的bean
     * @return -- 包装了bean的map
     */
    public static <K, T> Map<K, T> wrapToMap(Object bean) {

        if (Map.class.isInstance(bean)) {
            return (Map<K, T>)bean;
        } else {
            Map<K, T> target = new ConcurrentHashMap<K, T>();
            return target;
        }
    }

    /**
     * 将bean转成int类型
     * @param bean -- 被转换的bean
     * @param defaultValue -- 默认值,如果bean为null时返回
     * @return -- bean转成的in值
     */
    public static int toInt(Object bean, int defaultValue) {
        try{
            return (null != bean) ? Integer.parseInt(bean.toString()) : defaultValue;
        }catch(Exception e){
            e.printStackTrace();
        }
        return defaultValue;
    }

    /**
     * 将bean转成int类型
     * @param bean -- 被转换的bean
     * @param defaultValue -- 默认值,如果bean为null时返回
     * @return -- bean转成的in值
     */
    public static int toInt(String bean, int defaultValue) {
        try{
            return (null != bean) ? Integer.parseInt(bean.toString()) : defaultValue;
        }catch(Exception e){
            e.printStackTrace();
        }
        return defaultValue;
    }

    /**
     * 将bean转成int类型
     * @param bean -- 被转换的bean
     * @param defaultValue -- 默认值,如果bean为null时返回
     * @return -- bean转成的in值
     */
    public static int toInt(Integer bean, int defaultValue) {
        return (null != bean) ? bean.intValue() : defaultValue;
    }

    /**
     * 根据value来找map中的key
     * @param <T> -- key多态类型
     * @param value -- map中的value
     * @param map -- 被查找的map
     * @return -- value在map中的key
     */
    public static <T> T getKeyByValue(Object value, Map map) {
        Object v = null;
        for (Object key : map.keySet()) {
            v = map.get(key);
            if(value == null){
                if(v == null){
                    return (T)key;
                } else {
                    continue;
                }
            } else if(value.equals(v)){
                return (T)key;
            }
        }
        return null;
    }

    /**
     * 将Object对象转成String
     * @param obj -- Object对象
     * @return -- String对象
     */
    public static String toNullableString(Object obj) {
        return (null != obj) ? obj.toString() : null;
    }

    /**
     * 将Object对象转成String, 如果obj为null, 返回""字符串
     * @param obj -- Object对象
     * @return -- String对象
     */
    public static String toNotNullString(Object obj) {
        return (null != obj) ? obj.toString() : "";
    }

    /**
     * 判断str是否是null或者是""字符串
     * @param str -- 字符串
     * @return -- true, 是null或者是""字符串
     */
    public static boolean isNullOrEmpty(String str) {
        return null == str || "".equals(str);
    }

    /**
     * 判断str非null和""字符串
     * @param str -- 字符串
     * @return -- true, str不为null, 并且不等于(equals)""字符串
     */
    public static boolean isNotNullAndEmpty(String str) {
        return (null != str && !"".equals(str));
    }

    /**
     * 根据一组key,和一组升降序标记,按顺序对一个map list进行排序
     * @param list -- map list
     * @param sortKeys -- key数组
     * @param ascOrders -- 升降序标记数组
     * @return -- 排序后的map list
     */
    public static List<Map<String, Object>> sort(List<Map<String, Object>> list, final String[] sortKeys, final boolean[] ascOrders) {

        Comparator<Map<String, Object>> cmp = new Comparator<Map<String, Object>>() {
            public int compare(Map m1, Map m2) {

                for (int i = 0; i < sortKeys.length; i++) {
                    Object o1 = m1.get(sortKeys[i]);
                    Object o2 = m2.get(sortKeys[i]);
                    if (null == o1 || null == o2) {
                        throw new IllegalStateException("被排序的map的key的值不能为空");
                    }
                    Comparable c = (Comparable) o1;
                    int r = c.compareTo(o2);
                    if (0 != r) {
                        return ascOrders[i] ? r : -r;
                    }
                }
                return 0;
            }
        };
        Collections.sort(list, cmp);
        return list;
    }

    /**
     * 根据map中一个key的值,对一个map list进行排序, 默认顺序:asc
     * @param list -- map list
     * @param sortKey -- 用于排序的值对应的key
     * @return -- 排序后的map list
     */
    public static List<Map<String, Object>> sort(List<Map<String, Object>> list, final String sortKey) {

        Comparator cmp = new Comparator<Map>() {
            public int compare(Map m1, Map m2) {

                Object o1 = m1.get(sortKey);
                Object o2 = m2.get(sortKey);
                if (null == o1 || null == o2) {
                    throw new IllegalStateException("被排序的map的key的值不能为空");
                }
                Comparable c = (Comparable) o1;
                return c.compareTo(o2);
            }
        };
        Collections.sort(list, cmp);
        return list;
    }

    /**
     * 防止非法实例化
     */
    private BeanUtil() {}

    /**
     *
     * 根据配置文件配置的类，动态创建对象
     *
     * @param className -- 类名
     * @return 返回一个类名所代表的一个对象
     * @exception 参数为空或者没有找到相应类的时候抛出Exception
     */
    public static Object createBean(String className) throws Exception {
        if (StrUtil.isBlank(className)) {
            String msg = "Cannot find the class " + ":" + className;
            throw new Exception(msg);
        } else {
            try {
                return Class.forName(className).newInstance();
            } catch (Throwable ex) {
                String msg ="Cannot create this class " + ":"  + className;
                throw new Exception(msg, ex);
            }
        }
    }

    /**
     * key转换驼峰
     * @param map
     * @return
     */
    public static Map<String, Object> mapKeyCovertion(Map<String, Object> map){
        if(map == null){
            return null;
        }
        Map<String, Object> nMap = Maps.newHashMap();
        map.forEach((k, v)->{nMap.put(NameUtil.getCamelName(k), v);});
        map = null;
        return nMap;
    }




}