package cn.dolphin.core.util;

import org.apache.commons.lang3.ArrayUtils;

import java.lang.reflect.Array;
import java.util.*;

/**
 * 数组的工具类
 * 可以先到ArrayUtils（cpmmoms-lang-*.jar）找
 */
public class ArrayUtil extends ArrayUtils{

    /**
     * 数组为空
     * @param o
     * @return 空true，否则false
     */
    public static boolean isBlank(Object[] o) {
        return o == null || o.length <= 0;
    }

    /**
     * 数字不为空
     * @param o
     * @return 不为空true，否则false
     */
    public static boolean isNotBlank(Object[] o) {
        return !isBlank(o);
    }

    /**
     * 对象是否为数组对象
     *
     * @param obj 对象
     * @return 是否为数组对象，如果为{@code null} 返回false
     */
    public static boolean isArray(Object obj) {
        if (null == obj) {
            return false;
        }
        return obj.getClass().isArray();
    }

    /**
     * 排除数组中重复值
     *
     * @param <T>
     *
     * @param sources
     * @return
     */
    public static <T> T[] distinct(T[] sources) {
        if (null == sources) {
            return sources;
        }
        List<T> list = new ArrayList<>();
        for (T obj : sources) {
            if (!list.contains(obj)) {
                list.add(obj);
            }
        }
        return (T[]) list.toArray();
    }

    /**
     * 指定数组中追加元素
     *
     * @param <T>
     *
     * @param sources
     * @param target
     * @return
     */
    public static <T> T[] join(T[] sources, T target) {
        if (null == sources) {
            return (T[]) new Object[] { target };
        }
        T[] tmp = Arrays.copyOf(sources, sources.length + 1);
        tmp[sources.length] = target;
        return tmp;
    }

    /**
     * 移除数组中---指定元素
     *
     * @param <T>
     *
     * @param sources
     * @param target
     * @return
     */
    public static <T> T[] remove(T[] sources, T target) {
        List<T> list = new ArrayList<>();
        if (null == sources) {
            return (T[]) list.toArray();
        }
        for (T obj : sources) {
            list.add(obj);
            list.remove(target);
        }
        return (T[]) list.toArray();
    }

    /**
     * 集合是否有交叉
     *
     * @param arg0
     * @param arg1
     * @return
     */
    public static <T> boolean cross(T[] arg0, T[] arg1) {
        if (arg0 == null && arg1 == null) {
            return true;
        }
        List<T> arg0List = new ArrayList<>();
        if (null != arg0) {
            arg0List = Arrays.asList(arg0);
        }
        List<T> arg1List = new ArrayList<>();
        if (null != arg1) {
            arg1List = Arrays.asList(arg1);
        }

        if (arg0List.size() > arg1List.size()) {
            for (T obj : arg0List) {
                if (arg1List.contains(obj)) {
                    return true;
                }
            }
        } else {
            for (T obj : arg1List) {
                if (arg0List.contains(obj)) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * 聚合多个数组合并为一个数组
     *
     * @param source
     * @param targets
     * @return
     */
    public static <T> T[] concat(T[] source, T[]... targets) {
        if (CommonUtil.isEmpty(targets)) {
            return source;
        }
        int offset = null == source ? 0 : source.length;
        int index = offset;
        for (T[] objs : targets) {
            index += null == objs ? 0 : objs.length;
        }
        T[] result = null == source ? (T[]) new Object[index] : Arrays.copyOf(source, index);
        for (T[] objs : targets) {
            if (null != objs) {
                System.arraycopy(objs, 0, result, offset, objs.length);
                offset += objs.length;
            }
        }
        return result;
    }

    /**
     * 是否在数组中
     *
     * @param array
     * @param str
     * @return
     */
    public static boolean isInArray(String str, String[] array) {
        for (String s : array) {
            if (s.equalsIgnoreCase(str)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 新建一个空数组
     *
     * @param <T> 数组元素类型
     * @param componentType 元素类型
     * @param newSize 大小
     * @return 空数组
     */
    @SuppressWarnings("unchecked")
    public static <T> T[] newArray(Class<?> componentType, int newSize) {
        return (T[]) Array.newInstance(componentType, newSize);
    }

    /**
     * 将集合转为数组
     *
     * @param <T> 数组元素类型
     * @param iterator {@link Iterator}
     * @param componentType 集合元素类型
     * @return 数组
     */
    public static <T> T[] toArray(Iterator<T> iterator, Class<T> componentType) {
        return toArray(newArrayList(iterator), componentType);
    }

    /**
     * 新建一个ArrayList<br>
     * 提供的参数为null时返回空{@link ArrayList}
     *
     * @param <T> 集合元素类型
     * @param iter {@link Iterator}
     * @return ArrayList对象
     */
    private static <T> ArrayList<T> newArrayList(Iterator<T> iter) {
        final ArrayList<T> list = new ArrayList<>();
        if (null == iter) {
            return list;
        }
        while (iter.hasNext()) {
            list.add(iter.next());
        }
        return list;
    }

    /**
     * 获取数组长度<br>
     * 如果参数为{@code null}，返回0
     *
     * <pre>
     * ArrayUtil.length(null)            = 0
     * ArrayUtil.length([])              = 0
     * ArrayUtil.length([null])          = 1
     * ArrayUtil.length([true, false])   = 2
     * ArrayUtil.length([1, 2, 3])       = 3
     * ArrayUtil.length(["a", "b", "c"]) = 3
     * </pre>
     *
     * @param array 数组对象
     * @return 数组长度
     * @throws IllegalArgumentException 如果参数不为数组，抛出此异常
     */
    public static int length(Object array) throws IllegalArgumentException {
        if (null == array) {
            return 0;
        }
        return Array.getLength(array);
    }

    /**
     * 数组或集合转String
     *
     * @param obj 集合或数组对象
     * @return 数组字符串，与集合转字符串格式相同
     */
    public static String toString(Object obj) {
        if (null == obj) {
            return null;
        }
        if (ArrayUtil.isArray(obj)) {
            try {
                return Arrays.deepToString((Object[]) obj);
            } catch (Exception e) {
                final String className = obj.getClass().getComponentType().getName();
                switch (className) {
                    case "long":
                        return Arrays.toString((long[]) obj);
                    case "int":
                        return Arrays.toString((int[]) obj);
                    case "short":
                        return Arrays.toString((short[]) obj);
                    case "char":
                        return Arrays.toString((char[]) obj);
                    case "byte":
                        return Arrays.toString((byte[]) obj);
                    case "boolean":
                        return Arrays.toString((boolean[]) obj);
                    case "float":
                        return Arrays.toString((float[]) obj);
                    case "double":
                        return Arrays.toString((double[]) obj);
                    default:
                        throw new RuntimeException(e);
                }
            }
        }
        return obj.toString();
    }

    /**
     * 首先尝试强转，强转失败则构建一个新的{@link ArrayList}
     * @param <E> 集合元素类型
     * @param iterable {@link Iterable}
     * @return {@link Collection} 或者 {@link ArrayList}
     */
    private static <E> Collection<E> toCollection(Iterable<E> iterable) {
        return (iterable instanceof Collection) ? (Collection<E>) iterable : newArrayList(iterable.iterator());
    }
    /**
     * 将集合转为数组
     *
     * @param <T> 数组元素类型
     * @param iterable {@link Iterable}
     * @param componentType 集合元素类型
     * @return 数组
     */
    public static <T> T[] toArray(Iterable<T> iterable, Class<T> componentType) {
        return toArray(toCollection(iterable), componentType);
    }

    /**
     * 将集合转为数组
     *
     * @param <T> 数组元素类型
     * @param collection 集合
     * @param componentType 集合元素类型
     * @return 数组
     */
    public static <T> T[] toArray(Collection<T> collection, Class<T> componentType) {
        final T[] array = newArray(componentType, collection.size());
        return collection.toArray(array);
    }



    // ---------------------------------------------------------------------- toXxxArray
    public static String[] toStringArray(int[] array) {
        String[] newArray = new String[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = String.valueOf(array[i]);
        }
        return newArray;
    }

    public static String[] toStringArray(double[] array) {
        String[] newArray = new String[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = String.valueOf(array[i]);
        }
        return newArray;
    }

    public static String[] toStringArray(float[] array) {
        String[] newArray = new String[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = String.valueOf(array[i]);
        }
        return newArray;
    }

    public static String[] toStringArray(short[] array) {
        String[] newArray = new String[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = String.valueOf(array[i]);
        }
        return newArray;
    }

    public static String[] toStringArray(byte[] array) {
        String[] newArray = new String[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = String.valueOf(array[i]);
        }
        return newArray;
    }

    public static String[] toStringArray(long[] array) {
        String[] newArray = new String[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = String.valueOf(array[i]);
        }
        return newArray;
    }

    public static String[] toStringArray(boolean[] array) {
        String[] newArray = new String[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = String.valueOf(array[i]);
        }
        return newArray;
    }

    public static String[] toStringArray(char[] array) {
        String[] newArray = new String[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = String.valueOf(array[i]);
        }
        return newArray;
    }

    public static String[] toStringArray(Object[] array) {
        String[] newArray = new String[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = String.valueOf(array[i]);
        }
        return newArray;
    }

    public static String[] toStringArray(Collection<String> collection) {
        if (collection == null) {
            return null;
        }
        return collection.toArray(new String[collection.size()]);
    }

    public static int[] toIntArray(String[] array) {
        int[] newArray = new int[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = Integer.valueOf(array[i]);
        }
        return newArray;
    }

    public static int[] toIntArray(double[] array) {
        int[] newArray = new int[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (int) array[i];
        }
        return newArray;
    }

    public static int[] toIntArray(float[] array) {
        int[] newArray = new int[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (int) array[i];
        }
        return newArray;
    }

    public static int[] toIntArray(short[] array) {
        int[] newArray = new int[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = array[i];
        }
        return newArray;
    }

    public static int[] toIntArray(byte[] array) {
        int[] newArray = new int[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = array[i];
        }
        return newArray;
    }

    public static int[] toIntArray(long[] array) {
        int[] newArray = new int[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (int) array[i];
        }
        return newArray;
    }

    public static int[] toIntArray(char[] array) {
        int[] newArray = new int[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (int) array[i];
        }
        return newArray;
    }

    /**
     * 转 int 数组,[false:0,true:1]
     *
     * @param array
     * @return
     */
    public static int[] toIntArray(boolean[] array) {
        int[] newArray = new int[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            if (array[i] == false) {
                newArray[i] = 0;
            } else {
                newArray[i] = 1;
            }
        }
        return newArray;
    }

    public static int[] toIntArray(Object[] array) {
        int[] newArray = new int[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (int) array[i];
        }
        return newArray;
    }

    public static double[] toDoubleArray(String[] array) {
        double[] newArray = new double[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = Double.valueOf(array[i]);
        }
        return newArray;
    }

    public static double[] toDoubleArray(int[] array) {
        double[] newArray = new double[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = array[i];
        }
        return newArray;
    }

    public static double[] toDoubleArray(float[] array) {
        double[] newArray = new double[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = array[i];
        }
        return newArray;
    }

    public static double[] toDoubleArray(long[] array) {
        double[] newArray = new double[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = array[i];
        }
        return newArray;
    }

    public static double[] toDoubleArray(short[] array) {
        double[] newArray = new double[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = array[i];
        }
        return newArray;
    }

    public static double[] toDoubleArray(byte[] array) {
        double[] newArray = new double[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = array[i];
        }
        return newArray;
    }

    public static double[] toDoubleArray(char[] array) {
        double[] newArray = new double[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = array[i];
        }
        return newArray;
    }

    public static double[] toDoubleArray(Object[] array) {
        double[] newArray = new double[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (double) array[i];
        }
        return newArray;
    }

    public static float[] toFloatArray(String[] array) {
        float[] newArray = new float[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = Float.valueOf(array[i]);
        }
        return newArray;
    }

    public static float[] toFloatArray(int[] array) {
        float[] newArray = new float[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = array[i];
        }
        return newArray;
    }

    public static float[] toFloatArray(double[] array) {
        float[] newArray = new float[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (float) array[i];
        }
        return newArray;
    }

    public static float[] toFloatArray(short[] array) {
        float[] newArray = new float[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = array[i];
        }
        return newArray;
    }

    public static float[] toFloatArray(byte[] array) {
        float[] newArray = new float[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = array[i];
        }
        return newArray;
    }

    public static float[] toFloatArray(long[] array) {
        float[] newArray = new float[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = array[i];
        }
        return newArray;
    }

    public static float[] toFloatArray(char[] array) {
        float[] newArray = new float[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = array[i];
        }
        return newArray;
    }

    public static float[] toFloatArray(Object[] array) {
        float[] newArray = new float[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (float) array[i];
        }
        return newArray;
    }

    public static long[] toLongArray(String[] array) {
        long[] newArray = new long[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = Long.valueOf(array[i]);
        }
        return newArray;
    }

    public static long[] toLongArray(int[] array) {
        long[] newArray = new long[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = array[i];
        }
        return newArray;
    }

    public static long[] toLongArray(double[] array) {
        long[] newArray = new long[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (long) array[i];
        }
        return newArray;
    }

    public static long[] toLongArray(float[] array) {
        long[] newArray = new long[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (long) array[i];
        }
        return newArray;
    }

    public static long[] toLongArray(short[] array) {
        long[] newArray = new long[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = array[i];
        }
        return newArray;
    }

    public static long[] toLongArray(byte[] array) {
        long[] newArray = new long[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = array[i];
        }
        return newArray;
    }

    public static long[] toLongArray(char[] array) {
        long[] newArray = new long[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = array[i];
        }
        return newArray;
    }

    public static long[] toLongArray(Object[] array) {
        long[] newArray = new long[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (long) array[i];
        }
        return newArray;
    }

    public static short[] toShortArray(String[] array) {
        short[] newArray = new short[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = Short.valueOf(array[i]);
        }
        return newArray;
    }

    public static short[] toShortArray(int[] array) {
        short[] newArray = new short[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (short) array[i];
        }
        return newArray;
    }

    public static short[] toShortArray(double[] array) {
        short[] newArray = new short[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (short) array[i];
        }
        return newArray;
    }

    public static short[] toShortArray(float[] array) {
        short[] newArray = new short[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (short) array[i];
        }
        return newArray;
    }

    public static short[] toShortArray(long[] array) {
        short[] newArray = new short[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (short) array[i];
        }
        return newArray;
    }

    public static short[] toShortArray(byte[] array) {
        short[] newArray = new short[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = array[i];
        }
        return newArray;
    }

    public static short[] toShortArray(char[] array) {
        short[] newArray = new short[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (short) array[i];
        }
        return newArray;
    }

    public static short[] toShortArray(Object[] array) {
        short[] newArray = new short[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (short) array[i];
        }
        return newArray;
    }

    public static byte[] toByteArray(String[] array) {
        byte[] newArray = new byte[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = Byte.valueOf(array[i]);
        }
        return newArray;
    }

    public static byte[] toByteArray(int[] array) {
        byte[] newArray = new byte[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (byte) array[i];
        }
        return newArray;
    }

    public static byte[] toByteArray(double[] array) {
        byte[] newArray = new byte[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (byte) array[i];
        }
        return newArray;
    }

    public static byte[] toByteArray(float[] array) {
        byte[] newArray = new byte[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (byte) array[i];
        }
        return newArray;
    }

    public static byte[] toByteArray(long[] array) {
        byte[] newArray = new byte[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (byte) array[i];
        }
        return newArray;
    }

    public static byte[] toByteArray(short[] array) {
        byte[] newArray = new byte[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (byte) array[i];
        }
        return newArray;
    }

    public static byte[] toByteArray(char[] array) {
        byte[] newArray = new byte[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (byte) array[i];
        }
        return newArray;
    }

    public static byte[] toByteArray(Object[] array) {
        byte[] newArray = new byte[array.length];
        if (isEmpty(array))
            return newArray;
        for (int i = 0; i < array.length; i++) {
            newArray[i] = (byte) array[i];
        }
        return newArray;
    }





    /**
     * 数组转列表
     *
     * @param array
     *            an array of T objects.
     * @param <T>
     *            a T object.
     * @return a {@link List} object.
     */
    public static final <T> List<T> convertList(T[] array) {
        if (isEmpty(array)) {
            return null;
        }
        return Arrays.asList(array);
    }

    /**
     * 转Map
     *
     * @param array
     *            String[][]、int[][]、double[][] ...
     * @return
     */
    public static Map convertMap(Object[] array) {
        if (array == null) {
            return null;
        }
        Map map = new HashMap((int) (array.length * 1.5D));
        for (int i = 0; i < array.length; i++) {
            Object object = array[i];
            if ((object instanceof Map.Entry)) {
                Map.Entry entry = (Map.Entry) object;
                map.put(entry.getKey(), entry.getValue());
            } else if ((object instanceof Object[])) {
                Object[] entry = (Object[]) object;
                if (entry.length < 2) {
                    throw new IllegalArgumentException("Array element " + i
                            + ", '" + object + "', has a length less than 2");
                }

                map.put(entry[0], entry[1]);
            } else {
                throw new IllegalArgumentException("Array element " + i + ", '"
                        + object
                        + "', is neither of type Map.Entry nor an Array");
            }
        }

        return map;
    }


    // ---------------------------------------------------------------------- join
    /**
     * 将数组转为字符串，逗号分隔，如 [1,2,3]
     *
     * @param array
     * @return
     */
    public static <T> String join(T[] array) {
        if (array == null)
            return "null";
        int iMax = array.length - 1;
        if (iMax == -1)
            return "[]";

        StringBuilder sb = new StringBuilder();
        sb.append('[');
        for (int i = 0; ; i++) {
            sb.append(array[i]);
            if (i == iMax)
                return sb.append(']').toString();
            sb.append(", ");
        }
    }

    /**
     * 将数组转为字符串，逗号分隔，如 [1,2,3]
     *
     * @param array
     * @return
     */
    public static String join(int[] array) {
        return join(toStringArray(array));
    }

    /**
     * 将数组转为字符串，逗号分隔，如 [1,2,3]
     *
     * @param array
     * @return
     */
    public static String join(double[] array) {
        return join(toStringArray(array));
    }

    /**
     * 将数组转为字符串，逗号分隔，如 [1,2,3]
     *
     * @param array
     * @return
     */
    public static String join(float[] array) {
        return join(toStringArray(array));
    }

    /**
     * 将数组转为字符串，逗号分隔，如 [1,2,3]
     *
     * @param array
     * @return
     */
    public static String join(long[] array) {
        return join(toStringArray(array));
    }

    /**
     * 将数组转为字符串，逗号分隔，如 [1,2,3]
     *
     * @param array
     * @return
     */
    public static String join(short[] array) {
        return join(toStringArray(array));
    }

    /**
     * 将数组转为字符串，逗号分隔，如 [1,2,3]
     *
     * @param array
     * @return
     */
    public static String join(byte[] array) {
        return join(toStringArray(array));
    }

    /**
     * 将数组转为字符串，逗号分隔，如 [1,2,3]
     *
     * @param array
     * @return
     */
    public static String join(boolean[] array) {
        return join(toStringArray(array));
    }

    /**
     * 将数组转为字符串,并以某个字符相连，如 1#2#3
     *
     * @param array
     * @param splitStr
     * @return
     */
    public static <T> String join(T[] array, String splitStr) {
        StringBuilder sb = new StringBuilder();
        if (isEmpty(array)) {
            return "";
        }
        if (array.length == 1)
            return String.valueOf(array[0]);
        int iMax = array.length - 1;
        for (int i = 0; i < array.length; i++) {
            if (i == iMax) {
                sb.append(array[i]);
            } else {
                sb.append(array[i] + splitStr);
            }
        }

        return sb.toString();
    }

    /**
     * 将数组转为字符串,并以某个字符相连，如 1#2#3
     *
     * @param array
     * @return
     */
    public static String join(int[] array, String splitStr) {
        return join(toStringArray(array), splitStr);
    }

    /**
     * 将数组转为字符串,并以某个字符相连，如 1#2#3
     *
     * @param array
     * @return
     */
    public static String join(double[] array, String splitStr) {
        return join(toStringArray(array), splitStr);
    }

    /**
     * 将数组转为字符串,并以某个字符相连，如 1#2#3
     *
     * @param array
     * @return
     */
    public static String join(float[] array, String splitStr) {
        return join(toStringArray(array), splitStr);
    }

    /**
     * 将数组转为字符串,并以某个字符相连，如 1#2#3
     *
     * @param array
     * @return
     */
    public static String join(long[] array, String splitStr) {
        return join(toStringArray(array), splitStr);
    }

    /**
     * 将数组转为字符串,并以某个字符相连，如 1#2#3
     *
     * @param array
     * @return
     */
    public static String join(short[] array, String splitStr) {
        return join(toStringArray(array), splitStr);
    }

    /**
     * 将数组转为字符串,并以某个字符相连，如 1#2#3
     *
     * @param array
     * @return
     */
    public static String join(byte[] array, String splitStr) {
        return join(toStringArray(array), splitStr);
    }

    /**
     * 将数组转为字符串,并以某个字符相连，如 1#2#3
     *
     * @param array
     * @return
     */
    public static String join(boolean[] array, String splitStr) {
        return join(toStringArray(array), splitStr);
    }

    /**
     * 拼接Set集合，并以某个字符相连，如 1#2#3
     * @param set
     * @return
     */
    public static <T> String joinSet(Set<T> set, String splitStr) {
        if (set == null || set.size() == 0)
            return "";
        if (set.size() == 1){
            for (T t : set) {
                return String.valueOf(t);
            }
        }

        StringBuilder sb = new StringBuilder();

        for (T t : set) {
            sb.append(String.valueOf(t) + splitStr);
        }
        return sb.toString().substring(0, sb.length() - 1);
    }

    /**
     * 拼接List集合,并以某个字符相连，如 1#2#3
     * @param list
     * @return
     */
    public static <T> String joinList(List<T> list, String splitStr) {
        if (list == null || list.size() == 0)
            return "";
        if (list.size() == 1){
            return String.valueOf(list.get(0));
        }

        StringBuilder sb = new StringBuilder();

        for (int i = 0; i < list.size(); i++) {
            if (i == list.size() - 1) {
                sb.append(String.valueOf(list.get(i)));
            } else {
                sb.append(String.valueOf(list.get(i)) + splitStr);
            }
        }
        return sb.toString();
    }

    /**
     * 数组截取
     *
     * @param arr
     * @param len
     * @return
     */
    public static int[] subArray(int[] arr, int len) {
        if (arr.length <= len) {
            return arr;
        }
        int[] newArr = new int[len];
        for (int i = 0; i < len; i++) {
            newArr[i] = arr[i];
        }
        return newArr;
    }

    /**
     * 数组截取
     *
     * @param arr
     * @param len
     * @return
     */
    public static Object[] subArray(Object[] arr, int len) {
        if (arr.length <= len) {
            return arr;
        }
        Object[] newArr = new Object[len];
        for (int i = 0; i < len; i++) {
            newArr[i] = arr[i];
        }
        return newArr;
    }



    /**
     * 拼装数组。
     * @param aryStr
     * @param separator
     * @return
     */
    public static String contact(String[] aryStr,String separator){
        if(aryStr.length==1) return aryStr[0];
        String out="";
        for(int i=0;i<aryStr.length;i++){
            if(i==0){
                out+=aryStr[i];
            }
            else{
                out+=separator + aryStr[i];
            }
        }
        return out;
    }

    /**
     * 添加引号。
     * 例如 1,2,3 修改为 '1','2','3'.
     * @param val
     * @return
     */
    public static String addQuote(String val){
        String[] aryVal=val.split(",");

        if(aryVal.length==1) return "'" + val +"'";

        String tmp="";
        for(int i=0;i<aryVal.length;i++){
            if(i==0){
                tmp+="'" + aryVal[i] +"'";
            }
            else{
                tmp+=",'" + aryVal[i] +"'";
            }
        }
        return tmp;
    }


}
