package cn.dolphin.core.util;

import org.springframework.core.BridgeMethodResolver;
import org.springframework.core.DefaultParameterNameDiscoverer;
import org.springframework.core.MethodParameter;
import org.springframework.core.ParameterNameDiscoverer;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.core.annotation.SynthesizingMethodParameter;
import org.springframework.lang.Nullable;
import org.springframework.stereotype.Service;
import org.springframework.web.method.HandlerMethod;

import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;

/**
 * 类注释
 */
public class AnnotationUtil extends org.springframework.util.ClassUtils{

    private static final ParameterNameDiscoverer PARAMETER_NAME_DISCOVERER = new DefaultParameterNameDiscoverer();

    /** 获取指定类所有注解 */
    public static <T> Annotation[] getAnnotations(Class<T> srcClass) {
        return srcClass == null ? null : srcClass.getAnnotations();
    }

//    /**
//     * 获取类中的某种注解
//     *
//     * @param <A>             the type parameter
//     * @param clazz           the clazz
//     * @param annotationClazz the annotation clazz
//     * @return the a
//     */
//    public static <A extends Annotation> A getAnnotation(Class<?> clazz, Class<A> annotationClazz) {
//        return clazz.isAnnotationPresent(annotationClazz) ? clazz.getAnnotation(annotationClazz) : null;
//    }


    /** 获取指定注解 */
    public static <T, A extends Annotation> A getAnnotation(Class<T> srcClass, Class<A> annotationClass) {
        return srcClass == null ? null : srcClass.getAnnotation(annotationClass);
    }

    /** 获取spring.service注解 */
    public static <T> Service getSpringServiceAnnotation(Class<T> srcClass) {
        return srcClass == null ? null : srcClass.getAnnotation(Service.class);
    }

    /** 获取spring.service注解的value属性值 */
    public static <T> String getSpringServiceAnnotationValue(Class<T> srcClass) {
        Service service = getSpringServiceAnnotation(srcClass);
        return service == null ? null : service.value();
    }


    /**
     * 获取方法参数信息
     * @param constructor 构造器
     * @param parameterIndex 参数序号
     * @return {MethodParameter}
     */
    public static MethodParameter getMethodParameter(Constructor<?> constructor, int parameterIndex) {
        MethodParameter methodParameter = new SynthesizingMethodParameter(constructor, parameterIndex);
        methodParameter.initParameterNameDiscovery(PARAMETER_NAME_DISCOVERER);
        return methodParameter;
    }

    /**
     * 获取方法参数信息
     * @param method 方法
     * @param parameterIndex 参数序号
     * @return {MethodParameter}
     */
    public static MethodParameter getMethodParameter(Method method, int parameterIndex) {
        MethodParameter methodParameter = new SynthesizingMethodParameter(method, parameterIndex);
        methodParameter.initParameterNameDiscovery(PARAMETER_NAME_DISCOVERER);
        return methodParameter;
    }

    /**
     * 获取Annotation
     * @param method Method
     * @param annotationType 注解类
     * @param <A> 泛型标记
     * @return {Annotation}
     */
    @Nullable
    public static <A extends Annotation> A getAnnotation(Method method, Class<A> annotationType) {
        Class<?> targetClass = method.getDeclaringClass();
        // The method may be on an interface, but we need attributes from the target class.
        // If the target class is null, the method will be unchanged.
        Method specificMethod = AnnotationUtil.getMostSpecificMethod(method, targetClass);
        // If we are dealing with method with generic parameters, find the original method.
        specificMethod = BridgeMethodResolver.findBridgedMethod(specificMethod);
        // 先找方法，再找方法上的类
        A annotation = AnnotatedElementUtils.findMergedAnnotation(specificMethod, annotationType);;
        if (null != annotation) {
            return annotation;
        }
        // 获取类上面的Annotation，可能包含组合注解，故采用spring的工具类
        return AnnotatedElementUtils.findMergedAnnotation(specificMethod.getDeclaringClass(), annotationType);
    }

    /**
     * 获取Annotation
     * @param handlerMethod HandlerMethod
     * @param annotationType 注解类
     * @param <A> 泛型标记
     * @return {Annotation}
     */
    @Nullable
    public static <A extends Annotation> A getAnnotation(HandlerMethod handlerMethod, Class<A> annotationType) {
        // 先找方法，再找方法上的类
        A annotation = handlerMethod.getMethodAnnotation(annotationType);
        if (null != annotation) {
            return annotation;
        }
        // 获取类上面的Annotation，可能包含组合注解，故采用spring的工具类
        Class<?> beanType = handlerMethod.getBeanType();
        return AnnotatedElementUtils.findMergedAnnotation(beanType, annotationType);
    }

}
