package cn.dolphin.core.spring;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.aop.framework.AdvisedSupport;
import org.springframework.aop.framework.AopProxy;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Component;
import org.springframework.web.context.support.XmlWebApplicationContext;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Spring Context 工具类
 */
@Component
@Slf4j
@SuppressWarnings("all")
public class SpringContextUtil implements ApplicationContextAware , DisposableBean {

    private static final Map<String, Object> CACHE_MAP = new HashMap<>();

    private static ApplicationContext applicationContext;

    //使用volatile关键字保其可见性
    volatile private static SpringContextUtil instance = null;

    private SpringContextUtil(){}

    public static SpringContextUtil getInstance() {
        try {
            if (instance == null) {//懒汉式
                //创建实例之前可能会有一些准备性的耗时工作
                Thread.sleep(300);
                synchronized (SpringContextUtil.class) {
                    if (instance == null) {//二次检查
                        instance = new SpringContextUtil();
                    }
                }
            }
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        return instance;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
        log.info("@Springboot Starting [SpringContext处理机制] ");
    }

    /**获取spring上下文*/
    public static ApplicationContext getApplicationContext() {
        return applicationContext;
    }

    /**
     * 根据Class从Spring容器中获取Bean对象
     * @param name
     * @param <T>
     * @return
     */
    public static <T> T getBean(Class<T> clazz) {
        return getApplicationContext().getBean(clazz);
    }


    /**
     * 根据Bean的名称从Spring容器中获取Bean对象
     * @param name
     * @param <T>
     * @return
     */
    public static <T> T getBean(String name) {
        return (T)getApplicationContext().getBean(name);
    }

    /**
     * 根据Bean名称、Class从Spring容器中获取Bean对象
     * @param name
     * @param requiredType
     * @param <T>
     * @return
     */
    public static <T> T getBean(String name, Class<T> requiredType) {
        return getApplicationContext().getBean(name, requiredType);
    }

    /**
     * 根据类名获取到bean(实际对象)
     *
     * @param <T>
     * @param clazz
     * @return
     * @throws BeansException
     */
    public static <T> T getBeanByClass(Class<T> clazz){
        try {
            return (T)getTarget(getApplicationContext().getBean(clazz));
        } catch (Exception e) {
            log.error("获取对象错误:",e);
            return null;
        }
    }
    /**
     * 根据class 从beanfactory中找到对应的实现类集合
     *
     * @param clazz clazz
     * @param <T>   class的类型
     * @return beanfactory中找到对应的实现类集合
     */
    public static <T> List<T> getBeansByClass(Class<T> clazz) {
        String[] names = getApplicationContext().getBeanNamesForType(clazz);
        List<T> result = new ArrayList<T>();
        for (String name : names) {
            result.add((T) SpringContextUtil.getBean(name));
        }
        return result;
    }

    /**
     * 获取微服务接口的实现类
     *
     * @param clazz 微服务接口
     * @return
     */
    @SuppressWarnings("unchecked")
    public static <T> T getBeanByClassForApi(Class<T> clazz) {
        String[] names = SpringContextUtil.getApplicationContext().getBeanNamesForType(clazz);
        T resultBean = null;
        String serviceName = clazz.getName();
        if (names.length > 1) {
            for (String name : names) {
                if (name.equals(serviceName)) {
                    continue;
                }
                resultBean = (T) SpringContextUtil.getBean(name);
                break;
            }
        }
        if (resultBean == null) {
            resultBean = (T) SpringContextUtil.getBean(serviceName);
        }
        return resultBean;
    }

    /**
     * 泛型注入
     *
     * @param clazz
     * @param actualTypeArguments
     * @return
     */
    @SuppressWarnings("unchecked")
    public static <T> T getBeanByClass(Class<T> clazz, String[] actualTypeArguments) {
        StringBuilder sBuilder = new StringBuilder();
        String cacheKey = null;
        for (String actualTypeArgument : actualTypeArguments) {
            sBuilder.append("_" + actualTypeArgument);
        }
        cacheKey = clazz.getName() + sBuilder.toString();
        if (CACHE_MAP.containsKey(cacheKey)) {
            return (T) CACHE_MAP.get(cacheKey);
        }
        // 获取候选人id
        String[] candidateNames = applicationContext.getBeanNamesForType(clazz);
        Object object = null;
        Type[] types = null;

        //遍历所有的候选人，看候选人的泛型
        for (String candidateName : candidateNames) {
            object = getBean(candidateName);

            if (object.getClass().getName().contains("EnhancerBySpringCGLIB")) {
                types = ((ParameterizedType) object.getClass().getSuperclass().getGenericSuperclass()).getActualTypeArguments();
            } else {
                types = ((ParameterizedType) object.getClass().getGenericSuperclass()).getActualTypeArguments();

            }
            boolean isThisObj = true;
            for (int i = 0; i < actualTypeArguments.length; i++) {

                if (!actualTypeArguments[i].equals(types[i].getTypeName())) {
                    isThisObj = false;
                    break;
                }

            }
            if (!isThisObj) {
                continue;
            }
            CACHE_MAP.put(cacheKey, object);
            return (T) object;
        }
        return null;
    }



    /**
     * 是否包含bean
     *
     * @param name
     * @return
     */
    public static boolean containsBean(String name) {
        return getApplicationContext().containsBean(name);
    }

    /**
     * 是否是单例
     * 判断以给定名字注册的bean定义是一个singleton还是一个prototype。 如果与给定名字相应的bean定义没有被找到，将会抛出一个异常（NoSuchBeanDefinitionException）
     * @param name
     * @return
     */
    public static boolean isSingleton(String name) {
        return getApplicationContext().isSingleton(name);
    }

    /**
     * bean的类型
     *
     * @param name
     * @return
     */
    public static Class<? extends Object> getType(String name) {
        return getApplicationContext().getType(name);
    }

    /**
     * 如果给定的bean名字在bean定义中有别名，则返回这些别名
     *
     * @return
     */
    public static String[] getAliases(String beanId) {
        return getApplicationContext().getAliases(beanId);
    }


    /**获取环境信息*/
    public static Environment getEnvironment() {
        return getApplicationContext().getEnvironment();
    }

    /**
     * 获取Spring容器中的BeanDefinition Map
     * @return 设定文件 Map<String,BeanDefinition> 返回类型
     */
    public static Map<String, BeanDefinition> getApplicationBeanDefinitions() {
        Map<String, BeanDefinition> map = new HashMap<String, BeanDefinition>();
        XmlWebApplicationContext context = (XmlWebApplicationContext)applicationContext;
        ConfigurableListableBeanFactory factory = context.getBeanFactory();
        String[] names = factory.getBeanDefinitionNames();
        for (String name : names) {
            map.put(name, factory.getBeanDefinition(name));
        }
        return map;
    }

    /**
     * 实现DisposableBean接口, 在Context关闭时清理静态变量.
     */
    @Override
    public void destroy() throws Exception {
        SpringContextUtil.cleanHolder();
    }

    /**
     * 清除Spring容器的持有
     */
    public static void cleanHolder() {
        applicationContext = null;
    }


    /**
     * 检查ApplicationContext不为空.
     */
    private static void assertContextInjected() {
        Validate.validState(applicationContext != null, "applicaitonContext属性未注入, 请在applicationContext.xml中定义SpringContextHolder.");
    }


    /**
     * 获取 目标对象
     *
     * @param proxy 代理对象
     * @return
     * @throws Exception
     */
    public static Object getTarget(Object proxy)  {
        if (!AopUtils.isAopProxy(proxy)) {
            return proxy;//不是代理对象
        }

        if (AopUtils.isJdkDynamicProxy(proxy)) {
            try {
                proxy = getJdkDynamicProxyTargetObject(proxy);
            } catch (Exception e) {
                log.error("获取对象错误:",e);
                return proxy;
            }
        } else { //cglib
            try {
                proxy = getCglibProxyTargetObject(proxy);
            } catch (Exception e) {
                log.error("获取对象错误:",e);
                return proxy;
            }
        }
        return getTarget(proxy);
    }

    private static Object getCglibProxyTargetObject(Object proxy) throws Exception {
        Field h = proxy.getClass().getDeclaredField("CGLIB$CALLBACK_0");
        h.setAccessible(true);
        Object dynamicAdvisedInterceptor = h.get(proxy);
        Field advised = dynamicAdvisedInterceptor.getClass().getDeclaredField("advised");
        advised.setAccessible(true);
        Object target = ((AdvisedSupport) advised.get(dynamicAdvisedInterceptor)).getTargetSource().getTarget();
        return target;
    }


    private static Object getJdkDynamicProxyTargetObject(Object proxy) throws Exception {
        Field h = proxy.getClass().getSuperclass().getDeclaredField("h");
        h.setAccessible(true);
        AopProxy aopProxy = (AopProxy) h.get(proxy);
        Field advised = aopProxy.getClass().getDeclaredField("advised");
        advised.setAccessible(true);
        Object target = ((AdvisedSupport) advised.get(aopProxy)).getTargetSource().getTarget();
        return target;
    }
}
