package cn.dolphin.core.sec;

import cn.dolphin.core.consts.CharsetConstant;
import cn.dolphin.core.util.StrUtil;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import java.security.SecureRandom;


/**
 * AES 对称加密(单密钥算法，密钥可用于加密解密)
 *
 */
@SuppressWarnings("all")
public class AESUtil {

    /**
     * 加密算法
     */
    public static String ALGORITHM = "AES";
    /**
     * 密钥签名算法
     */
    public static String SHA1PRNG = "SHA1PRNG";

    /**
     * 密钥长度(AES密钥长度128、192、256位，但受地区出口限制，只能用128位)
     */
    public static int KEY_SIZE = 128;


    /**
     * 生成AES密钥(固定)
     * @param keyCode (相同的KEY_CODE永远获取相同的密钥，所以KEY_CODE等同于密钥)
     * @return 密钥
     */
    public static String getKey(String keyCode) {
        try {
            // 实例化密钥生成器
            KeyGenerator kg = KeyGenerator.getInstance(ALGORITHM);
            // SHA1密码随机数供应者（强随机数密钥，如果指定了KEY_CODE则随机数固定）
            SecureRandom secureRandom = SecureRandom.getInstance(SHA1PRNG);
            secureRandom.setSeed(keyCode.getBytes(CharsetConstant.UTF8));
            // 初始化
            kg.init(KEY_SIZE, secureRandom);
            // 生成秘密密钥
            SecretKey secretKey = kg.generateKey();
            // 获得密钥的二进制编码形式，再转换为16进制
            return HexUtil.encode(secretKey.getEncoded()) ;
        } catch (Exception e) {
            throw new RuntimeException("生成AES密钥失败",e);
        }
    }

    /**
     * 生成AES密钥(随机)
     * @return 密钥
     */
    public static String getKey() {
        try {
            // 实例化密钥生成器
            KeyGenerator kg = KeyGenerator.getInstance(ALGORITHM);
            // 初始化
            kg.init(KEY_SIZE, new SecureRandom());
            // 生成秘密密钥
            SecretKey secretKey = kg.generateKey();
            // 获得密钥的二进制编码形式，再转换为16进制
            return HexUtil.encode(secretKey.getEncoded()) ;
        } catch (Exception e) {
            throw new RuntimeException("生成AES密钥失败",e);
        }
    }

    /**
     * AES加密
     * @param data 源数据
     * @param key 密钥
     * @return 加密数据
     */
    public static byte[] encode(byte[] data, String key) {
        // 还原密钥
        SecretKey secretKey = new SecretKeySpec(HexUtil.decode(key), ALGORITHM);
        try {
            // 创建密码器
            Cipher cipher = Cipher.getInstance(ALGORITHM);
            // 初始化密码器，设置为加密模式
            cipher.init(Cipher.ENCRYPT_MODE, secretKey);
            // 加密原数据
            return cipher.doFinal(data);
        } catch (Exception e) {
            throw new RuntimeException("AES加密",e);
        }
    }


    /**
     * AES解密
     * @param data 加密数据
     * @param key 密钥
     * @return 源数据
     */
    public static byte[] decode(byte[] data, String key) {
        // 还原为密钥
        SecretKey secretKey = new SecretKeySpec(HexUtil.decode(key), ALGORITHM);
        try {
            // 创建密码器
            Cipher cipher = Cipher.getInstance(ALGORITHM);
            // 初始化密码器，设置为解密模式
            cipher.init(Cipher.DECRYPT_MODE, secretKey);
            // 解密加密数据
            return cipher.doFinal(data);
        } catch (Exception e) {
            throw new RuntimeException("AES解密失败",e);
        }
    }


    /**
     * AES加密
     * @param text 源数据
     * @param key 密钥
     * @return 加密数据
     */
    public static String encode(String text, String key) {
        return Base64Util.encodeUrl(encode(StrUtil.getBytes(text),key));
    }

    /**
     * AES解密
     * @param text 加密数据
     * @param key 密钥
     * @return 源数据
     */
    public static String decode(String text, String key) {
        return StrUtil.getStr(decode(Base64Util.decode(text), key));
    }

}
