package cn.dolphin.core.regex;


import com.google.common.collect.Lists;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;
import org.apache.commons.lang3.StringUtils;

import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 正则表达式
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public class RegexUtil {

    /**
     * 非法字符正则校验
     */
    private static final Pattern LEGAL_PATTERN = Pattern.compile("^[\\u4e00-\\u9fa5\\w\\d_.%@!！\\[\\]\\\\|\\-,/,\\[\\]\\s，。;；:：“”?？]+$");

    /**
     * 纯数字校验
     */
    private static final Pattern NUMBER_PATTERN = Pattern.compile("^\\d+$");

    /**
     * 用户名正则表达式
     */
    private static final Pattern USER_NAME_PATTERN = Pattern.compile("^[a-zA-Z0-9]{4,20}$");
    /**
     * 正则：用户名，取值范围为a-z,A-Z,0-9,"_",汉字，不能以"_"结尾,用户名必须是6-20位
     */
    public static final Pattern REGEX_USERNAME = Pattern.compile("^[A-Za-z\\u4e00-\\u9fa5]{4,30}(?<!_)$");

    /**
     * 手机号正则表达式
     */
    private static final Pattern PHONE_NUMBER_PATTERN = Pattern.compile("^1[345789]\\d{9}$");

    /**
     * 正则：手机号（简单）
     */
    public static final Pattern REGEX_MOBILE_SIMPLE = Pattern.compile("^[1]\\d{10}$");
    /**
     * 正则：手机号（精准）
     * 13 14 15 16 17 18 19 开头都可以
     */
    public static final Pattern REGEX_MOBILE_EXACT = Pattern.compile("^1[3|4|5|6|7|8|9][0-9]\\d{8}$");
    /**
     * 正则：电话号码
     */
    public static final Pattern REGEX_TEL = Pattern.compile("^0\\d{2,3}[- ]?\\d{7,8}");

    /**
     * 字母数字正则表达式
     */
    private static final Pattern ALPHA_NUM = Pattern.compile("^[a-zA-Z0-9]+$");

    /**
     * 空串校验
     */
    private static final Pattern EMPTY = Pattern.compile("^\\s*$");

    /**
     * 正则：金额
     */
    public static final Pattern REGEX_MONEY = Pattern.compile("^(([1-9]\\d*)|0)(\\.\\d{1,2})?$");

    /**
     * 正则：英文单词或者数字
     */
    public static final Pattern REGEX_WORD_OR_NUMBER = Pattern.compile("^[A-Za-z0-9]+$");
    /**
     * 正则：身份证号码15位
     */
    public static final Pattern REGEX_ID_CARD15 = Pattern.compile("^[1-9]\\d{7}((0\\d)|(1[0-2]))(([0|1|2]\\d)|3[0-1])\\d{3}$");
    /**
     * 正则：身份证号码18位
     */
    public static final Pattern REGEX_ID_CARD18 = Pattern.compile("^[1-9]\\d{5}[1-9]\\d{3}((0\\d)|(1[0-2]))(([0|1|2]\\d)|3[0-1])\\d{3}([0-9Xx])$");
    /**
     * 正则：邮箱
     */
    public static final Pattern REGEX_EMAIL = Pattern.compile("^\\w+([-+.]\\w+)*@\\w+([-.]\\w+)*\\.\\w+([-.]\\w+)*$");
    /**
     * 正则：URL
     */
    public static final Pattern REGEX_URL = Pattern.compile("^(http|https)://.*$");
    /**
     * 正则：汉字
     */
    public static final Pattern REGEX_ZH = Pattern.compile("^[\\u4e00-\\u9fa5]+$");
    /**
     * 正则：yyyy-MM-dd格式的日期校验，已考虑平闰年
     */
    public static final Pattern REGEX_DATE = Pattern.compile("^(?:(?!0000)[0-9]{4}-(?:(?:0[1-9]|1[0-2])-(?:0[1-9]|1[0-9]|2[0-8])|(?:0[13-9]|1[0-2])-(?:29|30)|(?:0[13578]|1[02])-31)|(?:[0-9]{2}(?:0[48]|[2468][048]|[13579][26])|(?:0[48]|[2468][048]|[13579][26])00)-02-29)$");
    /**
     * 正则：IP地址
     */
    public static final Pattern REGEX_IP = Pattern.compile("((2[0-4]\\d|25[0-5]|[01]?\\d\\d?)\\.){3}(2[0-4]\\d|25[0-5]|[01]?\\d\\d?)");
    /**
     * 正则：双字节字符(包括汉字在内)
     */
    public static final Pattern REGEX_DOUBLE_BYTE_CHAR = Pattern.compile("[^\\x00-\\xff]");
    /**
     * 正则：空白行
     */
    public static final Pattern REGEX_BLANK_LINE = Pattern.compile("\\n\\s*\\r");
    /**
     * 正则：QQ号
     */
    public static final Pattern REGEX_QQ_NUM = Pattern.compile("[1-9][0-9]{4,}");
    /**
     * 正则：中国邮政编码
     */
    public static final Pattern REGEX_ZIP_CODE = Pattern.compile("[1-9]\\d{5}(?!\\d)");

    //region 以下摘自http://tool.oschina.net/regex
    /**
     * 正则：正整数
     */
    public static final Pattern REGEX_POSITIVE_INTEGER = Pattern.compile("^[1-9]\\d*$");
    /**
     * 正则：负整数
     */
    public static final Pattern REGEX_NEGATIVE_INTEGER = Pattern.compile("^-[1-9]\\d*$");
    /**
     * 正则：整数
     */
    public static final Pattern REGEX_INTEGER = Pattern.compile("^-?[1-9]\\d*$");
    /**
     * 正则：非负整数(正整数 + 0)
     */
    public static final Pattern REGEX_NOT_NEGATIVE_INTEGER = Pattern.compile("^[1-9]\\d*|0$");
    /**
     * 正则：非正整数（负整数 + 0）
     */
    public static final Pattern REGEX_NOT_POSITIVE_INTEGER = Pattern.compile("^-[1-9]\\d*|0$");
    /**
     * 正则：正浮点数
     */
    public static final Pattern REGEX_POSITIVE_FLOAT = Pattern.compile("^[1-9]\\d*\\.\\d*|0\\.\\d*[1-9]\\d*$");
    /**
     * 正则：负浮点数
     */
    public static final Pattern REGEX_NEGATIVE_FLOAT = Pattern.compile("^-[1-9]\\d*\\.\\d*|-0\\.\\d*[1-9]\\d*$");

    /**
     * 验证字符串是否匹配指定正则表达式
     *
     * @param content 字符串
     * @param regExp  正则表达式
     * @return
     */
    public static boolean matcher(String content, String regExp){
        Pattern pattern = Pattern.compile(regExp);
        Matcher matcher = pattern.matcher(content);
        return matcher.matches();
    }

    /**
     * 验证字符串是否匹配指定正则表达式
     * @param content 字符串
     * @param pattern 正则表达式
     * @return
     */
    public static boolean matcher(String content,Pattern pattern){
        Matcher matcher = pattern.matcher(content);
        return matcher.matches();
    }

    /**
     * 是否包含中文的校验
     * @param s 需要校验的字符串
     * @return true 包含, false 不包含
     */
    public static boolean containChinese(String s) {
        if (s != null && !s.isEmpty()) {
            Matcher matcher = Pattern.compile(Regex.CHINESE.getValue()).matcher(s);
            if (matcher.find()) {
                return true;
            }
        }
        return false;
    }

    /**
     * 判断是否包含中文, 且长度小于length
     * @param s 需要校验的字符串
     * @param length 需要判断的长度
     * @return true 包含中文, 且长度小于length, false 不包含中文, 或长度大于length
     */
    public static boolean containChinese(String s, int length) {
        if (containChinese(s)) {
            return s.length() <= length;
        }
        return false;
    }

    /**
     * 字符串长度限制
     * @param s 待检验的字符串
     * @param length 限制长度 <=
     * @return false 未超出, true 超出
     */
    public static boolean overLengthLimit(String s, int length) {
        if (s != null) {
            return s.length() > length;
        }
        return false;
    }

    /**
     * 不包含中文
     * @param s 需要校验的字符串
     * @return true 不包含, false 包含
     */
    public static boolean notContainChinese(String s) {
        return !containChinese(s);
    }

    /**
     * 不包含非法字符校验 (除 中文, 英文, 数字, _ 以外的都属于非法字符)
     * @param s 需要校验的字符串
     * @return false 包含非法字符,
     */
    public static boolean notContainIllegalParameter(String s) {
        if (s == null || s.isEmpty()) {
            // 如果为空, 判断不包含非法字符
            return true;
        }
        Matcher matcher = LEGAL_PATTERN.matcher(s);
        return matcher.matches();
    }

    /**
     * 只有整数的校验方法
     * @param s 待校验的字符串
     * @return
     * true 只有数字
     * false 不含有数字, 为空, 或者含有其他字符
     */
    public static boolean onlyIntegerNumber(String s) {
        if (s == null || s.isEmpty()) {
            return false;
        }
        Matcher matcher = NUMBER_PATTERN.matcher(s);
        return matcher.matches();
    }

    /**
     * 判断输入的字符串是否符合Email格式.
     *
     * @param email
     * 				传入的字符串
     * @return 符合Email格式返回true，否则返回false
     */
    public static boolean isEmail(String email) {
        if (email == null || email.length() < 1 || email.length() > 256) {
            return false;
        }
        Pattern pattern = Pattern.compile("^\\w+([-+.]\\w+)*@\\w+([-.]\\w+)*\\.\\w+([-.]\\w+)*$");
        return pattern.matcher(email).matches();
    }

    /**
     * 判断是否为整数
     *
     * @param value
     * 			传入的字符串
     * @return
     */
    public static boolean isInteger(String value) {
        Pattern pattern = Pattern.compile("^[-\\+]?[\\d]+$");
        return pattern.matcher(value).matches();
    }

    /**
     * 判断是否为浮点数，包括double和float
     *
     * @param value
     * 			传入的字符串
     * @return
     */
    public static boolean isDouble(String value) {
        Pattern pattern = Pattern.compile("^[-\\+]?\\d+\\.\\d+$");
        return pattern.matcher(value).matches();
    }

    /**
     * 手机号校验
     * @param s
     * @return
     * true 是合法手机号
     * false 不是手机号或者为空
     */
    public static boolean isPhoneNumber(String s) {
        if (s == null || s.isEmpty()) {
            return false;
        }
        Matcher matcher = PHONE_NUMBER_PATTERN.matcher(s);
        return matcher.matches();
    }

    /**
     * 判断输入的字符串是否为纯汉字
     * @autor:chenssy
     * @date:2014年8月7日
     *
     * @param value
     * 				传入的字符串
     * @return
     */
    public static boolean isChinese(String value) {
        Pattern pattern = Pattern.compile("[\u0391-\uFFE5]+$");
        return pattern.matcher(value).matches();
    }

    public static boolean isUserName(String un){
        if(StringUtils.isBlank(un)){
            return false;
        }
        Matcher matcher = USER_NAME_PATTERN.matcher(un);
        return matcher.matches();
    }

    /**
     * 仅仅包含字母数字的校验
     * @param s
     */
    public static boolean onlyAlphaNum (String s) {
        if (StringUtils.isBlank(s)) {
            return false;
        }
        Matcher matcher = ALPHA_NUM.matcher(s);
        return matcher.matches();
    }


    /**
     * 空判断
     * @param s
     * @return
     */
    public static boolean isEmpty(String s) {
        if (null == s) {
            return true;
        }
        if (EMPTY.matcher(s).matches()) {
            return true;
        }
        return false;
    }

    public static String urlDecode(String param){
        try {
            if (StringUtils.isNotBlank(param)){
                String result = URLDecoder.decode(param,"UTF-8");
                if( StringUtils.isNotBlank(result) ){
                    return result;
                }
            }
        }catch (Exception e){
            e.printStackTrace();
        }
        return param;
    }

    public static String urlEncode(String param){
        try {
            if (StringUtils.isNotBlank(param)){
                String result = URLEncoder.encode(param,"UTF-8");
                if( StringUtils.isNotBlank(result) ){
                    return result;
                }
            }
        }catch (Exception e){
            e.printStackTrace();
        }

        return param;
    }

    public static String urlEncodeOneByOne(String param){
        String[] list = param.split("");
        StringBuilder result = new StringBuilder();
        for ( String item : list ) {
            if( containChinese(item) ){
                result.append(urlEncode(item));
            }else{
                result.append(item);
            }
        }
        return result.toString();
    }

    public static boolean containsOne(String string,String regex){
        Pattern p1 = Pattern.compile(regex);
        Matcher m1 = p1.matcher(string);
        Integer count = 0;
        while(m1.find()) {
            count ++ ;
            if( count > 1 ){
                return false;
            }
        }
        return true;
    }

    public static boolean containsMore(String string,String regex,Integer time){
        if( time <= 0 ){ return false; }
        Pattern p1 = Pattern.compile(regex);
        Matcher m1 = p1.matcher(string);
        Integer count = 0;
        do {
            count ++ ;
            if( count > time ){
                return true;
            }
        }while(m1.find());
        return false;
    }

    /**
     * 统计字符在文本中出现的次数
     **/
    public static List<String> gatherStringCount(String content,String regex) {
        Pattern p1 = Pattern.compile(regex);
        Matcher m1 = p1.matcher(content);
        List<String> colgroups = Lists.newArrayList();
        while (m1.find()) {
            for (int i = 0; i < m1.groupCount(); i++) {
                String test1 = m1.group(i);
                colgroups.add(test1);
            }
        }
        return colgroups;
    }


}
