package cn.dolphin.core.page;

import cn.dolphin.core.map.MapUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 页码计算实用函数
 */
@Slf4j
@SuppressWarnings("all")
public class PageUtils
{
    private PageUtils() {
    }

    /**
     * 分页参数，起始位置，从0开始
     */
    public static final String PAGE = "page";
    /**
     * 分页参数，每页数据条数
     */
    public static final String LIMIT = "pageSize";

    /**
     * 转换并校验分页参数<br>
     * mybatis中limit #{start, JdbcType=INTEGER}, #{length,
     * JdbcType=INTEGER}里的类型转换貌似失效<br>
     * 我们这里先把他转成Integer的类型
     *
     * @param params 参数
     * @param required 是否分页 true 不分页 false 分页
     * 分页参数是否是必填
     */
    public static void pageParamConver(Map<String, Object> params, boolean required) {
        if (required) {// 分页参数必填时，校验参数
            if (params == null || !params.containsKey(PAGE) || !params.containsKey(LIMIT)) {
                throw new IllegalArgumentException("请检查分页参数," + PAGE + "," + LIMIT);
            }
        }

        if (!CollectionUtils.isEmpty(params)) {
            if (params.containsKey(PAGE)) {
                Integer start = MapUtil.getInteger(params, PAGE,1);
                Integer length = MapUtil.getInteger(params, LIMIT,10);
                if (start < 0) {
                    log.error("page：{}，重置为0", start);
                    start = 0;
                }
                params.put(PAGE, (start-1)*length);
            }

            if (params.containsKey(LIMIT)) {
                Integer length = MapUtil.getInteger(params, LIMIT,10);
                if (length < 0) {
                    log.error("limit：{}，重置为0", length);
                    length = 0;
                }
                params.put(LIMIT, length);
            }
        }
    }

    /**
     * 取得首页记录数
     * @param currentPage
     * @param pageSize
     * @return
     */
    public static int getFirstNumber(int currentPage, int pageSize)
    {
        if(currentPage <=0){
            currentPage = 1;
        }
        if (pageSize <= 0)
            throw new IllegalArgumentException("[pageSize] must great than zero");
        return (currentPage - 1)*pageSize;
    }

    public static int getLastNumber(int currentPage,int pageSize,int totalCount)
    {
        int last=currentPage * pageSize;
        if(last > totalCount) return totalCount;
        return last;
    }

    /**
     * 取得数字导航，例如 2,3,4,6
     * @param currentPage
     * @param totalPage
     * @param count
     * @return
     */
    public static List<Integer> getPageNumbers(int currentPage, int totalPage, int count)
    {
        int avg = count / 2;
        int startPage = currentPage - avg;
        if (startPage <= 0){
            startPage = 1;
        }
        int endPage = startPage + count - 1;
        if (endPage > totalPage){
            endPage = totalPage;
        }
        if (endPage - startPage < count){
            startPage = endPage - count;
            if (startPage <= 0){
                startPage = 1;
            }
        }
        List<Integer> result = new ArrayList<Integer>();
        for (int i = startPage; i <= endPage; i++){
            result.add(new Integer(i));
        }
        return result;
    }

    /**
     * 取得尾页页码
     * @param totalCount  总的记录数
     * @param pageSize	页大小
     * @return
     */
    public static int getTotalPage(int totalCount , int pageSize)
    {
        int result = totalCount  % pageSize == 0 ? totalCount  / pageSize : totalCount  / pageSize + 1;
        if (result <= 1)
            result = 1;
        return result;
    }

    /**
     * 取得实际的页码
     * @param currentPage	当前页
     * @param pageSize		页大小
     * @param totalCount	总记录数
     * @return
     */
    public static int getPageNumber(int currentPage, int pageSize, int totalCount)
    {
        if (currentPage <= 1)
        {
            return 1;
        }
        if (Integer.MAX_VALUE == currentPage || currentPage > getTotalPage(totalCount, pageSize))
        { // last page
            return getTotalPage(totalCount, pageSize);
        }
        return currentPage;
    }


}

