package cn.dolphin.core.mybatis;

import cn.dolphin.core.jdbc.JdbcDao;
import cn.dolphin.core.spring.SpringContextUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.ibatis.mapping.*;
import org.apache.ibatis.reflection.MetaObject;
import org.apache.ibatis.session.Configuration;
import org.apache.ibatis.session.SqlSession;
import org.apache.ibatis.session.SqlSessionFactory;
import org.apache.ibatis.type.TypeHandlerRegistry;
import org.mybatis.spring.SqlSessionTemplate;

import java.io.Serializable;
import java.util.Collection;
import java.util.List;


/**
 * mybatis 操作基类，实现mybatis 基本操作
 * 继承了JDBC扩展
 */
public abstract class BaseMyBatisDao<T, ID extends Serializable> extends JdbcDao<T,ID> {
    protected final Log log = LogFactory.getLog(getClass());

    /**
     * 数据库连接ThreadLocal
     */
    ThreadLocal<SqlSessionFactory> sqlSessionFactoryThreadLocal = new ThreadLocal<SqlSessionFactory>();

    /*
     * 用户维护线程资源的一个控制对象.
     * 可以保证,相同的线程中,只有唯一的一个资源对象,与之对应.
     * 功能类似自定义的属性currentSessions
     * 结构和currentSessions几乎一样.
     * ThreadLocal中有属性,Map集合类型的属性.
     * 集合的key是当前线程对象, 集合的value是要绑定的资源对象.
     * 常用方法 :
     * set(T t) -> map.put(Thread.currentThread(), t);
     * get() -> map.get(Thread.currentThread());
     * remove() -> map.remove(Thread.currentThread());
     */
    private static ThreadLocal<SqlSession> localSessions = new ThreadLocal<SqlSession>();


    /**
     * 注入sqlSessionFactory
     */
    private SqlSessionFactory sqlSessionFactory;

    /**
     * 注入sqlSessionTemplate
     */
    private SqlSessionTemplate sqlSessionTemplate;

    /**
     * 检查注入是sqlSessionFactory
     * @throws IllegalArgumentException
     */
    @Override
    protected void checkDaoConfig() throws IllegalArgumentException {
//        notNull(this.openSession(), "Property 'sqlSessionFactory' or 'sqlSessionTemplate' are required");

//        if(sqlSessionFactory == null){
//            sqlSessionFactory = sqlSessionFactoryThreadLocal.get();
//            if(sqlSessionFactory == null) {
//                sqlSessionFactory = SpringContextUtil.getBean(SqlSessionFactory.class);
//                sqlSessionFactoryThreadLocal.set(sqlSessionFactory);
//            }
//        }

        //Assert.notNull(sqlSessionFactory, "sqlSessionFactory must be not null");
    }

    /**
     * 返回SqlSessionFactory
     * @return
     */
    public SqlSessionFactory getSqlSessionFactory() {
        if(sqlSessionFactory == null){
            sqlSessionFactory = sqlSessionFactoryThreadLocal.get();
            if(sqlSessionFactory == null) {
                sqlSessionFactory = SpringContextUtil.getBean(SqlSessionFactory.class);
                setSqlSessionFactory(sqlSessionFactory);
                sqlSessionFactoryThreadLocal.set(sqlSessionFactory);
            }
        }
        return sqlSessionFactory;
    }


    /**
     * 注入sqlSessionFactory
     * @param sqlSessionFactory
     */
    public void setSqlSessionFactory(SqlSessionFactory sqlSessionFactory) {
        this.sqlSessionFactory = sqlSessionFactory;
        this.sqlSessionTemplate = new SqlSessionTemplate(sqlSessionFactory);
    }


    /**
     * 返回sqlSessionTemplate
     * @return
     */
    public SqlSessionTemplate getSqlSessionTemplate() {
        if(sqlSessionFactory == null){
            sqlSessionFactory = sqlSessionFactoryThreadLocal.get();
            if(sqlSessionFactory == null) {
                sqlSessionFactory = SpringContextUtil.getBean(SqlSessionFactory.class);
                setSqlSessionFactory(sqlSessionFactory);
                sqlSessionFactoryThreadLocal.set(sqlSessionFactory);
            }
        }
        return sqlSessionTemplate;
    }


    /*
     * 会话对象创建方法
     * 功能增强. 一个线程,多次访问当前方法,只创建唯一的一个会话对象.
     * 线程有对象. Thread类型的对象就是线程.
     * 如何获取当前线程? Thread.currentThread();
     * 使用当前线程对象作为key,SqlSession对象作为value,创建一个Map集合.管理会话对象.
     */
    public SqlSession openSession(){
        // 1. 获取当前线程中对应的会话对象.
        // SqlSession session = currentSessions.get(Thread.currentThread());
        SqlSession session = localSessions.get();
        // 2. 判断获取的会话对象是否为null
        if(session == null){
            // 当前线程未创建过会话对象.
            session =  getSqlSessionFactory().openSession();
            // currentSessions.put(Thread.currentThread(), session  );
            localSessions.set(session);
        }
        // 3. 返回结果
        return session;
        // return sqlSessionFactory.openSession();
    }
    /**
     * 回收资源方法. 避免关闭会话对象后,在集合中仍然有会话对象的引用. 造成过时对象的应用.
     */
    public void close(){
        SqlSession session = localSessions.get();
        if(session != null){
            session.close();
        }
        // 将关闭后的资源从集合中删除
        localSessions.remove();
    }


    /**
     * 根据mapper文件的Id 和参数对象获取IbatisSql对象
     * 用于获取SQL对象
     * @param id
     * @param parameterObject
     * @return
     */
    @SuppressWarnings("unused")
    public IbatisSql getIbatisSql(String id, Object parameterObject) {
        IbatisSql ibatisSql = new IbatisSql();
        Configuration configuration = getSqlSessionTemplate().getConfiguration();
        Collection<String> coll= configuration.getMappedStatementNames();

        MappedStatement ms = configuration.getMappedStatement(id);
        BoundSql boundSql = ms.getBoundSql(parameterObject);

        TypeHandlerRegistry typeHandlerRegistry= configuration.getTypeHandlerRegistry();

        List<ResultMap> ResultMaps=ms.getResultMaps();
        if(ResultMaps!=null&&ResultMaps.size()>0){
            ResultMap ResultMap = ms.getResultMaps().get(0);
            ibatisSql.setResultClass(ResultMap.getType());
        }
        ibatisSql.setSql(boundSql.getSql());

        List<ParameterMapping> parameterMappings = boundSql.getParameterMappings();
        if (parameterMappings != null) {
            Object[] parameterArray = new Object[parameterMappings.size()];
            MetaObject metaObject = parameterObject == null ? null : configuration.newMetaObject(parameterObject);

            for (int i = 0; i < parameterMappings.size(); i++) {
                ParameterMapping parameterMapping = parameterMappings.get(i);
                if (parameterMapping.getMode() != ParameterMode.OUT) {
                    Object value;

                    String propertyName = parameterMapping.getProperty();
                    if (boundSql.hasAdditionalParameter(propertyName)) { // issue #448 ask first for additional params
                        value = boundSql.getAdditionalParameter(propertyName);
                    } else if (parameterObject == null) {
                        value = null;
                    } else if (typeHandlerRegistry.hasTypeHandler(parameterObject.getClass())) {
                        value = parameterObject;
                    } else {
                        value = metaObject == null ? null : metaObject.getValue(propertyName);
                    }
                    parameterArray[i] = value;
                }
            }
            ibatisSql.setParameters(parameterArray);
        }

        return ibatisSql;
    }




}