package cn.dolphin.core.map;


import cn.dolphin.core.util.CastUtil;
import cn.dolphin.wechat.mp.model.message.WxMpTemplateData;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletRequest;
import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URLEncoder;
import java.text.NumberFormat;
import java.text.ParseException;
import java.util.*;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;

/**
 * Map转换工具
 */
@SuppressWarnings("all")
public final class MapUtil  {

    private static Logger logger = LoggerFactory.getLogger(MapUtil.class);

    private MapUtil(){}

    public static <K,V> void putMultiValue(Map<K,List<V>> map, K k, V v){
        if(map.get(k) == null){
            List<V> values = new ArrayList<V>();
            values.add(v);
            map.put(k,values);
        }else{
            map.get(k).add(v);
        }
    }

    /**
     * Map是否为空
     *
     * @param map 集合
     * @return 是否为空
     */
    public static boolean isEmpty(Map<?, ?> map) {
        return null == map || map.isEmpty();
    }

    /**
     * Map是否为非空
     *
     * @param map 集合
     * @return 是否为非空
     */
    public static boolean isNotEmpty(Map<?, ?> map) {
        return null != map && false == map.isEmpty();
    }

    /**
     * Map判断属性是否存在
     * @param map 集合
     * @param key 属性
     * @return true存在，false不存在
     */
    public static boolean isContainsKey(Map<?, ?> map,String key){
        return map.containsKey(key);
    }


    public static HashMap newHashMap(Object... keyValues) {
        return newMap(HashMap.class, keyValues);
    }

    public static ConcurrentMap newConcurrentMap(Object... keyValues) {
        return newMap(ConcurrentMap.class, keyValues);
    }

    public static TreeMap newTreeMap(Object... keyValues) {
        return newMap(TreeMap.class, keyValues);
    }

    public static LinkedHashMap newLinkedHashMap(Object... keyValues) {
        return newMap(LinkedHashMap.class, keyValues);
    }

    public static IdentityHashMap newIdentityHashMap(Object... keyValues) {
        return newMap(IdentityHashMap.class, keyValues);
    }

    /**
     * map转换string
     * 例子：name=hjz,age=25
     * @param map 数据
     * @param separator 分割符号
     * @return
     */
    public static String maptoString(Map<?, ?> map, char separator) {
        List<String> result = new ArrayList<>();
        map.entrySet().parallelStream().reduce(result, (first, second)->{
            first.add(second.getKey() + "=" + second.getValue());
            return first;
        }, (first, second)->{
            if (first == second) {
                return first;
            }
            first.addAll(second);
            return first;
        });

        return StringUtils.join(result, separator);
    }

    /**
     * map转换map<string,string>
     * @param map
     * @return
     */
    public static Map<String, String> maptoMapString(Map<String, ?> map) {
        return map.entrySet().stream().collect(Collectors.toMap(
                (entry) -> {
                    return entry.getKey();
                },
                (entry) -> {
                    if(null == entry.getValue()){
                        return "";
                    }
                    if(entry.getValue().getClass().isArray()) {
                        StringBuilder sb = new StringBuilder();
                        for(int i = 0; i< Array.getLength(entry.getValue()); ++i) {
                            Object obj = Array.get(entry.getValue(), i);
                            sb.append(obj.toString()).append(",");
                        }
                        if (sb.length() > 0) {
                            sb.deleteCharAt(sb.length() - 1);
                        }

                        return sb.toString();
                    } else {
                        return entry.getValue().toString();
                    }
                }
        ));
    }

    /**
     * 取参数值
     *
     * @param <T>
     *
     */
    @SuppressWarnings("unchecked")
    public static <T> T getTParameter(Map<?,?> params, String name, T defaultValue) {
        T value = defaultValue;
        try {
            Object temp = params.get(name);
            if (temp != null) {
                value = (T) temp;
            }
        } catch (Exception e) {
        }
        return value;
    }

    /**
     * Map获取Object
     * @param map
     * @param key
     * @param <K>
     * @param <V>
     * @return
     */
    public static <K, V> V getObject(final Map<? super K, V> map, final K key) {
        if(isNotEmpty(map)){
            return map.get(key);
        }
        return null;
    }

    /**
     * Map获取String
     * @param map
     * @param key
     * @param <K>
     * @return
     */
    public static <K> String getString(final Map<? super K, ?> map, final K key) {
        if(isNotEmpty(map)){
            final Object answer = map.get(key);
            if (answer != null) {
                return CastUtil.castString(answer);
            }
        }
        return null;
    }

    /**
     * Map获取boolean
     * @param map
     * @param key
     * @param <K>
     * @return
     */
    public static <K> Boolean getBoolean(final Map<? super K, ?> map, final K key) {
        if(isNotEmpty(map)){

            final Object answer = map.get(key);
            if (answer != null) {
                if (answer instanceof Boolean) {
                    return (Boolean) answer;
                }
                if (answer instanceof String) {
                    return Boolean.valueOf((String) answer);
                }
                if (answer instanceof Number) {
                    final Number n = (Number) answer;
                    return n.intValue() != 0 ? Boolean.TRUE : Boolean.FALSE;
                }
            }
        }
        return null;
    }

    /**
     * Map获取Number
     * @param map
     * @param key
     * @param <K>
     * @return
     */
    public static <K> Number getNumber(final Map<? super K, ?> map, final K key) {
        if (isNotEmpty(map)) {
            final Object answer = map.get(key);
            if (answer != null) {
                if (answer instanceof Number) {
                    return (Number) answer;
                }
                if (answer instanceof String) {
                    try {
                        final String text = (String) answer;
                        return NumberFormat.getInstance().parse(text);
                    } catch (final ParseException e) { // NOPMD
                        // failure means null is returned
                    }
                }
            }
        }
        return null;
    }

    /**
     * Map获取Byte
     * @param map
     * @param key
     * @param <K>
     * @return
     */
    public static <K> Byte getByte(final Map<? super K, ?> map, final K key) {
        final Number answer = getNumber(map, key);
        if (answer == null) {
            return null;
        }
        if (answer instanceof Byte) {
            return (Byte) answer;
        }
        return Byte.valueOf(answer.byteValue());
    }

    /**
     * Map获取Short
     * @param map
     * @param key
     * @param <K>
     * @return
     */
    public static <K> Short getShort(final Map<? super K, ?> map, final K key) {
        final Number answer = getNumber(map, key);
        if (answer == null) {
            return null;
        }
        if (answer instanceof Short) {
            return (Short) answer;
        }
        return Short.valueOf(answer.shortValue());
    }

    /**
     * Map获取Integer
     * @param map
     * @param key
     * @param <K>
     * @return
     */
    public static <K> Integer getInteger(final Map<? super K, ?> map, final K key) {
        final Number answer = getNumber(map, key);
        if (answer == null) {
            return null;
        }
        if (answer instanceof Integer) {
            return (Integer) answer;
        }
        return Integer.valueOf(answer.intValue());
    }

    /**
     * Map获取Long
     * @param map
     * @param key
     * @param <K>
     * @return
     */
    public static <K> Long getLong(final Map<? super K, ?> map, final K key) {
        final Number answer = getNumber(map, key);
        if (answer == null) {
            return null;
        }
        if (answer instanceof Long) {
            return (Long) answer;
        }
        return Long.valueOf(answer.longValue());
    }

    /**
     * Map获取Float
     * @param map
     * @param key
     * @param <K>
     * @return
     */
    public static <K> Float getFloat(final Map<? super K, ?> map, final K key) {
        final Number answer = getNumber(map, key);
        if (answer == null) {
            return null;
        }
        if (answer instanceof Float) {
            return (Float) answer;
        }
        return Float.valueOf(answer.floatValue());
    }

    /**
     * Map获取Double
     * @param map
     * @param key
     * @param <K>
     * @return
     */
    public static <K> Double getDouble(final Map<? super K, ?> map, final K key) {
        final Number answer = getNumber(map, key);
        if (answer == null) {
            return null;
        }
        if (answer instanceof Double) {
            return (Double) answer;
        }
        return Double.valueOf(answer.doubleValue());
    }

    /**
     * Map获取Map
     * @param map
     * @param key
     * @param <K>
     * @return
     */
    public static <K> Map<?, ?> getMap(final Map<? super K, ?> map, final K key) {
        if (isNotEmpty(map)) {
            final Object answer = map.get(key);
            if (answer != null && answer instanceof Map) {
                return (Map<?, ?>) answer;
            }
        }
        return null;
    }

    /**
     * Map获取Object并设置默认值
     * @param map
     * @param key
     * @param defaultValue
     * @param <K>
     * @param <V>
     * @return
     */
    public static <K, V> V getObject(final Map<K, V> map, final K key, final V defaultValue) {
        if (isNotEmpty(map)) {
            final V answer = map.get(key);
            if (answer != null) {
                return answer;
            }
        }
        return defaultValue;
    }

    /**
     * Map获取String并设置默认值
     * @param map
     * @param key
     * @param defaultValue
     * @param <K>
     * @return
     */
    public static <K> String getString(final Map<? super K, ?> map, final K key, final String defaultValue) {
        String answer = getString(map, key);
        if (answer == null) {
            answer = defaultValue;
        }
        return answer;
    }

    /**
     * Map获取Boolean并设置默认值
     * @param map
     * @param key
     * @param defaultValue
     * @param <K>
     * @return
     */
    public static <K> Boolean getBoolean(final Map<? super K, ?> map, final K key, final Boolean defaultValue) {
        Boolean answer = getBoolean(map, key);
        if (answer == null) {
            answer = defaultValue;
        }
        return answer;
    }

    /**
     * Map获取Number并设置默认值
     * @param map
     * @param key
     * @param defaultValue
     * @param <K>
     * @return
     */
    public static <K> Number getNumber(final Map<? super K, ?> map, final K key, final Number defaultValue) {
        Number answer = getNumber(map, key);
        if (answer == null) {
            answer = defaultValue;
        }
        return answer;
    }

    /**
     * Map获取Byte并设置默认值
     * @param map
     * @param key
     * @param defaultValue
     * @param <K>
     * @return
     */
    public static <K> Byte getByte(final Map<? super K, ?> map, final K key, final Byte defaultValue) {
        Byte answer = getByte(map, key);
        if (answer == null) {
            answer = defaultValue;
        }
        return answer;
    }

    /**
     * Map获取Short并设置默认值
     * @param map
     * @param key
     * @param defaultValue
     * @param <K>
     * @return
     */
    public static <K> Short getShort(final Map<? super K, ?> map, final K key, final Short defaultValue) {
        Short answer = getShort(map, key);
        if (answer == null) {
            answer = defaultValue;
        }
        return answer;
    }

    /**
     * Map获取Integer并设置默认值
     * @param map
     * @param key
     * @param defaultValue
     * @param <K>
     * @return
     */
    public static <K> Integer getInteger(final Map<? super K, ?> map, final K key, final Integer defaultValue) {
        Integer answer = getInteger(map, key);
        if (answer == null) {
            answer = defaultValue;
        }
        return answer;
    }

    /**
     * Map获取Long并设置默认值
     * @param map
     * @param key
     * @param defaultValue
     * @param <K>
     * @return
     */
    public static <K> Long getLong(final Map<? super K, ?> map, final K key, final Long defaultValue) {
        Long answer = getLong(map, key);
        if (answer == null) {
            answer = defaultValue;
        }
        return answer;
    }

    /**
     * Map获取Float并设置默认值
     * @param map
     * @param key
     * @param defaultValue
     * @param <K>
     * @return
     */
    public static <K> Float getFloat(final Map<? super K, ?> map, final K key, final Float defaultValue) {
        Float answer = getFloat(map, key);
        if (answer == null) {
            answer = defaultValue;
        }
        return answer;
    }

    /**
     * Map获取Double并设置默认值
     * @param map
     * @param key
     * @param defaultValue
     * @param <K>
     * @return
     */
    public static <K> Double getDouble(final Map<? super K, ?> map, final K key, final Double defaultValue) {
        Double answer = getDouble(map, key);
        if (answer == null) {
            answer = defaultValue;
        }
        return answer;
    }

    /**
     * Map获取Map并设置默认值
     * @param map
     * @param key
     * @param defaultValue
     * @param <K>
     * @return
     */
    public static <K> Map<?, ?> getMap(final Map<? super K, ?> map, final K key, final Map<?, ?> defaultValue) {
        Map<?, ?> answer = getMap(map, key);
        if (answer == null) {
            answer = defaultValue;
        }
        return answer;
    }


    /**
     * RequestMap转map的方法
     * request.getParameterMap()获取的参数Map转成 (参数,参数值)
     *
     * @param requestParams
     * @return
     */
    public static Map requestParameterMapToMap(Map requestParams) {
        Map resultMap = new HashMap();
        for (Iterator iter = requestParams.keySet().iterator(); iter.hasNext(); ) {
            String name = (String) iter.next();
            String[] values = (String[]) requestParams.get(name);
            String valueStr = "";
            for (int i = 0; i < values.length; i++) {
                valueStr = (i == values.length - 1) ? valueStr + values[i] : valueStr + values[i] + ",";
            }
            //乱码解决，这段代码在出现乱码时使用。
            //valueStr = new String(valueStr.getBytes("ISO-8859-1"), "gbk");
            resultMap.put(name, valueStr);
        }
        return resultMap;
    }

    /**
     * Map转换成url参数
     *
     * @param map
     * @param isSort    是否排序
     * @param removeKey 不包含的key元素集
     * @return
     */
    public static String getOrderedQueryStr(Map<String, Object> map, boolean isSort, Set<String> removeKey) {
        StringBuffer param = new StringBuffer();
        List<String> msgList = new ArrayList<String>();
        for (Iterator<String> it = map.keySet().iterator(); it.hasNext(); ) {
            String key = it.next();
            Object value = map.get(key);
            if (removeKey != null && removeKey.contains(key)) {
                continue;
            }
            msgList.add(key + "=" + value);
        }
        if (isSort) {
            // 排序
            Collections.sort(msgList);
        }
        for (int i = 0; i < msgList.size(); i++) {
            String msg = msgList.get(i);
            if (i > 0) {
                param.append("&");
            }
            param.append(msg);
        }
        return param.toString();
    }

    /**
     * 转换时对map中的key里的字符串会做忽略处理的正则串
     */
    private static final String OMIT_REG = "_";

    /**
     * 将map集合转换成Bean集合，Bean的属性名与map的key值对应时不区分大小写，并对map中key做忽略OMIT_REG正则处理
     *
     * @param <E>
     * @param cla
     * @param mapList
     * @return
     */
    public static <E> List<E> toBeanList(Class<E> cla, List<Map<String, Object>> mapList) {

        List<E> list = new ArrayList<E>(mapList.size());

        for (Map<String, Object> map : mapList) {
            E obj = toBean(cla, map);

            list.add(obj);
        }

        return list;
    }

    /**
     * 将map转换成Bean，Bean的属性名与map的key值对应时不区分大小写，并对map中key做忽略OMIT_REG正则处理
     *
     * @param <E>
     * @param cla
     * @param map
     * @return
     */
    @SuppressWarnings({"rawtypes"})
    public static <E> E toBean(Class<E> cla, Map<String, Object> map) {
        // 创建对象
        E obj = null;

        try {
            obj = cla.newInstance();

            if (obj == null) {
                throw new Exception();
            }
        } catch (Exception e) {
            logger.error("类型实例化对象失败,类型:" + cla);

            return null;
        }

        // 处理map的key
        Map<String, Object> newmap = new HashMap<String, Object>();

        for (Map.Entry<String, Object> en : map.entrySet()) {
            newmap.put("set" + en.getKey().trim().replaceAll(OMIT_REG, "").toLowerCase(),
                    en.getValue());
        }

        // 进行值装入
        Method[] ms = cla.getMethods();

        for (Method method : ms) {
            String mname = method.getName().toLowerCase();

            if (mname.startsWith("set")) {
                Class[] clas = method.getParameterTypes();

                Object v = newmap.get(mname);

                if (v != null && clas.length == 1) {
                    try {
                        method.invoke(obj, v);
                    } catch (Exception e) {
                        logger.error("属性值装入失败,装入方法：" + cla + "." + method.getName() + ".可装入类型"
                                + clas[0] + ";欲装入值的类型:" + v.getClass());
                    }
                }
            }
        }

        return obj;
    }

    /**
     * 将一个 JavaBean 对象转化为一个 Map
     *
     * @param bean 要转化的JavaBean 对象
     * @return 转化出来的 Map 对象
     * @throws IntrospectionException 如果分析类属性失败
     * @throws IllegalAccessException 如果实例化 JavaBean 失败
     * @throws InvocationTargetException 如果调用属性的 setter 方法失败
     */
    @SuppressWarnings({"rawtypes", "unchecked"})
    public static Map convertBeanToMap(Object bean) throws IntrospectionException,
            IllegalAccessException, InvocationTargetException {
        Class type = bean.getClass();

        Map returnMap = new HashMap();

        BeanInfo beanInfo = Introspector.getBeanInfo(type);

        PropertyDescriptor[] propertyDescriptors = beanInfo.getPropertyDescriptors();

        for (int i = 0; i < propertyDescriptors.length; i++ ) {
            PropertyDescriptor descriptor = propertyDescriptors[i];

            String propertyName = descriptor.getName();

            if ( !propertyName.equals("class")) {
                Method readMethod = descriptor.getReadMethod();

                Object result = readMethod.invoke(bean, new Object[0]);

                if (result != null) {
                    returnMap.put(propertyName, result);
                } else {
                    returnMap.put(propertyName, "");
                }
            }
        }
        return returnMap;
    }

    /**
     * request转map
     * @param request
     * @return
     */
    public static Map<String, Object> convertDataMap(HttpServletRequest request){
        Map<String, String[]> properties = request.getParameterMap();
        Map<String, Object> returnMap = new HashMap<String, Object>();
        Iterator<?> entries = properties.entrySet().iterator();
        Map.Entry<?, ?> entry;
        String name = "";
        String value = "";
        while (entries.hasNext())
        {
            entry = (Map.Entry<?, ?>) entries.next();
            name = (String) entry.getKey();
            Object valueObj = entry.getValue();
            if (null == valueObj)
            {
                value = "";
            }
            else if (valueObj instanceof String[])
            {
                String[] values = (String[]) valueObj;
                for (int i = 0; i < values.length; i++)
                {
                    value = values[i] + ",";
                }
                value = value.substring(0, value.length() - 1);
            }
            else
            {
                value = valueObj.toString();
            }
            returnMap.put(name, value);
        }
        return returnMap;
    }

    /**
     * 将一个 Map 对象转化为一个 JavaBean 〈一句话功能简述〉 〈功能详细描述〉
     *
     * @param type
     * @param map
     * @return
     * @throws IntrospectionException
     * @throws IllegalAccessException
     * @throws InstantiationException
     * @throws InvocationTargetException Object 如果有违例，请使用@exception/throws [违例类型]
     *             [违例说明：异常的注释必须说明该异常的含义及什么条件下抛出该
     * @see [类、类#方法、类#成员]
     */
    @SuppressWarnings("rawtypes")
    public static Object convertMapToBean(Class type, Map map) throws IntrospectionException,
            IllegalAccessException, InstantiationException, InvocationTargetException {
        BeanInfo beanInfo = Introspector.getBeanInfo(type); // 获取类属性

        Object obj = type.newInstance(); // 创建 JavaBean 对象

        // 给 JavaBean 对象的属性赋值
        PropertyDescriptor[] propertyDescriptors = beanInfo.getPropertyDescriptors();

        for (int i = 0; i < propertyDescriptors.length; i++ ) {
            PropertyDescriptor descriptor = propertyDescriptors[i];

            String propertyName = descriptor.getName();

            if (map.containsKey(propertyName)) {
                // 下面一句可以 try 起来，这样当一个属性赋值失败的时候就不会影响其他属性赋值。
                Object value = map.get(propertyName);

                Object[] args = new Object[1];

                args[0] = value;

                descriptor.getWriteMethod().invoke(obj, args);
            }
        }
        return obj;
    }

    /**
     * 输入Map，输出Bean对象 new Bean(); MapObjectUtil.mapToObject(入参Map,出参Bean); 〈一句话功能简述〉 〈功能详细描述〉
     *
     * @param map
     * @param classPath
     * @param obj void 如果有违例，请使用@exception/throws [违例类型] [违例说明：异常的注释必须说明该异常的含义及什么条件下抛出该
     * @see [类、类#方法、类#成员]
     */
    public static void mapToObject(Map<String, Object> map, String classPath, Object obj) {
        Class clazz;

        try {
            clazz = Class.forName(obj.getClass().getName());

            Field[] fields = clazz.getDeclaredFields();

            for (int i = 0; i < fields.length; i++ ) {
                String varName = fields[i].getName();

                Object value = map.get(varName);

                if (null != value) {
                    Method method = clazz.getMethod("set" + varName.toUpperCase().charAt(0)
                                    + varName.subSequence(1, varName.length()),
                            String.class);

                    method.invoke(obj, value);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private static <R extends Map> R newMap(Class<? extends Map> type, Object... keyValues) {
        if (null == keyValues) {
            logger.warn("MapUtil#newMap() keyValues is null.");
            return null;
        }

        int length = keyValues.length;
        if (length % 2 != 0) {
            throw new IllegalArgumentException("参数不符合预期, 可变数组应该成对出现。");
        }
        Map resultMap;
        if (type == HashMap.class) {
            resultMap = Maps.newHashMap();
        } else if (type == ConcurrentMap.class) {
            resultMap = Maps.newConcurrentMap();
        } else if (type == TreeMap.class) {
            resultMap = Maps.newTreeMap();
        } else if (type == LinkedHashMap.class) {
            resultMap = Maps.newLinkedHashMap();
        } else if (type == IdentityHashMap.class) {
            resultMap = Maps.newIdentityHashMap();
        } else {
            throw new IllegalArgumentException("MapUtil#newMap() type = " + type.getSimpleName() + " 不支持。");
        }
        for (int i = 0; i < length; i = i + 2) {
            resultMap.put(keyValues[i], keyValues[i + 1]);
        }
        return (R) resultMap;
    }

    /**
     * 排序-按照key进行排序
     * @param map
     * @return
     */
    public static Map<String, String> sortMapByKey(Map<String, String> map) {
        if (map == null || map.isEmpty()) {
            return null;
        }

        Map<String, String> sortMap = new TreeMap<String, String>(new MapKeyComparator());

        sortMap.putAll(map);
        return sortMap;
    }

    /**
     * Map key 升序排序
     *
     * 排序方式：ASCII码从小到大排序（字典序）
     *
     * @param map 需排序的map集合
     * @return 排序后的map集合
     */
    public static Map<String, String> sortAsc(Map<String, String> map) {
        HashMap<String, String> tempMap = new LinkedHashMap<String, String>();
        List<Map.Entry<String, String>> infoIds = new ArrayList<Map.Entry<String, String>>(map.entrySet());
        //排序
        Collections.sort(infoIds, new Comparator<Map.Entry<String, String>>() {
            @Override
            public int compare(Map.Entry<String, String> o1, Map.Entry<String, String> o2) {
                return o1.getKey().compareTo(o2.getKey());
            }
        });

        for (int i = 0; i < infoIds.size(); i++) {
            Map.Entry<String, String> item = infoIds.get(i);
            tempMap.put(item.getKey(), item.getValue());
        }
        return tempMap;
    }


    /**
     * 排序-按照value进行排序
     * @param map
     * @return
     */
    public static Map<String, String> sortMapByValue(Map<String, String> map) {
        if (map == null || map.isEmpty()) {
            return null;
        }
        Map<String, String> sortedMap = new LinkedHashMap<String, String>();
        List<Map.Entry<String, String>> entryList = new ArrayList<Map.Entry<String, String>>(map.entrySet());
        Collections.sort(entryList, new MapValueComparator());

        Iterator<Map.Entry<String, String>> iter = entryList.iterator();
        Map.Entry<String, String> tmpEntry = null;
        while (iter.hasNext()) {
            tmpEntry = iter.next();
            sortedMap.put(tmpEntry.getKey(), tmpEntry.getValue());
        }
        return sortedMap;
    }

    /**
     * url 参数串连
     *
     * @param map 需拼接的参数map
     * @param valueUrlEncode 是否需要对map的value进行url编码
     * @return 拼接后的URL键值对字符串
     */
    public static String mapJoin(Map<String, String> map, boolean valueUrlEncode) {
        StringBuilder sb = new StringBuilder();
        for (String key : map.keySet()) {
            if (map.get(key) != null && !"".equals(map.get(key))) {
                try {
                    String temp = (key.endsWith("_") && key.length() > 1) ? key.substring(0, key.length() - 1) : key;
                    sb.append(temp);
                    sb.append("=");
                    //获取到map的值
                    String value = map.get(key);
                    //判断是否需要url编码
                    if (valueUrlEncode) {
                        value = URLEncoder.encode(map.get(key), "utf-8").replace("+", "%20");
                    }
                    sb.append(value);
                    sb.append("&");
                } catch (UnsupportedEncodingException e) {
                    e.printStackTrace();
                }
            }
        }
        if (sb.length() > 0) {
            sb.deleteCharAt(sb.length() - 1);
        }
        return sb.toString();
    }


    public static void main(String[] args) {
        // 传递多个map
        System.out.println(MapUtil.newHashMap("key1", new WxMpTemplateData(),"key2",new WxMpTemplateData()));
    }

}




