package cn.dolphin.core.json;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.reflect.TypeToken;
import com.google.gson.*;
import lombok.NoArgsConstructor;

import java.io.IOException;
import java.lang.reflect.Type;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * Gson是google开发的一个开源Json解析库
 */
@NoArgsConstructor
@SuppressWarnings("all")
public class GsonUtil {

    private static Gson gson = null;
    static{
        if(gson==null){
            gson = new GsonBuilder()
                    .setDateFormat("yyyy-MM-dd HH:mm:ss")
                    .create();
        }
    }

    /**
     * 将对象转换成json格式
     * @param ts
     * @return
     */
    public static String objectToJson(Object ts){
        String jsonStr=null;
        if(gson!=null){
            jsonStr=gson.toJson(ts);
        }
        return jsonStr;
    }

    public static String  StringJoin(Collection<?> collection, String separator){
        Joiner joiner =  Joiner.on(separator).skipNulls();
        return joiner.join(collection);
    }

    public static List<String> separatorString (String source,String separator){
        Splitter splitter = Splitter.on(separator).trimResults().omitEmptyStrings();
        return splitter.splitToList(source);
    }


    /**
     * 将对象转换成json格式(并自定义日期格式)
     * @param ts
     * @return
     */
    public static String objectToJsonDateSerializer(Object ts,final String dateformat){
        String jsonStr=null;
        gson=new GsonBuilder().registerTypeHierarchyAdapter(Date.class, new JsonSerializer<Date>() {
            public JsonElement serialize(Date src, Type typeOfSrc,
                                         JsonSerializationContext context) {
                SimpleDateFormat format = new SimpleDateFormat(dateformat);
                return new JsonPrimitive(format.format(src));
            }
        }).setDateFormat(dateformat).create();
        if(gson!=null){
            jsonStr=gson.toJson(ts);
        }
        return jsonStr;
    }

    /**
     * 将json格式转换成list对象
     * @param jsonStr
     * @return
     */
    public static List<?> jsonToList(String jsonStr){
        List<?> objList=null;
        if(gson!=null){
            Type type=new com.google.gson.reflect.TypeToken<List<?>>(){}.getType();
            objList=gson.fromJson(jsonStr, type);
        }
        return objList;
    }

    /**
     * 将json格式转换成list对象
     * @param <T>
     * @param jsonStr
     * @return
     */
    public static <T> List<T> jsonToList(String jsonStr, Class<T> T){
        List<T> objList=null;
        if(gson!=null){
            Type type=new com.google.gson.reflect.TypeToken<List<T>>(){}.getType();
            objList=gson.fromJson(jsonStr, type);
        }
        return objList;
    }

    /**
     * 将json格式转换成list对象
     * 解决stringMap问题
     * @param s
     * @param clazz
     * @param <T>
     * @return
     */
    public static <T> List<T> stringToArray(String s, Class<T[]> clazz) {
        T[] arr = new Gson().fromJson(s, clazz);
        return Arrays.asList(arr);
    }

    /**
     * 将json格式转换成list对象
     * @param jsonStr
     * @return
     */
    public static List<Map<String, String>> jsonToListStr(String jsonStr){
        List<Map<String, String>> objList=null;
        if(gson!=null){
            Type type=new com.google.gson.reflect.TypeToken<List<Map<String, String>>>(){}.getType();
            objList=gson.fromJson(jsonStr, type);
        }
        return objList;
    }

    /**
     * 将json转换成bean对象
     * @param jsonStr
     * @return
     */
    public static Object  jsonToBean(String jsonStr,Class<?> cl){
        Object obj=null;
        if(gson!=null){
            obj=gson.fromJson(jsonStr, cl);
        }
        return obj;
    }

    /**
     * 将json转换成bean对象
     * @param jsonStr
     * @param cl
     * @return
     */
    @SuppressWarnings("unchecked")
    public static <T> T  jsonToBeanDateSerializer(String jsonStr,Class<T> cl,final String pattern){
        Object obj=null;
        gson=new GsonBuilder().registerTypeAdapter(Date.class, new JsonDeserializer<Date>() {
            public Date deserialize(JsonElement json, Type typeOfT,
                                    JsonDeserializationContext context)
                    throws JsonParseException {
                SimpleDateFormat format=new SimpleDateFormat(pattern);
                String dateStr=json.getAsString();
                try {
                    return format.parse(dateStr);
                } catch (ParseException e) {
                    e.printStackTrace();
                }
                return null;
            }
        }).setDateFormat(pattern).create();
        if(gson!=null){
            obj=gson.fromJson(jsonStr, cl);
        }
        return (T)obj;
    }

    /**
     * 根据
     * @param jsonStr
     * @param key
     * @return
     */
    public static Object  getJsonValue(String jsonStr,String key){
        Object resultObj = null;
        Map<?,?> resultMap = jsonToMap(jsonStr);
        if (null != resultMap && !resultMap.isEmpty() && resultMap.containsKey(key)) {
            resultObj = resultMap.get(key);
        }
        return resultObj;
    }

    /**
     * 将json格式转换成map对象
     * @param jsonStr
     * @return
     */
    public static Map<?,?> jsonToMap(String jsonStr){
        Map<?,?> objectMap = null;
        if (null != gson) {
            Type type = new com.google.gson.reflect.TypeToken<Map<?,?>>(){}.getType();
            objectMap=gson.fromJson(jsonStr, type);
        }
        return objectMap;
    }

    /**
     * 将json格式转换成map对象
     * @param json
     * @param clz
     * @param <T>
     * @return
     */
    public static <T> Map<String, T> jsonToMap(String json, Class<T> clz){
        Map<String, JsonObject> map = gson.fromJson(json, new TypeToken<Map<String,JsonObject>>(){}.getType());
        Map<String, T> result = new HashMap<String, T>();
        for(String key:map.keySet()){
            result.put(key,gson.fromJson(map.get(key),clz) );
        }
        return result;
    }


    /**
     * 转成bean
     *
     * @param json
     * @param cls
     * @return
     */
    public static <T> T fromJsonString(String json, Class<T> classOfT) {
        T t = null;
        if (gson != null) {
            t = gson.fromJson(json, classOfT);
        }
        return t;
    }

    /**
     * 待完善
     *
     * @param json
     * @param key
     * @return
     */
    public static String fromJsonString(String json, String key) {
        ObjectMapper mapper = new ObjectMapper();
        String value;
        JsonNode root;
        try {
            root = mapper.readTree(json);
            JsonNode data = root.path(key);
            value = data.asText();
        } catch (IOException e) {
            value = null;
        }
        return value;
    }

    /**
     * 转成list
     *
     * @param json
     * @param cls
     * @return
     */
    public static <T> List<T> toList(String json, Class<T> cls) {
        List<T> list = null;
        if (gson != null) {
            list = gson.fromJson(json, new TypeToken<List<T>>() {
            }.getType());
        }
        return list;
    }

    /**
     * 转成list中有map的
     *
     * @param json
     * @return
     */
    public static <T> List<Map<String, T>> toListMaps(String json) {
        List<Map<String, T>> list = null;
        if (gson != null) {
            list = gson.fromJson(json, new TypeToken<List<Map<String, T>>>() {
            }.getType());
        }
        return list;
    }

}
