package cn.dolphin.core.json;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.*;
import com.fasterxml.jackson.databind.node.MissingNode;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateDeserializer;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateSerializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalTimeSerializer;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;

/**
 * 阿里云fastjson
 */
@NoArgsConstructor
@Slf4j
@SuppressWarnings("all")
public class FastJsonUtil {

     /* ---------------API------------------

     // Jackson 忽略序列化字段，注解字段或get()/is()方法
     @JsonIgnore

     // FastJSON 忽略序列化字段，注解字段或get()/is()方法，可以使用format等
     @JSONField(serialize = false)

     */

//    private static final FastJsonUtil defaultMapper = new FastJsonUtil(JsonInclude.Include.NON_NULL);
//
//    public FastJsonUtil(JsonInclude.Include inclusion) {
//        ObjectMapper mapper = new ObjectMapper();
//        //设置输出时包含属性的风格
//        mapper.setSerializationInclusion(inclusion);
//        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
//        //禁止使用int代表Enum的order()來反序列化Enum,非常危險
//        mapper.configure(DeserializationFeature.FAIL_ON_NUMBERS_FOR_ENUMS, true);
//        mapper.configure(JsonParser.Feature.ALLOW_UNQUOTED_CONTROL_CHARS, true);
//    }
//
//    public FastJsonUtil(JsonInclude.Include inclusion, String timeFormat) {
//        ObjectMapper mapper = new ObjectMapper();
//        //设置输出时包含属性的风格
//        mapper.setSerializationInclusion(inclusion);
//        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
//        //禁止使用int代表Enum的order()來反序列化Enum,非常危險
//        mapper.configure(DeserializationFeature.FAIL_ON_NUMBERS_FOR_ENUMS, true);
//        mapper.configure(JsonParser.Feature.ALLOW_UNQUOTED_CONTROL_CHARS, true);
//        mapper.setDateFormat(new SimpleDateFormat(timeFormat));
//    }
//
//    /**
//     * 创建只输出非空属性到Json字符串的Mapper.
//     */
//    public static FastJsonUtil buildNotNullMapper() {
//        return defaultMapper;
//    }

    private static ObjectMapper objectMapper = null;

    static {
        if(objectMapper == null){
            objectMapper = new ObjectMapper();
            objectMapper.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
            //去掉默认的时间戳格式
            objectMapper.configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false);
//            //设置为中国上海时区
//            objectMapper.setTimeZone(TimeZone.getTimeZone("GMT+8"));
            //设置为中国上海时区
            objectMapper.setTimeZone(TimeZone.getTimeZone(ZoneId.systemDefault()));
            objectMapper.configure(SerializationFeature.WRITE_NULL_MAP_VALUES, false);
            //空值不序列化
            //objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
            // null替换为""
            objectMapper.getSerializerProvider().setNullValueSerializer(new JsonSerializer<Object>() {
                @Override
                public void serialize(Object arg0, JsonGenerator arg1, SerializerProvider arg2) throws IOException, JsonProcessingException {
                    arg1.writeString("");
                }
            });
            //反序列化时，属性不存在的兼容处理
            objectMapper.getDeserializationConfig().withoutFeatures(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
            //序列化时，日期的统一格式
            objectMapper.setDateFormat(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss"));
            //失败处理
            objectMapper.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
            //忽略未知属性 防止解析报错
            objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
            //单引号处理
            objectMapper.configure(JsonParser.Feature.ALLOW_SINGLE_QUOTES, true);
            objectMapper.configure(JsonParser.Feature.ALLOW_UNQUOTED_CONTROL_CHARS, true);
            objectMapper.configure(JsonParser.Feature.ALLOW_UNQUOTED_FIELD_NAMES, true);
            objectMapper.configure(JsonParser.Feature.ALLOW_COMMENTS, true);

            JavaTimeModule javaTimeModule = new JavaTimeModule();

            javaTimeModule.addSerializer(LocalDateTime.class, new LocalDateTimeSerializer(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));
            javaTimeModule.addSerializer(LocalDate.class, new LocalDateSerializer(DateTimeFormatter.ofPattern("yyyy-MM-dd")));
            javaTimeModule.addSerializer(LocalTime.class, new LocalTimeSerializer(DateTimeFormatter.ofPattern("HH:mm:ss")));
            javaTimeModule.addDeserializer(LocalDateTime.class, new LocalDateTimeDeserializer(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));
            javaTimeModule.addDeserializer(LocalDate.class, new LocalDateDeserializer(DateTimeFormatter.ofPattern("yyyy-MM-dd")));
            javaTimeModule.addDeserializer(LocalTime.class, new LocalTimeDeserializer(DateTimeFormatter.ofPattern("HH:mm:ss")));
            objectMapper.registerModule(javaTimeModule);

        }
    }


    // 通用序列化格式
    public static final SerializerFeature[] FEATURES = new SerializerFeature[] {
            // 不使用引用符，便于JS读取
            SerializerFeature.DisableCircularReferenceDetect,
            // 空值处理
            SerializerFeature.WriteNullStringAsEmpty, // 字符类型字段如果为null，输出为""，而不是null
            SerializerFeature.WriteNullListAsEmpty, // list字段如果为null，输出为[]，而不是null
            SerializerFeature.WriteNullNumberAsZero, // 数值字段如果为null，输出为0，而不是null
            // null键保留
            SerializerFeature.WriteMapNullValue, // 输出空置字段
            // 日期格式化：yyyy-MM-dd HH:mm:ss
            SerializerFeature.WriteDateUseDateFormat
    };

    // 通用序列化格式-带美化
    public static final SerializerFeature[] PRETTY_FEATURES = new SerializerFeature[] {
            // 不使用引用符，便于JS读取
            SerializerFeature.DisableCircularReferenceDetect,
            // 空值处理
            SerializerFeature.WriteNullStringAsEmpty,
            SerializerFeature.WriteNullListAsEmpty,
            SerializerFeature.WriteNullNumberAsZero,
            // null键保留
            SerializerFeature.WriteMapNullValue,
            // 日期格式化：yyyy-MM-dd HH:mm:ss
            SerializerFeature.WriteDateUseDateFormat,
            // 美化
            SerializerFeature.PrettyFormat
    };



    /**
     * 判断字符串是否是JSON对象字符串
     * @param str
     * @return
     */
    public static boolean isJsonObjectString(String str) {
        return str != null && str.matches("^\\{.*\\}$");
    }

    /**
     * 判断字符串是否是JSON数组字符串
     * @param str
     * @return
     */
    public static boolean isJsonArrayString(String str) {
        return str != null && str.matches("^\\[.*\\]$");
    }




    /**
     * 转换对象为JSON串
     * @param obj
     * @return
     *
     */
    public static final String toString(Object obj){
        return JSON.toJSONString(obj,FEATURES);
    }

    /**
     * 转换对象为JSON串
     * @param obj
     * @return
     */
    public static final String toString2(Object obj){
        try {
            return objectMapper.writeValueAsString(obj);
        } catch (IOException e) {
            log.error("write to json string error:" + obj, e);
            return null;
        }
    }

    /**
     *
     * 转换对象为JSON串，格式化日期类型参数(默认为时间戳)
     *
     * @param obj
     * @param dateFormat 日期格式化"yyyy-MM-dd HH:mm:ss.SSS"，{ @link DateFormatEnum }
     * @return
     *
     */
    public static final String toString(Object obj, String dateFormat){
        return JSON.toJSONStringWithDateFormat(obj, dateFormat, FEATURES);
    }

    /**
     * 转换对象为JSON串并美化
     *
     * @param obj
     * @return
     *
     */
    public static final String toFormatString(Object obj){
        return JSON.toJSONString(obj,PRETTY_FEATURES);
    }

    /**
     * JSON串转换为JSONObject(Map<String, Object>)
     *
     * @param json
     * @return
     *
     */
    public static final JSONObject parseObject(String json){
        return JSON.parseObject(json);
    }

    /**
     * JSON串转换为JSONArray（List<Object>）
     *
     * @param json
     * @return
     *
     */
    public static final JSONArray parseArray(String json){
        return JSON.parseArray(json);
    }

    /**
     *
     * 转换JSON串为对象
     *
     * @param json
     * @param clazz
     * @return
     *
     */
    public static final <T> T parse(String json,Class<T> clazz){
        if (StringUtils.isBlank(json)) {
            return null;
        }
        return JSON.parseObject(json, clazz);
    }
    /**
     *
     * 转换JSON串为对象集合
     *
     * @param json
     * @param clazz	集合元素的类型
     * @return
     *
     */
    public static final <T> List<T> parseList(String json,Class<T> clazz){
        return JSON.parseArray(json, clazz);
    }

    /**
     *
     * 转换JSONObject对象为Java对象
     *
     * 注：如果FastJSON在转换对象的JSON串时，不知道对象的属性的类型，就会转换为JSON对象。如果是对象类型，会转换为JSONObject(Map<String, Object>)
     *
     * @param obj
     * @param clazz
     * @return
     *
     */
    public static final <T> T parse(JSONObject obj,Class<T> clazz){
        if(obj == null){
            return null;
        }
        return JSON.toJavaObject(obj, clazz);
    }
    /**
     *
     * 转换JSONArray对象为数组
     *
     * 注：如果FastJSON在转换对象的JSON串时，不知道对象的属性的类型，就会转换为JSON对象。如果是数组类型，会转换为JSONArray(List<Object>)
     *
     * @param array
     * @param t
     * @return
     *
     */
    public static final <T> T[] parseArray(JSONArray array,T[] t){
        return array.toArray(t);
    }

    /**
     *
     * 转换json串为数组
     *
     * @param json
     * @param arr
     * @return
     *
     */
    public static final <T> T[] parseArray(String json,T[] arr){
        return JSON.parseArray(json).toArray(arr);
    }

    /**
     *
     * 转换JSONArray对象为Java列表
     *
     * 注：如果FastJSON在转换对象的JSON串时，不知道对象的属性的类型，就会转换为JSON对象。如果是数组类型，会转换为JSONArray(List<Object>)
     *
     * @param array
     * @param clazz
     * @return
     *
     */
    public static final <T> List<T> parseList(JSONArray array,Class<T> clazz){
        if(array == null){
            return null;
        }
        return array.toJavaList(clazz);
    }

    /**
     * 设置时区
     *
     * @param tz 时区
     */
    public void setTimeZone(TimeZone tz) {
        objectMapper.setTimeZone(tz);
    }

    /**
     * Java对象转成Json字符串
     *
     * @param obj Java对象
     * @return Json字符串
     */
    public static String toJson(Object obj){
        if(obj == null){
            return null;
        }
        try {
            /**
             * ObjectMapper是JSON操作的核心，Jackson的所有JSON操作都是在ObjectMapper中实现。
             * ObjectMapper有多个JSON序列化的方法，可以把JSON字符串保存File、OutputStream等不同的介质中。
             * writeValue(File arg0, Object arg1)把arg1转成json序列，并保存到arg0文件中。
             * writeValue(OutputStream arg0, Object arg1)把arg1转成json序列，并保存到arg0输出流中。
             * writeValueAsBytes(Object arg0)把arg0转成json序列，并把结果输出成字节数组。
             * writeValueAsString(Object arg0)把arg0转成json序列，并把结果输出成字符串。
             */
            String json = objectMapper.writeValueAsString(obj);
            return json;
        } catch (JsonProcessingException e) {
            throw new RuntimeException("Jackson转字符串出现异常：",e);
        }
    }


    /**
     * Java对象转成JsonNode
     *
     * @param obj Java对象
     * @return JsonNode
     * @throws RuntimeException
     */
    public static JsonNode toJsonNode(Object obj) {
        if (obj instanceof String) {
            try {
                return objectMapper.readTree((String) obj);
            } catch (Exception e) {
                throw new RuntimeException("Java对象转成JsonNode出现异常：",e);
            }
        } else {
            return objectMapper.valueToTree(obj);
        }
    }
    /**
     * Java对象转成Json
     *
     * @param obj
     * @param toPrettify
     * @return
     */
    public static String toJson(Object obj, boolean toPrettify) {
        return JSON.toJSONString(obj, toPrettify);
    }

    /**
     * Json转成Java对象
     *
     * @param <T>
     * @param json
     * @param type
     * @return
     */
    public static <T> T toObject(String json, Class<T> type){
        if(json==null || "".equals(json.trim())){
            return null;
        }
        try {
            /**
             * ObjectMapper支持从byte[]、File、InputStream、字符串等数据的JSON反序列化。
             */
            T value = objectMapper.readValue(json, type);
            return value;
        } catch (Exception e) {
            throw new RuntimeException("Jackson转Java对象出现异常：",e);
        }
    }

    /**
     * 转成List泛型对象
     *
     * @param obj   源数据，可以是Json字符串、JsonNode或其它Java对象
     * @param clazz 目标对象类型
     * @return 目标对象
     * @throws RuntimeException
     */
    public static <E> List<E> toList(Object obj, Class<E> clazz) {
        return (List<E>) toGeneric(obj, List.class, clazz);
    }

    /**
     * 转成泛型对象
     *
     * @param obj              源数据，可以是Json字符串、JsonNode或其它Java对象
     * @param parametrized     目标对象类型
     * @param parameterClasses 目标对象泛型类型
     * @return 目标对象
     * @throws RuntimeException
     */
    public static Object toGeneric(Object obj, Class<?> parametrized, Class... parameterClasses) {
        JavaType type = objectMapper.getTypeFactory().constructParametricType(parametrized, parameterClasses);
        return toGeneric(obj, type);
    }
    /**
     * 转成Set泛型对象
     *
     * @param obj   源数据，可以是Json字符串、JsonNode或其它Java对象
     * @param clazz 目标对象类型
     * @return 目标对象
     * @throws RuntimeException
     */
    public static  <E> Set<E> toSet(Object obj, Class<E> clazz) {
        return (Set<E>) toGeneric(obj, Set.class, clazz);
    }


    /**
     * 转成Map泛型对象
     *
     * @param obj        源数据，可以是Json字符串、JsonNode或其它Java对象
     * @param keyClazz   目标对象类型Key
     * @param valueClazz 目标对象类型Value
     * @return 目标对象
     * @throws RuntimeException
     */
    public static <K, V> Map<K, V> toMap(Object obj, Class<K> keyClazz, Class<V> valueClazz) {
        return (Map<K, V>) toGeneric(obj, Map.class, keyClazz, valueClazz);
    }

    /**
     * Bean转换为Map，忽略里面的Null转为空字符串
     * 可以解决：
     * 1、fastjson生成json时Null属性不显示
     * 2、ES里面比较常用的缺失字段不进索引的问题
     *
     * @param obj
     * @return
     */
    public static Map toMap(Object obj) {
        return JSON.parseObject(JSON.toJSONString(obj, SerializerFeature.WriteMapNullValue, SerializerFeature.WriteNullStringAsEmpty), Map.class);
    }

    public static JSONObject toJSONObject(String json) {
        return JSONObject.parseObject(json);
    }

    public static JSONObject toJSONObject(Object obj) {
        return toJSONObject(obj);
    }

    public static JSONArray toJSONArray(String json) {
        return JSONObject.parseArray(json);
    }

    /**
     * JSON转Map格式
     * 废弃，转换如果有null值会忽略字段
     * @param json
     * @return
     */
    @Deprecated
    public static Map toMap(String json) {
        return JSON.parseObject(json, new TypeReference<Map<String, String>>() {
        });
    }


    /**
     * 转成目标对象
     *
     * @param obj   源数据，可以是Json字符串、JsonNode或其它Java对象
     * @param clazz 目标对象类型
     * @return 目标对象
     * @throws RuntimeException
     */
    public static <E> E toObject(Object obj, Class<E> clazz) {
        try {
            if (obj instanceof String) {
                if (clazz == String.class) {
                    return (E) obj;
                } else {
                    return objectMapper.readValue((String) obj, clazz);
                }
            } else if (obj instanceof JsonNode) {
                return objectMapper.readValue(obj.toString(), clazz);
            } else {
                return objectMapper.readValue(objectMapper.writeValueAsString(obj), clazz);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 获取对应路径下的Json
     *
     * @param jsonNode Json对象
     * @param pathStr  路径
     * @return 对应的Json对象
     */
    public static JsonNode path(JsonNode jsonNode, String pathStr) {
        String[] splitPaths = pathStr.split("\\.");
        jsonNode = jsonNode.path(splitPaths[0]);
        if (jsonNode instanceof MissingNode) {
            return null;
        } else if (splitPaths.length == 1) {
            return jsonNode;
        } else {
            return path(jsonNode, pathStr.substring(pathStr.indexOf(".") + 1));
        }
    }


    private static Object toGeneric(Object obj, JavaType type) {
        try {
            if (obj instanceof String) {
                return objectMapper.readValue((String) obj, type);
            } else if (obj instanceof JsonNode) {
                return objectMapper.readValue(obj.toString(), type);
            } else {
                return objectMapper.readValue(objectMapper.writeValueAsString(obj), type);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public static String writeValue(Object object) {
        StringWriter result = new StringWriter();
        try {
            ObjectMapper mapper = new JsonMapper();
            mapper.writeValue(result, object);
        }
        catch (Exception e) {
            log.error("write json value error:"+ e.getMessage());
        }
        return result.toString().length()!=0 ? result.toString(): "{}";
    }

    public static <T> T readValue(String json, Class<T> clazz) {
        T result = null;
        try {
            if (json!=null) {
                ObjectMapper mapper = new JsonMapper();
                result = mapper.readValue(json, clazz);
            }
        }
        catch (Exception e) {
            log.error("read json value error:"+ e.getMessage());
        }
        return result;
    }






}
