package cn.dolphin.core.jdbc.support;

import cn.dolphin.core.jdbc.JdbcTemplatePlus;
import cn.dolphin.core.page.cache.Cache;
import cn.dolphin.core.page.cache.CacheFactory;
import cn.dolphin.core.util.CamelCaseUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.DefaultResourceLoader;
import org.springframework.core.io.Resource;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.BeanPropertySqlParameterSource;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;

import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * 基类
 */
public abstract class AbstractJdbcDao<T, ID extends Serializable> extends SQL{
    protected static Logger log = LoggerFactory.getLogger(AbstractJdbcDao.class);
    /**
     * spring jdbcTemplate 对象
     */
    public abstract JdbcTemplatePlus getJdbcTemplate();


    /**
     * RowMapper
     * @return
     */
    protected abstract RowMapper<T> getRowMapper();


    /**
     * 泛型
     */
    protected Class<T> entityClass;

    /**
     * 表名
     */
    protected String tableName;

    /**
     * 主键
     */
    protected String primaryKey;


    /**
     * 参数转换工具类
     * @param mapORpojo 要么是一个map，要么是一个pojo
     * @return
     */
    protected SqlParameterSource transToSource(Object mapORpojo) {
        if(mapORpojo == null){
            return new MapSqlParameterSource();
        }

        SqlParameterSource source = null;
        if (mapORpojo instanceof Map) {
            Map<String, ?> map = (Map<String, ?>) mapORpojo;
            source = new MapSqlParameterSource(map);
        }
        else {
            source = new BeanPropertySqlParameterSource(mapORpojo);
        }

        return source;
    }


    /**
     * 组装tableName，因为表名如果有数字，会造成异常
     *
     * @param tableName
     * @return
     */
    protected String buildTableName(String tableName) {
        return  String.format("%s"+tableName+"%s","`","`");
    }


    /**
     * 组装字段集
     *
     * @param fields
     * @return
     */
    protected StringBuffer buildFields(String[] fields) {
        StringBuffer fieldTemp = new StringBuffer();
        if (fields != null && fields.length > 0) {
            for (String field : fields) {
                //转换小写驼峰
                fieldTemp.append(CamelCaseUtil.toUnderlineName("field",false)).append(",");
            }
            fieldTemp.deleteCharAt(fieldTemp.length() - 1);
        }
        return fieldTemp;
    }

    /**
     * 组装字段到？分割 的列表，比如5个字段会组成：?,?,?,?
     *
     * @param fields 字段列表
     * @return
     */
    protected String buildQuestion(String[] fields) {

        String questionList = StringUtils.repeat("?,", fields.length);
        if (questionList.endsWith(","))
            questionList = questionList.substring(0, questionList.length() - 1);

        return questionList;
    }


    /**
     * 基础类型验证
     * @param clazz	String||Integer||Float||Double||Object
     * @return
     */
    protected Boolean isBaseType(Class clazz) {
        return clazz == String.class
                || clazz == Integer.class
                || clazz == Long.class
                || clazz == Float.class
                || clazz == Double.class
                || clazz == Object.class;
    }

    /**
     * hashmap转为对象
     *
     * @param map
     * @param clazz
     * @return
     */
    public <T> T hash2Class(HashMap map, Class<T> clazz) {
        if (null == clazz) {
            return null;
        }
        Object u = null;
        try {
            u = clazz.newInstance();
            // 遍历hashmap.取到key的值用来取字段
            for (Object key : map.keySet()) {

                // 用key值找到对应的字段
                Field field = clazz.getDeclaredField(key.toString());
                // 强制设置private 字段
                field.setAccessible(true);
                Object obj = field.getType();
                Object value;

                // boolean类型不能直接赋值,所以要做判断转换
                if (Boolean.class == obj) {
                    value = map.get(key).equals("1") ? true : false;
                } else {
                    value = map.get(key);
                }
                // 赋值到新的里
                field.set(u, value);

            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        return (T) u;
    }


    /**
     * 对象转为map
     *
     * @param entity
     * @return
     */
    public <T> HashMap class2Hash(Object entity) {
        if (null == entity) {
            return null;
        }
        Class<T> clazz = (Class<T>) entity.getClass();
        Field[] field = clazz.getDeclaredFields();

        HashMap<String, Object> hashmap = new HashMap<String, Object>();
        try {
            for (Field f : field) {
                f.setAccessible(true);
                hashmap.put(f.getName(), f.get(entity));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return hashmap;
    }


    //分页缓存
    private static Cache<String, Integer> pageCache = null;

    //分页缓存创建
    protected static Cache<String, Integer> getPageCache() {
        if (pageCache != null) {
            return pageCache;
        }
        Properties properties = new Properties();
        InputStream is = null;
        String location = "classpath:page.properties";
        try {
            Resource resource = new DefaultResourceLoader().getResource(location);
            is = resource.getInputStream();
            properties.load(is);
        } catch (IOException ex) {
            log.warn("Could not load properties from path:" + location);
        } finally {
            try {
                if (is != null) {
                    is.close();
                }
            } catch (IOException ioe) {
                // ignore
            }
        }
        pageCache = CacheFactory.createCache(properties.getProperty("pageCacheClass"), "page.", properties);
        return pageCache;
    }





}
