package cn.dolphin.core.jdbc;

import cn.dolphin.core.consts.SqlTypeConstant;
import cn.dolphin.core.date.DateFormatUtil;
import cn.dolphin.core.dialect.Dialect;
import cn.dolphin.core.dialect.util.DialectUtil;
import cn.dolphin.core.exception.DaoRuntimException;
import cn.dolphin.core.jdbc.enums.FieldType;
import cn.dolphin.core.jdbc.framework.FrameworkNamedParameterJdbcTemplate;
import cn.dolphin.core.jdbc.framework.PreparedStatementCreatorImpl;
import cn.dolphin.core.jdbc.model.SaveOrUpdateModel;
import cn.dolphin.core.jdbc.support.ColumnMap;
import cn.dolphin.core.jdbc.util.JdbcUtil;
import cn.dolphin.core.reflect.BeanField;
import cn.dolphin.core.reflect.ReflectUtil;
import cn.dolphin.core.util.ArrayUtil;
import cn.dolphin.core.util.CamelCaseUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONException;
import com.alibaba.fastjson.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.dao.IncorrectResultSizeDataAccessException;
import org.springframework.jdbc.core.*;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.jdbc.support.GeneratedKeyHolder;
import org.springframework.jdbc.support.KeyHolder;

import javax.sql.DataSource;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.sql.Date;
import java.sql.*;
import java.util.*;




/**
 * jdbc扩展封装
 * 1.查询不到数据时:
 * queryForBean返回null；
 * queryForBeanList返回值size() == 0；
 * queryForBeanPagination返回值getTotalCount() == 0；
 * queryForMapListPagination返回值getTotalCount() == 0；
 * queryForJson返回"{}"
 * queryForJsonList返回"[]"
 *
 */
@SuppressWarnings("all")
public class JdbcTemplatePlus extends JdbcTemplate {

    protected static Logger log = LoggerFactory.getLogger(JdbcTemplatePlus.class);
    /**
     * 数据库类型
     * 分页插件使用
     */
    private String dbType;

    public String getDbType() {
        return dbType;
    }

    public void setDbType(String dbType) {
        this.dbType = dbType;
    }

    /**
     * 重写父类继承，重写注入dataSource
     * @param dataSource
     */
    public JdbcTemplatePlus(DataSource dataSource) {
        super(dataSource);

    }

    /**
     * 重写父类继承，重写注入dataSource
     * @param dataSource
     * @param lazyInit
     */
    public JdbcTemplatePlus(DataSource dataSource, boolean lazyInit) {
        super(dataSource,lazyInit);
    }

    /**
     * 重写JdbcTemplate类，将getDataSource()方法重写，让其返回的是具体的子数据源 而不是整个的AbstractRoutingDataSource
     * @return
     */
    @Override
    public DataSource getDataSource() {
        return super.getDataSource();
    }

    /**
     * jdbc 具名参数查询扩展
     */
    private NamedParameterJdbcTemplate namedParameterJdbcTemplate;


    /**
     * 获取NamedParameterJdbcTemplate操作对象
     */
    public NamedParameterJdbcTemplate getNamedParamJdbcTemplate() {
        if(null == namedParameterJdbcTemplate) {
            namedParameterJdbcTemplate = new FrameworkNamedParameterJdbcTemplate(this);
        }
        return namedParameterJdbcTemplate;
    }


    /**
     * 重写JdbcTemplate里面的queryForObject方法源码调用的requiredSingleResult，当查询到的结果为空时返回null(原来是抛出异常)
     */
    @Override
    public <T> T queryForObject(String sql, Class<T> requiredType) throws DataAccessException {
        return queryForObject(sql, getSingleColumnRowMapper(requiredType));
    }

    /**
     * 重写JdbcTemplate里面的queryForObject方法源码调用的requiredSingleResult，当查询到的结果为空时返回null(原来是抛出异常)
     */
    @Override
    public <T> T queryForObject(String sql, Object[] args, RowMapper<T> rowMapper) throws DataAccessException {
        List<T> results = (List)this.query((String)sql, (Object[])args, (ResultSetExtractor)(new RowMapperResultSetExtractor(rowMapper, 1)));
        return requiredSingleResult(results);
    }

    /**
     * 重写JdbcTemplate里面的queryForObject方法源码调用的requiredSingleResult，当查询到的结果为空时返回null(原来是抛出异常)
     */
    @Override
    public <T> T queryForObject(String sql, RowMapper<T> rowMapper) throws DataAccessException {
        List<T> results = query(sql, rowMapper);
        return requiredSingleResult(results);
    }

    @Override
    public <T> T queryForObject(String sql, RowMapper<T> rowMapper, Object... args) throws DataAccessException {
        List<T> results = query(sql, args, new RowMapperResultSetExtractor<T>(rowMapper, 1));
        return requiredSingleResult(results);
    }

    // 重写JdbcTemplate，当queryForObject查询结果为0时，返回null而不是抛出异常
    @Override
    public Map<String, Object> queryForMap(String sql, Object... args) throws DataAccessException {
        try {
            return super.queryForMap(sql, args);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 重写当查询到的结果为空时返回null(原来是抛出异常)
     */
    public static  <T> T requiredSingleResult(Collection<T> results) throws IncorrectResultSizeDataAccessException {
        int size = (results != null ? results.size() : 0);
        if (size == 0) {
            return null;
        }
        if (results.size() > 1) {
            throw new IncorrectResultSizeDataAccessException(1, size);
        }
        return results.iterator().next();
    }

    /**
     * 重写map的字段K值为驼峰
     * @return
     */
    protected RowMapper<Map<String, Object>> getOverrideColumnMapRowMapper() {
        return new ColumnMap();
    }

    /**
     * 新增集合嵌套map返回结果查询
     * 自动转换驼峰命名规则
     * @param sql
     * @return
     */
    public List<Map<String, Object>> queryForMapList(String sql) throws DataAccessException {
        return this.query(sql, this.getOverrideColumnMapRowMapper());
    }

    /**
     * 返回int类型
     *
     * @param wsql
     * @param values
     * @param types
     * @return
     */
    public int queryForInt(String wsql, Object[] values, int[] types) {
        String logstr = null;
        if (log.isDebugEnabled()) {
            logstr = "queryForInt: sql=" + wsql + "; parameters=" + arrayToString(values);
            log.debug(logstr);
        }
        Integer result = 0;
        try {
            if (values == null || values.length == 0) {
                result = this.queryForObject(wsql, Integer.class);
            } else if (types == null || types.length == 0) {
                result = this.queryForObject(wsql, values, Integer.class);
            } else {
                result = this.queryForObject(wsql, values, types, Integer.class);
            }
        } catch (EmptyResultDataAccessException e) {
        }
        if (log.isInfoEnabled()) {
            if (logstr == null) {
                logstr = "queryForInt: sql=" + wsql + "; parameters=" + arrayToString(values);
            }
            log.info(logstr + "; result=" + result);
        }
        return (result == null ? 0 : result.intValue());
    }

    /**
     * 返回long类型
     *
     * @param wsql
     * @param values
     * @param types
     * @return
     */
    public long queryForLong(String wsql, Object[] values, int[] types) {
        String logstr = null;
        if (log.isDebugEnabled()) {
            logstr = "queryForLong: sql=" + wsql + "; parameters=" + arrayToString(values);
            log.debug(logstr);
        }
        Long result = 0l;
        try {
            if (values == null || values.length == 0) {
                result = this.queryForObject(wsql, Long.class);
            } else if (types == null || types.length == 0) {
                result = this.queryForObject(wsql, values, Long.class);
            } else {
                result = this.queryForObject(wsql, values, types, Long.class);
            }
        } catch (EmptyResultDataAccessException e) {
        }
        if (log.isInfoEnabled()) {
            if (logstr == null) {
                logstr = "queryForLong: sql=" + wsql + "; parameters=" + arrayToString(values);
            }
            log.info(logstr + "; result=" + result);
        }
        return (result == null ? 0 : result.longValue());
    }


    /**
     * 返回String类型
     *
     * @param wsql
     * @param values
     * @param types
     * @return
     */
    public String queryForString(String wsql, Object[] values, int[] types) {
        String logstr = null;
        if (log.isDebugEnabled()) {
            logstr = "queryForString: sql=" + wsql + "; parameters=" + arrayToString(values);
            log.debug(logstr);
        }
        String rows = null;
        try {
            if (values == null || values.length == 0) {
                rows = this.queryForObject(wsql, String.class);
            } else if (types == null || types.length == 0) {
                rows = this.queryForObject(wsql, values, String.class);
            } else {
                rows = this.queryForObject(wsql, values, types, String.class);
            }
        } catch (EmptyResultDataAccessException e) {
        }
        if (log.isInfoEnabled()) {
            if (logstr == null) {
                logstr = "queryForString: sql=" + wsql + "; parameters=" + arrayToString(values);
            }
            log.info(logstr + "; result=" + rows);
        }
        return rows;
    }


    /**
     * * 通过SQL查询一个对象的所有信息，只返回一条记录
     *
     * @param clazz  Object 需要查询的对象，这个对象是对数据库映射的一个javabean
     * @param wsql   String 查询的语句
     * @param values
     * @param types
     * @return
     */
    public <T> T queryForObject(final Class<T> clazz, String wsql, Object[] values, int[] types) {
        ResultSetExtractor<T> rse = rs -> {
            if (rs.next()) {
                return convert2Object(rs, clazz);
            }
            return null;
        };
        if (wsql == null || (wsql = wsql.trim()).isEmpty()) {
            wsql = "SELECT * FROM " + JdbcUtil.getTableName(clazz);
            if(SqlTypeConstant.ORACLE.equals(getDbType().toLowerCase())){
                wsql += " WHERE ROWNUM<=1";
            }else {
                wsql += " LIMIT 1";
            }
        } else {
            if (!(wsql.startsWith("SELECT ") || wsql.startsWith("select "))) {
                wsql = "SELECT * FROM " + JdbcUtil.getTableName(clazz) + " WHERE " + wsql;
            }
        }

        String logstr = null;
        if (log.isDebugEnabled()) {
            logstr = "queryForObject: sql=" + wsql + "; parameters=" + arrayToString(values);
            log.debug(logstr);
        }
        T t = null;
        if (values == null || values.length == 0) {
            t = this.query(wsql, rse);
        } else if (types == null || types.length == 0) {
            t = this.query(wsql, values, rse);
        } else {
            t = this.query(wsql, values, types, rse);
        }
        if (log.isInfoEnabled()) {
            if (logstr == null) {
                logstr = "queryForObject: sql=" + wsql + "; parameters=" + arrayToString(values);
            }
            log.info(logstr + "; result=" + t);
        }
        return t;
    }

    /**
     * 返回Map<String, Object>对象
     *
     * @param wsql
     * @param values
     * @param types
     * @return
     */
    public Map<String, Object> queryForMap(String wsql, Object[] values, int[] types) {
        ResultSetExtractor<Map<String, Object>> rse = rs -> {
            if (!rs.next()) {
                return null;
            }
            ResultSetMetaData meta = rs.getMetaData();
            int colCount = meta.getColumnCount();
            // columns[0]-字段名, columns[1]-字段类型
            List<Object[]> columns = new ArrayList<>();
            for (int i = 1; i <= colCount; i++) {
                columns.add(new Object[]{meta.getColumnName(i), meta.getColumnType(i)});
            }
            return convert2Map(rs, columns);
        };

        String logstr = null;
        if (log.isDebugEnabled()) {
            logstr = "queryForMap: sql=" + wsql + "; parameters=" + arrayToString(values);
            log.debug(logstr);
        }
        Map<String, Object> map = null;
        if (values == null || values.length == 0) {
            map = this.query(wsql, rse);
        } else if (types == null || types.length == 0) {
            map = this.query(wsql, values, rse);
        } else {
            map = this.query(wsql, values, types, rse);
        }
        if (log.isInfoEnabled()) {
            if (logstr == null) {
                logstr = "queryForMap: sql=" + wsql + "; parameters=" + arrayToString(values);
            }
            log.info(logstr + "; result=" + map);
        }
        return map;
    }


    /**
     * 从指定表中查询某个对象的列表，返回多条记录
     *
     * @param clazz   <T>
     * @param wsql
     * @param values
     * @param orderBy
     * @return
     */
    public <T> List<T> queryForList(final Class<T> clazz, String wsql, Object[] values, String orderBy) {
        ResultSetExtractor<List<T>> rse = new ResultSetExtractorImpl(clazz);
        if (wsql == null || (wsql = wsql.trim()).isEmpty()) {
            wsql = "SELECT * FROM " + JdbcUtil.getTableName(clazz) + "";
        } else if (!(wsql.startsWith("SELECT ") || wsql.startsWith("select "))) {
            wsql = "SELECT * FROM " + JdbcUtil.getTableName(clazz) + " WHERE " + wsql;
        }
        if (orderBy != null && !orderBy.isEmpty()) {
            wsql += " ORDER BY " + orderBy;
        }

        String logstr = null;
        if (log.isDebugEnabled()) {
            logstr = "queryForList: sql=" + wsql + "; parameters=" + arrayToString(values);
            log.debug(logstr);
        }
        List<T> list = null;
        if (values == null || values.length == 0) {
            list = this.query(wsql, rse);
        } else {
            list = this.query(wsql, values, rse);
        }
        if (log.isInfoEnabled()) {
            if (logstr == null) {
                logstr = "queryForList: sql=" + wsql + "; parameters=" + arrayToString(values);
            }
            log.info(logstr + "; size=" + (list == null ? 0 : list.size()));
        }
        return list;
    }


    /**
     * 返回一个List<Map>
     *
     * @param sql
     * @param values
     * @param types
     * @return
     */
    public List<Map<String, Object>> queryForMapList(String sql, Object[] values, int[] types) {
        final List<Map<String, Object>> list = new ArrayList<>();
        final List<Object[]> columns = new ArrayList<>();
        RowCallbackHandler hander = rs -> {
            if (columns.isEmpty()) {
                ResultSetMetaData meta = rs.getMetaData();
                int colCount = meta.getColumnCount();
                // columns[0]-字段名, columns[1]-字段类型
                for (int i = 1; i <= colCount; i++) {
                    columns.add(new Object[]{meta.getColumnName(i), meta.getColumnType(i)});
                }
            }
            list.add(convert2Map(rs, columns));
        };
        String logstr = null;
        if (log.isDebugEnabled()) {
            logstr = "queryForList<Map>: sql=" + sql + "; parameters=" + arrayToString(values);
            log.debug(logstr);
        }
        if (values == null || values.length == 0) {
            this.query(sql, hander);
        } else if (types == null || types.length == 0) {
            this.query(sql, values, hander);
        } else {
            this.query(sql, values, types, hander);
        }

        if (log.isInfoEnabled()) {
            if (logstr == null) {
                logstr = "queryForList<Map>: sql=" + sql + "; parameters=" + arrayToString(values);
            }
            log.info(logstr + "; size=" + (list == null ? 0 : list.size()));
        }
        return list;
    }

    /**
     * 返回一个List<Integer>
     *
     * @param sql
     * @param values
     * @param unique 返回结果是否去重 ture是
     * @return List<Integer>
     */
    public List<Integer> queryForIntegerList(String sql, Object[] values, final boolean unique) {
        final List<Integer> list = new ArrayList<>();
        RowCallbackHandler hander = rs -> {
            ResultSetMetaData meta = rs.getMetaData();
            if (meta != null) {
                int colCount = meta.getColumnCount();
                if (colCount >= 1) {
                    int i = rs.getInt(1);
                    if (unique && list.contains(i)) {
                        return;
                    }
                    list.add(i);
                }
            }
        };
        String logstr = null;
        if (log.isDebugEnabled()) {
            logstr = "queryForList<Integer>: sql=" + sql + "; parameters=" + arrayToString(values);
            log.debug(logstr);
        }
        if (values == null || values.length == 0) {
            this.query(sql, hander);
        } else {
            this.query(sql, values, hander);
        }
        if (log.isInfoEnabled()) {
            if (logstr == null) {
                logstr = "queryForList<Integer>: sql=" + sql + "; parameters=" + arrayToString(values);
            }
            log.info(logstr + "; size=" + (list == null ? 0 : list.size()));
        }
        return list;
    }

    /**
     * 返回一个List<Long>
     *
     * @param sql
     * @param values
     * @param unique 返回结果是否去重 ture是
     * @return List<Long>
     */
    public List<Long> queryForLongList(String sql, Object[] values, final boolean unique) {
        final List<Long> list = new ArrayList<>();
        RowCallbackHandler hander = rs -> {
            ResultSetMetaData meta = rs.getMetaData();
            if (meta != null) {
                int colCount = meta.getColumnCount();
                if (colCount >= 1) {
                    long i = rs.getLong(1);
                    if (unique && list.contains(i)) {
                        return;
                    }
                    list.add(i);
                }
            }
            meta = null;
        };
        String logstr = null;
        if (log.isDebugEnabled()) {
            logstr = "queryForList<Long>: sql=" + sql + "; parameters=" + arrayToString(values);
            log.debug(logstr);
        }
        if (values == null || values.length == 0) {
            this.query(sql, hander);
        } else {
            this.query(sql, values, hander);
        }
        if (log.isInfoEnabled()) {
            if (logstr == null) {
                logstr = "queryForList<Long>: sql=" + sql + "; parameters=" + arrayToString(values);
            }
            log.info(logstr + "; size=" + (list == null ? 0 : list.size()));
        }
        return list;
    }

    /**
     * 返回一个List<String>
     *
     * @param sql
     * @param values
     * @param unique 返回结果是否去重 ture是
     * @return List<String>
     */
    public List<String> queryForStringList(String sql, Object[] values, final boolean unique) {
        final List<String> list = new ArrayList<>();
        RowCallbackHandler hander = rs -> {
            ResultSetMetaData meta = rs.getMetaData();
            if (meta != null) {
                int colCount = meta.getColumnCount();
                if (colCount >= 1) {
                    String i = rs.getString(1);
                    if (unique && list.contains(i)) {
                        return;
                    }
                    list.add(i);
                }
            }
            meta = null;
        };
        String logstr = null;
        if (log.isDebugEnabled()) {
            logstr = "queryForList<String>: sql=" + sql + "; parameters=" + arrayToString(values);
            log.debug(logstr);
        }
        if (values == null || values.length == 0) {
            this.query(sql, hander);
        } else {
            this.query(sql, values, hander);
        }
        if (log.isInfoEnabled()) {
            if (logstr == null) {
                logstr = "queryForList<String>: sql=" + sql + "; parameters=" + arrayToString(values);
            }
            log.info(logstr + "; size=" + (list == null ? 0 : list.size()));
        }
        return list;
    }

    /**
     * 执行无返回值的存储过程
     *
     * @param callString 存储过程语句
     */
    public void execute(String callString){
        if (log.isDebugEnabled()) {
            log.debug("execute: sql=" + callString);
        }
        this.execute(callString);
    }


    /**
     * 执行无返回值的存储过程
     *
     * @param callString 存储过程语句
     * @param args       参数
     */
    public void execute(final String callString, final Object[] args) {
        if (args == null || args.length == 0) {
            execute(callString);
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("execute: sql=" + callString + "; parameters=" + arrayToString(args));
        }
        this.execute(connection -> {
            CallableStatement cs = connection.prepareCall(callString);
            int len = args.length;
            for (int i = 1; i <= len; i++) {
                cs.setObject(i, args[i - 1]);// 设置输入参数的值
            }
            return cs;
        }, (CallableStatementCallback<String>) callableStatement -> {
            callableStatement.execute();
            return null;
        });
    }

    /**
     * 执行存储过程，返回String类型
     *
     * @param callString 存储过程语句
     */
    public String executeForString(String callString) {
        return executeForString(callString, null);
    }

    /**
     * 执行存储过程，返回String类型
     *
     * @param callString 存储过程语句
     * @param args       参数
     */
    public String executeForString(final String callString, final Object[] args) {
        if (log.isDebugEnabled()) {
            log.debug("executeForString: sql=" + callString + "; parameters=" + arrayToString(args));
        }
        final int len = (args == null ? 0 : args.length);
        return this.execute(connection -> {
            CallableStatement cs = connection.prepareCall(callString);
            for (int i = 1; i <= len; i++) {
                cs.setObject(i, args[i - 1]);// 设置输入参数的值
            }
            cs.registerOutParameter(len + 1, Types.VARCHAR);// 注册输出参数的类型
            return cs;
        }, (CallableStatementCallback<String>) callableStatement -> {
            callableStatement.execute();
            return callableStatement.getString(len + 1);// 获取输出参数的值
        });
    }


    /**
     * 执行存储过程，返回int类型
     *
     * @param callString 存储过程语句
     */
    public int executeForInt(String callString) {
        return executeForInt(callString, null);
    }

    /**
     * 执行存储过程，返回int类型
     *
     * @param callString 存储过程语句
     * @param args       参数
     */
    public int executeForInt(final String callString, final Object[] args) {
        if (log.isDebugEnabled()) {
            log.debug("executeForInt: sql=" + callString + "; parameters=" + arrayToString(args));
        }
        final int len = (args == null ? 0 : args.length);
        return this.execute((CallableStatementCreator) (connection -> {
            String storedProc = callString;// 调用的sql
            CallableStatement cs = connection.prepareCall(storedProc);
            for (int i = 1; i <= len; i++) {
                cs.setObject(i, args[i - 1]);// 设置输入参数的值
            }
            cs.registerOutParameter(len + 1, Types.BIGINT);// 注册输出参数的类型
            return cs;
        }), callableStatement -> {
            callableStatement.execute();
            return callableStatement.getInt(len + 1);// 获取输出参数的值
        });
    }

    /**
     * 执行存储过程，返回List类型
     *
     * @param callString 存储过程语句
     */
    public List<Map<String, Object>> executeForList(String callString) {
        return executeForList(callString, null);
    }

    /**
     * 执行存储过程，返回List类型
     *
     * @param callString 存储过程语句
     * @param args       参数
     */
    public List<Map<String, Object>> executeForList(final String callString, final Object[] args) {
        if (log.isDebugEnabled()) {
            log.debug("executeForList: sql=" + callString + "; parameters=" + arrayToString(args));
        }
        final int len = (args == null ? 0 : args.length);
        return this.execute((CallableStatementCreator) (connection -> {
            String storedProc = callString;// 调用的sql
            CallableStatement cs = connection.prepareCall(storedProc);
            for (int i = 1; i <= len; i++) {
                cs.setObject(i, args[i - 1]);// 设置输入参数的值
            }
            cs.registerOutParameter(len + 1, Types.REF);// 注册输出参数的类型
            return cs;
        }), callableStatement -> {
            callableStatement.execute();
            ResultSet rs = callableStatement.getObject(len + 1, ResultSet.class);// 获取游标一行的值
            if (!rs.next()) {
                return null;
            }
            List<Map<String, Object>> list = new ArrayList<>();
            ResultSetMetaData meta = rs.getMetaData();
            int colCount = meta.getColumnCount();
            // columns[0]-字段名, columns[1]-字段类型
            List<Object[]> columns = new ArrayList<>();
            for (int i = 1; i <= colCount; i++) {
                columns.add(new Object[]{meta.getColumnName(i), meta.getColumnType(i)});
            }
            meta = null;
            do {// 转换每行的返回值到Map中
                list.add(convert2Map(rs, columns));
            } while (rs.next());
            rs.close();
            return list;
        });
    }

    /**
     * 执行存储过程，返回Map类型
     *
     * @param callString 存储过程语句
     */
    public Map<String, Object> executeForMap(String callString) {
        return executeForMap(callString, null);
    }

    /**
     * 执行存储过程，返回Map类型
     *
     * @param callString 存储过程语句
     * @param args       参数
     */
    public Map<String, Object> executeForMap(final String callString, final Object[] args) {
        if (log.isDebugEnabled()) {
            log.debug("executeForMap: sql=" + callString + "; parameters=" + arrayToString(args));
        }
        final int len = (args == null ? 0 : args.length);
        return this.execute((CallableStatementCreator) (connection -> {
            String storedProc = callString;// 调用的sql
            CallableStatement cs = connection.prepareCall(storedProc);
            for (int i = 1; i <= len; i++) {
                cs.setObject(i, args[i]);// 设置输入参数的值
            }
            cs.registerOutParameter(len + 1, Types.REF);// 注册输出参数的类型
            return cs;
        }), callableStatement -> {
            callableStatement.execute();
            ResultSet rs = callableStatement.getObject(len + 1, ResultSet.class);// 获取游标一行的值
            if (!rs.next()) {
                return null;
            }
            ResultSetMetaData meta = rs.getMetaData();
            int colCount = meta.getColumnCount();
            // columns[0]-字段名, columns[1]-字段类型
            List<Object[]> columns = new ArrayList<>();
            for (int i = 1; i <= colCount; i++) {
                columns.add(new Object[]{meta.getColumnName(i), meta.getColumnType(i)});
            }
            meta = null;
            Map<String, Object> m = convert2Map(rs, columns);
            rs.close();
            return m;
        });
    }


    /**
     * 插入或更新
     *
     * @param sql
     * @param values
     * @param types
     * @param pkName
     * @param <T>
     * @return
     */
    public <T> long save(final String sql, final Object[] values, final int[] types, final String pkName) {
        String logstr = null;
        if (log.isDebugEnabled()) {
            logstr = "save: sql=" + sql + "; parameters=" + arrayToString(values);
            log.debug(logstr);
        }
        if (pkName == null || pkName.isEmpty()) {
            int row = this.update(new PreparedStatementCreatorImpl(sql, values, types));
            if (log.isInfoEnabled()) {
                if (logstr == null) {
                    logstr = "save: sql=" + sql + "; parameters=" + arrayToString(values);
                }
                log.info(logstr + "; row=" + row);
            }
            return row;
        } else {
            KeyHolder keyHolder = new GeneratedKeyHolder();
            this.update(new PreparedStatementCreatorImpl(sql, values, types, pkName), keyHolder);
            Number n = keyHolder.getKey();
            if (n == null) {
                return 0;
            }
            long id = n.longValue();
            if (log.isInfoEnabled()) {
                if (logstr == null) {
                    logstr = "save: sql=" + sql + "; parameters=" + arrayToString(values);
                }
                log.info(logstr + "; pk=" + id);
            }
            return id;
        }
    }

    /**
     * 插入或更新
     *
     * @param sql
     * @param values
     * @param pkName
     * @return
     */
    public long save(String sql, Object[] values, String pkName) {
        return save(sql, values, null, pkName);
    }

    /**
     * 保存并返回影响行数
     *
     * @param sql
     * @param values
     * @param types
     * @return
     */
    public <T> boolean insert(final String sql, final Object[] values, final int[] types) {
        String logstr = null;
        if (log.isDebugEnabled()) {
            logstr = "insert: sql=" + sql + "; parameters=" + arrayToString(values);
            log.debug(logstr);
        }
        int row = 0;
        if (types == null) {
            row = this.update(sql, values);
        } else {
            row = this.update(new PreparedStatementCreatorImpl(sql, values, types));
        }
        if (log.isInfoEnabled()) {
            if (logstr == null) {
                logstr = "insert: sql=" + sql + "; parameters=" + arrayToString(values);
            }
            log.info(logstr + "; rows=" + row);
        }

        return row > 0;
    }


    /**
     * 增加、删除、修改表时，调用此方法
     *
     * @param wsql DDL语句
     * @param args 参数
     * @param types 可以为null
     */
    public int update(String wsql, Object[] args, int[] types) {
        String logstr = null;
        if (log.isDebugEnabled()) {
            logstr = "update: sql=" + wsql + "; parameters=" + arrayToString(args);
            log.debug(logstr);
        }
        int rows = 0;
        if (types == null) {
            rows = this.update(wsql, args);
        } else {
            rows = this.update(wsql, args, types);
        }
        if (log.isInfoEnabled()) {
            if (logstr == null) {
                logstr = "update: sql=" + wsql + "; parameters=" + arrayToString(args);
            }
            log.info(logstr + "; rows=" + rows);
        }
        return rows;
    }

    /**
     * 新增或更新对象
     *
     * @param t
     * @return
     * @throws SecurityException
     * @throws InvocationTargetException
     * @throws IllegalArgumentException
     * @throws IllegalAccessException
     */
    public <T> boolean saveOrUpdate(T t) throws IllegalAccessException, IllegalArgumentException, InvocationTargetException,
            SecurityException {
        Dialect dialect = DialectUtil.getDialect(this);
        SaveOrUpdateModel model = JdbcUtil.buildSaveOrUpdateModel(t);
        boolean isNew = true;
        if (model.pkValue != null && !"".equals(model.pkValue + "") && !"0".equals(model.pkValue + "")) {
            String s = "SELECT COUNT(1) FROM " + model.table + " WHERE " + dialect.getMark() + model.pkName + dialect.getMark() + "=?";
            if (this.queryForObject(s, new Object[]{model.pkValue}, String.class) != null) {
                isNew = false;
            }
        }
        if (!isNew) {
            StringBuilder sql_set = new StringBuilder();
            int size = model.columns.size();
            int[] types = new int[size];
            Object[] values = new Object[size];
            int j = 0;
            Object val;
            String column;
            for (int i = 0; i < size; i++) {
                column = model.columns.get(i);
                if (column.equalsIgnoreCase(model.pkName)) {
                    values[size - 1] = model.values.get(i);
                    types[size - 1] = model.types.get(i).intValue();
                    continue;
                }
                val = model.values.get(i);
                if (String.valueOf(val).toUpperCase().endsWith(".NEXTVAL")) {
                    sql_set.append(",").append(dialect.getMark()).append(column).append(dialect.getMark()).append("=").append(val);
                } else {
                    sql_set.append(",").append(dialect.getMark()).append(column).append(dialect.getMark()).append("=?");
                    values[j] = val;
                    types[j] = model.types.get(i).intValue();
                    j++;
                }
            }
            if (j != size - 1) {
                values[j + 1] = values[size - 1];
                types[j + 1] = types[size - 1];
                values = ArrayUtil.subArray(values, j + 1);
                types = ArrayUtil.subArray(types, j + 1);
            }
            StringBuilder sql = new StringBuilder("UPDATE ");
            sql.append(" ").append(model.table);
            sql.append(" SET ").append(sql_set.substring(1));
            sql.append(" WHERE ").append(dialect.getMark()).append(model.pkName).append(dialect.getMark()).append("=?");
            return update(sql.toString(), values, types) >= 0;
        } else {
            StringBuilder sql_fields = new StringBuilder();
            StringBuilder sql_values = new StringBuilder();
            int size = model.columns.size();
            int[] types = new int[size];
            Object[] values = new Object[size];
            int j = 0;
            Object val;
            for (int i = 0; i < size; i++) {
                sql_fields.append(",").append(dialect.getMark()).append(model.columns.get(i)).append(dialect.getMark());
                val = model.values.get(i);
                if (String.valueOf(val).endsWith(".NEXTVAL")) {
                    sql_values.append(",").append(val);
                } else {
                    sql_values.append(",?");
                    values[j] = val;
                    types[j] = model.types.get(i).intValue();
                    j++;
                }
            }
            if (j != size) {
                values = ArrayUtil.subArray(values, j);
                types = ArrayUtil.subArray(types, j);
            }
            StringBuilder sql = new StringBuilder("INSERT INTO");
            sql.append(" ").append(model.table);
            sql.append(" (").append(sql_fields.substring(1)).append(")");
            sql.append(" VALUES (").append(sql_values.substring(1)).append(")");
            return save(sql.toString(), values, types, model.pkName) > 0;
        }
    }

    /**
     * 批量更新
     *
     * @param sql
     * @param valueList
     * @param types     java.sql.Types
     * @return
     */
    public int batchUpdateNew(String sql, final List<Object[]> valueList, final int[] types) {
        BatchPreparedStatementSetter setter = new BatchPreparedStatementSetter() {
            public void setValues(PreparedStatement ps, int index) throws SQLException {
                Object[] values = valueList.get(index);
                PreparedStatementCreatorImpl.setValues(ps, values, types);
            }

            public int getBatchSize() {
                return valueList.size();
            }
        };
        String logstr = null;
        if (log.isDebugEnabled()) {
            logstr = "batchUpdate: sql=" + sql + "; parameters=" + arrayListToString(valueList);
            log.debug(logstr);
        }
        int[] rows = null;
        if (types == null) {
            rows = this.batchUpdate(sql, valueList);
        } else {
            rows = this.batchUpdate(sql, setter);
        }
        int total = 0;
        for (int r : rows) {
            if (r == -2) {
                total += 1;
            } else if (r > 0) {
                total += r;
            }
        }
        if (log.isInfoEnabled()) {
            if (logstr == null) {
                logstr = "batchUpdate: sql=" + sql + "; parameters=" + arrayListToString(valueList);
            }
            log.info(logstr + "; rows=" + total);
        }
        return total;
    }


    /**
     * Object[] 转成 String
     *
     * @param objs 对象数组
     * @return
     */
    protected static String arrayToString(Object[] objs) {
        if (objs == null) {
            return "[]";
        }
        StringBuffer buf = new StringBuffer();
        buf.append("[");
        for (int j = 0; j < objs.length; j++) {
            if (j > 0) {
                buf.append(", ");
            }
            buf.append(String.valueOf(objs[j]));
        }
        buf.append("]");
        return buf.toString();
    }

    protected static String arrayListToString(List<Object[]> list) {
        if (list == null || list.isEmpty()) {
            return "[]";
        }
        StringBuffer buf = new StringBuffer();
        for (Object[] obj : list) {
            buf.append(", ").append(arrayToString(obj));
        }
        return "[" + buf.substring(2) + "]";
    }



    /**
     * 动态调用setter方法设置值
     *
     * @param rs
     * @param object
     * @param method
     * @param fieldType
     * @param columnName
     * @param annotation
     */
    private static final void invoke(ResultSet rs, Object object, Method method, Class fieldType, String columnName, cn.dolphin.core.jdbc.annotation.Field
            annotation) throws SQLException {
        try {
            if (fieldType == String.class) {
                if (annotation != null && annotation.format() != null && !annotation.format().isEmpty()) {
                    method.invoke(object, DateFormatUtil.formatDate(rs.getString(columnName), annotation.format()));
                } else {
                    method.invoke(object, rs.getString(columnName));
                }
            } else if (fieldType == Integer.class || fieldType == int.class) {
                method.invoke(object, rs.getInt(columnName));
            } else if (fieldType == Long.class || fieldType == long.class) {
                method.invoke(object, rs.getLong(columnName));
            } else if (fieldType == Float.class || fieldType == float.class) {
                if (annotation != null && annotation.scale() > 0) {
                    method.invoke(object, formatFloat(rs.getFloat(columnName), annotation.scale()));
                } else {
                    method.invoke(object, rs.getFloat(columnName));
                }
            } else if (fieldType == Double.class || fieldType == double.class) {
                if (annotation != null && annotation.scale() > 0) {
                    method.invoke(object, formatDouble(rs.getDouble(columnName), annotation.scale()));
                } else {
                    method.invoke(object, rs.getDouble(columnName));
                }
            } else if (fieldType == java.util.Date.class || fieldType == Timestamp.class) {
                method.invoke(object, rs.getTimestamp(columnName));
            } else if (fieldType == Date.class) {
                method.invoke(object, rs.getDate(columnName));
            } else if (fieldType == byte[].class) {
                method.invoke(object, rs.getBytes(columnName));
            } else if (fieldType == Short.class || fieldType == short.class) {
                method.invoke(object, rs.getShort(columnName));
            }
        } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
            log.error("method=" + method.getName() + ", column=" + columnName + ", error=" + e.toString(), e);
        }
    }

    protected static double formatDouble(double d, int scale) {
        if (d == 0) {
            return d;
        }
        try {
            return new BigDecimal(d).setScale(scale, BigDecimal.ROUND_HALF_UP).doubleValue();
        } catch (Exception e) {
            e.getMessage();
            return d;
        }
    }

    protected static float formatFloat(float f, int scale) {
        if (f == 0) {
            return f;
        }
        try {
            return new BigDecimal(f).setScale(scale, BigDecimal.ROUND_HALF_UP).floatValue();
        } catch (Exception e) {
            e.getMessage();
            return f;
        }
    }

    /**
     * 获取对应的数据库字段名
     *
     * @param annotation
     * @param field
     * @return
     */
    private static final String getColumnName(cn.dolphin.core.jdbc.annotation.Field annotation, Field field) {
        if (annotation == null) {
            return CamelCaseUtil.toUnderlineName(field.getName());
        }
        if (!annotation.isColumn()) {
            return null;
        }
        if (annotation.value() != null && !annotation.value().isEmpty()) {
            return annotation.value();
        } else {
            return CamelCaseUtil.toUnderlineName(field.getName());
        }
    }


    /**
     * 将结果集转成实体对象
     *
     * @param rs
     * @param clazz
     * @return
     * @throws Exception
     */
    @SuppressWarnings("unchecked")
    protected static <T> T convert2Object(ResultSet rs, Class<T> clazz) throws SQLException, DataAccessException {
        ResultSetMetaData meta = rs.getMetaData();
        int colCount = meta.getColumnCount();
        // columns[0]-数据集中的原始名, columns[1]-对应子类中的属性名
        List<String[]> columns = new ArrayList<>();
        for (int i = 1; i <= colCount; i++) {
            columns.add(new String[]{meta.getColumnName(i), meta.getColumnName(i)});
        }
        return (T) convert2Object(rs, clazz, columns);
    }

    /**
     * 将结果集转成实体对象
     *
     * @param rs
     * @param clazz
     * @param columns
     * @return
     * @throws SQLException
     * @throws DataAccessException
     */
    private static Object convert2Object(ResultSet rs, Class<?> clazz, List<String[]> columns) throws SQLException, DataAccessException {
        Object object = null;
        Class<?> fieldType;
        List<BeanField> fields = ReflectUtil.getBeanFields(clazz);
        try {
            object = clazz.newInstance();
        } catch (InstantiationException | IllegalAccessException e) {
            e.printStackTrace();
            return null;
        }
        Method method;
        String column, subObjName;
        cn.dolphin.core.jdbc.annotation.Field annotation = null;
        String[] columnName;
        for (BeanField field : fields) {
            method = field.getSetterMethod();
            if (method == null) {
                continue;
            }
            annotation = field.getAnnotation(cn.dolphin.core.jdbc.annotation.Field.class);
            column = getColumnName(annotation, field.field);
            if (column == null) {
                continue;
            }

            List<String[]> subColumns = null;//子对象字段，String[]{数据库字段名,实体属性名}
            for (int i = 0; i < columns.size(); i++) {
                columnName = columns.get(i);
                if (column.equalsIgnoreCase(columnName[1]) || field.getName().equalsIgnoreCase(columnName[1])) {
                    fieldType = method.getParameterTypes()[0];
                    if (annotation != null && annotation.isJson()) {
                        String json = rs.getString(columnName[0]);
                        try {
                            if (json != null && !(json = json.trim()).isEmpty()) {
                                if (fieldType.isAssignableFrom(List.class)) {
                                    Class<?> genericClazz = ReflectUtil.getGenericType(field.field);
                                    method.invoke(object, JSONArray.parseArray(json, genericClazz));
                                } else {
                                    method.invoke(object, JSONObject.parseObject(json, fieldType));
                                }
                            }
                        } catch (JSONException | IllegalAccessException | InvocationTargetException e) {
                            log.error("field=" + field.getName() + ", method=" + method.getName() + ", column=" + columnName[0] + ", json=" + json + ", " +
                                    "error=" + e
                                    .toString(), e);
                        }
                    } else {
                        invoke(rs, object, method, fieldType, columnName[0], annotation);
                    }
                    columns.remove(i);//移除已匹配过的字段，减少下次循环次数
                    break;
                }
                int ii = columnName[1].indexOf(".");
                if (ii > 0) {// 子对象字段
                    subObjName = columnName[1].substring(0, ii);
                    if (column.equalsIgnoreCase(subObjName) || field.getName().equalsIgnoreCase(subObjName)) {
                        if (subColumns == null) {
                            subColumns = new ArrayList<>();
                        }
                        subColumns.add(new String[]{columnName[0], columnName[1].substring(ii + 1)});
                        columns.remove(i);//移除已匹配过的字段，减少下次循环次数
                        i--;
                    }
                }
            }
            if (subColumns != null) {
                Object val = convert2Object(rs, method.getParameterTypes()[0], subColumns);
                if (val != null) {
                    try {
                        method.invoke(object, val);
                    } catch (Exception e) {
                        throw new DaoRuntimException(e);
                    }
                }
            }
        }

        return object;
    }


    protected static Object convert2Entity(ResultSet rs, Class<?> clazz, List<EntityField> fields) throws SQLException, DataAccessException {
        Object object = null;
        Class<?> fieldType;
        try {
            object = clazz.newInstance();
        } catch (InstantiationException | IllegalAccessException e) {
            e.printStackTrace();
            return null;
        }
        String columnName = null;
        Object obj = null;
        Method method;
        for (EntityField entityfield : fields) {
            method = entityfield.setter;
            fieldType = method.getParameterTypes()[0];
            columnName = entityfield.columnName;
            if (entityfield.type == FieldType.ENTITY) {
                obj = convert2Entity(rs, fieldType, entityfield.entityFields);
                if (obj != null) {
                    try {
                        method.invoke(object, obj);
                    } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
                        log.error("method=" + method + ", fieldType=" + fieldType + ", error=" + e.toString(), e);
                    }
                }
            } else if (entityfield.type == FieldType.JSON_ARRAY) {
                String json = rs.getString(entityfield.columnName);
                if (json != null && !(json = json.trim()).isEmpty()) {
                    try {
                        method.invoke(object, JSONArray.parseArray(json, entityfield.genericClazz));
                    } catch (JSONException e) {
                        log.error("method=" + method + ", column=" + columnName + ", json=" + json + ", error=" + e.toString(), e);
                    } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
                        log.error("method=" + method + ", column=" + columnName + ", error=" + e.toString(), e);
                    }
                }
            } else if (entityfield.type == FieldType.JSON_OBJECT) {
                String json = rs.getString(entityfield.columnName);
                try {
                    method.invoke(object, JSONObject.parseObject(json, fieldType));
                } catch (JSONException e) {
                    log.error("method=" + method + ", column=" + columnName + ", json=" + json + ", error=" + e.toString(), e);
                } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
                    log.error("method=" + method + ", column=" + columnName + ", error=" + e.toString(), e);
                }
            } else {
                invoke(rs, object, method, fieldType, columnName, entityfield.annotation);
            }
        }

        return object;
    }

    protected static Map<String, Object> convert2Map(ResultSet rs, List<Object[]> columns) throws SQLException {
        Map<String, Object> m = new HashMap<>();
        for (Object[] col : columns) {
            String fieldName = (String) col[0];
            int type = (int) col[1];
            Object fieldValue = null;
            if (type == Types.VARCHAR) {
                fieldValue = rs.getString(fieldName);
            } else if (type == Types.INTEGER) {
                fieldValue = rs.getInt(fieldName);
            } else if (type == Types.BIGINT) {
                fieldValue = rs.getLong(fieldName);
            } else if (type == Types.DATE) {
                fieldValue = DateFormatUtil.format(rs.getTimestamp(fieldName));
            } else if (type == Types.BLOB) {
                fieldValue = rs.getBytes(fieldName);
            } else if (type == Types.CLOB) {
                fieldValue = rs.getString(fieldName);
            } else if (type == Types.TIMESTAMP) {
                Timestamp tt = rs.getTimestamp(fieldName);
                if (tt != null) {
                    fieldValue = tt.getTime();
                }
            } else if (type == Types.SMALLINT) {
                fieldValue = rs.getShort(fieldName);
            } else if (type == Types.LONGVARCHAR) {
                String value = rs.getString(fieldName);
                if (value != null && !value.isEmpty()) {
                    try {
                        if (value.matches("\\{.*\\}")) {
                            fieldValue = JSONObject.parseObject(value, Map.class);
                        } else if (value.matches("\\[.*\\]")) {
                            fieldValue = JSONArray.parseArray(value, Map.class);
                        } else {
                            fieldValue = value;
                        }
                    } catch (Exception e) {
                        log.warn(e.getMessage(), e);
                        fieldValue = value;
                    }
                }
            } else {
                fieldValue = rs.getObject(fieldName);
            }
            m.put(fieldName, fieldValue);
        }
        return m;
    }



    /**
     * 结果集处理
     *
     * @param <T>
     */
    protected static class ResultSetExtractorImpl<T> implements ResultSetExtractor<List<T>> {
        private Class<T> clazz;

        public ResultSetExtractorImpl(Class<T> clazz) {
            this.clazz = clazz;
        }

        @SuppressWarnings({"unchecked"})
        public List<T> extractData(ResultSet rs) throws SQLException, DataAccessException {
            if (!rs.next()) {
                return null;
            }
            ResultSetMetaData meta = rs.getMetaData();
            int colCount = meta.getColumnCount();
            // columns[0]-数据集中的原始名, columns[1]-对应子类中的属性名
            List<String[]> columns = new ArrayList<>();
            for (int i = 1; i <= colCount; i++) {
                columns.add(new String[]{meta.getColumnName(i), meta.getColumnName(i)});
            }
            List<EntityField> fields = buildEntityModel(clazz, columns);
            if (fields == null) {
                return null;
            }
            List<T> list = new ArrayList<T>();
            do {
                list.add((T) convert2Entity(rs, clazz, fields));
            } while (rs.next());
            return list;
        }
    }

    /**
     * 生成实体对象模型
     *
     * @param clazz
     * @param columns
     * @return
     */
    protected static List<EntityField> buildEntityModel(Class<?> clazz, List<String[]> columns) {
        Class<?> fieldType;
        List<BeanField> fields = ReflectUtil.getBeanFields(clazz);
        Method method;
        String column, subObjName;
        cn.dolphin.core.jdbc.annotation.Field annotation = null;
        String[] columnName;
        List<EntityField> entityFields = new ArrayList<>();// 普通字段
        for (BeanField field : fields) {
            method = field.getSetterMethod();
            if (method == null) {
                continue;
            }
            annotation = field.getAnnotation(cn.dolphin.core.jdbc.annotation.Field.class);
            column = getColumnName(annotation, field.field);
            if (column == null) {
                continue;
            }

            List<String[]> subColumns = null;//子对象字段，String[]{数据库字段名,实体属性名}
            for (int i = 0; i < columns.size(); i++) {
                columnName = columns.get(i);
                if (column.equalsIgnoreCase(columnName[1]) || field.getName().equalsIgnoreCase(columnName[1])) {
                    fieldType = method.getParameterTypes()[0];
                    if (annotation != null && annotation.isJson()) {
                        if (fieldType.isAssignableFrom(List.class)) {
                            EntityField entityField = new EntityField(annotation, method, columnName[0], FieldType.JSON_ARRAY);
                            entityField.genericClazz = ReflectUtil.getGenericType(field.field);
                            entityFields.add(entityField);
                        } else {
                            entityFields.add(new EntityField(annotation, method, columnName[0], FieldType.JSON_OBJECT));
                        }
                    } else {
                        entityFields.add(new EntityField(annotation, method, columnName[0], FieldType.JAVA));
                    }
                    columns.remove(i);//移除已匹配过的字段，减少下次循环次数
                    break;
                }
                int ii = columnName[1].indexOf(".");
                if (ii > 0) {// 子对象字段
                    subObjName = columnName[1].substring(0, ii);
                    if (column.equalsIgnoreCase(subObjName) || field.getName().equalsIgnoreCase(subObjName)) {
                        if (subColumns == null) {
                            subColumns = new ArrayList<>();
                        }
                        subColumns.add(new String[]{columnName[0], columnName[1].substring(ii + 1)});
                        columns.remove(i);//移除已匹配过的字段，减少下次循环次数
                        i--;
                    }
                }
            }
            if (subColumns != null) {
                List<EntityField> lst = buildEntityModel(method.getParameterTypes()[0], subColumns);
                if (lst != null) {
                    entityFields.add(new EntityField(lst, method, FieldType.ENTITY));
                }
            }
        }

        if (entityFields.isEmpty()) {
            return null;
        }
        return entityFields;
    }


    protected static class EntityField {
        public cn.dolphin.core.jdbc.annotation.Field annotation;// annotation类
        public String columnName;
        public FieldType type; // 字段类型
        public Class<?> genericClazz;// json数组元素对象
        public List<EntityField> entityFields;
        public Method setter;

        public EntityField(cn.dolphin.core.jdbc.annotation.Field annotation, Method setter, String columnName, FieldType type) {
            this.annotation = annotation;
            this.setter = setter;
            this.columnName = columnName;
            this.type = type;
        }

        public EntityField(List<EntityField> entityFields, Method setter, FieldType type) {
            this.entityFields = entityFields;
            this.setter = setter;
            this.type = type;
        }
    }



}
