package cn.dolphin.core.jdbc;

import cn.dolphin.core.dialect.Dialect;
import cn.dolphin.core.dialect.util.DialectUtil;
import cn.dolphin.core.exception.DaoRuntimException;
import cn.dolphin.core.jdbc.enums.SqlMethod;
import cn.dolphin.core.jdbc.model.BatchModel;
import cn.dolphin.core.jdbc.model.SqlModel;
import cn.dolphin.core.jdbc.support.AbstractJdbcDao;
import cn.dolphin.core.jdbc.support.IRollBack;
import cn.dolphin.core.jdbc.util.FrameworkDataAccessUtil;
import cn.dolphin.core.jdbc.util.JdbcUtil;
import cn.dolphin.core.jdbc.util.QueryAndNamedParams;
import cn.dolphin.core.jdbc.util.QueryAndParams;
import cn.dolphin.core.page.PageBean;
import cn.dolphin.core.page.PageList;
import cn.dolphin.core.spring.SpringContextUtil;
import cn.dolphin.core.util.CamelCaseUtil;
import cn.dolphin.core.util.CollectionUtil;
import com.alibaba.fastjson.JSONObject;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.*;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;
import org.springframework.jdbc.core.namedparam.SqlParameterSourceUtils;
import org.springframework.jdbc.support.JdbcUtils;
import org.springframework.stereotype.Component;

import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.*;

/**
 * jdbc模板封装扩展
 * @param <T> 对象
 * @param <ID> ID类型
 */
@Component
@SuppressWarnings("all")
public abstract class JdbcDao<T, ID extends Serializable> extends AbstractJdbcDao<T,ID> {


    /**
     * 构造方法
     */
    @SuppressWarnings("unchecked")
    public JdbcDao() {
        try {
            Type type = getClass().getGenericSuperclass();
            if (type instanceof ParameterizedType) {
                entityClass = (Class<T>) ((ParameterizedType) type).getActualTypeArguments()[0];
            } else {
                entityClass = (Class<T>) type;
            }
            tableName = JdbcUtil.getTableName(entityClass);//表名
            primaryKey = JdbcUtil.getPk(entityClass);//主键

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 将一个对象插入数据库
     *
     * @param object t 需要插入的对象，这个对象是对数据库映射的一个javabean
     * @return 返回主键ID
     */
    public long saveJdbc(T object) throws Exception{
        SqlModel sqlModel;
        try {
            sqlModel = JdbcUtil.buildSaveModel(object, DialectUtil.getDialect(getJdbcTemplate().getDbType()).getMark());
        } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException | NoSuchMethodException
                | SecurityException e) {
            throw new DaoRuntimException(e);
        }
        return getJdbcTemplate().save(sqlModel.sql, sqlModel.values, sqlModel.types, sqlModel.pk);
    }

    /**
     * 批量新增对象
     *
     * @param list List<T> 需要插入的对象，这个对象是对数据库映射的一个javabean
     * @return 返回影响行数
     */
    public int batchSaveJdbc(List<T> list)throws Exception{
        BatchModel model;
        try {
            model = JdbcUtil.buildBatchSaveModel(list, DialectUtil.getDialect(getJdbcTemplate().getDbType()).getMark());
        } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException | NoSuchMethodException
                | SecurityException e) {
            throw new DaoRuntimException(e);
        }
        return getJdbcTemplate().batchUpdateNew(model.sql, model.valueList, model.types);
    }

    /**
     * 保存或更新对象,返回对象ID
     *
     * @param object
     * @return 返回对象ID
     */
    public boolean saveOrUpdateJdbc(T object) throws Exception{
        try {
            return getJdbcTemplate().saveOrUpdate(object);
        } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException
                | SecurityException e) {
            throw new DaoRuntimException(e);
        }
    }

    /**
     * 根据1个java对象更新数据库对应的表，表名跟类名一样
     *
     * @param object Object 需要更新的对象，这个对象是对数据库映射的一个javabean
     */
    public int updateJdbc(T object)throws Exception{
        return updateJdbc(object, null, false);
    }

    /**
     * update语句
     *
     * @param wsql DDL语句
     */
    public int updateJdbc(String wsql) throws Exception{
        return updateJdbc(wsql, null);
    }

    /**
     * 更新记录，返回影响行数
     * @param sql 以 :=propertyName 作为占位符
     * @param mapORpojo
     * @return 成功：i=1，失败：i=0
     */
    public int updateJdbc(String sql, Object mapORpojo) throws Exception{
        int i = getJdbcTemplate().getNamedParamJdbcTemplate().update(sql, transToSource(mapORpojo));
        return i;
    }


    /**
     * 增加、删除、修改表时，调用此方法
     *
     * @param wsql SQL语句
     * @param args 参数
     */
    public int updateJdbc(String wsql, Object[] args)throws Exception {
        if (!wsql.matches("^ *(?i)(INSERT|MERGE|DELETE|UPDATE) .*")) {
            wsql = "UPDATE " + tableName + " SET " + wsql;
        }
        return getJdbcTemplate().update(wsql, args, null);
    }


    /**
     * 根据1个java对象更新数据库对应的表
     *
     * @param object  对象
     * @param columns 需要更新的字段名
     * @return
     */
    public int updateJdbc(T object, String[] columns)throws Exception{
        return updateJdbc(object, columns, false);
    }

    /**
     * 根据1个java对象更新数据库对应的表
     *
     * @param object  对象
     * @param columns 限定属性
     * @param ignore  true-忽略，false-指定
     * @return
     */
    public int updateJdbc(T object, String[] columns, boolean ignore)throws Exception {
        SqlModel sqlModel;
        try {
            sqlModel = JdbcUtil.buildUpdateModel(object, columns, ignore, DialectUtil.getDialect(getJdbcTemplate().getDbType()).getMark());
        } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException | NoSuchMethodException
                | SecurityException e) {
            throw new DaoRuntimException(e);
        }
        return getJdbcTemplate().update(sqlModel.sql, sqlModel.values, sqlModel.types);
    }

    /**
     * 批量更新对象
     *
     * @param list 对象列表
     * @return 返回影响行数
     */
    public int batchUpdateJdbc(List<T> list) throws Exception {
        return batchUpdateJdbc(list, null);
    }

    /**
     * 批量更新对象
     *
     * @param list    对象列表
     * @param columns 需要更新的列名
     * @return
     */
    public int batchUpdateJdbc(List<T> list, String[] columns)throws Exception {
        BatchModel model;
        try {
            model = JdbcUtil.buildBatchUpdateModel(list, columns, false, DialectUtil.getDialect(getJdbcTemplate().getDbType()).getMark());
        } catch (SecurityException | IllegalAccessException | IllegalArgumentException
                | InvocationTargetException | NoSuchMethodException e) {
            throw new DaoRuntimException(e);
        }
        return getJdbcTemplate().batchUpdateNew(model.sql, model.valueList, model.types);
    }

    /**
     * 根据主键删除
     * @param id 主键
     * @return 影响行数
     */
    public <ID extends Serializable> int deleteByIdJdbc(ID id) throws Exception {
        String deleteByIdSql =  String.format(SqlMethod.DELETE_BY_ID.getSql(), tableName, primaryKey,
                CamelCaseUtil.toCamelCase(primaryKey));
        Map<String, Object> paramMap = Collections.singletonMap(CamelCaseUtil.toCamelCase(primaryKey), id);
        return getJdbcTemplate().getNamedParamJdbcTemplate().update(deleteByIdSql, paramMap);
    }

    /**
     * 删除表中的记录
     *
     * @param wsql sql或删除条件 如：( id>? and time>?)
     * @return 返回删除的条数
     */

    public int deleteJdbc(String wsql) throws Exception{
        return deleteJdbc(wsql, null);
    }

    /**
     * 删除记录（命名参数)
     * @param sql 以 :=propertyName 作为占位符
     * @param mapORpojo
     * @return 成功：i=1，失败：i=0
     */
    public int deleteJdbc(String sql, Object mapORpojo) {
        int i = getJdbcTemplate().getNamedParamJdbcTemplate().update(sql, transToSource(mapORpojo));
        return i;
    }
    /**
     * * 删除表中的记录
     *
     * @param wsql      sql 或 删除条件 如：( id>? and time>?)
     * @param whereArgs 条件里的参数 用来替换"?" 第1个参数，代表第1个问号；第2个参数，代表第2个问号；依此类推......
     * @return 返回删除的条数
     */
    public int deleteJdbc(String wsql, Object[] whereArgs) throws Exception{
        if (wsql == null || (wsql = wsql.trim()).isEmpty()) {
            wsql = "DELETE FROM " + tableName;
        } else if (!wsql.matches("^ *(?i)DELETE .*")) {
            wsql = "DELETE FROM " + tableName + " WHERE " + wsql;
        }
        String logstr = null;
        if (log.isDebugEnabled()) {
            logstr = "delete: sql=" + wsql + "; parameters=" + getJdbcTemplate().arrayToString(whereArgs);
            log.debug(logstr);
        }
        int rows = 0;
        if (whereArgs != null && whereArgs.length > 0) {
            rows = getJdbcTemplate().update(wsql, whereArgs);
        } else {
            rows = getJdbcTemplate().update(wsql);
        }
        if (log.isInfoEnabled()) {
            if (logstr == null) {
                logstr = "delete: sql=" + wsql + "; parameters=" + getJdbcTemplate().arrayToString(whereArgs);
            }
            log.info(logstr + "; rows=" + rows);
        }
        return rows;
    }


    /**
     * 批量批量SQL进行增、删和改
     *
     * @param sql
     *            sql语句或者sql语句的模板
     * @param parameter
     *            可以把对象作为参数，但是sql中的参数名称必须和对象属性相同,如果属性是对象，则可以使用属性名.属性名来传入参数值
     */
    public <T> void batchExecuteJdbc(String sql, Collection<T> parameter) throws Exception {
        if (parameter == null) {
            throw new RuntimeException("parameter can not be null.");
        }
        getJdbcTemplate().getNamedParamJdbcTemplate().batchUpdate(sql, SqlParameterSourceUtils.createBatch(parameter.toArray()));
    }


    /**
     * 根据ID查询对象
     * @param id 主键
     * @return 返回当前对象
     */
    public <ID extends Serializable> T queryForByIdJdbc(ID id) throws Exception {
        //String sql = "SELECT * FROM " + entityClass.getSimpleName().toLowerCase() + " WHERE "+ sqlHelper.getPrimaryKey(entityClass)+"=?";
        String sql = "SELECT * FROM " + tableName + " WHERE "+ primaryKey+"=?";
        RowMapper<T> rowMapper = BeanPropertyRowMapper.newInstance(entityClass);
        List<T> list = getJdbcTemplate().query(sql, rowMapper, id);
        return CollectionUtil.isEmpty(list) ? null : list.get(0);
    }

    /**
     * 通用的select语句，适用于数组作为参数的类型
     * @param sql sql语句
     * @param objects 查询条件
     * @param bean 实体
     * @return 返回当前对象
     */
    public Object queryForObjectJdbc(String sql, Object[] objects, Class bean) throws Exception {
        List<Object> list=null;
        try {
            list = getJdbcTemplate().query(sql, objects, new BeanPropertyRowMapper(bean));
        }catch (Exception e){
            list=new ArrayList<Object>();
        }
        return list.get(0);
    }


    /**
     * 通用的select语句
     * @param sql sql语句
     * @param rowMapper 反射mapper
     * @return
     */
    public <T> T queryForObjectJdbc(String sql, RowMapper<T> rowMapper) throws Exception {
        List<T> results = getJdbcTemplate().query(sql, rowMapper);
        return FrameworkDataAccessUtil.requiredSingleResult(results);
    }


    /**
     * 通用的select语句
     * @param sql sql语句
     * @param mapORpojo sql语句的参数，要么是一个Map，要么是一个POJO
     * @param beanClazz sql查询结果的载体，是一个POJO
     * @return
     */
    public <T> T queryForObjectJdbc(String sql, Object mapORpojo, Class<T> beanClazz){
        if(isBaseType(beanClazz)){
            throw new RuntimeException("beanClazz是一个基础类型！");
        }

        SqlParameterSource source = transToSource(mapORpojo);

        try {
            T t = getJdbcTemplate().getNamedParamJdbcTemplate().queryForObject(
                    sql,
                    source,
                    new BeanPropertyRowMapper<T>(beanClazz)
            );
            return t;
        } catch (EmptyResultDataAccessException e) {
            return null;
        } finally {

        }
    }

    /**
     * 通用的select语句
     * @param sql sql语句
     * @param mapORpojo sql语句的参数，要么是一个Map，要么是一个POJO
     * @param baseClazz sql查询结果的载体，是一个POJO
     * @return
     */
    public <T> T queryForSingleValueJdbc(String sql, Object mapORpojo, Class<T> baseClazz){
        if(!isBaseType(baseClazz)){
            throw new RuntimeException("baseClazz不是一个基础类型！");
        }

        SqlParameterSource source = transToSource(mapORpojo);

        try {
            T t = getJdbcTemplate().getNamedParamJdbcTemplate().queryForObject(
                    sql,
                    source,
                    baseClazz);
            return t;
        } catch (EmptyResultDataAccessException e) {
            return null;
        } finally {

        }
    }

    /**
     * 根据条件查询JSON对象
     * @param sql
     * @param args
     * @return
     */
    public JSONObject queryForJSONJdbc(String sql, Object[] args) {
        try {
            return this.getJdbcTemplate().queryForObject(sql, args, new RowMapper<JSONObject>() {
                @Override
                public JSONObject mapRow(ResultSet rs, int columnIndex)
                        throws SQLException {
                    JSONObject jsonObject = new JSONObject();
                    ResultSetMetaData rsmd = rs.getMetaData();
                    int columnCount = rsmd.getColumnCount();
                    for (int index = 1; index <= columnCount; ++index) {
                        String column = JdbcUtils.lookupColumnName(rsmd, index);
                        jsonObject.put(column, rs.getString(index));
                    }
                    return jsonObject;
                }
            } );
        } catch (EmptyResultDataAccessException e) {
            return null;
        }
    }

    /**
     * 通用的select语句
     * @param sql sql语句
     * @param args 条件
     * @param argTypes
     * @param rowMapper 反射mapper
     * @return
     */
    public <T> T queryForObjectJdbc(String sql, Object[] args, int[] argTypes, RowMapper<T> rowMapper) throws Exception {
        List<T> results = getJdbcTemplate().query(sql, args, argTypes, new RowMapperResultSetExtractor<T>(rowMapper, 1));
        return FrameworkDataAccessUtil.requiredSingleResult(results);
    }

    /**
     * 通用的select语句
     * @param sql sql语句
     * @param args 条件
     * @param rowMapper 反射mapper
     * @return
     */
    public <T> T queryForObjectJdbc(String sql, Object[] args, RowMapper<T> rowMapper) throws Exception {
        List<T> results = getJdbcTemplate().query(sql, args, new RowMapperResultSetExtractor<T>(rowMapper, 1));
        return FrameworkDataAccessUtil.requiredSingleResult(results);
    }

    /**
     * 通用的select语句
     * @param sql sql语句
     * @param rowMapper 反射mapper
     * @param args
     * @return
     */
    public <T> T queryForObjectJdbc(String sql, RowMapper<T> rowMapper, Object... args) throws Exception {
        List<T> results = getJdbcTemplate().query(sql, args, new RowMapperResultSetExtractor<T>(rowMapper, 1));
        return FrameworkDataAccessUtil.requiredSingleResult(results);
    }


    /**
     * 通用的select语句
     * @param sql sql语句
     * @param mapORpojo sql语句的参数，要么是一个Map，要么是一个POJO
     * @return
     */
    public Map<String, Object> queryForMapJdbc(String sql, Object mapORpojo) {
        SqlParameterSource source = transToSource(mapORpojo);
        try {
            Map<String,Object> map = getJdbcTemplate().getNamedParamJdbcTemplate().queryForMap(sql, source);
            return map;
        } catch (EmptyResultDataAccessException e) {
            return null;
        } finally {

        }
    }

    /**
     * 输入查询语句和查询条件查询列表
     * @param sql 查询语句  示例：select * from sys_role where name=:name
     * @param parameter Map map=new HashMap(); 		map.put("name", "li");
     * @param rowMap  需要实现RowMapper接口,有两种实现方式，一种是匿名类，另外一种是实现RowMapper接口。
     * 使用方法如下：<br/>
     * <pre>
     * List<Role> list =dao.queryForList(sql, null, new RowMapper<Role>() {
     *		@Override
     *		public Role mapRow(ResultSet rs, int arg1) throws SQLException {
     *			Role role=new Role();
     *			role.setName(rs.getString("name"));
     *			return role;
     *		}
     *	});
     * </pre>
     * @return
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    public  List queryForListJdbc(String sql,Map parameter,RowMapper rowMap) {
        return  getJdbcTemplate().getNamedParamJdbcTemplate().query(sql, parameter, rowMap);
    }

    /**
     * 输入查询语句和查询条件查询列表
     * @param sql
     * @param parameter
     * @return
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    public  List queryForListJdbc(String sql,Map parameter) {
        return  getJdbcTemplate().getNamedParamJdbcTemplate().queryForList(sql, parameter);
    }

    /**
     * 根据条件查询JSON对象
     * @param sql
     * @param args
     * @return
     */
    public List<JSONObject> queryForListJSONJdbc(String sql, Object[] args) {
        try {
            return this.getJdbcTemplate().query(sql, args,new RowMapper<JSONObject>() {

                @Override
                public JSONObject mapRow(ResultSet rs, int columnIndex)
                        throws SQLException {
                    JSONObject jsonObject = new JSONObject();
                    ResultSetMetaData rsmd = rs.getMetaData();
                    int columnCount = rsmd.getColumnCount();
                    for (int index = 1; index <= columnCount; ++index) {
                        String column = JdbcUtils.lookupColumnName(rsmd, index);
                        jsonObject.put(column, rs.getString(index));

                    }
                    return jsonObject;
                }} );
        } catch (EmptyResultDataAccessException e) {
            return null;
        }
    }

    /**
     * 输入查询语句和查询条件查询列表
     * @param sql	以 :=propertyName 作为占位符
     * @param mapORpojo	sql语句的参数，要么是一个Map，要么是一个POJO
     * @param beanClazz	sql查询结果的载体，是一个POJO
     * @return 不为null，但是size可能为0
     */
    public <T> List<T> queryForBeanListJdbc(String sql, Object mapORpojo, Class<T> beanClazz){
        if(isBaseType(beanClazz)){
            throw new RuntimeException("beanClazz是一个基础类型！");
        }

        SqlParameterSource source = transToSource(mapORpojo);

        List<T> list = getJdbcTemplate().getNamedParamJdbcTemplate().query(
                sql,
                source,
                new BeanPropertyRowMapper(beanClazz));
        return list;
    }

    /**
     * 输入查询语句和查询条件查询列表
     * @param sql
     * @param mapORpojo
     * @param baseClazz
     * @param <T>
     * @return
     */
    public <T> List<T> queryForSingleValueListJdbc(String sql, Object mapORpojo, Class<T> baseClazz){
        if(!isBaseType(baseClazz)){
            throw new RuntimeException("baseClazz不是一个基础类型！");
        }

        SqlParameterSource source = transToSource(mapORpojo);

        List<T> list = getJdbcTemplate().getNamedParamJdbcTemplate().query(
                sql,
                source,
                new SingleColumnRowMapper(baseClazz));
        return list;
    }


    /**
     * 输入查询语句和查询条件查询列表
     * @param sql
     * @param mapORpojo
     * @return
     */
    public List<Map<String, Object>> queryForMapListJdbc(String sql, Object mapORpojo) {
        SqlParameterSource source = transToSource(mapORpojo);
        List<Map<String, Object>> list = getJdbcTemplate().getNamedParamJdbcTemplate().queryForList(sql, source);
        return list;
    }

    /**
     * 获取查询记录结果条数
     * @param sql
     * @param mapORpojo
     * @return
     */
    public Integer queryForIntJdbc(String sql, Object mapORpojo) {
        SqlParameterSource source = transToSource(mapORpojo);
        Integer i = getJdbcTemplate().getNamedParamJdbcTemplate().queryForObject(sql, source, Integer.class);
        return i;
    }




    /**
     * 获取分页数据，List 为PageList。
     * 使用方法：<br>
     * <pre>
     * PageModel pageModel= dao.getPage(1, 2, "select * from sys_role" , null,new RowMapper<Role>() {
     *		@Override
     *		public Role mapRow(ResultSet rs, int arg1) throws SQLException {
     *			Role role=new Role();
     *			role.setName(rs.getString("name"));
     *			return role;
     *		}
     *	});
     * @param pageBean
     *            分页对象
     * @param sql
     *            分页sql语句 如果需要参数，请输入如下sql语句，sql="select * from user where name=:name";
     * @param rowMap
     *            对象映射接口。
     * @return
     */
    public PageList queryPageJdbc(PageBean pageBean, String sql, RowMapper rowMap) {

        int pageSize = pageBean.getPageSize();
        int offset = pageBean.getFirst();

        Map map = new HashMap();

        Dialect dialect = null;
        try {
            dialect = DialectUtil.getDialect(getJdbcTemplate().getDbType());
        } catch (Exception e) {
            return null;
        }
        String pageSql = dialect.getLimitString(sql, offset, pageSize);
        String totalSql = dialect.getCountSql(sql,primaryKey);
        List list = getJdbcTemplate().getNamedParamJdbcTemplate().query(pageSql, map, rowMap);
        int total = getJdbcTemplate().getNamedParamJdbcTemplate().queryForObject(totalSql, map,Integer.class);

        pageBean.setTotalCount(total);
        PageList pageList = new PageList();
        pageList.setPageBean(pageBean);
        pageList.addAll(list);

        return pageList;
    }

    public <T> T queryPageJdbc(PageBean pageBean, String sql, ResultSetExtractor<T> rse, Map<String, Object> params) {

        T result = null;
        if (pageBean != null) {
            int pageSize = pageBean.getPageSize();
            int offset = pageBean.getFirst();
            Dialect dialect = null;
            try {
                dialect = DialectUtil.getDialect(getJdbcTemplate().getDbType());
            } catch (Exception e) {
                return null;
            }
            String pageSql = dialect.getLimitString(sql, offset, pageSize);
            String totalSql = dialect.getCountSql(sql,primaryKey);
            result = getJdbcTemplate().getNamedParamJdbcTemplate().query(pageSql, params, rse);
            int total = getJdbcTemplate().getNamedParamJdbcTemplate().queryForObject(totalSql, params,Integer.class);
            pageBean.setTotalCount(total);
        } else {
            result = getJdbcTemplate().getNamedParamJdbcTemplate().query(sql, params, rse);
        }
        return result;
    }

    public PageList queryPageJdbc(int currentPage, int pageSize, String sql, Map paraMap) {

        int offset = (currentPage - 1) * pageSize;

        Dialect dialect = null;
        try {
            dialect = DialectUtil.getDialect(getJdbcTemplate().getDbType());
        } catch (Exception e) {
            return null;
        }

        String pageSql = dialect.getLimitString(sql, offset, pageSize);
        String totalSql = dialect.getCountSql(sql,primaryKey);
        List list = getJdbcTemplate().getNamedParamJdbcTemplate().queryForList(pageSql, paraMap);
        int total = getJdbcTemplate().getNamedParamJdbcTemplate().queryForObject(totalSql, paraMap,Integer.class);

        PageBean pageBean = new PageBean(currentPage, pageSize);

        pageBean.setTotalCount(total);

        PageList pageList = new PageList();

        pageList.addAll(list);

        pageList.setPageBean(pageBean);

        return pageList;
    }


    public List queryPageJdbc(PageBean pageBean,String sql, Map<?, ?> paraMap) {
        int currentPage = pageBean.getCurrentPage();
        int pageSize = pageBean.getPageSize();
        return queryPageJdbc( currentPage, pageSize, sql, paraMap);
    }

    /**执行sql，可使用sql
     * @param sql
     * @param args
     * @return
     */
    public int executeJdbc(String sql, Object... args) throws Exception {
        return getJdbcTemplate().update(sql, args);
    }


    /**
     * 执行一个带命名参数的sql语句
     * @param sql
     * @param params
     * @return
     * @throws Exception
     */
    public int executeJdbc(String sql , Map<String, Object> params)throws Exception{
        QueryAndParams qap = QueryAndParams.createFromQueryAndNamedParams(new QueryAndNamedParams(sql, params));
        return executeJdbc(qap.getQuery(),qap.getParams());
    }

    /**
     * 执行SQL,并指定是否可以回滚。
     * @param sql	：sql
     * @param rollback	:是否启动事务回滚
     * @return
     */
    public Boolean executeJdbc(String sql, boolean rollback) {
        //final JdbcTemplatePlus jdbcTemplate = (JdbcTemplatePlus) SpringContextUtil.getBean("jdbcTemplatePlus");
        RollbackJdbcTemplate rollbackJdbcTemplate = (RollbackJdbcTemplate) SpringContextUtil.getBean("rollbackJdbcTemplate");
        if (!rollback) {
            getJdbcTemplate().execute(sql);
            return true;
        }

        Map<String, Object> param = new HashMap<String, Object>();
        Boolean b = (Boolean) rollbackJdbcTemplate.executeRollBack(new IRollBack() {

            @Override
            public Object execute(String script, Map<String, Object> map) {
                try {
                    getJdbcTemplate().execute(script);
                } catch (Exception e) {
                    e.printStackTrace();
                    return false;
                }

                return true;
            }
        }, sql, param);

        return b;
    }


    /**
     * 批量批量SQL进行增、删和改
     *
     * @param sql
     *            SQL语句或者动态模板
     * @param parameter
     *            SQL语句内对应的参数
     */
    @SuppressWarnings("unchecked")
    public void executeBatchJdbc(String sql, Collection<Map<String, Object>> parameter) {
        if (parameter == null) {
            throw new RuntimeException("parameter can not be null.");
        }
        Map<String, Object>[] paramArray = new Map[parameter.size()];
        int index = 0;
        for (Map<String, Object> paramMap : parameter) {
            paramArray[index++] = paramMap;
        }
        getJdbcTemplate().getNamedParamJdbcTemplate().batchUpdate(sql, paramArray);
    }



    /**
     * 是否存在
     *
     * @param wsql
     * @return
     */
    public boolean existsJdbc(String wsql) throws Exception{
        return existsJdbc(wsql, null);
    }

    /**
     * 是否存在
     *
     * @param wsql
     * @param values
     * @return
     */
    public boolean existsJdbc(String wsql, Object[] values)throws Exception{
        if (wsql == null || (wsql = wsql.trim()).isEmpty()) {
            wsql = "SELECT COUNT(1) FROM " + tableName + "";
        } else if (!wsql.matches("^ *(?i)SELECT .*")) {
            wsql = "SELECT COUNT(1) FROM " + tableName + " WHERE " + wsql;
        }
        boolean flag = false;
        try {
            Integer r = getJdbcTemplate().queryForObject(wsql, values, Integer.class);
            if (r != null && r.intValue() > 0) {
                flag = true;
            }
        } catch (EmptyResultDataAccessException e) {
        }
        if (log.isInfoEnabled()) {
            log.info("exists: sql=" + wsql + "; parameters=" + getJdbcTemplate().arrayToString(values) + "; result=" + flag);
        }
        return flag;
    }



}
