package cn.dolphin.core.image;

import cn.dolphin.core.file.FileUtil;
import net.coobird.thumbnailator.Thumbnails;
import org.apache.commons.io.IOUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.imageio.ImageIO;
import javax.swing.*;
import java.awt.*;
import java.awt.color.ColorSpace;
import java.awt.geom.AffineTransform;
import java.awt.image.*;
import java.io.*;
import java.net.URL;
import java.util.Map;

/**
 * 图像处理
 * 对图片进行压缩、水印、伸缩变换、透明处理、格式转换操作
 */
@SuppressWarnings("all")
public class ImageUtil {

    public static final float DEFAULT_QUALITY = 0.2125f ;


    /**
     * 根据二进制流判断图片扩展类型
     * @param data
     * @return
     */
    public static String getType(byte[] data) {
        String type = "JPG";
        if (data[1] == 'P' && data[2] == 'N' && data[3] == 'G') {
            type = "PNG";
            return type;
        }
        if (data[0] == 'G' && data[1] == 'I' && data[2] == 'F') {
            type = "GIF";
            return type;
        }
        if (data[6] == 'J' && data[7] == 'F' && data[8] == 'I'
                && data[9] == 'F') {
            type = "JPG";
            return type;
        }
        return type;
    }

    /**
     *
     * 添加图片水印操作(物理存盘,使用默认格式)
     *
     * @param imgPath
     *            待处理图片
     * @param markPath
     *            水印图片
     * @param x
     *            水印位于图片左上角的 x 坐标值
     * @param y
     *            水印位于图片左上角的 y 坐标值
     * @param alpha
     *            水印透明度 0.1f ~ 1.0f
     * @param destPath
     *                 文件存放路径
     * @throws Exception
     *
     */
    public static void addWaterMark(String imgPath, String markPath, int x, int y, float alpha,String destPath) throws Exception{
        try {
            BufferedImage bufferedImage = addWaterMark(imgPath, markPath, x, y, alpha);
            ImageIO.write(bufferedImage, imageFormat(imgPath), new File(destPath));
        } catch (Exception e) {
            throw new RuntimeException("添加图片水印异常");
        }
    }


    /**
     *
     * 添加图片水印操作(物理存盘,自定义格式)
     *
     * @param imgPath
     *            待处理图片
     * @param markPath
     *            水印图片
     * @param x
     *            水印位于图片左上角的 x 坐标值
     * @param y
     *            水印位于图片左上角的 y 坐标值
     * @param alpha
     *            水印透明度 0.1f ~ 1.0f
     * @param format
     *                 添加水印后存储的格式
     * @param destPath
     *                 文件存放路径
     * @throws Exception
     *
     */
    public static void addWaterMark(String imgPath, String markPath, int x, int y, float alpha,String format,String destPath) throws Exception{
        try {
            BufferedImage bufferedImage = addWaterMark(imgPath, markPath, x, y, alpha);
            ImageIO.write(bufferedImage,format , new File(destPath));
        } catch (Exception e) {
            throw new RuntimeException("添加图片水印异常");
        }
    }


    /**
     *
     * 添加图片水印操作,返回BufferedImage对象
     *
     * @param imgPath
     *            待处理图片
     * @param markPath
     *            水印图片
     * @param x
     *            水印位于图片左上角的 x 坐标值
     * @param y
     *            水印位于图片左上角的 y 坐标值
     * @param alpha
     *            水印透明度 0.1f ~ 1.0f
     * @return
     *                 处理后的图片对象
     * @throws Exception
     *
     */
    public static BufferedImage addWaterMark(String imgPath, String markPath, int x, int y, float alpha) throws Exception{
        BufferedImage targetImage = null;
        try {
            // 加载待处理图片文件
            Image img = ImageIO.read(new File(imgPath));

            //创建目标图象文件
            targetImage = new BufferedImage(img.getWidth(null), img.getHeight(null), BufferedImage.TYPE_INT_RGB);
            Graphics2D g = targetImage.createGraphics();
            g.drawImage(img, 0, 0, null);

            // 加载水印图片文件
            Image markImg = ImageIO.read(new File(markPath));
            g.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_ATOP, alpha));
            g.drawImage(markImg, x, y, null);
            g.dispose();
        } catch (Exception e) {
            throw new RuntimeException("添加图片水印操作异常");
        }
        return targetImage;

    }

    /**
     *
     * 添加文字水印操作(物理存盘,使用默认格式)
     *
     * @param imgPath
     *            待处理图片
     * @param text
     *            水印文字
     * @param font
     *            水印字体信息    不写默认值为宋体
     * @param color
     *            水印字体颜色
     * @param x
     *            水印位于图片左上角的 x 坐标值
     * @param y
     *            水印位于图片左上角的 y 坐标值
     * @param alpha
     *            水印透明度 0.1f ~ 1.0f
     * @param destPath
     *                 文件存放路径
     * @throws Exception
     */
    public static void addTextMark(String imgPath, String text, Font font, Color color, float x, float y, float alpha,String destPath) throws Exception{
        try {
            BufferedImage bufferedImage = addTextMark(imgPath, text, font, color, x, y, alpha);
            ImageIO.write(bufferedImage, imageFormat(imgPath), new File(destPath));
        } catch (Exception e) {
            throw new RuntimeException("图片添加文字水印异常");
        }
    }

    /**
     *
     * 添加文字水印操作(物理存盘,自定义格式)
     *
     * @param imgPath
     *            待处理图片
     * @param text
     *            水印文字
     * @param font
     *            水印字体信息    不写默认值为宋体
     * @param color
     *            水印字体颜色
     * @param x
     *            水印位于图片左上角的 x 坐标值
     * @param y
     *            水印位于图片左上角的 y 坐标值
     * @param alpha
     *            水印透明度 0.1f ~ 1.0f
     * @param format
     *                 添加水印后存储的格式
     * @param destPath
     *                 文件存放路径
     * @throws Exception
     */
    public static void addTextMark(String imgPath, String text, Font font, Color color, float x, float y, float alpha,String format,String destPath) throws Exception{
        try {
            BufferedImage bufferedImage = addTextMark(imgPath, text, font, color, x, y, alpha);
            ImageIO.write(bufferedImage, format, new File(destPath));
        } catch (Exception e) {
            throw new RuntimeException("图片添加文字水印异常");
        }
    }

    /**
     *
     * 添加文字水印操作,返回BufferedImage对象
     *
     * @param imgPath
     *            待处理图片
     * @param text
     *            水印文字
     * @param font
     *            水印字体信息    不写默认值为宋体
     * @param color
     *            水印字体颜色
     * @param x
     *            水印位于图片左上角的 x 坐标值
     * @param y
     *            水印位于图片左上角的 y 坐标值
     * @param alpha
     *            水印透明度 0.1f ~ 1.0f
     * @return
     *                 处理后的图片对象
     * @throws Exception
     */

    public static BufferedImage addTextMark(String imgPath, String text, Font font, Color color, float x, float y, float alpha) throws Exception{
        BufferedImage targetImage = null;
        try {
            Font Dfont = (font == null) ? new Font("宋体", 20, 13) : font;
            Image img = ImageIO.read(new File(imgPath));
            //创建目标图像文件
            targetImage = new BufferedImage(img.getWidth(null), img.getHeight(null), BufferedImage.TYPE_INT_RGB);
            Graphics2D g = targetImage.createGraphics();
            g.drawImage(img, 0, 0, null);
            g.setColor(color);
            g.setFont(Dfont);
            g.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_ATOP, alpha));
            g.drawString(text, x, y);
            g.dispose();
        } catch (Exception e) {
            throw new RuntimeException("添加文字水印操作异常");
        }
        return targetImage;
    }

    /**
     *
     *
     *
     * 压缩图片操作(文件物理存盘,使用默认格式)
     *
     * @param imgPath
     *                 待处理图片
     * @param quality
     *                 图片质量(0-1之間的float值)
     * @param width
     *                 输出图片的宽度    输入负数参数表示用原来图片宽
     * @param height
     *                 输出图片的高度    输入负数参数表示用原来图片高
     * @param autoSize
     *                 是否等比缩放 true表示进行等比缩放 false表示不进行等比缩放
     * @param destPath
     *                 文件存放路径
     *
     * @throws Exception
     */
    public static void compressImage(String imgPath,float quality,int width, int height, boolean autoSize,String destPath)throws Exception{
        try {
            BufferedImage bufferedImage = compressImage(imgPath, quality, width, height, autoSize);
            ImageIO.write(bufferedImage, imageFormat(imgPath), new File(destPath));
        } catch (Exception e) {
            throw new RuntimeException("图片压缩异常");
        }

    }

    /**
     *
     * 压缩图片操作(文件物理存盘,可自定义格式)
     *
     * @param imgPath
     *                 待处理图片
     * @param quality
     *                 图片质量(0-1之間的float值)
     * @param width
     *                 输出图片的宽度    输入负数参数表示用原来图片宽
     * @param height
     *                 输出图片的高度    输入负数参数表示用原来图片高
     * @param autoSize
     *                 是否等比缩放 true表示进行等比缩放 false表示不进行等比缩放
     * @param format
     *                 压缩后存储的格式
     * @param destPath
     *                 文件存放路径
     *
     * @throws Exception
     */
    public static void compressImage(String imgPath,float quality,int width, int height, boolean autoSize,String format,String destPath)throws Exception{
        try {
            BufferedImage bufferedImage = compressImage(imgPath, quality, width, height, autoSize);
            ImageIO.write(bufferedImage, format, new File(destPath));
        } catch (Exception e) {
            throw new RuntimeException("图片压缩异常");
        }
    }

    /**
     *
     * 压缩图片操作,返回BufferedImage对象
     *
     * @param imgPath
     *                 待处理图片
     * @param quality
     *                 图片质量(0-1之間的float值)
     * @param width
     *                 输出图片的宽度    输入负数参数表示用原来图片宽
     * @param height
     *                 输出图片的高度    输入负数参数表示用原来图片高
     * @param autoSize
     *                 是否等比缩放 true表示进行等比缩放 false表示不进行等比缩放
     * @return
     *                 处理后的图片对象
     * @throws Exception
     */
    public static BufferedImage compressImage(String imgPath,float quality,int width, int height, boolean autoSize)throws Exception{
        BufferedImage targetImage = null;
        if(quality<0F||quality>1F){
            quality = DEFAULT_QUALITY;
        }
        try {
            Image img = ImageIO.read(new File(imgPath));
            //如果用户输入的图片参数合法则按用户定义的复制,负值参数表示执行默认值
            int newwidth =( width > 0 ) ? width : img.getWidth(null);
            //如果用户输入的图片参数合法则按用户定义的复制,负值参数表示执行默认值
            int newheight = ( height > 0 )? height: img.getHeight(null);
            //如果是自适应大小则进行比例缩放
            if(autoSize){
                // 为等比缩放计算输出的图片宽度及高度
                double Widthrate = ((double) img.getWidth(null)) / (double) width + 0.1;
                double heightrate = ((double) img.getHeight(null))/ (double) height + 0.1;
                double rate = Widthrate > heightrate ? Widthrate : heightrate;
                newwidth = (int) (((double) img.getWidth(null)) / rate);
                newheight = (int) (((double) img.getHeight(null)) / rate);
            }
            //创建目标图像文件
            targetImage = new BufferedImage(newwidth,newheight,BufferedImage.TYPE_INT_RGB);
            Graphics2D g = targetImage.createGraphics();
            g.drawImage(img, 0, 0, newwidth, newheight, null);
            //如果添加水印或者文字则继续下面操作,不添加的话直接返回目标文件----------------------
            g.dispose();

        } catch (Exception e) {
            throw new RuntimeException("图片压缩操作异常");
        }
        return targetImage;
    }

    /**
     * 图片黑白化操作(文件物理存盘,使用默认格式)
     *
     * @param imgPath
     *                 处理的图片对象
     * @param destPath
     *                 目标文件地址
     * @throws Exception
     *
     */
    public static void imageGray(String imgPath, String destPath)throws Exception{
        imageGray(imgPath, imageFormat(imgPath), destPath);
    }


    /**
     * 图片黑白化操作(文件物理存盘,可自定义格式)
     *
     * @param imgPath
     *                 处理的图片对象
     * @param format
     *                 图片格式
     * @param destPath
     *                 目标文件地址
     * @throws Exception
     *
     */
    public static void imageGray(String imgPath,String format, String destPath)throws Exception{
        try {
            BufferedImage bufferedImage = ImageIO.read(new File(imgPath));
            ColorSpace cs = ColorSpace.getInstance(ColorSpace.CS_GRAY);
            ColorConvertOp op = new ColorConvertOp(cs, null);
            bufferedImage = op.filter(bufferedImage, null);
            ImageIO.write(bufferedImage, format , new File(destPath));
        } catch (Exception e) {
            throw new RuntimeException("图片灰白化异常");
        }
    }

    /**
     * 图片透明化操作(文件物理存盘,使用默认格式)
     *
     * @param imgPath
     *                 图片路径
     * @param destPath
     *                 图片存放路径
     * @throws Exception
     */
    public static void imageLucency(String imgPath,String destPath)throws Exception{
        try {
            BufferedImage bufferedImage = imageLucency(imgPath);
            ImageIO.write(bufferedImage, imageFormat(imgPath), new File(destPath));
        } catch (Exception e) {
            throw new RuntimeException("图片透明化异常");
        }
    }

    /**
     * 图片透明化操作(文件物理存盘,可自定义格式)
     *
     * @param imgPath
     *                 图片路径
     * @param format
     *                 图片格式
     * @param destPath
     *                 图片存放路径
     * @throws Exception
     */
    public static void imageLucency(String imgPath,String format,String destPath)throws Exception{
        try {
            BufferedImage bufferedImage = imageLucency(imgPath);
            ImageIO.write(bufferedImage, format, new File(destPath));
        } catch (Exception e) {
            throw new RuntimeException("图片透明化异常");
        }
    }

    /**
     * 图片透明化操作返回BufferedImage对象
     *
     * @param imgPath
     *                 图片路径
     * @return
     *                 透明化后的图片对象
     * @throws Exception
     */
    public static BufferedImage imageLucency(String imgPath)throws Exception{
        BufferedImage targetImage = null;
        try {
            //读取图片
            BufferedImage img = ImageIO.read(new FileInputStream(imgPath));
            //透明度
            int alpha = 0;
            //执行透明化
            executeRGB(img, alpha);
            targetImage = new BufferedImage(img.getWidth(null), img.getHeight(null), BufferedImage.TYPE_INT_RGB);
            Graphics2D g = targetImage.createGraphics();
            g.drawImage(img, 0, 0, null);
            g.dispose();
        } catch (Exception e) {
            throw new RuntimeException("图片透明化执行异常");
        }
        return targetImage;
    }

    /**
     * 执行透明化的核心算法
     *
     * @param img
     *                 图片对象
     * @param alpha
     *                 透明度
     * @throws Exception
     */
    public static  void executeRGB(BufferedImage img, int alpha) throws Exception{
        int rgb = 0;//RGB值
        //x表示BufferedImage的x坐标，y表示BufferedImage的y坐标
        for(int x=img.getMinX();x<img.getWidth();x++){
            for(int y=img.getMinY();y<img.getHeight();y++){
                //获取点位的RGB值进行比较重新设定
                rgb = img.getRGB(x, y);
                int R =(rgb & 0xff0000 ) >> 16 ;
                int G= (rgb & 0xff00 ) >> 8 ;
                int B= (rgb & 0xff );
                if(((255-R)<30) && ((255-G)<30) && ((255-B)<30)){
                    rgb = ((alpha + 1) << 24) | (rgb & 0x00ffffff);
                    img.setRGB(x, y, rgb);
                }
            }
        }
    }

    /**
     * 图片格式转化操作(文件物理存盘)
     *
     * @param imgPath
     *                     原始图片存放地址
     * @param format
     *                     待转换的格式 jpeg,gif,png,bmp等
     * @param destPath
     *                     目标文件地址
     * @throws Exception
     */
    public static void formatConvert(String imgPath, String format, String destPath)throws Exception{
        try {
            BufferedImage bufferedImage = ImageIO.read(new File(imgPath));
            ImageIO.write(bufferedImage, format, new File(destPath));
        } catch (IOException e) {
            throw new RuntimeException("文件格式转换出错");
        }
    }

    /**
     * 图片格式转化操作返回BufferedImage对象
     *
     * @param bufferedImag
     *                     BufferedImage图片转换对象
     * @param format
     *                     待转换的格式 jpeg,gif,png,bmp等
     * @param destPath
     *                     目标文件地址
     * @throws Exception
     */
    public static void formatConvert(BufferedImage bufferedImag, String format, String destPath)throws Exception{
        try {
            ImageIO.write(bufferedImag, format, new File(destPath));
        } catch (IOException e) {
            throw new RuntimeException("文件格式转换出错");
        }
    }

    /**
     * 获取图片文件的真实格式信息
     *
     * @param imgPath
     *                     图片原文件存放地址
     * @return
     *                     图片格式
     * @throws Exception
     */
    public static String imageFormat(String imgPath)throws Exception{
        String[] filess = imgPath.split("\\\\");
        String[] formats = filess[filess.length - 1].split("\\.");
        return formats[formats.length - 1];
    }

    /**
     * 缩放图像（按比例缩放）
     *
     * @param sourceImg 源图片
     * @param output 缩放后输出的图片
     * @param scale 缩放倍数 如2就是2倍
     * @param flag
     * 缩放选择:true 放大; false 缩小;
     * @throws IOException
     */
    public final static void scale(File sourceImg, File output,
                                   int scale, boolean flag) throws IOException {
        BufferedImage src = ImageIO.read(sourceImg); // 读入文件
        int width = src.getWidth(); // 得到源图宽
        int height = src.getHeight(); // 得到源图长
        if (flag) {// 放大
            width = width * scale;
            height = height * scale;
        } else {// 缩小
            width = width / scale;
            height = height / scale;
        }
        Image image = src.getScaledInstance(width, height,
                Image.SCALE_DEFAULT);
        BufferedImage tag = new BufferedImage(width, height,
                BufferedImage.TYPE_INT_RGB);
        Graphics g = tag.getGraphics();
        g.drawImage(image, 0, 0, null); // 绘制缩小后的图
        g.dispose();
        ImageIO.write(tag, "JPEG", output);// 输出到文件流
    }

    /**
     * 缩放图像（按高度和宽度缩放）
     * 不按比例缩放，图片会失真
     * @param sourceImg
     *            源图像文件
     * @param output
     *            缩放后的图像
     * @param height
     *            缩放后的高度
     * @param width
     *            缩放后的宽度
     * @param flag
     *            比例不对时是否需要补白：true为补白; false为不补白;
     */
    public final static void scale(File sourceImg, File output,
                                   int height, int width, boolean flag) {
        try {
            double ratio = 0.0; // 缩放比例
            BufferedImage bi = ImageIO.read(sourceImg);
            Image itemp = bi.getScaledInstance(width, height, BufferedImage.SCALE_SMOOTH);
            // 计算比例
            if ((bi.getHeight() > height) || (bi.getWidth() > width)) {
                if (bi.getHeight() > bi.getWidth()) {
                    ratio = (new Integer(height)).doubleValue()
                            / bi.getHeight();
                } else {
                    ratio = (new Integer(width)).doubleValue() / bi.getWidth();
                }
                AffineTransformOp op = new AffineTransformOp(
                        AffineTransform.getScaleInstance(ratio, ratio), null);
                itemp = op.filter(bi, null);
            }
            if (flag) {// 补白
                BufferedImage image = new BufferedImage(width, height,
                        BufferedImage.TYPE_INT_RGB);
                Graphics2D g = image.createGraphics();
                g.setColor(Color.white);
                g.fillRect(0, 0, width, height);
                if (width == itemp.getWidth(null))
                    g.drawImage(itemp, 0, (height - itemp.getHeight(null)) / 2,
                            itemp.getWidth(null), itemp.getHeight(null),
                            Color.white, null);
                else
                    g.drawImage(itemp, (width - itemp.getWidth(null)) / 2, 0,
                            itemp.getWidth(null), itemp.getHeight(null),
                            Color.white, null);
                g.dispose();
                itemp = image;
            }
            ImageIO.write((BufferedImage) itemp, "JPEG",output);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 缩放图像（按高度和宽度缩放）
     * 不按比例缩放，图片会失真
     * @param sourceImg
     *            源图像文件
     * @param output
     *            缩放后的图像
     * @param height
     *            缩放后的高度
     * @param width
     *            缩放后的宽度
     * @param flag
     *            比例不对时是否需要补白：true为补白; false为不补白;
     */
    public final static void scale(InputStream sourceImg, OutputStream output,
                                   int height, int width, boolean flag) {
        try {
            double ratio = 0.0; // 缩放比例
            BufferedImage bi = ImageIO.read(sourceImg);
            Image itemp = bi.getScaledInstance(width, height, BufferedImage.SCALE_SMOOTH);
            // 计算比例
            if ((bi.getHeight() > height) || (bi.getWidth() > width)) {
                if (bi.getHeight() > bi.getWidth()) {
                    ratio = (new Integer(height)).doubleValue()
                            / bi.getHeight();
                } else {
                    ratio = (new Integer(width)).doubleValue() / bi.getWidth();
                }
                AffineTransformOp op = new AffineTransformOp(
                        AffineTransform.getScaleInstance(ratio, ratio), null);
                itemp = op.filter(bi, null);
            }
            if (flag) {// 补白
                BufferedImage image = new BufferedImage(width, height,
                        BufferedImage.TYPE_INT_RGB);
                Graphics2D g = image.createGraphics();
                g.setColor(Color.white);
                g.fillRect(0, 0, width, height);
                if (width == itemp.getWidth(null))
                    g.drawImage(itemp, 0, (height - itemp.getHeight(null)) / 2,
                            itemp.getWidth(null), itemp.getHeight(null),
                            Color.white, null);
                else
                    g.drawImage(itemp, (width - itemp.getWidth(null)) / 2, 0,
                            itemp.getWidth(null), itemp.getHeight(null),
                            Color.white, null);
                g.dispose();
                itemp = image;
            }
            ImageIO.write((BufferedImage) itemp, "JPEG",output);
            itemp.flush();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 根据指定宽度压缩图片
     * @param sourceImg
     * @param output
     * @param width
     */
    public final static void scaleByWidth(File sourceImg, File output,
                                          int width) {
        try {
            BufferedImage bi = ImageIO.read(sourceImg);
            int imageWidth = bi.getWidth();
            float sacle = (width * 1.0f) / (imageWidth * 1.0f);
            int height = (int) (bi.getHeight() * sacle);
            Image itemp = bi.getScaledInstance(width, height,
                    BufferedImage.SCALE_SMOOTH);
            // 宽度缩放比
            double widthRatio = (new Integer(width)).doubleValue()
                    / bi.getWidth();
            // 高度缩放比
            double heightRatio = (new Integer(height)).doubleValue()
                    / bi.getHeight();
            AffineTransformOp op = new AffineTransformOp(
                    AffineTransform.getScaleInstance(widthRatio, heightRatio),
                    null);
            itemp = op.filter(bi, null);
            ImageIO.write((BufferedImage) itemp, "JPEG", output);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }


    /**
     * 缩放图像（按高度和宽度缩放）
     *
     * @param img
     *            源图像文件地址
     * @param height
     *            缩放后的高度
     * @param width
     *            缩放后的宽度
     * @param flag
     *            比例不对时是否需要补白：true为补白; false为不补白;
     */
    public final static byte[] scale(byte[] img, int height, int width,
                                     boolean flag) {
        try {
            double ratio = 0.0; // 缩放比例
            InputStream inputStream = new ByteArrayInputStream(img);
            BufferedImage bi = ImageIO.read(inputStream);
            Image itemp = bi.getScaledInstance(width, height, BufferedImage.SCALE_SMOOTH);
            // 计算比例
            if ((bi.getHeight() > height) || (bi.getWidth() > width)) {
                if (bi.getHeight() > bi.getWidth()) {
                    ratio = (new Integer(height)).doubleValue()
                            / bi.getHeight();
                } else {
                    ratio = (new Integer(width)).doubleValue() / bi.getWidth();
                }
                AffineTransformOp op = new AffineTransformOp(
                        AffineTransform.getScaleInstance(ratio, ratio), null);
                itemp = op.filter(bi, null);
            }
            if (flag) {// 补白
                BufferedImage image = new BufferedImage(width, height,
                        BufferedImage.TYPE_INT_RGB);
                Graphics2D g = image.createGraphics();
                g.setColor(Color.white);
                g.fillRect(0, 0, width, height);
                if (width == itemp.getWidth(null))
                    g.drawImage(itemp, 0, (height - itemp.getHeight(null)) / 2,
                            itemp.getWidth(null), itemp.getHeight(null),
                            Color.white, null);
                else
                    g.drawImage(itemp, (width - itemp.getWidth(null)) / 2, 0,
                            itemp.getWidth(null), itemp.getHeight(null),
                            Color.white, null);
                g.dispose();
                itemp = image;
            }
            ByteArrayOutputStream outputstream = new ByteArrayOutputStream();
            ImageIO.write((BufferedImage) itemp, "JPEG", outputstream);
            return outputstream.toByteArray();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 图像切割(按指定起点坐标和宽高切割)
     *
     * @param srcImageFile
     *            源图像地址
     * @param result
     *            切片后的图像地址
     * @param x
     *            目标切片起点坐标X
     * @param y
     *            目标切片起点坐标Y
     * @param width
     *            目标切片宽度
     * @param height
     *            目标切片高度
     */
    public final static void cut(File sourceImg, File output, int x,
                                 int y, int width, int height) {
        try {
            // 读取源图像
            BufferedImage bi = ImageIO.read(sourceImg);
            int srcWidth = bi.getHeight(); // 源图宽度
            int srcHeight = bi.getWidth(); // 源图高度
            if (srcWidth > 0 && srcHeight > 0) {
                Image image = bi.getScaledInstance(srcWidth, srcHeight,
                        Image.SCALE_DEFAULT);
                // 四个参数分别为图像起点坐标和宽高
                // 即: CropImageFilter(int x,int y,int width,int height)
                ImageFilter cropFilter = new CropImageFilter(x, y, width,
                        height);
                Image img = Toolkit.getDefaultToolkit().createImage(
                        new FilteredImageSource(image.getSource(), cropFilter));
                BufferedImage tag = new BufferedImage(width, height,
                        BufferedImage.TYPE_INT_RGB);
                Graphics g = tag.getGraphics();
                g.drawImage(img, 0, 0, width, height, null); // 绘制切割后的图
                g.dispose();
                // 输出为文件
                ImageIO.write(tag, "JPEG", output);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 图像切割（指定切片的行数和列数）
     *
     * @param srcImageFile
     *            源图像地址
     * @param descDir
     *            切片目标文件夹
     * @param rows
     *            目标切片行数。默认2，必须是范围 [1, 20] 之内
     * @param cols
     *            目标切片列数。默认2，必须是范围 [1, 20] 之内
     */
    public final static void cut2(String srcImageFile, String descDir,
                                  int rows, int cols) {
        try {
            if (rows <= 0 || rows > 20)
                rows = 2; // 切片行数
            if (cols <= 0 || cols > 20)
                cols = 2; // 切片列数
            // 读取源图像
            BufferedImage bi = ImageIO.read(new File(srcImageFile));
            int srcWidth = bi.getHeight(); // 源图宽度
            int srcHeight = bi.getWidth(); // 源图高度
            if (srcWidth > 0 && srcHeight > 0) {
                Image img;
                ImageFilter cropFilter;
                Image image = bi.getScaledInstance(srcWidth, srcHeight,
                        Image.SCALE_DEFAULT);
                int destWidth = srcWidth; // 每张切片的宽度
                int destHeight = srcHeight; // 每张切片的高度
                // 计算切片的宽度和高度
                if (srcWidth % cols == 0) {
                    destWidth = srcWidth / cols;
                } else {
                    destWidth = (int) Math.floor(srcWidth / cols) + 1;
                }
                if (srcHeight % rows == 0) {
                    destHeight = srcHeight / rows;
                } else {
                    destHeight = (int) Math.floor(srcWidth / rows) + 1;
                }
                // 循环建立切片
                // 改进的想法:是否可用多线程加快切割速度
                for (int i = 0; i < rows; i++) {
                    for (int j = 0; j < cols; j++) {
                        // 四个参数分别为图像起点坐标和宽高
                        // 即: CropImageFilter(int x,int y,int width,int height)
                        cropFilter = new CropImageFilter(j * destWidth, i
                                * destHeight, destWidth, destHeight);
                        img = Toolkit.getDefaultToolkit().createImage(
                                new FilteredImageSource(image.getSource(),
                                        cropFilter));
                        BufferedImage tag = new BufferedImage(destWidth,
                                destHeight, BufferedImage.TYPE_INT_RGB);
                        Graphics g = tag.getGraphics();
                        g.drawImage(img, 0, 0, null); // 绘制缩小后的图
                        g.dispose();
                        // 输出为文件
                        ImageIO.write(tag, "JPEG", new File(descDir + "_r" + i
                                + "_c" + j + ".jpg"));
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 图像切割（指定切片的宽度和高度）
     *
     * @param srcImageFile
     *            源图像地址
     * @param descDir
     *            切片目标文件夹
     * @param destWidth
     *            目标切片宽度。默认200
     * @param destHeight
     *            目标切片高度。默认150
     */
    public final static void cut3(String srcImageFile, String descDir,
                                  int destWidth, int destHeight) {
        try {
            if (destWidth <= 0)
                destWidth = 200; // 切片宽度
            if (destHeight <= 0)
                destHeight = 150; // 切片高度
            // 读取源图像
            BufferedImage bi = ImageIO.read(new File(srcImageFile));
            int srcWidth = bi.getHeight(); // 源图宽度
            int srcHeight = bi.getWidth(); // 源图高度
            if (srcWidth > destWidth && srcHeight > destHeight) {
                Image img;
                ImageFilter cropFilter;
                Image image = bi.getScaledInstance(srcWidth, srcHeight,
                        Image.SCALE_DEFAULT);
                int cols = 0; // 切片横向数量
                int rows = 0; // 切片纵向数量
                // 计算切片的横向和纵向数量
                if (srcWidth % destWidth == 0) {
                    cols = srcWidth / destWidth;
                } else {
                    cols = (int) Math.floor(srcWidth / destWidth) + 1;
                }
                if (srcHeight % destHeight == 0) {
                    rows = srcHeight / destHeight;
                } else {
                    rows = (int) Math.floor(srcHeight / destHeight) + 1;
                }
                // 循环建立切片
                // 改进的想法:是否可用多线程加快切割速度
                for (int i = 0; i < rows; i++) {
                    for (int j = 0; j < cols; j++) {
                        // 四个参数分别为图像起点坐标和宽高
                        // 即: CropImageFilter(int x,int y,int width,int height)
                        cropFilter = new CropImageFilter(j * destWidth, i
                                * destHeight, destWidth, destHeight);
                        img = Toolkit.getDefaultToolkit().createImage(
                                new FilteredImageSource(image.getSource(),
                                        cropFilter));
                        BufferedImage tag = new BufferedImage(destWidth,
                                destHeight, BufferedImage.TYPE_INT_RGB);
                        Graphics g = tag.getGraphics();
                        g.drawImage(img, 0, 0, null); // 绘制缩小后的图
                        g.dispose();
                        // 输出为文件
                        ImageIO.write(tag, "JPEG", new File(descDir + "_r" + i
                                + "_c" + j + ".jpg"));
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 图像类型转换：GIF->JPG、GIF->PNG、PNG->JPG、PNG->GIF(X)、BMP->PNG
     *
     * @param srcImageFile
     *            源图像地址
     * @param formatName
     *            包含格式非正式名称的 String：如JPG、JPEG、GIF等
     * @param destImageFile
     *            目标图像地址
     */
    public final static void convert(String srcImageFile, String formatName,
                                     String destImageFile) {
        try {
            File f = new File(srcImageFile);
            f.canRead();
            f.canWrite();
            BufferedImage src = ImageIO.read(f);
            ImageIO.write(src, formatName, new File(destImageFile));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    /**
     * 彩色转为黑白
     *
     * @param srcImageFile
     *            源图像地址
     * @param destImageFile
     *            目标图像地址
     */
    public final static void gray(String srcImageFile, String destImageFile) {
        try {
            BufferedImage src = ImageIO.read(new File(srcImageFile));
            ColorSpace cs = ColorSpace.getInstance(ColorSpace.CS_GRAY);
            ColorConvertOp op = new ColorConvertOp(cs, null);
            src = op.filter(src, null);
            ImageIO.write(src, "JPEG", new File(destImageFile));
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 给图片添加文字水印
     *
     * @param pressText
     *            水印文字
     * @param srcImageFile
     *            源图像地址
     * @param destImageFile
     *            目标图像地址
     * @param fontName
     *            水印的字体名称
     * @param fontStyle
     *            水印的字体样式
     * @param color
     *            水印的字体颜色
     * @param fontSize
     *            水印的字体大小
     * @param x
     *            修正值
     * @param y
     *            修正值
     * @param alpha
     *            透明度：alpha 必须是范围 [0.0, 1.0] 之内（包含边界值）的一个浮点数字
     */
    public final static void pressText(String pressText, String srcImageFile,
                                       String destImageFile, String fontName, int fontStyle, Color color,
                                       int fontSize, int x, int y, float alpha) {
        try {
            File img = new File(srcImageFile);
            Image src = ImageIO.read(img);
            int width = src.getWidth(null);
            int height = src.getHeight(null);
            BufferedImage image = new BufferedImage(width, height,
                    BufferedImage.TYPE_INT_RGB);
            Graphics2D g = image.createGraphics();
            g.drawImage(src, 0, 0, width, height, null);
            g.setColor(color);
            g.setFont(new Font(fontName, fontStyle, fontSize));
            g.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_ATOP,
                    alpha));
            // 在指定坐标绘制水印文字
            g.drawString(pressText, (width - (getLength(pressText) * fontSize))
                    / 2 + x, (height - fontSize) / 2 + y);
            g.dispose();
            ImageIO.write((BufferedImage) image, "JPEG",
                    new File(destImageFile));// 输出到文件流
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 给图片添加文字水印
     *
     * @param pressText
     *            水印文字
     * @param srcImageFile
     *            源图像地址
     * @param destImageFile
     *            目标图像地址
     * @param fontName
     *            字体名称
     * @param fontStyle
     *            字体样式
     * @param color
     *            字体颜色
     * @param fontSize
     *            字体大小
     * @param x
     *            修正值
     * @param y
     *            修正值
     * @param alpha
     *            透明度：alpha 必须是范围 [0.0, 1.0] 之内（包含边界值）的一个浮点数字
     */
    public final static void pressText2(String pressText, String srcImageFile,
                                        String destImageFile, String fontName, int fontStyle, Color color,
                                        int fontSize, int x, int y, float alpha) {
        try {
            File img = new File(srcImageFile);
            Image src = ImageIO.read(img);
            int width = src.getWidth(null);
            int height = src.getHeight(null);
            BufferedImage image = new BufferedImage(width, height,
                    BufferedImage.TYPE_INT_RGB);
            Graphics2D g = image.createGraphics();
            g.drawImage(src, 0, 0, width, height, null);
            g.setColor(color);
            g.setFont(new Font(fontName, fontStyle, fontSize));
            g.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_ATOP,
                    alpha));
            // 在指定坐标绘制水印文字
            g.drawString(pressText, (width - (getLength(pressText) * fontSize))
                    / 2 + x, (height - fontSize) / 2 + y);
            g.dispose();
            ImageIO.write((BufferedImage) image, "JPEG",
                    new File(destImageFile));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 给图片添加图片水印
     *
     * @param pressImg
     *            水印图片
     * @param srcImageFile
     *            源图像地址
     * @param destImageFile
     *            目标图像地址
     * @param x
     *            修正值。 默认在中间
     * @param y
     *            修正值。 默认在中间
     * @param alpha
     *            透明度：alpha 必须是范围 [0.0, 1.0] 之内（包含边界值）的一个浮点数字
     */
    public final static void pressImage(String pressImg, String srcImageFile,
                                        String destImageFile, int x, int y, float alpha) {
        try {
            File img = new File(srcImageFile);
            Image src = ImageIO.read(img);
            int wideth = src.getWidth(null);
            int height = src.getHeight(null);
            BufferedImage image = new BufferedImage(wideth, height,
                    BufferedImage.TYPE_INT_RGB);
            Graphics2D g = image.createGraphics();
            g.drawImage(src, 0, 0, wideth, height, null);
            // 水印文件
            Image src_biao = ImageIO.read(new File(pressImg));
            int wideth_biao = src_biao.getWidth(null);
            int height_biao = src_biao.getHeight(null);
            g.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_ATOP,
                    alpha));
            g.drawImage(src_biao, (wideth - wideth_biao) / 2,
                    (height - height_biao) / 2, wideth_biao, height_biao, null);
            // 水印文件结束
            g.dispose();
            ImageIO.write((BufferedImage) image, "JPEG",
                    new File(destImageFile));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 计算text的长度（一个中文算两个字符）
     * @param text
     * @return
     */
    private final static int getLength(String text) {
        int length = 0;
        for (int i = 0; i < text.length(); i++) {
            if (new String(text.charAt(i) + "").getBytes().length > 1) {
                length += 2;
            } else {
                length += 1;
            }
        }
        return length / 2;
    }

    /**
     * 转换图片大小，不变形
     *
     * @param img 图片文件
     * @param width 图片宽
     * @param height 图片高
     */
    public static final void changeImge(File img, int width, int height) {
        try {
            Thumbnails.of(img).size(width, height).keepAspectRatio(false).toFile(img);
        } catch (IOException e) {
            e.printStackTrace();
            throw new IllegalStateException("图片转换出错！", e);
        }
    }
    /**
     * 根据比例缩放图片
     *
     * @param orgImg 源图片路径
     * @param scale 比例
     * @param targetFile 缩放后的图片存放路径
     * @throws IOException
     */
    public static final void scale(BufferedImage orgImg, double scale, String targetFile) throws IOException {
        Thumbnails.of(orgImg).scale(scale).toFile(targetFile);
    }

    public static final void scale(String orgImgFile, double scale, String targetFile) throws IOException {
        Thumbnails.of(orgImgFile).scale(scale).toFile(targetFile);
    }

    /**
     * 图片格式转换
     *
     * @param orgImgFile
     * @param width
     * @param height
     * @param suffixName
     * @param targetFile
     * @throws IOException
     */
    public static final void format(String orgImgFile, int width, int height, String suffixName, String targetFile)
            throws IOException {
        Thumbnails.of(orgImgFile).size(width, height).outputFormat(suffixName).toFile(targetFile);
    }

    /**
     * 根据宽度同比缩放
     *
     * @param orgImg 源图片
     * @param orgImg 原始宽度
     * @param targetWidth 缩放后的宽度
     * @param targetFile 缩放后的图片存放路径
     * @throws IOException
     */
    public static final double scaleWidth(BufferedImage orgImg, int targetWidth, String targetFile) throws IOException {
        int orgWidth = orgImg.getWidth();
        // 计算宽度的缩放比例
        double scale = targetWidth * 1.00 / orgWidth;
        // 裁剪
        scale(orgImg, scale, targetFile);

        return scale;
    }

    public static final void scaleWidth(String orgImgFile, int targetWidth, String targetFile) throws IOException {
        BufferedImage bufferedImage = ImageIO.read(new File(orgImgFile));
        scaleWidth(bufferedImage, targetWidth, targetFile);
    }

    /**
     * 根据高度同比缩放
     *
     * @param orgImg //源图片
     * @param targetHeight //原始高度
     * @param targetHeight //缩放后的高度
     * @param targetFile //缩放后的图片存放地址
     * @throws IOException
     */
    public static final double scaleHeight(BufferedImage orgImg, int targetHeight, String targetFile) throws IOException {
        int orgHeight = orgImg.getHeight();
        double scale = targetHeight * 1.00 / orgHeight;
        scale(orgImg, scale, targetFile);
        return scale;
    }

    public static final void scaleHeight(String orgImgFile, int targetHeight, String targetFile) throws IOException {
        BufferedImage bufferedImage = ImageIO.read(new File(orgImgFile));
        // int height = bufferedImage.getHeight();
        scaleHeight(bufferedImage, targetHeight, targetFile);
    }

    // 原始比例缩放
    public static final void scaleWidth(File file, Integer width) throws IOException {
        String fileName = file.getName();
        String filePath = file.getAbsolutePath();
        String postFix = fileName.substring(fileName.lastIndexOf(".")).toLowerCase();
        // 缩放
        BufferedImage bufferedImg = ImageIO.read(file);
        String targetFile = filePath + "_s" + postFix;
        scaleWidth(bufferedImg, width, targetFile);
        String targetFile2 = filePath + "@" + width;
        new File(targetFile).renameTo(new File(targetFile2));
    }

    /**
     * 按比例缩放
     * @param img
     * @return
     */
    public static BufferedImage getScaleZoom(BufferedImage img, int targetSize) {
        int srcWidth = img.getWidth();
        int srcHeight = img.getHeight();
        int targetWidth = targetSize;
        int targetHeight = targetSize;
        if (targetSize>srcWidth || targetSize>srcHeight){
            targetWidth = srcWidth;
            targetHeight = srcHeight;
        }
        double rate1 = ((double)srcWidth)/targetWidth;
        double rate2 = ((double)srcHeight)/targetHeight;
        // 根据缩放比率大的进行缩放控制
        double rate = rate1 > rate2 ? rate1 : rate2;
        int newWidth = (int)(((double)srcWidth)/rate);
        int newHeight = (int)(((double)srcHeight)/rate);
        int imageType = (img.getTransparency() == Transparency.OPAQUE) ? BufferedImage.TYPE_INT_RGB : BufferedImage.TYPE_INT_ARGB;
        BufferedImage tag = new BufferedImage(newWidth, newHeight, imageType);
        //Image.SCALE_SMOOTH 的缩略算法 生成缩略图片的平滑度的优先级比速度高 生成的图片质量比较好 但速度慢
        Image scaledImage = img.getScaledInstance(newWidth, newHeight, Image.SCALE_REPLICATE);
        tag.getGraphics().drawImage(scaledImage, 0, 0, null);

        return tag;
    }

    /**
     * 按比例裁剪
     * @param img
     * @param targetWidth
     * @param targetHeight
     * @return
     */
    public static BufferedImage getScaleCutscale(BufferedImage img, int targetWidth, int targetHeight) {
        int srcWidth = img.getWidth();
        int srcHeight = img.getHeight();
        //当超过图片最大宽高时候等比缩小所需要的图片规格
        while(targetWidth>srcWidth || targetHeight>srcHeight){
            double rt = (targetWidth>srcWidth)? ((double)targetWidth)/(srcWidth): ((double)targetHeight)/(srcHeight);
            targetWidth = (int)(((double)targetWidth)/rt);
            targetHeight = (int)(((double)targetHeight)/rt);
        }

        double rate1 = ((double)srcWidth)/(targetWidth);
        double rate2 = ((double)srcHeight)/(targetHeight);
        // 根据缩放比率大的进行缩放控制
        double rate = rate1 < rate2 ? rate1 : rate2;
        int newWidth = (int)(((double)srcWidth)/rate);
        int newHeight = (int)(((double)srcHeight)/rate);

        int x1 = newWidth/2 - targetWidth/2;
        int x2 = newWidth/2 + targetWidth/2;
        int y1 = newHeight/2 - targetHeight/2;
        int y2 = newHeight/2 + targetHeight/2;
        int imageType = (img.getTransparency() == Transparency.OPAQUE) ? BufferedImage.TYPE_INT_RGB : BufferedImage.TYPE_INT_ARGB;
        BufferedImage tag = new BufferedImage(targetWidth, targetHeight, imageType);
        Image scaledImage = img.getScaledInstance(newWidth, newHeight, Image.SCALE_REPLICATE);
        tag.getGraphics().drawImage(scaledImage, 0, 0, targetWidth, targetHeight, x1, y1, x2, y2, null);

        return tag;
    }


    /**
     * 获取图片尺寸信息
     *
     * @param filePath
     * @return [width, height]
     */
    public static int[] getSizeInfo(String filePath){
        File file = new File(filePath);
        return getSizeInfo(file);
    }

    /**
     * 获取图片尺寸信息
     *
     * @param url
     * @return [width,height]
     */
    public static int[] getSizeInfo(URL url){
        InputStream input = null;
        try {
            input = url.openStream();
            return getSizeInfo(input);
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            IOUtils.closeQuietly(input);
        }
    }

    /**
     * 获取图片尺寸信息
     *
     * @param file
     * @return [width,height]
     */
    public static int[] getSizeInfo(File file){
        if (!file.exists()) {
            throw new RuntimeException("file " + file.getAbsolutePath() + " doesn't exist.");
        }
        BufferedInputStream input = null;
        try {
            input = new BufferedInputStream(new FileInputStream(file));
            return getSizeInfo(input);
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            IOUtils.closeQuietly(input);
        }
    }

    /**
     * 获取图片尺寸
     *
     * @param input
     * @return [width,height]
     */
    public static int[] getSizeInfo(InputStream input) {
        try {
            BufferedImage img = ImageIO.read(input);
            int w = img.getWidth(null);
            int h = img.getHeight(null);
            return new int[] { w, h };
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }


//    public static void toThumbnail(String uploadPath,String fileName){
//        String uploadThumbPath =uploadPath+"/thumb/";
//        File file = new File(uploadThumbPath);
//        if (!file.isFile()) {
//            file.mkdirs();
//        }
//        System.out.println(uploadPath+fileName);
//        System.out.println(uploadThumbPath+fileName);
//        try {
//            createThumbnail(uploadPath+"/"+fileName, 390, 225, 100, uploadThumbPath+fileName);
//        } catch (FileNotFoundException e) {
//            // TODO Auto-generated catch block
//            e.printStackTrace();
//        } catch (InterruptedException e) {
//            // TODO Auto-generated catch block
//            e.printStackTrace();
//        } catch (IOException e) {
//            // TODO Auto-generated catch block
//            e.printStackTrace();
//        }
//    }
//
//
//    /**
//     * 创建缩略图
//     * @param filename
//     * @param thumbWidth
//     * @param thumbHeight
//     * @param quality
//     * @param outFilename
//     * @throws InterruptedException
//     * @throws FileNotFoundException
//     * @throws IOException
//     */
//    private static void createThumbnail(String filename, int thumbWidth,int thumbHeight, int quality, String outFilename)
//            throws InterruptedException,FileNotFoundException, IOException {
//
//        // 加载图片文件
//        Image image = Toolkit.getDefaultToolkit().getImage(filename);
//        MediaTracker mediaTracker = new MediaTracker(new Container());
//        mediaTracker.addImage(image, 0);
//        // 强制加载图像
//        mediaTracker.waitForID(0);
//        // 测试图片是否加载成功:
//        System.out.println(mediaTracker.isErrorAny());
//
//        // 通过参数判断缩略图大小
//        double thumbRatio = (double) thumbWidth / (double) thumbHeight;
//        int imageWidth = image.getWidth(null);
//        int imageHeight = image.getHeight(null);
//        double imageRatio = (double) imageWidth / (double) imageHeight;
//        // 缩小
//        if (thumbRatio < imageRatio) {
//            thumbHeight = (int) (thumbWidth / imageRatio);
//        } else {// 放大
//            thumbWidth = (int) (thumbHeight * imageRatio);
//        }
//
//        // 创建缩略图对象 并设置quality
//        BufferedImage thumbImage = new BufferedImage(thumbWidth, thumbHeight,BufferedImage.TYPE_INT_RGB);
//
//        Graphics2D graphics2D = thumbImage.createGraphics();
//
//        graphics2D.setRenderingHint(RenderingHints.KEY_INTERPOLATION,RenderingHints.VALUE_INTERPOLATION_BILINEAR);
//
//        graphics2D.drawImage(image, 0, 0,thumbWidth, thumbHeight, null);
//        // 输出缩略图
//        BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(outFilename));
//        JPEGImageEncoder encoder = JPEGCodec.createJPEGEncoder(out);
//        JPEGEncodeParam param = encoder.getDefaultJPEGEncodeParam(thumbImage);
//
//        quality = Math.max(0, Math.min(quality, 100));
//        param.setQuality((float) quality / 100.0f, false);
//        encoder.setJPEGEncodeParam(param);
//        encoder.encode(thumbImage);
//        out.close();
//    }


    /**
     * 头像图片
     * @param filePath
     * @param fileName
     * @param fileType
     * @param data
     * @param width
     * @param height
     * @return
     */
    public static boolean thumbnailHead(String filePath,String fileName,String fileType,byte[] data, int width, int height){
        // 判断目录是否存在
        try {

            File dir = new File(filePath);
            if(!dir.exists()) {
                dir.mkdir();
            }
            StringBuilder builder = new StringBuilder();
            builder.append(filePath).append("/").append(fileName).append(".").append(fileType);

            byte[] bytes = scale(data,height > 70 ? height : height * 2,width > 70 ? width : width * 2,true);
            InputStream is = new ByteArrayInputStream(data);
            Thumbnails.of(is).size(width,height).keepAspectRatio(false).toFile(builder.toString());
            return true;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
//            throw new RuntimeException(e);
        }
    }


    /**
     * 上传文件
     * @param file  文件
     * @param path  文件路径
     * @param fileName  文件名
     * @return
     */
    public static boolean upload(MultipartFile file, String path, String fileName, Map<String, Object> map){

        //文件路径
        String realPath = path + File.separator + fileName;
        System.out.println("realPath = " + realPath);

        File dest = new File(realPath);

        //判断文件目录是否存在
        if( !dest.getParentFile().exists()){
            dest.getParentFile().mkdir();   //不存在就创建
        }

        try{
            //保存文件
            file.transferTo(dest);
            ImageIcon image = new ImageIcon(dest.getAbsolutePath());
            map.put("height", image.getIconHeight());
            map.put("width", image.getIconWidth());
            String thumbnailImg = path + File.separator + "thumbnail" + File.separator + fileName;
            File thumbNail = new File(thumbnailImg);
            if(!thumbNail.getParentFile().exists()){
                thumbNail.getParentFile().mkdir();
            }
            Thumbnails.of(realPath)
                    .size(200,200)
                    .toFile(thumbnailImg);
            return true;
        }catch (IllegalStateException  e){
            //e.printStackTrace();
            return false;
        }catch (IOException e){
            //e.printStackTrace();
            return false;
        }
    }

    
    public static void main(String[] args)throws Exception {

        File file = new File("/Users/cloud/Desktop/a.png");
        FileUtil.Tobyte(file);
        thumbnailHead("/Users/cloud/Desktop","c","png",FileUtil.Tobyte(file),390,225);

//        createThumbnail("/Users/cloud/Desktop/a.png", 390, 225, 100, "/Users/cloud/Desktop/b.png");
//        String names[] = ImageIO.getReaderFormatNames();
//
//        Image img = ImageIO.read(new File("/Users/cloud/Desktop/20190904111746534.jpg"));
        System.out.println(1);
    }



}
