package cn.dolphin.core.html;

import cn.dolphin.core.util.CollectionUtil;
import cn.dolphin.core.util.StrUtil;
import org.jsoup.Connection;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


/**
 * Jsoup工具：用于解析.html
 * 结构：Html -> Document -> Element（Elements == ArrayList <Element>）
 * 通过Element可以获取和操作：相关Element、html内容、text内容、class名称、tag名称、id名称、attr名值
 *
 */
@SuppressWarnings("all")
public final class JsoupUtil {

    private static final Logger logger = LoggerFactory.getLogger(JsoupUtil.class);
    /**
     * 读取Html字符串，获得Document
     */
    public static final  Document getDocumentByHtmlStr(String html) {
        return Jsoup.parse(html);
    }

    /**
     * 读取Html文本，获得Document
     */
    public static final  Document getDocumentByHtmlFile(File html, String charset) throws IOException {
        return Jsoup.parse(html, charset);
    }

    /**
     * 读取Html文本，获得URL为绝对路径的Document
     * 注：Html通常使用相对路径的URL，可以传入网址的根路径("http://www.deomoncat.cn/")则自动解析为绝对URL
     */
    public static final  Document getDocumentByHtmlFile(File html, String charset, String baseUrl) throws IOException{
        return Jsoup.parse(html, charset, baseUrl);
    }

    /**
     * 以GET方式请求URL，获得Document：需要设置请求消息头(从浏览器请求中复制获取)。
     * 1、代理服务器：Host: 代理的网址
     * 2、防止403禁止访问资源：Referer: 来源的网址
     * 3、模仿浏览器：User-Agent: Mozilla/5.0 (Windows NT 6.1; WOW64; rv:44.0) Gecko/20100101 Firefox/44.0
     * 4、支持的数据：Accept:星/星
     * 5、支持的语言：Accept-Language:zh-CN,zh;q=0.8,en-US;q=0.5,en;q=0.3
     * 6、支持压缩：Accept-Encoding:gzip,deflate
     * 7、支持长连接：Connection: keep-alive
     * 8、设置Cookie：Cookie: vjuids=1959250ff.153734600b1.0.d6248074945348; vjlast=1457927881.1457927881.30; f_city=%E5%8C%97%E4%BA%AC%7C101010100%7C; Hm_lvt_080dabacb001ad3dc8b9b9049b36d43b=1457927882,1457966987,1457967842,1457967848; lzstat_uv=38717466241319606294|3496259; Hm_lpvt_080dabacb001ad3dc8b9b9049b36d43b=1457967852; lzstat_ss=1963542000_3_1457996652_3496259; BIGipServerd1src_pool=3625155696.20480.0000
     */
    public static final  Document getDocumentByGetUrlAndHeaders(String url, Map<String,String> headers) throws IOException{
        Connection connect = Jsoup.connect(url);
        if (CollectionUtil.isNotEmpty(headers)){
            Set<Entry<String, String>> entrySet = headers.entrySet();
            for(Entry<String,String> entry : entrySet){
                connect.header(entry.getKey(), entry.getValue());
            }
        }
        return connect.get();
    }

    /**
     * 以GET方式请求URL，获得Document：需要传递cookies
     */
    public static final  Document getDocumentByGetUrl(String url, Map<String,String> cookies) throws IOException{
        Connection connect = Jsoup.connect(url);
        if (CollectionUtil.isNotEmpty(cookies)){
            connect.cookies(cookies);
        }
        return connect.get();
    }
    /**
     * 以GET方式请求URL，获得Document：需要传递cookies、headers
     */
    public static final  Document getDocumentByGetUrl(String url, Map<String,String> cookies, Map<String,String> headers) throws IOException{
        Connection connect = Jsoup.connect(url);
        if (CollectionUtil.isNotEmpty(cookies)){
            connect.cookies(cookies);
        }
        if (CollectionUtil.isNotEmpty(headers)){
            Set<Entry<String, String>> entrySet = headers.entrySet();
            for(Entry<String,String> entry : entrySet){
                connect.header(entry.getKey(), entry.getValue());
            }
        }
        return connect.get();
    }
    /**
     * 以GET方式请求URL，获得Document
     */
    public static final  Document getDocumentByGetUrl(String url) throws IOException{
        return Jsoup.connect(url).get();
    }

    /**
     * 以POST方式请求URL，获得Document
     * 参数：URL 、参数Map
     */
    public static final  Document getDocumentByPostUrl(String url,Map<String, String> params) throws IOException{
        Connection connect = Jsoup.connect(url);
        if (CollectionUtil.isNotEmpty(params)){
            connect.data(params);
        }
        return connect.post();
    }
    /**
     * 以POST方式请求URL，获得Document
     * 参数：URL 、参数Map 、CookieMap
     */
    public static final  Document getDocumentByPostUrl(String url,Map<String, String> params, Map<String,String> cookies) throws IOException{
        Connection connect = Jsoup.connect(url);
        if (CollectionUtil.isNotEmpty(params)){
            connect.data(params);
        }
        if (CollectionUtil.isNotEmpty(cookies)){
            connect.cookies(cookies);
        }
        return connect.post();
    }
    /**
     * 以POST方式请求URL，获得Document
     * 参数：URL 、参数Map 、CookieMap、HeadersMap
     */
    public static final  Document getDocumentByPostUrl(String url,Map<String, String> params, Map<String,String> cookies,Map<String,String> headers) throws IOException{
        Connection connect = Jsoup.connect(url);
        if (CollectionUtil.isNotEmpty(params)){
            connect.data(params);
        }
        if (CollectionUtil.isNotEmpty(cookies)){
            connect.cookies(cookies);
        }
        if (CollectionUtil.isNotEmpty(headers)){
            Set<Entry<String, String>> entrySet = headers.entrySet();
            for(Entry<String,String> entry : entrySet){
                connect.header(entry.getKey(), entry.getValue());
            }
        }
        return connect.post();
    }
    /**
     * 根据id，获取Document的Element
     */
    public static final  Element getElementById(Document doc, String id){
        return doc.getElementById(id);
    }

    /**
     * 根据id，获取Element的Element
     */
    public static final  Element getElementById(Element ele, String id){
        return ele.getElementById(id);
    }

    /**
     * 根据id，获取Elements的Elements
     */
    public static final  List<Element> getElementByElementsId(List<Element> eles, String id){
        List<Element> es = new ArrayList<Element>();
        for(Element e : eles){
            es.add(getElementById(e,id));
        }
        return es;
    }

    /**
     * 根据ids，获取Document的Elements
     */
    public static final  List<Element> getElementsByIds(Document doc, List<String> ids){
        List<Element> list = new ArrayList<Element>();
        for(String id : ids){
            CollectionUtil.addNotNullElement(list,doc.getElementById(id));
        }
        return list;
    }

    /**
     * 根据ids，获取Element的Elements
     */
    public static final  List<Element> getElementsByIds(Element ele, List<String> ids){
        List<Element> list = new ArrayList<Element>();
        for(String id : ids){
            CollectionUtil.addNotNullElement(list,ele.getElementById(id));
        }
        return list;
    }

    /**
     * 根据ids，获取Elements的Elements
     */
    public static final  List<Element> getElementByElementsIds(List<Element> eles, List<String> ids){
        List<Element> es = new ArrayList<Element>();
        for(Element e : eles){
            es.addAll(getElementsByIds(e,ids));
        }
        return es;
    }

    /**
     * 根据tag，获取Document的Elements
     */
    public static final  List<Element> getElementsByTag(Document doc, String tag){
        List<Element> list = new ArrayList<Element>();
        CollectionUtil.addNotNullElements(list, doc.getElementsByTag(tag));
        return list;
    }

    /**
     * 根据tag，获取Element的Elements
     */
    public static final  List<Element> getElementsByTag(Element ele, String tag){
        List<Element> list = new ArrayList<Element>();
        CollectionUtil.addNotNullElements(list, ele.getElementsByTag(tag));
        return list;
    }

    /**
     * 根据tag，获取Elements的Elements
     */
    public static final  List<Element> getElementByElementsTag(List<Element> eles, String tag){
        List<Element> es = new ArrayList<Element>();
        for(Element e : eles){
            es.addAll(getElementsByTag(e,tag));
        }
        return es;
    }

    /**
     * 根据tags，获取Document的Elements
     */
    public static final  List<Element> getElementsByTags(Document doc, List<String> tags){
        List<Element> list = new ArrayList<Element>();
        for(String tag : tags){
            CollectionUtil.addNotNullElements(list,getElementsByTag(doc,tag));
        }
        return list;
    }

    /**
     * 根据tags，获取Element的Elements
     */
    public static final  List<Element> getElementsByTags(Element ele, List<String> tags){
        List<Element> list = new ArrayList<Element>();
        for(String tag : tags){
            CollectionUtil.addNotNullElements(list,getElementsByTag(ele,tag));
        }
        return list;
    }

    /**
     * 根据tags，获取Elements的Elements
     */
    public static final  List<Element> getElementByElementsTags(List<Element> eles, List<String> tags){
        List<Element> es = new ArrayList<Element>();
        for(Element e : eles){
            es.addAll(getElementsByTags(e,tags));
        }
        return es;
    }

    /**
     * 根据class，获取Document的Elements
     */
    public static final  List<Element> getElementsByClass(Document doc, String clazz){
        List<Element> list = new ArrayList<Element>();
        CollectionUtil.addNotNullElements(list,doc.getElementsByClass(clazz));
        return list;
    }

    /**
     * 根据class，获取Element的Elements
     */
    public static final  List<Element> getElementsByClass(Element ele, String clazz){
        List<Element> list = new ArrayList<Element>();
        CollectionUtil.addNotNullElements(list,ele.getElementsByClass(clazz));
        return list;
    }

    /**
     * 根据class，获取Elements的Elements
     */
    public static final  List<Element> getElementByElementsClass(List<Element> eles, String clazz){
        List<Element> es = new ArrayList<Element>();
        for(Element e : eles){
            es.addAll(getElementsByClass(e,clazz));
        }
        return es;
    }

    /**
     * 根据classes，获取Document的Elements
     */
    public static final  List<Element> getElementsByClasses(Document doc, List<String> classes){
        List<Element> list = new ArrayList<Element>();
        for(String clazz : classes){
            CollectionUtil.addNotNullElements(list,getElementsByClass(doc,clazz));
        }
        return list;
    }

    /**
     * 根据classes，获取Element的Elements
     */
    public static final  List<Element> getElementsByClasses(Element ele, List<String> classes){
        List<Element> list = new ArrayList<Element>();
        for(String clazz : classes){
            CollectionUtil.addNotNullElements(list,getElementsByClass(ele,clazz));
        }
        return list;
    }

    /**
     * 根据classes，获取Elements的Elements
     */
    public static final  List<Element> getElementByElementsClasses(List<Element> eles, List<String> classes){
        List<Element> es = new ArrayList<Element>();
        for(Element e : eles){
            es.addAll(getElementsByClasses(e,classes));
        }
        return es;
    }

    /**
     * 根据attr，获取Document的Elements
     */
    public static final  List<Element> getElementsByAttr(Document doc, String attr){
        List<Element> list = new ArrayList<Element>();
        CollectionUtil.addNotNullElements(list, doc.getElementsByAttribute(attr));
        return list;
    }

    /**
     * 根据attr，获取Element的Elements
     */
    public static final  List<Element> getElementsByAttr(Element ele, String attr){
        List<Element> list = new ArrayList<Element>();
        CollectionUtil.addNotNullElements(list, ele.getElementsByAttribute(attr));
        return list;
    }

    /**
     * 根据attr，获取Elements的Elements
     */
    public static final  List<Element> getElementByElementsAttr(List<Element> eles, String attr){
        List<Element> es = new ArrayList<Element>();
        for(Element e : eles){
            es.addAll(getElementsByAttr(e,attr));
        }
        return es;
    }

    /**
     * 根据attrs，获取Document的Elements
     */
    public static final  List<Element> getElementsByAttrs(Document doc, List<String> attrs){
        List<Element> list = new ArrayList<Element>();
        for(String attr : attrs){
            CollectionUtil.addNotNullElements(list, getElementsByAttr(doc,attr));
        }
        return list;
    }

    /**
     * 根据attrs，获取Element的Elements
     */
    public static final  List<Element> getElementsByAttrs(Element ele, List<String> attrs){
        List<Element> list = new ArrayList<Element>();
        for(String attr : attrs){
            CollectionUtil.addNotNullElements(list, getElementsByAttr(ele,attr));
        }
        return list;
    }

    /**
     * 根据attr，获取Elements的Elements
     */
    public static final  List<Element> getElementByElementsAttrs(List<Element> eles, List<String> attrs){
        List<Element> es = new ArrayList<Element>();
        for(Element e : eles){
            es.addAll(getElementsByAttrs(e,attrs));
        }
        return es;
    }

    /**
     * 根据attr/value，获取Document的Elements
     */
    public static final  List<Element> getElementsByAttrValue(Document doc, String attr, String value) {
        List<Element> list = new ArrayList<Element>();
        CollectionUtil.addNotNullElements(list, doc.getElementsByAttributeValue(attr, value));
        return list;
    }

    /**
     * 根据attr/value，获取Element的Elements
     */
    public static final  List<Element> getElementsByAttrValue(Element ele, String attr, String value) {
        List<Element> list = new ArrayList<Element>();
        CollectionUtil.addNotNullElements(list, ele.getElementsByAttributeValue(attr, value));
        return list;
    }

    /**
     * 根据attr/value，获取Elements的Elements
     */
    public static final  List<Element> getElementByElementsAttrValue(List<Element> eles, String attr, String value) {
        List<Element> es = new ArrayList<Element>();
        for (Element e : eles) {
            es.addAll(getElementsByAttrValue(e, attr, value));
        }
        return es;
    }

    /**
     * 根据attrs/values，获取Document的Elements
     */
    public static final  List<Element> getElementsByAttrValues(Document doc, Map<String, String> attrValues) {
        List<Element> list = new ArrayList<Element>();
        Set<String> avs = attrValues.keySet();
        for (String attr : avs) {
            CollectionUtil.addNotNullElements(list, getElementsByAttrValue(doc, attr, attrValues.get(attr)));
        }
        return list;
    }

    /**
     * 根据attrs/values，获取Element的Elements
     */
    public static final  List<Element> getElementsByAttrValues(Element ele, Map<String, String> attrValues) {
        List<Element> list = new ArrayList<Element>();
        Set<String> avs = attrValues.keySet();
        for (String attr : avs) {
            CollectionUtil.addNotNullElements(list, getElementsByAttrValue(ele, attr, attrValues.get(attr)));
        }
        return list;
    }

    /**
     * 根据attrs/values，获取Elements的Elements
     */
    public static final  List<Element> getElementByElementsAttrValues(List<Element> eles, Map<String, String> attrValues) {
        List<Element> es = new ArrayList<Element>();
        for (Element e : eles) {
            es.addAll(getElementsByAttrValues(e, attrValues));
        }
        return es;
    }

    /**
     * 根据反向attr/value，获取Document的Elements
     */
    public static final  List<Element> getElementsByNotAttrValue(Document doc, String attr, String value) {
        List<Element> list = new ArrayList<Element>();
        CollectionUtil.addNotNullElements(list, doc.getElementsByAttributeValueNot(attr, value));
        return list;
    }

    /**
     * 根据反向attr/value，获取Element的Elements
     */
    public static final  List<Element> getElementsByNotAttrValue(Element ele, String attr, String value) {
        List<Element> list = new ArrayList<Element>();
        CollectionUtil.addNotNullElements(list, ele.getElementsByAttributeValueNot(attr, value));
        return list;
    }

    /**
     * 根据反向attr/value，获取Elements的Elements
     */
    public static final  List<Element> getElementByNotElementsAttrValue(List<Element> eles, String attr, String value) {
        List<Element> es = new ArrayList<Element>();
        for (Element e : eles) {
            es.addAll(getElementsByNotAttrValue(e, attr, value));
        }
        return es;
    }

    /**
     * 根据反向attrs/values，获取Document的Elements
     */
    public static final  List<Element> getElementsByNotAttrValues(Document doc, Map<String, String> attrValues) {
        List<Element> list = new ArrayList<Element>();
        Set<String> avs = attrValues.keySet();
        for (String attr : avs) {
            CollectionUtil.addNotNullElements(list, getElementsByNotAttrValue(doc, attr, attrValues.get(attr)));
        }
        return list;
    }

    /**
     * 根据反向attrs/values，获取Element的Elements
     */
    public static final  List<Element> getElementsByNotAttrValues(Element ele, Map<String, String> attrValues) {
        List<Element> list = new ArrayList<Element>();
        Set<String> avs = attrValues.keySet();
        for (String attr : avs) {
            CollectionUtil.addNotNullElements(list, getElementsByNotAttrValue(ele, attr, attrValues.get(attr)));
        }
        return list;
    }

    /**
     * 根据反向attrs/values，获取Elements的Elements
     */
    public static final  List<Element> getElementByNotElementsAttrValues(List<Element> eles, Map<String, String> attrValues) {
        List<Element> es = new ArrayList<Element>();
        for (Element e : eles) {
            es.addAll(getElementsByNotAttrValues(e, attrValues));
        }
        return es;
    }

    /**
     * 根据attr/正则value，获取Document的Elements
     */
    public static final  List<Element> getElementsByRegAttrValue(Document doc, String attr, String valueReg) {
        List<Element> list = new ArrayList<Element>();
        CollectionUtil.addNotNullElements(list, doc.getElementsByAttributeValueMatching(attr, valueReg));
        return list;
    }

    /**
     * 根据attr/正则value，获取Element的Elements
     */
    public static final  List<Element> getElementsByRegAttrValue(Element ele, String attr, String valueReg) {
        List<Element> list = new ArrayList<Element>();
        CollectionUtil.addNotNullElements(list, ele.getElementsByAttributeValueMatching(attr, valueReg));
        return list;
    }

    /**
     * 根据attr/正则value，获取Elements的Elements
     */
    public static final  List<Element> getElementByRegElementsAttrValue(List<Element> eles, String attr, String value) {
        List<Element> es = new ArrayList<Element>();
        for (Element e : eles) {
            es.addAll(getElementsByRegAttrValue(e, attr, value));
        }
        return es;
    }

    /**
     * 根据attrs/正则values，获取Document的Elements
     */
    public static final  List<Element> getElementsByRegAttrValues(Document doc, Map<String, String> attrValueRegs) {
        List<Element> list = new ArrayList<Element>();
        Set<String> avs = attrValueRegs.keySet();
        for (String attr : avs) {
            CollectionUtil.addNotNullElements(list, getElementsByRegAttrValue(doc, attr, attrValueRegs.get(attr)));
        }
        return list;
    }

    /**
     * 根据attrs/正则values，获取Element的Elements
     */
    public static final  List<Element> getElementsByRegAttrValues(Element ele, Map<String, String> attrValueRegs) {
        List<Element> list = new ArrayList<Element>();
        Set<String> avs = attrValueRegs.keySet();
        for (String attr : avs) {
            CollectionUtil.addNotNullElements(list, getElementsByRegAttrValue(ele, attr, attrValueRegs.get(attr)));
        }
        return list;
    }

    /**
     * 根据attrs/正则values，获取Elements的Elements
     */
    public static final  List<Element> getElementByRegElementsAttrValues(List<Element> eles,
                                                                         Map<String, String> attrValueRegs) {
        List<Element> es = new ArrayList<Element>();
        for (Element e : eles) {
            es.addAll(getElementsByRegAttrValues(e, attrValueRegs));
        }
        return es;
    }

    /**
     * 根据正则text，获取Document的Elements
     */
    public static final  List<Element> getElementsByRegText(Document doc, String textReg) {
        List<Element> list = new ArrayList<Element>();
        CollectionUtil.addNotNullElements(list, doc.getElementsMatchingText(textReg));
        return list;
    }

    /**
     * 根据正则text，获取Element的Elements
     */
    public static final  List<Element> getElementsByRegText(Element ele, String textReg) {
        List<Element> list = new ArrayList<Element>();
        CollectionUtil.addNotNullElements(list, ele.getElementsMatchingText(textReg));
        return list;
    }

    /**
     * 根据正则text，获取Elements的Elements
     */
    public static final  List<Element> getElementByRegElementsText(List<Element> eles, String textReg) {
        List<Element> es = new ArrayList<Element>();
        for (Element e : eles) {
            es.addAll(getElementsByRegText(e, textReg));
        }
        return es;
    }

    /**
     * 根据正则texts，获取Document的Elements
     */
    public static final  List<Element> getElementsByRegTexts(Document doc, List<String> textRegs) {
        List<Element> list = new ArrayList<Element>();
        for (String textReg : textRegs) {
            CollectionUtil.addNotNullElements(list, getElementsByRegText(doc, textReg));
        }
        return list;
    }

    /**
     * 根据正则texts，获取Element的Elements
     */
    public static final  List<Element> getElementsByRegTexts(Element ele, List<String> textRegs) {
        List<Element> list = new ArrayList<Element>();
        for (String textReg : textRegs) {
            CollectionUtil.addNotNullElements(list, getElementsByRegText(ele, textReg));
        }
        return list;
    }

    /**
     * 根据正则texts，获取Elements的Elements
     */
    public static final  List<Element> getElementByRegElementsText(List<Element> eles, List<String> textRegs) {
        List<Element> es = new ArrayList<Element>();
        for (Element e : eles) {
            es.addAll(getElementsByRegTexts(e, textRegs));
        }
        return es;
    }

    /**
     * 根据Selector，获取Document的Elements
     *
     * 选择器： 语法示例： 结果示例：
     * *************************************************************************
     * 所有元素 			"*" 						所有元素
     * 组合查询(拼接) 		"jsp|a[href*=/url/]#id" 	所有<jsp:a id="id" href="xxx/url/xxx">
     * 多个查询(,) 		"a,div,#id" 				所有<a.>、<div.>和<xx id="id">
     * 标签查询 			"a" 						所有<a...>
     * 命名标签查询 		"jsp|a" 					所有<jsp:a>
     * ID 查询 			"#id" 						所有<xx id="id">
     * Class查询 			".clazz" 					所有<xx class="clazz">
     * 属性查询 			"[attrName]" 				所有<xx attrName="xx">
     * 属性前缀匹配 		"[^attr]" 					所有<xx attrXx="xx">
     * 属性和值查询			"[attr=value]" 				所有<xx attr="value">
     * 属性值前缀匹配		"[attr^=value]" 			所有<xx attr="valueXx">
     * 属性值后缀匹配 		"[attr&=Value]" 			所有<xx attr="XxValue">
     * 属性值包含匹配		"[attr*=Value]" 			所有<xx attr="XxValueXx">
     * 属性值正则匹配		"[attr~=\.(png|jpg)]" 		所有 <xx attr="XxValueXx">
     * 后代查询(空格) 		".clazz div a[name=me]" 	所有<xx class="clazz"><div.>下包含的<a name="me">
     * 子级查询( > ) 		"div > p" 					所有 <div.>下的子级<p.>
     * 之前同级( ~ ) 		"div#id ~ [name="me"]" 		所有<div id="id">的之前同级<xx name="me">
     * 前个同级( + ) 		"input#pwd + input" 		只找<input id="pwd">前面的一个<input...>
     *
     * ************************* 伪选择器：在已选元索中再次筛选 ************************
     * 元素数量查询		"div:empty" 			所有没有子元素的<div.>
     * 元素位置查询 	"div > p:first-child" 	所有<div.>的子级中第1个<p.>
     * "ol li:last-child" 	所有<ol.>的第1个<li.>
     * "td:eq(0)" 			所有在同级元素中索引为0的
     * <td.>，索引0表示首个元素 "div p:gt(2)" 所有<div.>下索引大于2的<p.> "form > input:lt(3)" 所有
     * <form.>的子级中索引小于3的<input.> 反向查询 "div:not(.clazz)" 所有class不为clazz的
     * <div.> 包含查询 "div:has(p)" 所有包含<p.>的<div.> 包含指定文本 "div:contains(text)"
     * 所有标签体文本中包含text的<div.>，不区分大小写 直接包含指定文本 "div:containsOwn(text)"
     * 所有标签体(不包含后代的标签体)文本中包含text的<div.> 包含正则匹配文本 "div:matches(\.(a|b))"
     * 所有标签体文本中包含.a或.b的<div.> 直接包含正则文本 "div:matchesOwn((a|b))"
     * 所有标签体(不包含后代的标签体)文本中包含.a或.b的<div.>
     */
    public static final  List<Element> getElementsBySelector(Document doc, String selector) {
        List<Element> list = new ArrayList<Element>();
        CollectionUtil.addNotNullElements(list, doc.select(selector));
        return list;
    }

    /**
     * 根据Selector，获取Element的Elements
     */
    public static final  List<Element> getElementsBySelector(Element ele, String selector) {
        List<Element> list = new ArrayList<Element>();
        CollectionUtil.addNotNullElements(list, ele.select(selector));
        return list;
    }

    /**
     * 根据Selector，获取Elements的Elements
     */
    public static final  List<Element> getElementsByElementsSelector(List<Element> eles, String selector) {
        List<Element> es = new ArrayList<Element>();
        for (Element e : eles) {
            es.addAll(getElementsBySelector(e, selector));
        }
        return es;
    }

    /**
     * 通过Document获取带链接的Element，包括src/href/link属性中
     */
    public static final  List<Element> getUrlElements(Document doc, boolean isSrc, boolean isHref, boolean isLink) {
        List<Element> list = new ArrayList<Element>();
        Elements links = doc.select("a[href]");
        Elements media = doc.select("[src]");
        Elements imports = doc.select("link[href]");
        list.addAll(links);
        list.addAll(media);
        list.addAll(imports);
        return list;
    }

    /**
     * 通过Element获取带链接的Element，包括src/href/link属性中
     */
    public static final  List<Element> getUrlElements(Element ele, boolean isSrc, boolean isHref, boolean isLink) {
        List<Element> list = new ArrayList<Element>();
        if(isSrc){
            Elements media = ele.select("[src]");
            list.addAll(media);
        }
        if(isHref){
            Elements links = ele.select("a[href]");
            list.addAll(links);
        }
        if(isLink){
            Elements imports = ele.select("link[href]");

            list.addAll(imports);
        }
        return list;
    }

    /**
     * 通过Elements获取带链接的Element，包括src/href/link属性中
     */
    public static final  List<Element> getUrlElementsByElementsSelector(List<Element> eles, boolean isSrc, boolean isHref,
                                                                        boolean isLink) {
        List<Element> es = new ArrayList<Element>();
        for (Element e : eles) {
            es.addAll(getUrlElements(e, isSrc, isHref, isLink));
        }
        return es;
    }

    /**
     * 获取链接型Element的绝对URL路径，包括href/src/link属性中
     * 注：html中URL通常使用相对路径，直接使用attr("href")获取的是"/*"之类的相对URL。
     * 要获取绝对URL，需使用attr("abs:href") 或 absUrl("src")。
     */
    public static final  String getAbsUrl(Element urlElement) {
        if (urlElement.hasAttr("src")) {
            return urlElement.attr("abs:src");
        } else if (urlElement.hasAttr("href")) {
            return urlElement.attr("abs:href");
        } else if (urlElement.hasAttr("link")) {
            return urlElement.attr("abs:link");
        } else {
            return "";
        }
    }

    /**
     * 获取数据路径页面信息
     *
     * @return
     */
    public static Document getHtml(String url) {
        Document doc = null;
        try {
            doc = Jsoup.connect(url).timeout(5 * 60000).get();
        } catch (Exception e) {
            logger.info("获取" + url + "数据：" + e);
        }
        return doc;
    }

    /**
     * 去除标签中的 id和class
     */
    public String removeLable(Element element) {
        if (element == null) {
            return "";
        }
        element.removeAttr("class").removeAttr("id");
        element.select("p").removeAttr("class");

        element.select("script").remove();

        // 移除注释
        Pattern p = Pattern.compile("\\<!--(.+)--\\>");
        Matcher m = p.matcher(element.toString());
        return m.replaceAll("");
    }





    public static String getPostfix(String filename) {
        filename = StrUtil.substringAfterLast(filename, ".");
        filename = StrUtil.substringBefore(filename, "?");
        filename = StrUtil.substringBefore(filename, "/");
        filename = StrUtil.substringBefore(filename, "\\");
        filename = StrUtil.substringBefore(filename, "&");
        filename = StrUtil.substringBefore(filename, "$");
        filename = StrUtil.substringBefore(filename, "%");
        filename = StrUtil.substringBefore(filename, "#");
        filename = StrUtil.substringBefore(filename, "@");
        return filename;
    }

    /**
     * 去除点击事件
     *
     * @param doc
     */
    public static void removeHref(Document doc) {
        doc.select("img").removeAttr("onclick");
        doc.select("input[type=submit]").attr("type", "button");
        doc.select("input[type=button]").removeAttr("onclick");
        doc.select("area").removeAttr("href");
    }


}
