package cn.dolphin.core.freemarker;


import freemarker.ext.beans.BeansWrapper;
import freemarker.ext.beans.BeansWrapperBuilder;
import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import freemarker.template.TemplateHashModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

/**
 * 模板封装
 */
public class FreemarkerUtil {

    private static Logger logger = LoggerFactory.getLogger(FreemarkerUtil.class);

    /** 是否已初始化 */
    private static boolean isInit = false;

    /** 应用所在路径 */
    private static String templatePath = null;

    /** 编码格式 UTF-8 */
    private static final String ENCODING = "UTF-8";

    /** FreeMarker配置 */
    private static Configuration freemarkerConfig = new Configuration(Configuration.VERSION_2_3_28);
    /** 项目路径 */
    //private static final String path = FreemarkerUtil.class.getClassLoader().getResource("").getPath();

    /** 路径分割符 */
    public static final String PATH_SEPARATOR = "/";

    private static final Integer TEMPLATE_LOADING_FILE = 1;
    private static final Integer TEMPLATE_LOADING_PROJECT = 2;

    /**
     * 初始化FreeMarker配置，项目路径
     * @param applicationPath 应用所在路径
     */
    public static void initFreeMarker(String applicationPath){
        initFreeMarker(applicationPath,2);
    }


    /**
     * 初始化FreeMarker配置
     * @param applicationPath 应用所在路径
     * @param type 模板文件夹路径类型，1：文件系统路径；2：项目路径
     */
    public static void initFreeMarker(String applicationPath,Integer type){

        try {

            if(!isInit){
                templatePath = applicationPath;
                /**
                 * 设置模板本件夹路径：
                 * 1、setDirectoryForTemplateLoading，文件系统绝对路径
                 * 2、setClassForTemplateLoading项目相对路径
                 */
                if(TEMPLATE_LOADING_FILE == type){
                    freemarkerConfig.setDirectoryForTemplateLoading(new File(applicationPath));
                }
                if(TEMPLATE_LOADING_PROJECT == type){
                    freemarkerConfig.setClassForTemplateLoading(FreemarkerUtil.class,applicationPath);
                }
                // 设置文件编码为UTF-8
                freemarkerConfig.setEncoding(Locale.getDefault(), ENCODING);
                isInit = true;
            }

        }catch (Exception e){
            logger.error("初始化FreeMarker配置出错",e);
            //throw new RuntimeException("初始化FreeMarker配置出错",e);
        }

    }

    /**
     * 指定要在ftl页面使用的静态包名
     *
     * @param packageName 静态包名
     * @return TemplateHashModel
     */
    public static TemplateHashModel useStaticPackage(String packageName) {
        try {
            BeansWrapper wrapper = new BeansWrapperBuilder(Configuration.DEFAULT_INCOMPATIBLE_IMPROVEMENTS).build();
            TemplateHashModel staticModels = wrapper.getStaticModels();
            TemplateHashModel fileStatics = (TemplateHashModel) staticModels.get(packageName);
            return fileStatics;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 获取HTML文件字符串
     * @param templateName
     * @param params
     * @return
     */
    public static String crateHTML(String templateName, Map<String, Object> params){
        String htmlText = "";
        // 通过指定模板名获取FreeMarker模板实例
        Template template =  getTemplate(templateName);
        try {
            // FreeMarker通过Map传递动态数据
            // 解析模板并替换动态数据，最终content将替换模板文件中的${content}标签。
            htmlText = processTemplateIntoString(template, params);
        }catch (Exception e){
            logger.error("转换Freemarker模板为HTML文件出错",e);
        }
        return htmlText;
    }



    /**
     * 获取模板json，项目文件路径
     * @param templateName  模板名称
     * @param templateDirectory 模板地址
     * @param datas
     * @return
     */
    public static String generateString(String templateName, String templateDirectory, Map<String, Object> datas){
        String json = "";
        try {

            // 设置默认编码
            freemarkerConfig.setDefaultEncoding("UTF-8");

            // 设置模板所在文件夹
            freemarkerConfig.setClassForTemplateLoading(FreemarkerUtil.class,templateDirectory);

            // 生成模板对象
            Template template = freemarkerConfig.getTemplate(templateName);

            // 将datas写入模板并返回
            json = processTemplateIntoString(template, datas);
            //json.replaceAll("\\s+", "").replaceAll("\"", "'");
            // 解析echartsConvert响应
            return json;
        } catch (Exception e) {
            logger.error("转换Freemarker模板文件出错",e);
//             throw  new UtilsRuntimeException(e);
            return json;

        }

    }


    /**
     * 获取模板json，系统文件路径
     * @param templateName  模板名称
     * @param templateDirectory 模板地址
     * @param datas
     * @return
     */
    public static String generateStringField(String templateName, String templateDirectory, Map<String, Object> datas){
        String json = "";
        try {

            // 设置默认编码
            freemarkerConfig.setDefaultEncoding("UTF-8");

            // 设置模板所在文件夹
            //freemarkerConfig.setDirectoryForTemplateLoading(new File(templateDirectory));
            freemarkerConfig.setClassForTemplateLoading(FreemarkerUtil.class,templateDirectory);
            // 生成模板对象
            Template template = freemarkerConfig.getTemplate(templateName);

            // 将datas写入模板并返回
            json = processTemplateIntoString(template, datas);
            //json.replaceAll("\\s+", "").replaceAll("\"", "'");
            // 解析echartsConvert响应
            return json;
        } catch (Exception e) {
            logger.error("转换Freemarker模板文件出错",e);
//             throw  new UtilsRuntimeException(e);
            return json;
        }

    }

    /**
     * 获取Freemarker模板文件
     * @param name 文件名
     * @return
     */
    public static Template getTemplate(String name) {
        Template template = null;
        try {
            template = freemarkerConfig.getTemplate(name,ENCODING);
        } catch (Exception e) {
            logger.error("获取Freemarker模板文件出错",e);
        }
        return template;

    }

    /**
     * 将解析之后的文件内容返回字符串
     * @param template 模板文件名
     * @param model 数据Map
     * @return
     */
    private static String processTemplateIntoString(Template template,Object model) throws IOException, TemplateException {
        StringWriter result = new StringWriter();
        try {
            //通过一个文件输出流，就可以写到相应的文件中
            template.process(model, result);
            result.flush();
        } catch (Exception e) {
            logger.error("将解析之后的文件内容返回字符串",e);
        } finally {
            try {
                if(result!=null) result.close();
            } catch (IOException e) {
                //e.printStackTrace();
                throw new RuntimeException("将解析之后的文件内容返回字符串",e);
            }
        }
        return result.toString();
    }


    /**
     * 执行解析操作
     *
     * @param model 数据
     * @param templateName 模板文件名
     * @param storeFilePath 解析输出路径
     * @return
     */
    public static void exec(Map<String, Object> model, String templateName,String storeFilePath) {
        FileOutputStream fileOutputStream=null;
        try {
            fileOutputStream = new FileOutputStream(storeFilePath);
            Template template = getTemplate(templateName);
            template.process(model, new OutputStreamWriter(fileOutputStream));
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            try {
                fileOutputStream.close();
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    /**
     * 将解析之后的文件内容打印在控制台
     * @param templateName 模板文件名
     * @param params 数据Map
     */
    public static void printConsole(String templateName,Map<String,Object> params) {
        System.out.println(crateHTML(templateName, params));
    }



    public static void main(String[] args) {
        FreemarkerUtil.initFreeMarker("/template");
        Map<String,Object> params = new HashMap<String, Object>();
        params.put("userName","测试");
        params.put("age",12);
        FreemarkerUtil.printConsole("user.ftl",params);
    }


}
