package cn.dolphin.core.db;

import cn.dolphin.core.bean.BaseModel;
import cn.dolphin.core.map.MapUtil;
import cn.dolphin.core.mybatis.BaseMyBatisDao;
import cn.dolphin.core.mybatis.IbatisSql;
import cn.dolphin.core.mybatis.SqlMapper;
import cn.dolphin.core.page.PageBean;
import cn.dolphin.core.page.PageList;
import cn.dolphin.core.page.QueryFilter;
import cn.dolphin.core.util.StrUtil;
import cn.dolphin.core.util.SystemClock;
import org.apache.ibatis.session.ExecutorType;
import org.apache.ibatis.session.RowBounds;
import org.apache.ibatis.session.SqlSession;
import org.apache.ibatis.session.SqlSessionFactory;
import org.mybatis.spring.SqlSessionUtils;

import javax.annotation.Resource;
import java.io.Serializable;
import java.sql.Connection;
import java.util.*;

@SuppressWarnings("all")
public abstract class GenericDao<E, PK extends Serializable> extends BaseMyBatisDao<E, PK> implements IEntityDao<E, PK>{

    /**
     * 获取当前dao 需要访问的model类名，在实现类中实现这个方法。
     * 主要的功能：用于获取类的完整路径，以提供给mybatis使用。
     */
    @SuppressWarnings("rawtypes")
    public abstract Class<?> getEntityClass();

    /**
     * 扩展动态sql查询，可以使用SQL类组装原生态sql语句直接运行
     */
    public abstract SqlMapper sqlMapper();


    @Resource
    Properties configproperties;

    /**
     * 获取当前数据库类型。
     * @return
     */
    protected String getDbType(){
        return configproperties.getProperty("jdbc.dbType");
    }

    /**
     * 根据出入的POJO对象获取ibatis的命名空间。<br>
     * 返回如：com.test.platform.model.role.Role
     * @return
     */
    private String getIbatisMapperNamespace() {
        return getEntityClass().getName();
    }

    /**
     * 根据主键取得对象。
     */
    public E getMyBatisById(PK primaryKey) throws Exception{
        String getStatement= getIbatisMapperNamespace() + ".getById";
        E object = (E)getSqlSessionTemplate().selectOne(getStatement, primaryKey);

        return object;
    }


    /**
     * 返回一条数据对象。
     * @param sqlKey  <a href="#Mapper键值ID">Mapper键值ID</a>。
     * @param params 参数为单个对象或Map类型
     * @return
     */
    public E getMyBatisUnique(String sqlKey,Object params) throws Exception{
        String getStatement= getIbatisMapperNamespace() + "." + sqlKey;
        E object = (E)getSqlSessionTemplate().selectOne(getStatement,params);
        return object;
    }



    /**
     * 返回单条数据，如 select count(*) from table_a
     * @param sqlKey
     * @param params
     * @return
     */
    public Object getMyBatisOne(String sqlKey,Object params) throws Exception{
        String getStatement=getIbatisMapperNamespace() + "." + sqlKey;
        Object object = getSqlSessionTemplate().selectOne(getStatement,params);
        return object;
    }

    /**
     * 按Query Filter进行数据过滤查询，并且可返回对应的页面中的分页结果
     * @param sqlKey   <a href="#Mapper键值ID">Mapper键值ID</a>
     * @param queryFilter {@link QueryFilter QueryFilter}
     * @return
     */
    public List<E> getMyBatisBySqlKey(String sqlKey, QueryFilter queryFilter)throws Exception{
        List<E> list= new ArrayList<E>();
        if(null ==queryFilter){
            list=getMyBatisBySqlKey(sqlKey);
        }else if(queryFilter.getPageBean()==null){
            list=getMyBatisBySqlKey(sqlKey,queryFilter.getFilters());
        }
        else{
            list=getMyBatisBySqlKey(sqlKey,queryFilter.getFilters(),queryFilter.getPageBean());
        }
        return list;
    }

    /**
     * 按SqlKey获取分页的数据。<br>
     * @param sqlKey   <a href="#Mapper键值ID">Mapper键值ID</a>。
     * @param params 参数  参数为单个对象或Map类型 ,若要排序，请在该参数Map里加上两参数key为orderField与orderSeq的值对。
     * @param pageBean  分页对象 {@link PageBean pageBean}
     * @return
     */
    public List<E> getMyBatisBySqlKey(String sqlKey,Object params,PageBean pageBean) throws Exception{
        return getMyBatisList(sqlKey, params, pageBean);
    }

    /**
     * 取得列表类型返回不使用范型
     * @param sqlKey Mapper键值ID
     * @param params 参数类型 可为Object类型，也可为Map类型,若要排序，请在该参数Map里加上两参数key为orderField与orderSeq的值对
     * @return
     */
    @SuppressWarnings("rawtypes")
    public List getMyBatisBySqlKeyGenericity(String sqlKey,Object params) throws Exception{
        String statement= getIbatisMapperNamespace() + "." + sqlKey;

        return getSqlSessionTemplate().selectList(statement,params);
    }

    /**
     * 取得列表类型返回不使用范型
     * @param sqlKey Mapper键值ID
     * @return
     */
    @SuppressWarnings("rawtypes")
    public List getMyBatisBySqlKeyGenericity(String sqlKey) throws Exception{
        String statement= getIbatisMapperNamespace() + "." + sqlKey;

        return getSqlSessionTemplate().selectList(statement);
    }

    /**
     * 根据mybatis映射文件的ID获取分页数据。
     * @param sqlKey
     * @param params 过滤参数
     * @param pageBean 分页
     *
     */
    public <T> List<T> getMyBatisBySqlKeyGenericity(String sqlKey,Object params,PageBean pageBean) throws Exception{

        if(pageBean==null){
            return getMyBatisBySqlKeyGenericity(sqlKey, params);
        }
        // 拼接dao层地址
        String statementName = getIbatisMapperNamespace() + "."+ sqlKey;
        Map<String,Object> filters = new HashMap<String, Object>();
        if(params!=null){
            if (params instanceof Map){
                filters.putAll((Map) params); //旧写法
            }else{ // 屏蔽下边写法，是因为影响了扩展基类传递map取值问题
                //Map parameterObject = BeanUtil.describe(params); //旧写法
//                Map parameterObject = BeanMap2Util.map(params,HashMap.class);
//                if(MapUtil.isContainsKey(parameterObject,"params")){
//                    if(MapUtil.isNotEmpty(MapUtil.getMap(parameterObject,"params"))){
//                        filters.putAll(MapUtil.getMap(parameterObject,"params"));
//                    }
//                }
//                filters.putAll(parameterObject);

            }
        }
        if(pageBean.isShowTotal()){
            //获取SQL语句
            IbatisSql ibatisSql=this.getIbatisSql(statementName, (MapUtil.isNotEmpty(filters)?filters:params));
            //使用插件输出sql
            //log.info(ibatisSql.getSql());
            //取得返回数量
            if(pageBean.isDistinct()){//count去重
                if(StrUtil.isEmpty(primaryKey)){
                    int totalCount = getJdbcTemplate().queryForObject(ibatisSql.getCountSqlDistinct(), ibatisSql.getParameters(), Integer.class);
                    pageBean.setTotalCount(totalCount);
                }else {
                    int totalCount = getJdbcTemplate().queryForObject(ibatisSql.getCountSqlDistinct(primaryKey), ibatisSql.getParameters(), Integer.class);
                    pageBean.setTotalCount(totalCount);
                }
            }else { //count不去重
                if(StrUtil.isEmpty(primaryKey)){
                    int totalCount = getJdbcTemplate().queryForObject(ibatisSql.getCountSql(), ibatisSql.getParameters(), Integer.class);
                    pageBean.setTotalCount(totalCount);
                }else {
                    int totalCount = getJdbcTemplate().queryForObject(ibatisSql.getCountSql(primaryKey), ibatisSql.getParameters(), Integer.class);
                    pageBean.setTotalCount(totalCount);
                }
            }
        }
        RowBounds rowBounds=new RowBounds(pageBean.getFirst(),pageBean.getPageSize());

        List<T> list = getSqlSessionTemplate().selectList(statementName, (MapUtil.isNotEmpty(filters)?filters:params), rowBounds);

        PageList<T> pageList=new PageList<T>();
        pageList.addAll(list);
        pageList.setPageBean(pageBean);

        return pageList;
    }


    /**
     * 按Map.xml中的select指定的id获取数据列表
     * @param sqlKey  <a href="#Mapper键值ID">Mapper键值ID</a>
     * @param params 参数类型 可为Object类型，也可为Map类型,若要排序，请在该参数Map里加上两参数key为orderField与orderSeq的值对。
     * @return
     */
    public List<E> getMyBatisBySqlKey(String sqlKey,Object params) throws Exception{
        String statement= getIbatisMapperNamespace() + "." + sqlKey;

        return getSqlSessionTemplate().selectList(statement,params);
    }

    /**
     * 按Map.xml中的select指定的id获取数据列表。
     * 没有参数。
     * @param sqlKey <a href="#Mapper键值ID">Mapper键值ID</a>
     * @return
     */
    public List<E> getMyBatisBySqlKey(String sqlKey) throws Exception{
        String statement= getIbatisMapperNamespace() + "." + sqlKey;

        List<E> list=getSqlSessionTemplate().selectList(statement);

        return list;
    }

    /**
     * 根据主键ID删除记录。
     */
    public int deleteMyBatisById(PK id) throws Exception{
        String delStatement=getIbatisMapperNamespace() + ".delById";
        int affectCount = getSqlSessionTemplate().delete(delStatement, id);
        return affectCount;
    }

    /**
     * 按Map.xml中的delete指定的id删除数据
     * @param sqlKey
     * @param params
     * @return
     */
    public int deleteMyBatisBySqlKey(String sqlKey,Object params) throws Exception{
        String delStatement=getIbatisMapperNamespace() + "."+ sqlKey;
        int affectCount = getSqlSessionTemplate().delete(delStatement, params);
        return affectCount;
    }


    /**
     * 执行新增语句
     */
    public void saveMyBatis(E entity) throws Exception{
        String addStatement=getIbatisMapperNamespace() + ".add";

        if(entity instanceof BaseModel) {
            BaseModel baseModel=((BaseModel) entity);
            if(baseModel.getCreatedAt()==null){
                baseModel.setCreatedAt(SystemClock.nowDate());
            }
        }
        getSqlSessionTemplate().insert(addStatement, entity);
    }

    /**
     * 执行新增语句
     * @param sqlKey 按Map.xml中的update指定的id
     * @param params 传入的参数 ,一般为map,object
     * @return
     */
    public int saveMyBatis(String sqlKey,Object params) throws Exception{
        String addStatement=getIbatisMapperNamespace() + "."+ sqlKey;
        int affectCount = getSqlSessionTemplate().insert(addStatement, params);
        return affectCount;
    }

    /**
     * 更新对象。
     * @return 返回更新的记录数
     */
    @Override
    public int updateMyBatis(E entity) throws Exception{
        String updStatement=getIbatisMapperNamespace() + ".update";
        if(entity instanceof BaseModel) {
            BaseModel baseModel=((BaseModel) entity);
            baseModel.setUpdatedAt(SystemClock.nowDate());
        }
        int affectCount = getSqlSessionTemplate().update(updStatement, entity);
        return affectCount;
    }

    /**
     * 执行更新语句
     * @param sqlKey 按Map.xml中的update指定的id
     * @param params 传入的参数 ,一般为map,object
     * @return 返回更新的记录数
     */
    public int updateMyBatis(String sqlKey,Object params) throws Exception{
        String updStatement=getIbatisMapperNamespace() + "."+ sqlKey;
        int affectCount = getSqlSessionTemplate().update(updStatement, params);
        return affectCount;
    }

    /**
     * 批量新增
     * @param objs
     * @return
     * @throws Exception
     */
    public Object batchSaveMyBatis(List objs)throws Exception{
        String addStatement=getIbatisMapperNamespace() + "."+ "batchSave";
        return getSqlSessionTemplate().insert(addStatement, objs);
    }

    /**
     * 批量新增
     * @param str
     * @param objs
     * @return
     * @throws Exception
     */
    public Object batchSaveMyBatis(String sqlKey, List objs )throws Exception{
        String addStatement=getIbatisMapperNamespace() + "."+ sqlKey;
        return getSqlSessionTemplate().insert(addStatement, objs);
    }

    /**
     * 批量更新
     * @param str
     * @param objs
     * @return
     * @throws Exception
     */
    public void batchUpdateMyBatis(String sqlKey, List objs )throws Exception{
        String addStatement=getIbatisMapperNamespace() + "."+ sqlKey;
        SqlSessionFactory sqlSessionFactory = getSqlSessionTemplate().getSqlSessionFactory();
        //批量执行器
        SqlSession sqlSession = sqlSessionFactory.openSession(ExecutorType.BATCH,false);
        try{
            if(objs!=null){
                for(int i=0,size=objs.size();i<size;i++){
                    sqlSession.update(addStatement, objs.get(i));
                }
                sqlSession.flushStatements();
                sqlSession.commit();
                sqlSession.clearCache();
            }
        }finally{
            sqlSession.close();
        }
    }

    /**
     * 批量删除
     * @param sqlKey
     * @param objs
     * @return
     * @throws Exception
     */
    public Object batchDeleteMyBatis(String sqlKey, List objs )throws Exception{
        String delStatement=getIbatisMapperNamespace() + "."+ sqlKey;
        return getSqlSessionTemplate().delete(delStatement, objs);
    }


    /**
     * 根据mybatis映射文件的ID获取分页数据。
     * @param sqlKey
     * @param params 过滤参数
     * @param pageBean 分页
     *
     */
    public List<E> getMyBatisList(String sqlKey,Object params,PageBean pageBean) throws Exception{

        if(pageBean==null){
            return getMyBatisList(sqlKey, params);
        }
        // 拼接dao层地址
        String statementName = getIbatisMapperNamespace() + "."+ sqlKey;
        Map<String,Object> filters = new HashMap<String, Object>();
        if(params!=null){
            if (params instanceof Map){
                filters.putAll((Map) params); //旧写法
            }else{ // 屏蔽下边写法，是因为影响了扩展基类传递map取值问题
                //Map parameterObject = BeanUtil.describe(params); //旧写法
//                Map parameterObject = BeanMap2Util.map(params,HashMap.class);
//                if(MapUtil.isContainsKey(parameterObject,"params")){
//                    if(MapUtil.isNotEmpty(MapUtil.getMap(parameterObject,"params"))){
//                        filters.putAll(MapUtil.getMap(parameterObject,"params"));
//                    }
//                }
//                filters.putAll(parameterObject);

            }
        }
        if(pageBean.isShowTotal()){
            //获取SQL语句
            IbatisSql ibatisSql=this.getIbatisSql(statementName, (MapUtil.isNotEmpty(filters)?filters:params));
            //使用插件输出sql
            //log.info(ibatisSql.getSql());
            //取得返回数量
            if(pageBean.isDistinct()){//count去重
                if(StrUtil.isEmpty(primaryKey)){
                    int totalCount = getJdbcTemplate().queryForObject(ibatisSql.getCountSqlDistinct(), ibatisSql.getParameters(), Integer.class);
                    pageBean.setTotalCount(totalCount);
                }else {
                    int totalCount = getJdbcTemplate().queryForObject(ibatisSql.getCountSqlDistinct(primaryKey), ibatisSql.getParameters(), Integer.class);
                    pageBean.setTotalCount(totalCount);
                }
            }else { //count不去重
                if(StrUtil.isEmpty(primaryKey)){
                    int totalCount = getJdbcTemplate().queryForObject(ibatisSql.getCountSql(), ibatisSql.getParameters(), Integer.class);
                    pageBean.setTotalCount(totalCount);
                }else {
                    int totalCount = getJdbcTemplate().queryForObject(ibatisSql.getCountSql(primaryKey), ibatisSql.getParameters(), Integer.class);
                    pageBean.setTotalCount(totalCount);
                }
            }
        }
        RowBounds rowBounds=new RowBounds(pageBean.getFirst(),pageBean.getPageSize());

        List<E> list = getSqlSessionTemplate().selectList(statementName, (MapUtil.isNotEmpty(filters)?filters:params), rowBounds);

        PageList<E> pageList=new PageList<E>();
        pageList.addAll(list);
        pageList.setPageBean(pageBean);

        return pageList;
    }

    /**
     * 取得列表数据。
     * @param sqlKey
     * @param params 参数可以是map 或者 pojo对象
     * @return
     */
    public List<E> getMyBatisList(String sqlKey,Object params) throws Exception{

        String statementName = getIbatisMapperNamespace() + "."+ sqlKey;

        Map filters = new HashMap();
        if(params!=null){
            if (params instanceof Map){
                filters.putAll((Map) params); //旧写法
            }else{// 屏蔽下边写法，是因为影响了扩展基类传递map取值问题
                //Map parameterObject = BeanUtil.describe(params); //旧写法
//                Map parameterObject = BeanMap2Util.map(params,HashMap.class);
//                if(MapUtil.isContainsKey(parameterObject,"params")){
//                    if(MapUtil.isNotEmpty(MapUtil.getMap(parameterObject,"params"))){
//                        filters.putAll(MapUtil.getMap(parameterObject,"params"));
//                    }
//                }
//                filters.putAll(parameterObject);

            }
        }

        //获取SQL语句,下面两句pkq加的，为了查看sql语句
        //使用插件输出sql
//        if(log.isDebugEnabled()){
//            IbatisSql ibatisSql=this.getIbatisSql(statementName, filters);
//            log.debug(ibatisSql.getSql());
//        }
        return getSqlSessionTemplate().selectList(statementName, (MapUtil.isNotEmpty(filters)?filters:params));
    }


    /**
     * 获取所有的记录,不分页。
     */
    public List<E> getMyBatisAll() throws Exception{
        String statementName = getIbatisMapperNamespace() + ".getAll";
        return getSqlSessionTemplate().selectList(statementName, null);
    }

    /**
     * 计算记录总数
     *
     * @param sqlKey
     * @param params
     * @return int
     */
    public Integer getMyBatisCount(String sqlKey, Object params) throws Exception{
        try {
            String statementName = getIbatisMapperNamespace() + "." + sqlKey;
            return (Integer) getSqlSessionTemplate().selectOne(statementName, params);
        } catch (Exception e) {
            logger.error(getClass().getName() + " count exception and key is: " + sqlKey, e);
            return Integer.valueOf(0);
        }
    }


    /**
     * 返回数据库连接
     *
     * @return Connection
     */
    public Connection getConnection() throws Exception{
        Connection connection = SqlSessionUtils
                .getSqlSession(getSqlSessionTemplate().getSqlSessionFactory(), getSqlSessionTemplate().getExecutorType(), getSqlSessionTemplate().getPersistenceExceptionTranslator())
                .getConnection();
        return connection;
        // return getSqlSession().getConnection();
    }

}
