package cn.dolphin.core.date;

import cn.dolphin.core.util.StrUtil;
import cn.dolphin.core.util.SystemClock;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;

import java.io.Serializable;
import java.lang.management.ManagementFactory;
import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 时间过去验证工具
 */
@Slf4j
public class TimeUtil implements Serializable {

    public static final String  YYYY = "yyyy";

    public static final String  YYYYMM = "yyyyMM";

    public static final String  YYYYMMDD = "yyyyMMdd";

    public static final String  YYYYMMDDHH = "yyyyMMddHH";

    public static final String  YYYYMMDDHHMMSS = "yyyyMMddHHmmss";

    public static final String  HHMMSS = "HHmmss";

    public static final String  YYYYMMDDHHMMSSSSS = "yyyyMMddHHmmssSSS";

    private static final long ONE_MINUTE = 60000L;
    private static final long ONE_HOUR = 3600000L;
    private static final long ONE_DAY = 86400000L;
    private static final long ONE_WEEK = 604800000L;

    private static final String ONE_SECOND_AGO = "秒前";
    private static final String ONE_MINUTE_AGO = "分钟前";
    private static final String ONE_HOUR_AGO = "小时前";
    private static final String ONE_DAY_AGO = "天前";
    private static final String ONE_MONTH_AGO = "月前";
    private static final String ONE_YEAR_AGO = "年前";

    private static String[] parsePatterns = {
            "yyyy-MM-dd", "yyyy-MM-dd HH:mm:ss", "yyyy-MM-dd HH:mm", "yyyy-MM",
            "yyyy/MM/dd", "yyyy/MM/dd HH:mm:ss", "yyyy/MM/dd HH:mm", "yyyy/MM",
            "yyyy.MM.dd", "yyyy.MM.dd HH:mm:ss", "yyyy.MM.dd HH:mm", "yyyy.MM"};

    /**
     * 获取当前时间
     * @return
     */
    public static Date getTime() {
        return new Date();
    }

    /**
     * 获取unix的long时间
     * 10位时间戳
     * @return
     */
    public static long unixTimestamp () {
        return SystemClock.now()/1000;
    }

    /**
     * Date类型转换为10位时间戳
     * @param time
     * @return
     */
    public static long unixTimestamp(Date time){
        Timestamp ts = new Timestamp(time.getTime());
        return ((ts.getTime())/1000);
    }

    /**
     * 获取unix的long的10位时间戳
     * @param time
     * @return
     */
    public static long unixTimestamp(long time){
        return toSeconds(time);
    }


    /**
     * 获取当前时间时间戳
     */
    public static Long getCurrentUnixTimeStamp(){
        return new DateTime().getMillis()/1000;
    }
    /**
     * 获取指定时间的时间戳
     */
    public static Long  getUnixTimeStamp(Date date){
        return new DateTime(date).getMillis()/1000;
    }

    /**
     * 万能日期转换解析
     * @param stringDate
     * @return
     */
    public static Date parseToDate(String stringDate) {
        if (StrUtil.isBlank(stringDate)) {
            return new Date();
        }
        List<String> patterns = Stream.of(
                "yyyy年MM月dd日",
                "yy-MM",
                "yyyy-MM",
                "yyyy-MM-dd",
                "yyyy/MM/dd",
                "yyyy-MM-dd HH:mm:ss",
                "yyyy-MM-dd HH:mm",
                "yyyy-MM-dd HH"
        ).collect(Collectors.toList());

        for (String pattern : patterns) {
            SimpleDateFormat df = new SimpleDateFormat(pattern);
            try {
                return df.parse(stringDate);
            } catch (Exception ignored) {
            }
        }
        log.error("转换日期发生异常，没有适配的pattern。日期为：{}", stringDate);
        return null;
    }


    /**
     * 获取当天开始时间
     *
     * @return
     */
    public static Date getTodayBegin() {
        Calendar currentDate = Calendar.getInstance();
        currentDate.set(Calendar.HOUR_OF_DAY, 0);
        currentDate.set(Calendar.MINUTE, 0);
        currentDate.set(Calendar.SECOND, 0);
        return currentDate.getTime();
    }


    /**
     * 获取时间戳，乘以1000后转换的时间字符
     * @param date
     * @return
     */
    public static String getDateStamp(long date){
        long times = date*1000L;
        Date d = new Date(times);
        return DateFormatUtil.DATETIME_FORMAT.format(d);
    }

    /**
     * 获取当天结束时间
     *
     * @return
     */
    public static Date getTodayEnd() {
        Calendar currentDate = Calendar.getInstance();
        currentDate.set(Calendar.HOUR_OF_DAY, 23);
        currentDate.set(Calendar.MINUTE, 59);
        currentDate.set(Calendar.SECOND, 59);
        return currentDate.getTime();
    }

    /**
     * 获取当前时间秒级
     * @param date
     * @return
     */
    public static long getSeconds(long date){
        return toSeconds(date);
    }

    /**
     * 间隔毫秒数
     * @param beforeDate
     * @param endDate
     * @return
     */
    public static long intervalMs (Date beforeDate,Date endDate) {
        return endDate.getTime() - beforeDate.getTime();
    }

    /**
     * 日期型字符串转化为日期 格式
     */
    public static Date parseDate(Object str) {
        try{
            if (str == null) {
                return null;
            }
            return DateUtil.parseDate(str.toString(), parsePatterns);
        }catch (ParseException e) {
            return null;
        }
    }


    /**
     * 根据给出的时间限制判断是最近访问时间是否过期，
     * secondLimit小于0则无限期，等于0表示过期
     * @param lastAccessedTime
     * @param secondLimit
     * @return
     */
    public static boolean isOverdueSecond(long lastAccessedTime, long secondLimit) {
        if(secondLimit == 0L) {
            return true;
        }
        if(secondLimit < 0L) {
            return false;
        }
        long interval = SystemClock.now();
        long interval_sec = interval / 1000L;
//		System.out.println("使用时间＝"+interval_sec);
        if(interval_sec > secondLimit) {
            return true;
        }
        return false;
    }

    /**
     * 计算两个时间差
     */
    public static String getDatePoor(Date endDate, Date nowDate)
    {
        long nd = 1000 * 24 * 60 * 60;
        long nh = 1000 * 60 * 60;
        long nm = 1000 * 60;
        // long ns = 1000;
        // 获得两个时间的毫秒时间差异
        long diff = endDate.getTime() - nowDate.getTime();
        // 计算差多少天
        long day = diff / nd;
        // 计算差多少小时
        long hour = diff % nd / nh;
        // 计算差多少分钟
        long min = diff % nd % nh / nm;
        // 计算差多少秒//输出结果
        // long sec = diff % nd % nh % nm / ns;
        return day + "天" + hour + "小时" + min + "分钟";
    }

    /**
     * 获取服务器启动时间
     */
    public static Date getServerStartDate()
    {
        long time = ManagementFactory.getRuntimeMXBean().getStartTime();
        return new Date(time);
    }

    //返回前num天的日期
    public static String getDate(Integer num) {
        if (num != null && num >= 1) {
            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
            return simpleDateFormat.format(new Date(SystemClock.now()- TimeUnit.DAYS.toMillis(num)));
        }
        return null;
    }

    /**
     * 日期格式化
     * @param date
     * @param format
     * @return
     */
    public static String format (Date date,String format) {
        return getSimpleDateFormat(format).format(date);
    }

    /**
     * 获取格式化对象
     * @param strFormat 格式化的格式 如"yyyy-MM-dd"
     * @return 格式化对象
     */
    public static SimpleDateFormat getSimpleDateFormat(String strFormat) {
        if (strFormat != null && !"".equals(strFormat.trim())) {
            return new SimpleDateFormat(strFormat);
        }
        else {
            return new SimpleDateFormat();
        }
    }

    /**
     * unix转Timestamp
     * @param time
     * @return
     */
    public static Timestamp getUnixTimestamp(String time){
        Long timestamp = Long.parseLong(time)*1000;
        String date = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date(timestamp));
        Timestamp ts = new Timestamp(SystemClock.now());
        try {
            ts = Timestamp.valueOf(date);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return ts;
    }

    /**
     * long转Timestamp
     * @param time
     * @return
     */
    public static Timestamp getTimestamp(long time){
        String date = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date(time));
        Timestamp ts = new Timestamp(SystemClock.now());
        try {
            ts = Timestamp.valueOf(date);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return ts;
    }

    /**
     * string转Timestamp
     * @param time
     * @return
     */
    public static Timestamp getTimestamp(String time){
        try {
            Date date = DateUtil.parseDate(time.trim(),parsePatterns);
            return new Timestamp(date.getTime());
        } catch (ParseException e) {
            return null;
        }
    }

    /**
     * 将long转为Timestamp
     */
//    public static Timestamp getTimestamp(long _long) {
//        return new Timestamp(_long);
//    }


    /**
     * 格林时间+8个小时long转Timestamp
     * @param time
     * @return
     */
    public static Timestamp getGMTTimestamp(long time){
        //加8个时区
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(time);
        //calendar.setTime(new Date(time));
        calendar.set(Calendar.HOUR, calendar.get(Calendar.HOUR) + 8);
        //calendar.add(Calendar.HOUR, +8);
        String date = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format( calendar.getTime());
        Timestamp ts = new Timestamp(SystemClock.now());
        try {
            ts = Timestamp.valueOf(date);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return ts;
    }

    /**
     * 格式化成多久以前
     *
     * @param date
     * @return
     */
    public static String formatAgo(Date date) {

        if (date == null) {
            return "未知";
        }

        long curTime = System.currentTimeMillis() - date.getTime();

        if (curTime < 1L * ONE_MINUTE) {
            long seconds = toSeconds(curTime);
            return (seconds <= 0 ? 1 : seconds) + ONE_SECOND_AGO;
        }
        if (curTime < 45L * ONE_MINUTE) {
            long minutes = toMinutes(curTime);
            return (minutes <= 0 ? 1 : minutes) + ONE_MINUTE_AGO;
        }
        if (curTime < 24L * ONE_HOUR) {
            long hours = toHours(curTime);
            return (hours <= 0 ? 1 : hours) + ONE_HOUR_AGO;
        }
        if (curTime < 48L * ONE_HOUR) {
            return "昨天";
        }
        if (curTime < 30L * ONE_DAY) {
            long days = toDays(curTime);
            return (days <= 0 ? 1 : days) + ONE_DAY_AGO;
        }
        if (curTime < 12L * 4L * ONE_WEEK) {
            long months = toMonths(curTime);
            return (months <= 0 ? 1 : months) + ONE_MONTH_AGO;
        } else {
            long years = toYears(curTime);
            return (years <= 0 ? 1 : years) + ONE_YEAR_AGO;
        }
    }

    /**
     * 比较日期大小 yyyyMMdd
     * @param date1 日期1
     * @param date2 日期2
     * @return
     */
    public static int dateCompare(Date date1, Date date2) {
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd");
        String dateFirst = dateFormat.format(date1);
        String dateLast = dateFormat.format(date2);
        int dateFirstIntVal = Integer.parseInt(dateFirst);
        int dateLastIntVal = Integer.parseInt(dateLast);
        if (dateFirstIntVal > dateLastIntVal) {
            return 1;
        } else if (dateFirstIntVal < dateLastIntVal) {
            return -1;
        }
        return 0;
    }

    /**
     * 获取时间差
     * @param beginTime	开始时间
     * @param endTime	结束时间
     * @param type		时间的类型[0秒/1分钟/2小时]
     * @return
     */
    public static Long getDateDiff(Date beginTime, Date endTime, Integer type) {
        long time = 0;
        long diffTime = endTime.getTime() - beginTime.getTime();
        switch (type) {
            case 0:
                time = diffTime/1000;
                break;
            case 1:
                time = diffTime/(60*1000);
                break;
            case 2:
                time = diffTime/(60*60*1000);
                break;
            default:
                break;
        }
        return time;
    }


    /**
     * 在指定日期上添加指定的年
     * @param date		：传入的日期
     * @param amount 	：添加的年
     * @return
     */
    public static Date addYears(Date date, Integer amount) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.YEAR, amount);
        return calendar.getTime();
    }

    /**
     * 在指定日期上添加指定的月份
     * @param date		：传入的日期
     * @param amount 	：添加的月份
     * @return
     */
    public static Date addMonths(Date date, Integer amount) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.MONTH, amount);
        return calendar.getTime();
    }

    /**
     * 在指定日期上添加指定的天数
     * @param date		：传入的日期
     * @param amount 	：添加的天数
     * @return
     */
    public static Date addDays(Date date, Integer amount) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.DAY_OF_YEAR, amount);
        return calendar.getTime();
    }

    /**
     * 在指定日期上添加指定的小时
     * @param date		：传入的日期
     * @param amount 	：添加的小时
     * @return
     */
    public static Date addHours(Date date, Integer amount) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.HOUR, amount);
        return calendar.getTime();
    }

    /**
     * 在指定日期上添加指定的分钟
     * @param date		：传入的日期
     * @param amount 	：添加的分钟
     * @return
     */
    public static Date addMinutes(Date date, Integer amount) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.MINUTE, amount);
        return calendar.getTime();
    }

    /**
     * 在指定日期上添加指定的秒数
     * @param date		：传入的日期
     * @param amount 	：添加的秒数
     * @return
     */
    public static Date addSeconds(Date date, Integer amount){
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.SECOND, amount);
        return calendar.getTime();
    }


    private static long toSeconds(long date) {
        return date / 1000L;
    }

    private static long toMinutes(long date) {
        return toSeconds(date) / 60L;
    }

    private static long toHours(long date) {
        return toMinutes(date) / 60L;
    }

    private static long toDays(long date) {
        return toHours(date) / 24L;
    }

    private static long toMonths(long date) {
        return toDays(date) / 30L;
    }

    private static long toYears(long date) {
        return toMonths(date) / 365L;
    }

}
