package cn.dolphin.core.date;

import cn.dolphin.core.consts.DateConstant;
import cn.dolphin.core.util.ConvertUtil;

import java.time.*;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAdjusters;
import java.util.Date;


/**
 * JDK8时间 的处理工具
 */
public final class LocalDateUtil {

    private static DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    /**
     * 获取瞬时实例(代表某个的时间，可以精确到纳秒)
     *
     * @param dateTime
     * @return
     *
     */
    public static Instant getInstant(LocalDateTime dateTime){
        // LocalDateTime的系统时区的实例
        return dateTime.atZone(DateConstant.ZONE_ID).toInstant();
    }

    /**
     * 获取瞬时实例
     *
     * @param timestamp	时间戳
     * @return
     *
     */
    public static Instant getInstant(long timestamp){
        return Instant.ofEpochMilli(timestamp);
    }

    /**
     * 获取瞬时实例
     *
     * @param date
     * @return
     *
     */
    public static Instant getInstant(Date date){
        return date.toInstant();
    }

    /**
     * 获取瞬时实例(日起始)
     *
     * @param date
     * @return
     *
     */
    public static Instant getInstant(LocalDate date){
        return date.atStartOfDay().atZone(DateConstant.ZONE_ID).toInstant();
    }

    /**
     * 获取时间之差
     *
     * 注：可以通过 Duration的toDays()等获取具体的时间单位之差
     *
     * @param startTime
     * @param endTime
     * @return
     *
     */
    public static Duration between(LocalDateTime startTime, LocalDateTime endTime){
        return Duration.between(startTime,endTime);
    }

    /**
     * 获取日期之差
     *
     * 注：可以通过 Period的getDays()等获取具体的时间单位之差
     *
     * @param startTime
     * @param endTime
     * @return
     *
     */
    public static Period between(LocalDate startTime, LocalDate endTime){
        return Period.between(startTime,endTime);
    }

    /**
     * 获取时间之差
     *
     * 注：可以通过 Duration的toDays()等获取具体的时间单位之差
     *
     * @param startTime
     * @param endTime
     * @return
     *
     */
    public static Duration between(LocalTime startTime, LocalTime endTime){
        return Duration.between(startTime,endTime);
    }

    /**
     * 字符串转换为日期
     *
     * @param date 	格式：yyyy-MM-dd HH:mm:ss
     * @return
     *
     */
    public static LocalDateTime toLocalDateTime(String date){
        return LocalDateTime.parse(date,DateConstant.DATE_TIME_FROMAT);
    }

    /**
     * 字符串转换为日期
     *
     * @param date	格式：yyyyMMddHHmmss
     * @return
     *
     */
    public static LocalDateTime toLocalDateTimeNum(String date){
        return LocalDateTime.parse(date,DateConstant.DATE_TIME_FROMAT_NUM);
    }

    /**
     * 字符串转换为日期
     *
     * @param date	格式：yyyy-MM-dd
     * @return
     *
     */
    public static LocalDate toLocalDate(String date){
        return LocalDate.parse(date,DateConstant.DATE_FROMAT);
    }

    /**
     * 字符串转换为日期
     *
     * @param date	格式：yyyyMMdd
     * @return
     *
     */
    public static LocalDate toLocalDateNum(String date){
        return LocalDate.parse(date,DateConstant.DATE_FROMAT_NUM);
    }

    /**
     * 日期转字符串
     *
     * @param date
     * @return	格式：yyyy-MM-dd HH:mm:ss
     *
     */
    public static String toStr(LocalDateTime date){
        return date.format(DateConstant.DATE_TIME_FROMAT);
    }

    /**
     * 日期转字符串
     *
     * @param date
     * @return	格式：yyyy-MM-dd
     *
     */
    public static String toStr(LocalDate date){
        return date.format(DateConstant.DATE_FROMAT);
    }

    /**
     * 日期转字符串
     *
     * @param date
     * @return	格式：HH:mm:ss
     *
     */
    public static String toStr(LocalTime date){
        return date.format(DateConstant.TIME_FROMAT);
    }

    /**
     * 当前时间转字符串
     *
     * @return	格式：yyyy-MM-dd HH:mm:ss
     *
     */
    public static String toStr(){
        return LocalDateTime.now().format(DateConstant.DATE_TIME_FROMAT);
    }

    /**
     * 当前时间转字符串
     *
     * @return	格式：yyyy-MM-dd
     *
     */
    public static String toDateStr(){
        return LocalDate.now().format(DateConstant.DATE_FROMAT);
    }

    /**
     * 当前时间转字符串
     *
     * @return	格式：HH:mm:ss
     *
     */
    public static String toTimeStr(){
        return LocalTime.now().format(DateConstant.TIME_FROMAT);
    }

    /**
     * 日期转字符串
     *
     * @param date
     * @return	格式：yyyyMMddHHmmss
     *
     */
    public static String toNum(LocalDateTime date){
        return date.format(DateConstant.DATE_TIME_FROMAT_NUM);
    }

    /**
     * 日期转字符串
     *
     * @param date
     * @return	格式：yyyyMMdd
     *
     */
    public static String toNum(LocalDate date){
        return date.format(DateConstant.DATE_FROMAT_NUM);
    }

    /**
     * 日期转字符串
     *
     * @param date
     * @return	格式：HHmmss
     *
     */
    public static String toNum(LocalTime date){
        return date.format(DateConstant.TIME_FROMAT_NUM);
    }

    /**
     * 当前时间转字符串
     *
     * @return	格式：yyyyMMddHHmmss
     *
     */
    public static String toNum(){
        return LocalDateTime.now().format(DateConstant.DATE_TIME_FROMAT_NUM);
    }

    /**
     * 当前时间转字符串
     *
     * @return	格式：yyyyMMdd
     *
     */
    public static String toDateNum(){
        return LocalDate.now().format(DateConstant.DATE_FROMAT_NUM);
    }

    /**
     * 当前时间转字符串
     *
     * @return	格式：HHmmss
     *
     */
    public static String toTimeNum(){
        return LocalTime.now().format(DateConstant.TIME_FROMAT_NUM);
    }

    /**
     * 获取当前时间戳
     *
     * @return
     *
     */
    public static long getTime(){
        return System.currentTimeMillis();
    }

    /**
     * 获取时间戳
     *
     * @param dateTime
     * @return
     *
     */
    public static long getTime(LocalDateTime dateTime){
        return getInstant(dateTime).toEpochMilli();
    }

    /**
     * 获取时间戳
     *
     * @param date	当天0点
     * @return
     *
     */
    public static long getTime(LocalDate date){
        return getTime(date.atStartOfDay());
    }

    /**
     * LocalDateTime 转换 Date
     *
     * @param dateTime
     * @return
     *
     */
    public static Date toDate(LocalDateTime dateTime){
        return Date.from(getInstant(dateTime));
    }
    /**
     * LocalDate转换 Date
     *
     * @param date
     * @return
     *
     */
    public static Date toDate(LocalDate date){
        return Date.from(getInstant(date));
    }

    /**
     * Date 转换 LocalDateTime
     *
     * @param date
     * @return
     *
     */
    public static LocalDateTime toLocalDateTime(Date date){
        return LocalDateTime.ofInstant(getInstant(date),DateConstant.ZONE_ID);
    }

    /**
     * 时间戳 转换 LocalDateTime
     *
     * @param timestamp
     * @return
     *
     */
    public static LocalDateTime toLocalDateTime(Long timestamp){
        return LocalDateTime.ofInstant(getInstant(timestamp),DateConstant.ZONE_ID);
    }

    /**
     * Date 转换 LocalDate
     *
     * @param date
     * @return
     *
     */
    public static LocalDate toLocalDate(Date date){
        return toLocalDateTime(date).toLocalDate();
    }

    /**
     * 时间戳 转换 LocalDate
     *
     * @param timestamp
     * @return
     *
     */
    public static LocalDate toLocalDate(Long timestamp){
        return toLocalDateTime(timestamp).toLocalDate();
    }

    /**
     * Date 转换 LocalTime
     *
     * @param date
     * @return
     *
     */
    public static LocalTime toLocalTime(Date date){
        return toLocalDateTime(date).toLocalTime();
    }

    /**
     * 时间戳 转换 LocalTime
     *
     * @param timestamp
     * @return
     *
     */
    public static LocalTime toLocalTime(Long timestamp){
        return toLocalDateTime(timestamp).toLocalTime();
    }

    /**
     * 获取当前时间
     * @return
     */
    public static String nowTimeStr() {
        return formatter.format(LocalDateTime.now().withNano(0));
    }

    /**
     * 获取当前时间
     * @return
     */
    public static Date nowTime() {
        return localDateTimeToDate(LocalDateTime.now());
    }

    public static String dayStartTimeStr(int days) {
        return formatter.format(LocalDate.now().minusDays(days).atTime(0, 0, 0).withNano(0));
    }

    public static Date dayStartTime(int days) {
        return localDateTimeToDate(LocalDate.now().minusDays(days).atTime(0, 0, 0).withNano(0));
    }

    public static String dayEndTimeStr(int days) {
        return formatter.format(LocalDate.now().minusDays(days).atTime(23, 59, 59).withNano(0));
    }

    public static Date dayEndTime(int days) {
        return localDateTimeToDate(LocalDate.now().minusDays(days).atTime(23, 59, 59).withNano(0));
    }

    /**
     * 当天开始时间
     * @return
     */
    public static String dayStartTimeStr() {
        return formatter.format(LocalDate.now().atTime(0, 0, 0).withNano(0));
    }

    /**
     * 当天开始时间
     * @return
     */
    public static Date dayStartTime() {
        return localDateTimeToDate(LocalDate.now().atTime(0, 0, 0).withNano(0));
    }

    /**
     * 当天结束时间
     * @return
     */
    public static String dayEndTimeStr() {
        return formatter.format(LocalDate.now().atTime(23, 59, 59).withNano(0));
    }

    /**
     * 当天结束时间
     * @return
     */
    public static Date dayEndTime() {
        return localDateTimeToDate(LocalDate.now().atTime(23, 59, 59).withNano(0));
    }

    /**
     * 昨天开始时间
     * @return
     */
    public static String lastDayStartTimeStr() {
        return formatter.format(LocalDate.now().atTime(0, 0, 0).minusDays(1L).withNano(0));
    }

    /**
     * 昨天开始时间
     * @return
     */
    public static Date lastDayStartTime() {
        return localDateTimeToDate(LocalDate.now().atTime(0, 0, 0).minusDays(1L).withNano(0));
    }

    /**
     * 昨天结束时间
     * @return
     */
    public static String lastDayEndTimeStr() {
        return formatter.format(LocalDate.now().atTime(23, 59, 59).minusDays(1L).withNano(0));
    }

    /**
     * 昨天结束时间
     * @return
     */
    public static Date lastDayEndTime() {
        return localDateTimeToDate(LocalDate.now().atTime(23, 59, 59).minusDays(1L).withNano(0));
    }

    /**
     * 本周开始时间
     * @return
     */
    public static String weekStartTimeStr() {
        LocalDate date = LocalDate.now();
        int week = date.getDayOfWeek().getValue();
        return formatter.format(date.atTime(0, 0, 0).minusDays(week - 1).withNano(0));
    }

    /**
     * 本周开始时间
     * @return
     */
    public static Date weekStartTime() {
        LocalDate date = LocalDate.now();
        int week = date.getDayOfWeek().getValue();
        return localDateTimeToDate(date.atTime(0, 0, 0).minusDays(week - 1).withNano(0));
    }

    /**
     * 本周结束时间
     * @return
     */
    public static String weekEndTimeStr() {
        LocalDate date = LocalDate.now();
        int week = date.getDayOfWeek().getValue();
        return formatter.format(date.atTime(23, 59, 59).plusDays(7 - week).withNano(0));
    }

    /**
     * 本周结束时间
     * @return
     */
    public static Date weekEndTime() {
        LocalDate date = LocalDate.now();
        int week = date.getDayOfWeek().getValue();
        return localDateTimeToDate(date.atTime(23, 59, 59).plusDays(7 - week).withNano(0));
    }

    /**
     * 上周开始时间
     * @return
     */
    public static String lastWeekStartTimeStr() {
        LocalDate date = LocalDate.now();
        int week = date.getDayOfWeek().getValue();
        return formatter.format(date.atTime(0, 0, 0).minusWeeks(1L).minusDays(week - 1).withNano(0));
    }

    /**
     * 上周开始时间
     * @return
     */
    public static Date lastWeekStartTime() {
        LocalDate date = LocalDate.now();
        int week = date.getDayOfWeek().getValue();
        return localDateTimeToDate(date.atTime(0, 0, 0).minusWeeks(1L).minusDays(week - 1).withNano(0));
    }

    /**
     * 上周结束时间
     * @return
     */
    public static String lastWeekEndTimeStr() {
        LocalDate date = LocalDate.now();
        int week = date.getDayOfWeek().getValue();
        return formatter.format(date.atTime(23, 59, 59).minusWeeks(1L).plusDays(7 - week).withNano(0));
    }

    /**
     * 上周结束时间
     * @return
     */
    public static Date lastWeekEndTime() {
        LocalDate date = LocalDate.now();
        int week = date.getDayOfWeek().getValue();
        return localDateTimeToDate(date.atTime(23, 59, 59).minusWeeks(1L).plusDays(7 - week).withNano(0));
    }

    /**
     * 本月开始时间
     * @return
     */
    public static String monthStartTimeStr() {
        return formatter.format(LocalDate.now().atTime(0, 0, 0).with(TemporalAdjusters.firstDayOfMonth()).withNano(0));
    }

    /**
     * 本月开始时间
     * @return
     */
    public static Date monthStartTime() {
        return localDateTimeToDate(LocalDate.now().atTime(0, 0, 0).with(TemporalAdjusters.firstDayOfMonth()).withNano(0));
    }

    /**
     * 本月结束时间
     * @return
     */
    public static String monthEndTimeStr() {
        return formatter.format(LocalDate.now().atTime(23, 59, 59).with(TemporalAdjusters.lastDayOfMonth()).withNano(0));
    }

    /**
     * 本月结束时间
     * @return
     */
    public static Date monthEndTime() {
        return localDateTimeToDate(LocalDate.now().atTime(23, 59, 59).with(TemporalAdjusters.lastDayOfMonth()).withNano(0));
    }

    /**
     * 上月开始时间
     * @return
     */
    public static String lastMonthStartTimeStr() {
        return formatter.format(LocalDate.now().atTime(0, 0, 0).minusMonths(1L).with(TemporalAdjusters.firstDayOfMonth()).withNano(0));
    }

    /**
     * 上月开始时间
     * @return
     */
    public static Date lastMonthStartTime() {
        return localDateTimeToDate(LocalDate.now().atTime(0, 0, 0).minusMonths(1L).with(TemporalAdjusters.firstDayOfMonth()).withNano(0));
    }

    /**
     * 上月结束时间
     * @return
     */
    public static String lastMonthEndTimeStr() {
        return formatter.format(LocalDate.now().atTime(23, 59, 59).minusMonths(1L).with(TemporalAdjusters.lastDayOfMonth()).withNano(0));
    }

    /**
     * 上月结束时间
     * @return
     */
    public static Date lastMonthEndTime() {
        return localDateTimeToDate(LocalDate.now().atTime(23, 59, 59).minusMonths(1L).with(TemporalAdjusters.lastDayOfMonth()).withNano(0));
    }

    public static Date localDateToDate(LocalDate date) {
        ZoneId zoneId = ZoneId.systemDefault();
        ZonedDateTime zdt = date.atStartOfDay(zoneId);
        return Date.from(zdt.toInstant());
    }

    public static Date localDateTimeToDate(LocalDateTime date) {
        ZoneId zoneId = ZoneId.systemDefault();
        Instant instant = date.atZone(zoneId).toInstant();
        return Date.from(instant);
    }

    public static int get2ZeroHour() {
        long nowTime = nowTime().getTime();
        long endTime = dayEndTime().getTime();
        long currentTime = endTime - nowTime;
        return ConvertUtil.toInteger(currentTime / 1000);
    }

    public static void main(String[] args) {
        System.out.println("获取当前时间:" + nowTimeStr());
        System.out.println("获取当前时间:" + nowTime().getTime());
        System.out.println("当天开始时间:" + dayStartTimeStr());
        System.out.println("当天开始时间:" + dayStartTime());
        System.out.println("当天结束时间:" + dayEndTimeStr());
        System.out.println("当天结束时间:" + dayEndTime());
        System.out.println("昨天开始时间:" + lastDayStartTimeStr());
        System.out.println("昨天开始时间" + lastDayStartTime());
        System.out.println("昨天结束时间" + lastDayEndTimeStr());
        System.out.println("昨天结束时间" + lastDayEndTime());
        System.out.println("本周开始时间" + weekStartTimeStr());
        System.out.println("本周开始时间" + weekStartTime());
        System.out.println("本周结束时间" + weekEndTimeStr());
        System.out.println("本周结束时间" + weekEndTime());
        System.out.println("上周开始时间" + lastWeekStartTimeStr());
        System.out.println("上周开始时间" + lastWeekStartTime());
        System.out.println("上周结束时间" + lastWeekEndTimeStr());
        System.out.println("上周结束时间" + lastWeekEndTime());
        System.out.println("本月开始时间" + monthStartTimeStr());
        System.out.println("本月开始时间" + monthStartTime());
        System.out.println("本月结束时间" + monthEndTimeStr());
        System.out.println("本月结束时间" + monthEndTime());
        System.out.println("上月开始时间" + lastMonthStartTimeStr());
        System.out.println("上月开始时间" + lastMonthStartTime());
        System.out.println("上月结束时间" + lastMonthEndTimeStr());
        System.out.println("上月结束时间" + lastMonthEndTime());
    }


}
