package cn.dolphin.core.date;


import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.joda.time.*;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import java.text.ParseException;
import java.util.*;


/**
 * JodaTime的封装，线程安全，比系统的好用
 */
@SuppressWarnings("all")
public class JodaUtil {


    /**
     * "yyyy-MM-dd格式类型
     * 2017-05-19 这样的固定日期格式
     */
    public static final String DATE_FORMAT = "yyyy-MM-dd";

    /**
     * "yyyy-MM-dd HH:mm:ss"格式类型
     * 2017-05-19 12:23:33 这样的固定时间格式
     */
    public static final String TIME_FORMAT = "yyyy-MM-dd HH:mm:ss";

    /**
     * 常用日期和时间格式
     */
    public final static String[] parsePatterns = {
            "yyyy-MM-dd", "yyyy-MM-dd HH:mm:ss", "yyyy-MM-dd HH:mm", "yyyy-MM",
            "yyyy/MM/dd", "yyyy/MM/dd HH:mm:ss", "yyyy/MM/dd HH:mm", "yyyy/MM",
            "yyyy.MM.dd", "yyyy.MM.dd HH:mm:ss", "yyyy.MM.dd HH:mm", "yyyy.MM",
            "yyyyMMddHHmmss", "yyyyMMdd", "yyyy-MM-dd'T'HH:mm:ss.SSS", "yyyy-MM-dd'T'HH:mm:ss:SSSZZ",
            "yyyy-MM-dd'T'HH:mm:ss.SSSX", "yyyy-MM-dd'T'HH:mm:ss.SSS Z", "dd MMM yyyy"};

    /**
     * 返回当前日期 yyyy-MM-dd格式
     *
     * @return
     */
    public static String getDate() {
        return LocalDate.now().toString(DATE_FORMAT);
    }

    /**
     * 获取当前系统时间
     * 格式 :yyyy-MM-dd HH:mm:ss
     * @return
     */
    public static String getCurSysTime() {
        // 初始化时间
        DateTime dateTime = new DateTime();
        return dateTime.toString(TIME_FORMAT);
    }

    /**
     * 解析Unix时间戳
     *
     * @param unixTimeStamp
     * @return
     */
    public static DateTime getUnixTimeStamp(Long unixTimeStamp) {

        if (String.valueOf(unixTimeStamp).length() > 10) {
            unixTimeStamp = unixTimeStamp / 1000;

        }
        try {
            return new DateTime(unixTimeStamp * 1000L);
        } catch (Exception ex) {
            return null;
        }
    }

    /**
     * 把字符串解析为日期格式，兼容各种格式
     *
     * @param input
     * @return
     */
    public static String parseStrToDate(String input) {
        DateTime dateTime = parseStr(input);
        if (null != dateTime) {
            return dateTime.toString(DATE_FORMAT);
        }
        return null;
    }

    /**
     * 使用常规格式解析日期和时间
     * 兼容基本的格式。但是如果超过1980年的日期，会做null处理
     *
     * @param input
     * @return
     */
    public static DateTime parseStr(String input) {
        DateTime dateTime = parseStr(input, parsePatterns);
        if (null == dateTime || dateTime.getYear() < 1980) {
            dateTime = getUnixTimeStamp(NumberUtils.toLong(input));
        }

        // 如果最后解析出来还是1900年之前的话，直接干掉不要，应该是Unix解析出错了
        if (null != dateTime && dateTime.getYear() < 1980) {
            dateTime = null;
        }

        return dateTime;
    }

    /**
     * 把字符串按照多种形式进行解析，并转换为Joda的时间格式
     *
     * @param input         日期字符串
     * @param parsePatterns 多种不同的日期格式
     * @return
     */
    public static DateTime parseStr(String input, String[] parsePatterns) {

        DateTime dateTime = null;
        try {
            Date date = DateUtils.parseDate(input, parsePatterns);
            dateTime = new DateTime(date);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return dateTime;
    }

    /**
     * 获取某时间段内所有周的列表（开始时间，结束时间）
     * @param startDate
     * @param endDate
     * @return
     */
    public static List<String[]> getWeekList(Date startDate, Date endDate){
        List<String[]> weekList = new ArrayList<>();
        //转换成joda-time的对象
        DateTime firstDay = new DateTime(startDate).dayOfWeek().withMinimumValue();
        DateTime lastDay = new DateTime(endDate).dayOfWeek().withMaximumValue();
        //计算两日期间的区间天数
        Period p = new Period(firstDay, lastDay, PeriodType.days());
        int days = p.getDays();
        if (days > 0){
            int weekLength = 7;
            for(int i=0;i<days;i=i+weekLength){
                String monDay = firstDay.plusDays(i).toString(DATE_FORMAT);
                String sunDay = firstDay.plusDays(i+6).toString(DATE_FORMAT);
                String [] week = {monDay,sunDay};
                weekList.add(week);
            }
        }
        return weekList;
    }


    /**
     * 获取某时间段内所有一周的时间列表（开始时间）
     * 时间格式：yyyy-MM-dd
     * @param startDate
     * @param isOrder true=asc false =desc
     * @return
     */
    public static List<String> getWeekList(Date startDate, int days ,boolean isOrder){
        List<String> weekList = new ArrayList<>();
        for(int i=0;i<days;i++){
            //转换成joda-time的对象
            DateTime sunDay = new DateTime(startDate).plusDays(-i);
            weekList.add(sunDay.toString(DATE_FORMAT));
        }

        Collections.sort(weekList, new Comparator<String>() {
            @Override
            public int compare(String o1, String o2) {
                if(isOrder){
                    return o1.compareTo(o2);
                }else{
                    return o2.compareTo(o1);
                }
            }
        });
        return weekList;
    }



    /**
     * 获取UTC时间
     * @return UTC时间
     */
    public static DateTime getUTCTime() {
        final Calendar cal = Calendar.getInstance();
        final int zoneOffset = cal.get(Calendar.ZONE_OFFSET);
        final int dstOffset = cal.get(Calendar.DST_OFFSET);
        cal.add(Calendar.MILLISECOND, -(zoneOffset + dstOffset));
        return new DateTime(cal);
    }


    /**
     * 判断两个日期是否在同一天
     * @param date
     * @param anotherDate
     * @return
     */
    public static boolean isSameDay(String date, String anotherDate) {

        boolean res = false;
        // 初始化时间
        DateTime dateTime = new DateTime();

        DateTime dt1 = new DateTime(date);
        DateTime dt2 = new DateTime(anotherDate);

        int intervalDays = Days.daysBetween(dt1, dt2).getDays();

        if (intervalDays == 0) {
            res = true;
        }

        return res;

    }

    /**
     * 计算两个时间相差多少天
     * @param startDate
     * @param endDate
     * @return
     */
    public static Integer diffDay(Date startDate, Date endDate) {
        if (startDate == null || endDate == null) {
            return null;
        }
        DateTime dt1 = new DateTime(startDate);
        DateTime dt2 = new DateTime(endDate);
        int day = Days.daysBetween(dt1, dt2).getDays();
        return Math.abs(day);
    }

    /**
     * 获取每天的零点
     * @return 2016-09-09 00:00:00
     */
    public static String day00(){
        DateTime dt=new DateTime().withMillisOfDay(0);
        return dt.toString("yyyy-MM-dd HH:mm:ss");
    }

    /**
     * 获取指定时间之后或者之前的某一天00:00:00 默认返回当天
     * @param days
     * @param date 2018-09-09
     * @param zimeZone 时区 GMT+8
     * @return
     */
    public static Date day00(Integer days, String date, String zimeZone){
        DateTime dt;
        TimeZone timeZone;
        try {
            if (StringUtils.isBlank(zimeZone)) {
                timeZone = TimeZone.getDefault();
            } else {
                timeZone = TimeZone.getTimeZone(zimeZone);
            }
            if (StringUtils.isBlank(date)) {
                dt = new DateTime().withZone(DateTimeZone.forTimeZone(timeZone)).toLocalDateTime().toDateTime();
            } else {
                dt = new DateTime(date).withZone(DateTimeZone.forTimeZone(timeZone)).toLocalDateTime().toDateTime();
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

        DateTime y = dt.minusDays(days).withHourOfDay(0).withMinuteOfHour(0).withSecondOfMinute(0);
        return y.toDate();
    }

    /**
     *获取指定时间之后或者之前的某一天23:59:59 默认返回当天
     * @param days 偏移量
     * @param date 2018-09-09
     * @param zimeZone 时区 GMT+8
     * @return
     */
    public static Date day59(Integer days, String date, String zimeZone) {
        DateTime dt;
        TimeZone timeZone;
        try {
            if (StringUtils.isBlank(zimeZone)) {
                timeZone = TimeZone.getDefault();
            } else {
                timeZone = TimeZone.getTimeZone(zimeZone);
            }
            if (StringUtils.isBlank(date)) {

                dt = new DateTime().withZone(DateTimeZone.forTimeZone(timeZone)).toLocalDateTime().toDateTime();
            } else {
                dt = new DateTime(date).withZone(DateTimeZone.forTimeZone(timeZone)).toLocalDateTime().toDateTime();
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        DateTime y = dt.minusDays(days).withHourOfDay(23).withMinuteOfHour(59).withSecondOfMinute(59);
        return y.toDate();
    }




    /**
     * 取2个时间范围内的每一天
     * @param start
     * @param end
     * @return
     */
    public static List<LocalDate> getDateBetween(String start, String end) {

        List<LocalDate> listDates = Lists.newArrayList();

        DateTime startTime = parseStr(start);
        DateTime endTime = parseStr(end);

        Integer betweenDays = Days.daysBetween(startTime, endTime).getDays();
        if (betweenDays > 0) {
            for (int i = 0; i <= betweenDays; i++) {
                listDates.add(startTime.plusDays(i).toLocalDate());
            }
        }
        return listDates;
    }

    /**
     * 取2个时间范围内的每一天
     * 格式：yyyy-MM-dd
     * @param start
     * @param end
     * @param isOrder true=asc false =desc
     * @return
     */
    public static List<String> getDateStrBetween(String start, String end,boolean isOrder) {

        List<String> listDates = Lists.newArrayList();

        DateTime startTime = parseStr(start);
        DateTime endTime = parseStr(end);

        Integer betweenDays = Days.daysBetween(startTime, endTime).getDays();
        if (betweenDays > 0) {
            for (int i = 0; i <= betweenDays; i++) {
                listDates.add(startTime.plusDays(i).toString(DATE_FORMAT));
            }
        }

        Collections.sort(listDates, new Comparator<String>() {
            @Override
            public int compare(String o1, String o2) {
                if(isOrder){
                    return o1.compareTo(o2);
                }else{
                    return o2.compareTo(o1);
                }
            }
        });

        return listDates;
    }

    /**
     * 将日期从fromPattern转换为toPattern
     *
     * @param dateStr     日期字符串
     * @param fromPattern 原始格式，比如yyyy-MM-dd HH:mm:ss
     * @param toPattern   返回格式，比如 yyyy-MM-dd
     * @return
     */
    public static String formatFrom(String dateStr, String fromPattern, String toPattern) {
        DateTimeFormatter dateTimeFormatter = DateTimeFormat.forPattern(fromPattern);
        DateTime localTime = dateTimeFormatter.parseDateTime(dateStr);
        return localTime.toString(toPattern);
    }

    /**
     * 解析日期 yyyy-MM-dd HH:mm:ss
     * @param timestamp
     * @return
     */
    public static String format(Long timestamp) {
        String dateStr = "";
        if (null == timestamp || timestamp.longValue() < 0) {
            return dateStr;
        }
        try {
            DateTimeFormatter dateTimeFormatter = DateTimeFormat.forPattern(TIME_FORMAT);
            dateStr = dateTimeFormatter.print(timestamp);
        } catch (Exception e) {
            // ignore
        }
        return dateStr;
    }






}
