package cn.dolphin.core.date;

import cn.dolphin.core.enums.DateEnum;
import cn.dolphin.core.exception.UtilsRuntimeException;
import cn.dolphin.core.util.CastUtil;
import cn.dolphin.core.util.StrUtil;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * 时间工具
 *
 * 本类将逐渐废弃，在更新版本时会大量删减，建议转用LocalDateUtil
 *
 */
@SuppressWarnings("all")
public class DateUtil extends org.apache.commons.lang3.time.DateUtils{

    private static final Calendar calendar = Calendar.getInstance();
    /**
     * 构造函数禁止new
     */
    private DateUtil() {
    }

    //获取指定时间的那天 00:00:00.000 的时间
    public static Date dayBegin(final Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        c.set(Calendar.HOUR_OF_DAY, 0);
        c.set(Calendar.MINUTE, 0);
        c.set(Calendar.SECOND, 0);
        c.set(Calendar.MILLISECOND, 0);
        return c.getTime();
    }

    //获取今天 00:00:00.000 的时间
    public static Date dayBegin() {
        return dayBegin(day());
    }

    //获取指定时间的那天 23:59:59.999 的时间
    public static Date dayEnd(final Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        c.set(Calendar.HOUR_OF_DAY, 23);
        c.set(Calendar.MINUTE, 59);
        c.set(Calendar.SECOND, 59);
        c.set(Calendar.MILLISECOND, 999);
        return c.getTime();
    }

    /**
     * 获取今天 23:59:59.999 的时间
     *
     * @param date
     * @return
     */
    public static Date dayEnd() {
        return dayEnd(day());
    }

    /**
     * 获取当前时间
     *
     * @return
     */
    public static Date day() {
        return new Date();
    }

    /**
     * 功能描述：返回毫
     *
     * @param date
     *            日期
     * @return 返回毫
     */
    public static long getMillis(Date date) {
        calendar.setTime(date);
        return calendar.getTimeInMillis();
    }

    /**新建时间对象**/
    public static DateTime create(){
        return DateTime.now();
    }

    /**注意:timestamp为13位,否则新建出来的日期是1970年1月18号**/
    public static DateTime create(long timestamp){
        // 解决10位时间戳
        String milStr = Long.toString(timestamp);
        if (milStr.length() == 10) {
            timestamp = timestamp * 1000;
        }
        return new DateTime(timestamp);
    }

    /**
     * 根据指定格式获取当前时间
     * @param format
     * @return String
     */
    public static String getCurrentTime(String format){
        SimpleDateFormat sdf = DateFormatUtil.getFormat(format);
        Date date = new Date();
        return sdf.format(date);
    }

    /**
     * 基本功能:获取当前时间
     * 一.日期的格式转换
     * 1.日期对象转字符串 {@link #format(DateTime)}
     * 2.日期对象转时间戳 {@link #create().getMillis() }
     * 3.时间戳转日期对象 {@link #create(long timestamp) }
     * 4.时间戳转字符串 {@link #create(long timestamp)} --> {@link #format(DateTime)}
     */

    /**默认日期转换格式:yyyy-MM-dd HH:mm:ss**/
    public static String format(DateTime dateTime){
        return format(dateTime, DateEnum.FORMAT_YYMMDDHMS_MID);
    }

    public static String format(DateTime dateTime,DateEnum fmt){
        return dateTime.toString(getFormatter(fmt));
    }

    public static String format(DateTime dateTime,String format){
        return dateTime.toString(getFormatter(format));
    }

    public static DateTime parse(String date){
        return parse(date,DateEnum.FORMAT_YYMMDDHMS_MID);
    }

    /**转换日期**/
    public static DateTime parse(String date,DateEnum fmt){
        return getFormatter(fmt).parseDateTime(date);
    }

    /**转换日期**/
    public static DateTime parse(String date,String format){
        return getFormatter(format).parseDateTime(date);
    }

    /**
     * 转换日期
     * @param date 时间字符串
     * @param format 时间转换格式
     * @return year 年，month 月，day日
     */
    public static Map<String,Object> getYearMonthDayParse(String date,String format){
        Map<String,Object> result = new HashMap<String, Object>();
        try {
            SimpleDateFormat sdf = DateFormatUtil.getFormat(format);
            Date d = sdf.parse(date);
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(d);
            result.put("year",calendar.get(Calendar.YEAR));
            result.put("month",calendar.get(Calendar.MONTH) + 1);
            result.put("day",calendar.get(Calendar.DAY_OF_MONTH));
        } catch (ParseException e) {
            throw new UtilsRuntimeException(e);
        }
        return result;
    }

    /**
     * 二.秒的加减
     * 0.当前日期加减N秒
     */
    public static DateTime addSeconds(DateTime dateTime,int seconds){
        return dateTime.plusSeconds(seconds);
    }

    /**
     * 二.分钟的加减
     * 0.当前日期加减N分钟
     */
    public static DateTime addMinutes(DateTime dateTime,int minutes){
        return dateTime.plusMinutes(minutes);
    }

    /**
     * 二.小时的加减
     * 0.当前日期加减N小时
     */
    public static DateTime addHours(DateTime dateTime,int hours){
        return dateTime.plusHours(hours);
    }

    /**
     * 二.日期的加减
     * 1.当前日期加减N天
     */
    public static DateTime addDays(DateTime dateTime,int days){
        return dateTime.plusDays(days);
    }

    /**
     * 2.当前日期加减N周
     */
    public static DateTime addWeeks(DateTime dateTime,int weeks){
        return dateTime.plusWeeks(weeks);
    }

    /**
     * 3.当前日期加减N月
     */
    public static DateTime addMonths(DateTime dateTime,int months){
        return dateTime.plusMonths(months);
    }

    /**
     * 4.当前日期加减N年
     */
    public static DateTime addYears(DateTime dateTime,int years){
        return dateTime.plusYears(years);
    }


    /**
     * 三.日期坐标转换
     * 1.指定日期是本周的周几
     * 2.指定日期是本年的第几周
     */

    public static int getDayOfWeek(DateTime dateTime){
        return dateTime.getDayOfWeek();
    }

    public static int getWeekOfYear(DateTime dateTime){
        return dateTime.getWeekOfWeekyear();
    }


    /**
     * 四.日期单位的特殊点
     * 1.根据日期,获取当前的起止时间 {@link #getDayOfBegin(DateTime)} {@link #getDayOfEnd(DateTime)}
     * 2.根据日期,获取本周的第一天的0点时间 {@link #getWeekOfBegin(DateTime)}
     * 3.根据日期,获取本周的最后一天的23:59:59的时间戳 {@link #getWeekOfEnd(DateTime)}
     * 4.根据日期,获取本月的第一天的0点时间 {@link #getMonthOfBegin(DateTime)}
     * 5.根据日期,获取本月的最后一天的23:59:59的时间 {@link #getMonthOfEnd(DateTime)}
     * 6.根据日期,获取本年的第一天的0点时间 {@link #getYearOfBegin(DateTime)}
     * 7.根据日期,获取本年的最后一天的23:59:59的时间 {@link #getYearOfEnd(DateTime)}
     */
    public static DateTime getDayOfBegin(DateTime dateTime){
        return dateTime.withTimeAtStartOfDay();
    }

    public static DateTime getDayOfEnd(DateTime dateTime){
        return dateTime.millisOfDay().withMaximumValue();
    }

    public static DateTime getWeekOfBegin(DateTime dateTime){
        return dateTime.dayOfWeek().withMinimumValue();
    }

    public static DateTime getWeekOfEnd(DateTime dateTime){
        return dateTime.dayOfWeek().withMaximumValue();
    }

    public static DateTime getMonthOfBegin(DateTime dateTime){
        return dateTime.dayOfMonth().withMinimumValue();
    }

    public static DateTime getMonthOfEnd(DateTime dateTime){
        return dateTime.dayOfMonth().withMaximumValue();
    }

    public static DateTime getYearOfBegin(DateTime dateTime){
        return dateTime.dayOfYear().withMinimumValue();
    }

    public static DateTime getYearOfEnd(DateTime dateTime){
        return dateTime.dayOfYear().withMaximumValue();
    }

    /**
     * 五.日期大小的比较
     * 1.两个日期,比较返回大小 {@link #compare(DateTime, DateTime)}
     * 2.两个日期,比较返回差额,判断是否满足某一个阈值 {@link #between(DateTime, DateTime, DateTime)}
     * left > right = 1
     * left = right = 0
     * left < right = -1
     */
    public static int compare(DateTime left, DateTime right){
        if( left.isBefore(right) ){
            return 1;
        }
        if( left.isEqual(right) ){
            return 0;
        }
        return -1;
    }

    public static boolean between(DateTime src,DateTime left, DateTime right){
        if( src.isAfter(left) && src.isBefore(right) ){
            return true;
        }
        return false;
    }

    /**
     * 六.日期截断
     * 1.获取指定日期的天的毫秒数
     *   2018-04-08 16:33:01 285 -> 2018-04-08 00:00:00 000
     * 2.获取指定日期的小时的毫秒数
     *   2018-04-08 16:33:01 285 -> 2018-04-08 16:00:00 000
     * 3.获取指定日期的分的毫秒数
     *   2018-04-08 16:33:01 285 -> 2018-04-08 16:33:00 000
     * 4.获取指定日期的秒的毫秒数
     *   2018-04-08 16:33:01 285 -> 2018-04-08 16:33:01 000
     */
    public static DateTime getHourOfBegin(DateTime dateTime){
        return dateTime.hourOfDay().roundFloorCopy();
    }

    public static DateTime getMinuteOfBegin(DateTime dateTime){
        return dateTime.minuteOfDay().roundFloorCopy();
    }

    public static DateTime getSecondOfBegin(DateTime dateTime){
        return dateTime.secondOfDay().roundFloorCopy();
    }

    /**
     * 七.获取时间段
     * 1.获取起止日期之间的所有日期列表
     * 2.获取起止日期之间的所有周列表
     * 3.获取起止日期之间的所有月列表
     * TODO 刘正航 待完善...
     */

    /*
     ****************************************私有方法区*******************************************
                   _               _                           _    _                 _
                  (_)             | |                         | |  | |               | |
      _ __   _ __  _ __   __ __ _ | |_  ___   _ __ ___    ___ | |_ | |__    ___    __| |
     | '_ \ | '__|| |\ \ / // _` || __|/ _ \ | '_ ` _ \  / _ \| __|| '_ \  / _ \  / _` |
     | |_) || |   | | \ V /| (_| || |_|  __/ | | | | | ||  __/| |_ | | | || (_) || (_| |
     | .__/ |_|   |_|  \_/  \__,_| \__|\___| |_| |_| |_| \___| \__||_| |_| \___/  \__,_|
     | |
     |_|
     ****************************************私有方法区*******************************************
     */

    /**
     * 获取格式解析器
     */
    private static DateTimeFormatter getFormatter(DateEnum fmt) {
        return DateTimeFormat.forPattern(fmt.getFmt());
    }

    /**
     * 获取格式解析器
     */
    private static DateTimeFormatter getFormatter(String format) {
        return DateTimeFormat.forPattern(format);
    }

    /**
     * 获取指定时间戳所在月份开始的时间戳/秒
     * @param dateTimeMillis 指定时间戳/毫秒
     * @return
     */
    public static Long getMonthBegin(Long dateTimeMillis) {
        calendar.setTime(new Date(dateTimeMillis));

        //设置为1号,当前日期既为本月第一天
        calendar.set(Calendar.DAY_OF_MONTH, 1);
        //将小时至0
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        //将分钟至0
        calendar.set(Calendar.MINUTE, 0);
        //将秒至0
        calendar.set(Calendar.SECOND,0);
        //将毫秒至0
        calendar.set(Calendar.MILLISECOND, 0);
        // 获取本月第一天的时间戳
        return calendar.getTimeInMillis() / 1000;
    }

    /**
     * 获取指定时间戳所在月份结束的时间戳/秒
     * @param dateTimeMillis 指定时间戳/毫秒
     * @return
     */
    public static Long getMonthEnd(Long dateTimeMillis) {
        calendar.setTime(new Date(dateTimeMillis));

        //设置为当月最后一天
        calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMaximum(Calendar.DAY_OF_MONTH));
        //将小时至23
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        //将分钟至59
        calendar.set(Calendar.MINUTE, 59);
        //将秒至59
        calendar.set(Calendar.SECOND,59);
        //将毫秒至999
        calendar.set(Calendar.MILLISECOND, 999);
        // 获取本月最后一天的时间戳
        return calendar.getTimeInMillis() / 1000;
    }

    /**
     * 获取指定时间戳所在月份15号的时间戳/秒
     * @param dateTimeMillis 指定时间戳/毫秒
     * @return
     */
    public static Long getMonthMiddle(Long dateTimeMillis) {
        calendar.setTime(new Date(dateTimeMillis));

        //设置为当月最后一天
        calendar.set(Calendar.DAY_OF_MONTH, 15);
        //将小时至23
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        //将分钟至59
        calendar.set(Calendar.MINUTE, 59);
        //将秒至59
        calendar.set(Calendar.SECOND,59);
        //将毫秒至999
        calendar.set(Calendar.MILLISECOND, 999);
        // 获取本月最后一天的时间戳
        return calendar.getTimeInMillis() / 1000;
    }




    /**
     * 获取当前系统时间的年份
     * @return
     */
    public static int  getYear() {
        calendar.setTime(new Date());
        return calendar.get(Calendar.YEAR);
    }

    /**
     * 获取指定时间的年份
     * @param date //指定时间
     * @return
     */
    public static int  getYear( Date date) {
        calendar.setTime(date);
        return calendar.get(Calendar.YEAR);
    }

    /**
     * 获取当前系统时间的月份
     * @return
     */
    public static int  getMonth() {
        calendar.setTime(new Date());
        return calendar.get(Calendar.MONTH)+1;//不知道是系统的BUG还是怎样，获取的月份要比时间的少1，于是加个1
    }

    /**
     * 获取指定时间的月份
     * @param date //指定时间
     * @return
     */
    public static int  getMonth( Date date) {
        calendar.setTime(date);
        return calendar.get(Calendar.MONTH)+1;
    }

    /**
     * 获取当前系统时间的日期
     * @return
     */
    public static int  getDay() {
        calendar.setTime(new Date());
        return calendar.get(Calendar.DATE);
    }

    /**
     * 获取指定时间的日期
     * @param date //指定时间
     * @return
     */
    public static int  getDay( Date date) {
        calendar.setTime(date);
        return calendar.get(Calendar.DATE);
    }



    /**
     * 计算所选月的最大天数
     * @param year
     * @param month
     * @return
     */
    public static int getMaxDay(int year,int month){
        calendar.set(year, month,1);
        calendar.add(Calendar.DAY_OF_YEAR, -1);
        return calendar.get(Calendar.DAY_OF_MONTH);
    }


    /**
     * 获取某年某月的第一天
     * @param year 目标年份
     * @param month 目标月份
     * @return
     */
    public static Date getFirstDayOfMonth(int year,int month){
        month = month-1;
        calendar.set(Calendar.YEAR, year);
        calendar.set(Calendar.MONTH, month);

        int day = calendar.getActualMinimum(calendar.DAY_OF_MONTH);

        calendar.set(Calendar.DAY_OF_MONTH, day);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTime();
    }

    /**
     * 获取某年某月的最后一天
     * @param year 目标年份
     * @param month 目标月份
     * @return
     */
    public static Date getLastDayOfMonth(int year,int month){
        month = month-1;
        calendar.set(Calendar.YEAR, year);
        calendar.set(Calendar.MONTH, month);
        int day = calendar.getActualMaximum(calendar.DAY_OF_MONTH);
        calendar.set(Calendar.DAY_OF_MONTH, day);
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        calendar.set(Calendar.MILLISECOND, 999);
        return calendar.getTime();
    }

    /**
     * 获取当前系统时间的小时
     * @return
     */
    public static int  getHour() {
        calendar.setTime(new Date());
        return calendar.get(Calendar.HOUR_OF_DAY);
    }

    /**
     * 获取指定时间的小时
     * @param date //指定时间
     * @return
     */
    public static int  getHour( Date date) {
        calendar.setTime(date);
        return calendar.get(Calendar.HOUR_OF_DAY);
    }

    /**
     * 获取当前系统时间的分钟
     * @return
     */
    public static int  getMinute() {
        calendar.setTime(new Date());
        return calendar.get(Calendar.MINUTE);
    }

    /**
     * 获取指定时间的分钟
     * @param date //指定时间
     * @return
     */
    public static int  getMinute( Date date) {
        calendar.setTime(date);
        return calendar.get(Calendar.MINUTE);
    }

    /**
     * 获取当前系统时间的秒钟
     * @return
     */
    public static int  getSecond() {
        calendar.setTime(new Date());
        return calendar.get(Calendar.SECOND);
    }

    /**
     * 获取指定时间的秒钟
     * @param date //指定时间
     * @return
     */
    public static int  getSecond( Date date) {
        calendar.setTime(date);
        return calendar.get(Calendar.SECOND);
    }

    /**
     * 10位时间戳转Date
     * @param time
     */
    public static Date unixTimestampToDate(Long time){
        long temp = (long)time*1000;
        Timestamp ts = new Timestamp(temp);
       return ts;
    }

    /**
     * Date类型转换为10位时间戳
     * @param time
     * @return
     */
    public static Integer dateToUnixTimestamp(Date time){
        Timestamp ts = new Timestamp(time.getTime());
        return (int) ((ts.getTime())/1000);
    }

    /**
     * 获得指定日期的前一天
     * @param date
     * @param format
     * @return
     */
    public static Date getDayBefore(Date date, String format) {
        calendar.setTime(date);
        int day = calendar.get(Calendar.DATE);
        calendar.set(Calendar.DATE, day - 1);

        String dayBeforeStr = new SimpleDateFormat(format).format(calendar.getTime());

        Date dayBefore = null;
        try {
            dayBefore = new SimpleDateFormat(format).parse(dayBeforeStr);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return dayBefore;
    }


    /**
     * 获得指定日期的后一天
     * @param date
     * @param format
     * @return
     */
    public static Date getDayAfter(Date date, String format) {
        calendar.setTime(date);
        int day = calendar.get(Calendar.DATE);
        calendar.set(Calendar.DATE, day + 1);
        String dayAfterStr = new SimpleDateFormat(format).format(calendar.getTime());
        Date dayAfter = null;
        try {
            dayAfter = new SimpleDateFormat(format).parse(dayAfterStr);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return dayAfter;
    }

    /**
     * 计算两个日期的差(结束日期减开始日期)
     *
     * @param startDate 开始日期
     * @param endDate   结束日期
     * @return 返回两个时间的差
     */
    public static Duration duration(Date startDate, Date endDate) {
        return Duration.between(startDate.toInstant(), endDate.toInstant());
    }

    /**
     * 计算两个日期相差的纳秒(结束日期减开始日期)
     *
     * @param startDate 开始日期
     * @param endDate   结束日期
     * @return 返回相差的纳秒
     */
    public static long durationNanos(Date startDate, Date endDate) {
        return duration(startDate, endDate).toNanos();
    }

    /**
     * 计算两个日期相差的毫秒(结束日期减开始日期)
     *
     * @param startDate 开始日期
     * @param endDate   结束日期
     * @return 返回相差的毫秒
     */
    public static long durationMillis(Date startDate, Date endDate) {
        return duration(startDate, endDate).toMillis();
    }

    /**
     * 计算两个日期相差的秒(结束日期减开始日期)
     *
     * @param startDate 开始日期
     * @param endDate   结束日期
     * @return 返回相差的秒
     */
    public static long durationSeconds(Date startDate, Date endDate) {
        return duration(startDate, endDate).toMillis() / 1000;
    }

    /**
     * 计算两个日期相差的分(结束日期减开始日期)
     *
     * @param startDate 开始日期
     * @param endDate   结束日期
     * @return 返回相差的分
     */
    public static long durationMinutes(Date startDate, Date endDate) {
        return duration(startDate, endDate).toMinutes();
    }

    /**
     * 计算两个日期相差的时(结束日期减开始日期)
     *
     * @param startDate 开始日期
     * @param endDate   结束日期
     * @return 返回相差的时
     */
    public static long durationHours(Date startDate, Date endDate) {
        return duration(startDate, endDate).toHours();
    }

    /**
     * 计算两个日期相差的日(结束日期减开始日期)
     *
     * @param startDate 开始日期
     * @param endDate   结束日期
     * @return 返回相差的日
     */
    public static long durationDays(Date startDate, Date endDate) {
        return duration(startDate, endDate).toDays();
    }


    /**
     * 获取今天星期几
     * @return
     */
    public static int getTodayWeek(){
        Calendar now = Calendar.getInstance();
        //一周第一天是否为星期天
        boolean isFirstSunday = (now.getFirstDayOfWeek() == Calendar.SUNDAY);
        //获取周几
        int weekDay = now.get(Calendar.DAY_OF_WEEK);
        //若一周第一天为星期天，则-1
        if(isFirstSunday){
            weekDay = weekDay - 1;
            if(weekDay == 0){
                weekDay = 7;
            }
        }
        //打印周几
        return weekDay;
    }

    /**
     * 判断当前日期是星期几
     * @param date 要判断的时间
     * @param format 日期格式
     * @return
     */
    public static int dayForWeek(String date,String format){
        try {
            SimpleDateFormat sdf = DateFormatUtil.getFormat(format);
            Date d = sdf.parse(date);//19700101
            long ms = d.getTime();
            long day=ms/(24 * 60 * 60 * 1000); //单位换算为天
            long week=(day+4)%7;//对7取模
            return CastUtil.castInt(week + 1);
        } catch (ParseException e) {
            throw new UtilsRuntimeException(e);
        }
    }

    /**
     * 是否为周末
     * @param milliseconds
     * @return true 周末（包含周六和周日），false 工作日
     */
    public static boolean isWeekend(long milliseconds){
        int dayForWeek = dayForWeek(milliseconds);
        if(dayForWeek == 6 ){
            return true;
        }else if(dayForWeek == 7){
            return true;
        }
        return false;
    }

    /**
     * 判断当前日期是星期几
     * @param milliseconds 要判断的时间戳
     * @return
     */
    public static int dayForWeek(long milliseconds){
        try {
            Calendar calendar = Calendar.getInstance(); // 获得一个日历
            calendar.setTimeInMillis(completMilliseconds(milliseconds));

            calendar.setTimeInMillis(completMilliseconds(milliseconds));
            // 将时，分，秒，以及毫秒值设置为0
            calendar.set(Calendar.HOUR_OF_DAY, 0);
            calendar.set(Calendar.MINUTE, 0);
            calendar.set(Calendar.SECOND, 0);
            calendar.set(Calendar.MILLISECOND, 0);
            Integer whatDay = 0;
            if(calendar.get(Calendar.DAY_OF_WEEK)==Calendar.SATURDAY){ //星期六
                whatDay = 6;
            }
            if(calendar.get(Calendar.DAY_OF_WEEK)==Calendar.SUNDAY){ // 星期日
                whatDay = 7;
            }
            if(calendar.get(Calendar.DAY_OF_WEEK) == Calendar.MONDAY){// 星期一
                whatDay = 1;
            }
            if(calendar.get(Calendar.DAY_OF_WEEK) == Calendar.TUESDAY){// 星期二
                whatDay = 2;
            }
            if(calendar.get(Calendar.DAY_OF_WEEK) == Calendar.WEDNESDAY){// 星期三
                whatDay = 3;
            }
            if(calendar.get(Calendar.DAY_OF_WEEK) == Calendar.THURSDAY){// 星期四
                whatDay = 4;
            }
            if(calendar.get(Calendar.DAY_OF_WEEK) == Calendar.FRIDAY){// 星期五
                whatDay = 5;
            }
            return whatDay;
        } catch (Exception e) {
            throw new UtilsRuntimeException(e);
        }
    }

    /**
     * 由于服务器返回的是10位，手机端使用需要补全3位
     *
     * @param milliseconds
     * @return
     */
    private static long completMilliseconds(long milliseconds) {
        String milStr = Long.toString(milliseconds);
        if (milStr.length() == 10) {
            milliseconds = milliseconds * 1000;
        }
        return milliseconds;
    }


    /**
     * 将24小时时间转换成分钟<br>流程:<br>
     * 将传来的String 类型的hh:MM 以:进行分割得到hh*60得到hh<br>
     * 得到hh的分钟，在加上MM得到最终求得的分钟<br>
     * 注意:<br> hh:MM中":"必须为英文":"
     * @param time
     *			需要转换的时间
     * @return
     */
    public static Integer toMm(String time) {
        if(StrUtil.isBlank(time)) {
            return 0;
        }
        String[] tim = StrUtil.split(time,":");
        return Integer.valueOf(tim[0])*60+Integer.valueOf(tim[1]);
    }



    //测试方法
    public static void main(String[] args) throws Exception {
        Date date = new Date();
        System.out.println("获取当前系统时间的年份：" + DateUtil.getYear());
        System.out.println("获取当前系统时间的月份：" + DateUtil.getMonth());
        System.out.println("获取当前系统时间的日期：" + DateUtil.getDay());
        System.out.println("获取时间戳的日期：" + DateUtil.getDay(DateUtil.unixTimestampToDate(1575216000l)));
        System.out.println("获取当前系统时间的小时：" + DateUtil.getHour());
        System.out.println("获取当前系统时间的分钟：" + DateUtil.getMinute());
        System.out.println("获取当前系统时间的秒钟：" + DateUtil.getSecond());
        System.out.println("计算所选月的最大天数：" + DateUtil. getMaxDay(2019,12));
        long currentTime = System.currentTimeMillis();
        System.out.println("格式化日期：" + DateFormatUtil.formatDate(currentTime, DateEnum.FORMAT_YYMMDD_MID));
        System.out.println("转换日期:"+DateUtil.getYearMonthDayParse("2019-12","yyyy-MM"));
        System.out.println("获取今天星期几:"+getTodayWeek());
        System.out.println("星期几:"+DateUtil.dayForWeek("2019-12-05","yyyy-MM-dd"));
        System.out.println("星期几:"+DateUtil.dayForWeek(1575527594));




    }





}
