package cn.dolphin.core.date;

import cn.dolphin.core.consts.DateConstant;
import cn.dolphin.core.consts.StringConstant;
import cn.dolphin.core.enums.DateEnum;
import cn.dolphin.core.util.StrUtil;
import lombok.extern.slf4j.Slf4j;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;

/**
 * DateFormatUtil是用来格式化时间的类
 */
@Slf4j
public class DateFormatUtil {

    /**
     * yyyy-MM-dd 时间格式
     */
    public static final DateFormat DATE_FORMAT = new SimpleDateFormat(
            StringConstant.DATE_FORMAT_DATE);
    /**
     * yyyy-MM-dd HH:mm:ss 时间格式
     */
    public static final DateFormat DATETIME_FORMAT = new SimpleDateFormat(
            StringConstant.DATE_FORMAT_DATETIME);
    /**
     * yyyy-MM-dd HH:mm 时间格式
     */
    public static final DateFormat DATETIME_NOSECOND_FORMAT = new SimpleDateFormat(
            StringConstant.DATE_FORMAT_DATETIME_NOSECOND);
    /**
     * HH:mm:ss 时间格式
     */
    public static final DateFormat TIME_FORMAT = new SimpleDateFormat(
            StringConstant.DATE_FORMAT_TIME);
    /**
     * HH:mm 时间格式
     */
    public static final DateFormat TIME_NOSECOND_FORMAT = new SimpleDateFormat(
            StringConstant.DATE_FORMAT_TIME_NOSECOND);
    /**
     * yyyy-MM-dd HH:mm:ss.SSS 时间格式
     */
    public static final DateFormat TIMESTAMP_FORMAT = new SimpleDateFormat(
            StringConstant.DATE_FORMAT_TIMESTAMP);

    private DateFormatUtil() {
        // nothing
    }

    /**
     * DateFormatUtil.SDF_DATETIME.get().format(new Date())
     * DateFormatUtil.SDF_DATETIME.get().parse("2013-12-12 00:00:00")
     */
    public static final ThreadLocal<SimpleDateFormat> SDF_DATETIME = new ThreadLocal<SimpleDateFormat>() {
        @Override
        protected SimpleDateFormat initialValue() {
            return new SimpleDateFormat(StringConstant.DATE_FORMAT_DATETIME);
        }
    };

    public static final ThreadLocal<SimpleDateFormat> SDF_DATETIME_NOSEP = new ThreadLocal<SimpleDateFormat>() {
        @Override
        protected SimpleDateFormat initialValue() {
            return new SimpleDateFormat(StringConstant.DATE_FORMAT_DATETIME_NOSEP);
        }
    };

    public static final ThreadLocal<SimpleDateFormat> SDF_DATE_MONTH_NOSEP = new ThreadLocal<SimpleDateFormat>() {
        @Override
        protected SimpleDateFormat initialValue() {
            return new SimpleDateFormat(StringConstant.DATE_FORMAT_DATE_MONTH_NOSEP);
        }
    };

    public static final ThreadLocal<SimpleDateFormat> SDF_DATE = new ThreadLocal<SimpleDateFormat>() {
        @Override
        protected SimpleDateFormat initialValue() {
            return new SimpleDateFormat(StringConstant.DATE_FORMAT_DATE);
        }
    };

    public static final ThreadLocal<SimpleDateFormat> SDF_DATE_NOSEP = new ThreadLocal<SimpleDateFormat>() {
        @Override
        protected SimpleDateFormat initialValue() {
            return new SimpleDateFormat(StringConstant.DATE_FORMAT_DATE_NOSEP);
        }
    };

    public static final ThreadLocal<SimpleDateFormat> SDF_TIME = new ThreadLocal<SimpleDateFormat>() {
        @Override
        protected SimpleDateFormat initialValue() {
            return new SimpleDateFormat(StringConstant.DATE_FORMAT_TIME);
        }
    };

    /**
     * 根据日期字符串是否含有时间决定转换为日期还是日期时间还是时间
     *
     * @param dateString 时间字符串
     * @return 格式化的时间
     * @throws ParseException
     */
    public static Date parse(String dateString) throws ParseException {
        if (dateString.trim().indexOf(" ") > 0
                && dateString.trim().indexOf(".") > 0) {
            return new java.sql.Timestamp(TIMESTAMP_FORMAT.parse(dateString)
                    .getTime());
        } else if (dateString.trim().indexOf(" ") > 0) {
            // 如果有两个:，则有时分秒,一个冒号只有时分
            if (dateString.trim().indexOf(":") != dateString.trim()
                    .lastIndexOf(":")) {
                return new java.sql.Timestamp(DATETIME_FORMAT.parse(dateString)
                        .getTime());
            } else {
                return new java.sql.Timestamp(DATETIME_NOSECOND_FORMAT.parse(
                        dateString).getTime());
            }
        } else if (dateString.indexOf(":") > 0) {
            // 如果有两个:，则有时分秒,一个冒号只有时分
            if (dateString.trim().indexOf(":") != dateString.trim()
                    .lastIndexOf(":")) {
                return new java.sql.Time(TIME_FORMAT.parse(dateString)
                        .getTime());
            } else {
                return new java.sql.Time(TIME_NOSECOND_FORMAT.parse(dateString)
                        .getTime());
            }
        }
        return new java.sql.Date(DATE_FORMAT.parse(dateString).getTime());
    }

    public static Date parseTime(String dateString) throws ParseException {
        if (dateString.trim().indexOf(" ") > 0) {
            String[] d = dateString.trim().split(" ");
            if (dateString.trim().indexOf(":") != dateString.trim()
                    .lastIndexOf(":")) {
                return new java.sql.Timestamp(TIME_FORMAT.parse(d[1])
                        .getTime());
            } else {
                return new java.sql.Timestamp(TIME_NOSECOND_FORMAT.parse(
                        d[1]).getTime());
            }
        } else if (dateString.indexOf(":") > 0) {
            // 如果有两个:，则有时分秒,一个冒号只有时分
            if (dateString.trim().indexOf(":") != dateString.trim()
                    .lastIndexOf(":")) {
                return new java.sql.Time(TIME_FORMAT.parse(dateString)
                        .getTime());
            } else {
                return new java.sql.Time(TIME_NOSECOND_FORMAT.parse(dateString)
                        .getTime());
            }
        }
        return new java.sql.Date(DATETIME_FORMAT.parse(dateString).getTime());
    }


    /**
     * 按格式输出date到string,按照日期类型自动判断
     *
     * @param date 按格式输出
     * @return 时间字符串
     */
    public static String format(Date date) {
        if (date instanceof java.sql.Timestamp) {
            return TIMESTAMP_FORMAT.format(date);
        } else if (date instanceof java.sql.Time) {
            return TIME_FORMAT.format(date);
        } else if (date instanceof java.sql.Date) {
            return DATE_FORMAT.format(date);
        }
        return DATETIME_FORMAT.format(date);
    }

    /**
     * 获取日期显示格式，为空默认为yyyy-mm-dd HH:mm:ss
     * @author chenssy
     * @date Dec 30, 2013
     * @param format
     * @return
     * @return SimpleDateFormat
     */
    protected static SimpleDateFormat getFormat(String format){
        if(format == null || "".equals(format)){
            format = DateConstant.DATE_TIME_PATTERN;
        }
        return new SimpleDateFormat(format);
    }

    /**
     * 按指定的格式输出string到date
     *
     * @param dateString 时间字符串
     * @param style      格式化参数
     * @return 格式化的时间
     * @throws ParseException
     */
    public static Date parse(String dateString, String style) {
        try {
            DateFormat dateFormat = new SimpleDateFormat(style);
            return dateFormat.parse(dateString);
        } catch (ParseException e) {
            return null;
        }
    }

    public static String formatDate(String str, String format) {
        if (str == null || str.isEmpty()) {
            return str;
        }
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        try {
            return sdf.format(sdf.parse(str));
        } catch (Exception e) {
            log.debug("Date format error: value=" + str + ", error=" + e.toString());
            return str;
        }
    }

    /**
     * 格式化输出date到string
     *
     * @param date
     * @param style 格式化参数
     * @return
     */
    public static String format(Date date, String style) {
        DateFormat dateFormat = new SimpleDateFormat(style);
        return dateFormat.format(date);
    }

    private static Date format(String value, String format, Locale locale) {
        if (StrUtil.isNotBlank(value) && StrUtil.isNotBlank(format)) {
            SimpleDateFormat df = new SimpleDateFormat(format, locale);
            try {
                return df.parse(value);
            } catch (ParseException e) {
            }
        }
        return null;
    }

    private static Date format(String value, String format) {
        if (StrUtil.isNotBlank(value) && StrUtil.isNotBlank(format)) {
            SimpleDateFormat df = new SimpleDateFormat(format);
            try {
                return df.parse(value);
            } catch (ParseException e) {
            }
        }
        return null;
    }

    /** 字符串转换成时间类型 */
    public static Date format(String value) {
        Date date = null;
        date = format(value, StringConstant.DATE_FORMAT_DATETIME);
        if (date != null) {
            return date;
        }
        date = format(value, StringConstant.DATE_FORMAT_DATE);
        if (date != null) {
            return date;
        }
        date = format(value, "yyyy/MM/dd");
        if (date != null) {
            return date;
        }
        date = format(value, StringConstant.DATE_FORMAT_DATETIME_MS);
        if (date != null) {
            return date;
        }
        date = format(value, StringConstant.DATE_FORMAT_DATETIME_NOSEP);
        if (date != null) {
            return date;
        }
        date = formatByLong(value);
        if (date != null) {
            return date;
        }
        date = format(value, StringConstant.DATE_FORMAT_DATE_NOSEP);
        if (date != null) {
            return date;
        }
        date = format(value, StringConstant.DATE_FORMAT_DATETIME_STRING, Locale.US);
        if (date != null) {
            return date;
        }
        date = format(value, "yyyy-MM-dd HH:mm:ss.SSS");
        if (date != null) {
            return date;
        }
        date = format(value, "yyyy/MM/dd HH:mm:ss");
        return date;
    }

    /** long型字符串时间格式转时间对象 */
    private static Date formatByLong(String value) {
        if (StrUtil.isNotBlank(value) && value.matches("[0-9]{13}")) {
            return new Date(Long.valueOf(value));
        }
        return null;
    }

    /** 秒数转成"136:25:30"格式的字符串 */
    public static String formatToHHmmss(long theSecond) {
        int timetiem = (int) theSecond;
        int minute = timetiem / 60;
        int hour = minute / 60;
        int second = timetiem % 60;
        minute %= 60;
        return String.format("%02d:%02d:%02d", hour, minute, second);
    }

    /** 将 Date 类型转换为 XMLGregorianCalendar */
    public static XMLGregorianCalendar formatToXmlDate(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        DatatypeFactory dtf = null;
        try {
            dtf = DatatypeFactory.newInstance();
        } catch (DatatypeConfigurationException e) {
        }
        XMLGregorianCalendar dateType = dtf.newXMLGregorianCalendar();
        dateType.setYear(cal.get(Calendar.YEAR));
        // 由于Calendar.MONTH取值范围为0~11,需要加1
        dateType.setMonth(cal.get(Calendar.MONTH) + 1);
        dateType.setDay(cal.get(Calendar.DAY_OF_MONTH));
        dateType.setHour(cal.get(Calendar.HOUR_OF_DAY));
        dateType.setMinute(cal.get(Calendar.MINUTE));
        dateType.setSecond(cal.get(Calendar.SECOND));
        return dateType;
    }

    /** 将 XMLGregorianCalendar 转换为 Date */
    public static Date format(XMLGregorianCalendar cal) {
        return cal.toGregorianCalendar().getTime();
    }

    /**
     * 将string(yyyy-MM-dd)转化为日期
     *
     * @param dateString
     * @return
     * @throws ParseException
     */
    public static Date parseDate(String dateString) throws ParseException {
        return DATE_FORMAT.parse(dateString);
    }

    /**
     * 按格式(yyyy-MM-dd)输出date到string
     *
     * @param date
     * @return
     */
    public static String formatDate(Date date) {
        return DATE_FORMAT.format(date);
    }

    /**
     * 将长整型数字转换为日期格式的字符串
     * @param time 毫秒
     * @param format
     * @return
     */
    public static String formatDate(long time, DateEnum format) {
        if (time > 0l) {
            if (StrUtil.isNotBlank(format.getFmt())){
                DateFormat dateFormat = new SimpleDateFormat(format.getFmt());
                Date date = new Date(time);
                return dateFormat.format(date);
            }
        }
        return "";
    }

    /**
     * 将string(yyyy-MM-dd HH:mm:ss)转化为日期
     *
     * @param dateString
     * @return
     * @throws ParseException
     */
    public static Date parseDateTime(String dateString) throws ParseException {
        return DATETIME_FORMAT.parse(dateString);
    }

    /**
     * 按格式(yyyy-MM-dd HH:mm:ss )输出date到string
     *
     * @param date
     * @return
     */
    public static String formaDatetTime(Date date) {
        return DATETIME_FORMAT.format(date);
    }

    /**
     * 按格式(yyyy-MM-dd HH:mm )输出date到string
     *
     * @param date
     * @return
     */
    public static String formatTimeNoSecond(Date date) {
        return DATETIME_NOSECOND_FORMAT.format(date);
    }

    /**
     * 按格式(yyyy-MM-dd HH:mm )输出 string到date
     *
     * @param dateString
     * @return
     * @throws ParseException
     */
    public static Date parseTimeNoSecond(String dateString)
            throws ParseException {
        return DATETIME_NOSECOND_FORMAT.parse(dateString);
    }

    /**
     * 取当前系统日期，并按指定格式或者是默认格式返回
     *
     * @param style
     * @return
     */
    public static String getNowByString(String style) {
        if (null == style || "".equals(style)) {
            style = "yyyy-MM-dd HH:mm:ss";
        }
        return format(new Date(), style);
    }


}