package cn.thinkingdata.tga.javasdk;

import cn.thinkingdata.tga.javasdk.exception.InvalidArgumentException;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONException;
import com.alibaba.fastjson.serializer.SerializerFeature;
import org.apache.commons.codec.binary.Base64;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.*;
import java.util.regex.Pattern;
import java.util.zip.GZIPOutputStream;


public class ThinkingDataAnalytics {
	private final Consumer consumer ;
	private final Map<String,Object> publicProperties ;
	private final static String libVersion = "1.1.15";
	private final static Pattern trackPattern = Pattern.compile("^(#[a-z][a-z0-9_]{0,49})|([a-z][a-z0-9_]{0,50})$",Pattern.CASE_INSENSITIVE);
	private final static Pattern userPattern = Pattern.compile("^(#[a-z][a-z0-9_]{0,49})|([a-z][a-z0-9_]{0,50})$",Pattern.CASE_INSENSITIVE);


	/**
	 * 实例化tga类，接收一个Consumer的一个实例化对象
	 * @param consumer	BatchConsumer,LoggerConsumer实例
	 * */
	public ThinkingDataAnalytics(final Consumer consumer) {
		this.consumer = consumer;
		this.publicProperties = new ConcurrentHashMap<String, Object>();
		this.clearSuperProperties();
	}

	/**
	 * 用户删除
	 * @param 	account_id	账号ID
	 * @param	distinct_id	匿名ID
	 * @throws	InvalidArgumentException	数据错误
	 * */
	public void user_del(String account_id,String distinct_id) throws InvalidArgumentException{
		Map<String,Object> properties = new HashMap<String,Object>();
		if(properties.get("#time") == null){
			properties.put("#time", new Date());
		}
		__add(distinct_id, account_id, "user_del", properties);
	}

	/**
	 * 用户属性修改，只支持数字属性增加的接口
	 * @param	account_id	账号ID
	 * @param	distinct_id	匿名ID
	 * @param	properties	增加的用户属性
	 * @throws	InvalidArgumentException	数据错误
	 * */
	public void user_add(String account_id,String distinct_id,Map<String,Object> properties) throws InvalidArgumentException{
		__add(distinct_id,account_id,"user_add",properties);
	}



	/**
	 * 设置用户属性，首次设置用户的属性,如果该属性已经存在,该操作为无效.
	 * @param	account_id	账号ID
	 * @param	distinct_id	匿名ID
	 * @param	properties	增加的用户属性
	 * @throws	InvalidArgumentException	数据错误
	 * */
	public void user_setOnce(String account_id,String distinct_id,Map<String,Object> properties) throws InvalidArgumentException{
		__add(distinct_id,account_id,"user_setOnce",properties);
	}

	/**
	 * 设置用户属性，如果已经存在，则覆盖，否则，新创建
	 * @param	account_id	账号ID
	 * @param	distinct_id	匿名ID
	 * @param	properties	增加的用户属性
	 * @throws InvalidArgumentException	数据错误
	 * */
	public void user_set(String account_id,String distinct_id,Map<String,Object> properties) throws InvalidArgumentException{
		__add(distinct_id,account_id,"user_set",properties);
	}

	/**
	 * 用户事件属性(注册)
	 * @param	account_id	账号ID
	 * @param	distinct_id	匿名ID
	 * @param	event_name	事件名称
	 * @param	properties	事件属性
	 * @throws	InvalidArgumentException	数据错误
	 * */
	public void track(String account_id,String distinct_id,String event_name,Map<String,Object> properties) throws InvalidArgumentException{
		Map all_properties = new HashMap();
		all_properties.putAll(publicProperties);
		if(properties != null){
			all_properties.putAll(properties);
		}
//		for (String key : properties.keySet()) {
//			all_properties.put(key, properties.get(key));
//		}
		__add(distinct_id,account_id,"track",event_name,all_properties);
	}

	//user
	private void __add(String distinct_id,String account_id,String type,Map<String,Object> properties) throws InvalidArgumentException{
		__add(distinct_id,account_id,type,null,properties);
	}

	private void __add(String distinct_id,String account_id,String type,String event_name,Map<String, Object> properties_add) throws InvalidArgumentException{
		Map<String,Object> properties = new HashMap<>();
		properties.putAll(properties_add);
		if(account_id == null && distinct_id ==null){
			throw new InvalidArgumentException("account_id and distinct_id Simultaneously are null ");
		}

		//properties check
		if(properties != null){
			assertProperties(type,properties);
		}

		Map<String,Object> event = new HashMap<String,Object>();

		event.put("#time", properties.get("#time"));
		properties.remove("#time");

		if(properties.containsKey("#ip")){
			event.put("#ip", properties.get("#ip"));
			properties.remove("#ip");
		} else {
			event.put("#ip", "");
		}

		event.put("#type", type);

		if(event_name != null){
			event.put("#event_name", event_name);
		}

		Map<String, Object> eventProperties = new HashMap<String, Object>();
//	    if (type.equals("track")) {
//	      eventProperties.putAll(this.publicProperties);
//	    }
		if (properties != null) {
			eventProperties.putAll(properties);
		}
		event.put("properties", eventProperties);


		if(distinct_id != null){
			event.put("#distinct_id", distinct_id);
		}
		if(account_id != null){
			event.put("#account_id", account_id);
		}

		this.consumer.add(event);

	}

	/**
	 * 判断属性是否满足要求
	 * @param	type	事件类型
	 * @param	properties	属性值
	 * */
	private void assertProperties(String type, Map<String, Object> properties) throws InvalidArgumentException {
		assertType("type catgory",type);

		Pattern pattern_key = getPatternKey(type);


		if(properties.containsKey("#time")){
			if(!(properties.get("#time") instanceof Date)){
				throw new InvalidArgumentException("type(#time) must be Date, #time is " + properties.get("#time"));
			}
		}else{
			properties.put("#time",new Date());
		}


		for(Entry<String, Object> property:properties.entrySet()){
			if(property.getValue() == null) {
				continue;
			}
			if(pattern_key.matcher(property.getKey()).matches()){
				if(!(property.getValue() instanceof Number) && !(property.getValue() instanceof Date) && !(property.getValue() instanceof String) && !(property.getValue() instanceof Boolean)){
					throw new InvalidArgumentException("The property value should be a basic type: Number, String, Date, Boolean.");
				}else{
					if(type.toLowerCase().equals("user_add") && !(property.getValue() instanceof Number) &&!(property.getKey().startsWith("#"))){
						throw new InvalidArgumentException("Type user_add only support Number");
					}
				}
			}else{
				throw new InvalidArgumentException("type "+type+"'key "+property.getKey()+" is invalid");
			}
		}

	}

	private boolean assertType(String description, String type) throws InvalidArgumentException {
		if ("user_set,user_setonce,user_add,user_del,track".contains(type.toLowerCase())){
			return true;
		}
		throw new InvalidArgumentException(description+" without support : "+type);
	}

	private Pattern getPatternKey(String type) {
		if(type.equals("track")){
			return trackPattern;
		}else{
			return userPattern;
		}
	}

	/**
	 * 清理掉公共属性，之后track属性将不会再加入公共属性
	 * */
	public void clearSuperProperties(){
		this.publicProperties.clear();
		this.publicProperties.put("#lib", "tga_java_sdk");
		this.publicProperties.put("#lib_version", libVersion);
	}

	/**
	 * 公共属性只用于track接口，其他接口无效，且每次都会自动向track事件中添加公共属性
	 * @param properties	公共属性
	 * */
	public void setSuperProperties(Map<String, Object> properties){
		this.publicProperties.putAll(properties);
	}


	/**
	 * 立即提交数据到相应的接收器
	 * */
	public void flush(){
		this.consumer.flush();
	}
	/**
	 * 关闭并退出sdk,阻塞式的
	 * */
	public void close(){
		this.consumer.close();
	}

	public static class LoggerConsumer implements Consumer{
		private static final Logger logger = LoggerFactory.getLogger(LoggerConsumer.class);
		private final Integer fileSize ;
		private final String log_directory;
		private final StringBuilder message_buffer;
		private final SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
		private LoggerWriter logger_writer ;
		private final Integer bufferSize = 81920;
		private boolean isClose = false;
		private Integer count = 0;



		public LoggerConsumer(String log_directory) throws FileNotFoundException {
			this(log_directory,1024);
		}
		/**
		 * @param 	log_directory	log数据保存的文件目录
		 * @param 	fileSize 		 一次写入数据的数据量
		 * @throws 	FileNotFoundException	如果设置的数据保存目录不存在，将会抛出错误
		 * **/
		public LoggerConsumer(String log_directory,int fileSize) throws FileNotFoundException {
			this.fileSize = fileSize;
			this.log_directory = log_directory;
			this.message_buffer = new StringBuilder();
			String file_name = getFileName();
			if (logger_writer != null){
				logger_writer.close();
				logger_writer = null;
			}
			logger_writer = new LoggerWriter(file_name);
			JSON.DEFFAULT_DATE_FORMAT = "yyyy-MM-dd HH:mm:ss.SSS";
		}

		@Override
		public synchronized void add(Map<String, Object> message) {
			if(this.isClose){
				try {
					logger.error("message throw away:"+JSON.toJSONString(message, SerializerFeature.WriteDateUseDateFormat));
				} catch (JSONException e) {
					logger.error("message throw away error :" +e.toString());
//					message.forEach((k,v) -> {logger.error(k.toString() + "-" + v.toString());});
				}
			}else{
				try {
					message_buffer.append(JSON.toJSONString(message, SerializerFeature.WriteDateUseDateFormat));
					message_buffer.append("\n");
				} catch (JSONException e) {
					throw new RuntimeException("Failed to become json",e);
				}
				if(message_buffer.length() >= bufferSize){
					this.flush();
					count++;
				}
//				if(message_buffer.length() < BUFFER_LIMITATION){
//					try {
//						message_buffer.append(JSON.toJSONString(message, SerializerFeature.WriteDateUseDateFormat));
//						message_buffer.append("\n");
//					} catch (JSONException e) {
//						throw new RuntimeException("Failed to become json",e);
//					}
//				}else{
//					throw new RuntimeException("Logging buffer exceeded the allowed limitation.");
//				}
//				if(message_buffer.length() >= batch_size){
//					this.flush();
//				}
			}
		}

		@Override
		public synchronized void flush() {
			if(message_buffer.length() == 0){
				return ;
			}
			if(count % 100 == 0){
				String file_name = getFileName();
				if(!logger_writer.isValid(file_name)){
					logger_writer.close();
					logger_writer = null;
				}
				count = 0;
				if (logger_writer == null){
					try {
						logger_writer = new LoggerWriter(file_name);
					} catch (FileNotFoundException e) {
						throw new RuntimeException(e);
					}
				}
			}
			if(logger_writer.write(message_buffer)){
				message_buffer.setLength(0);
			}

		}

		@Override
		public void close() {
			this.isClose = true;
			this.flush();
			this.logger_writer.close();
		}

		public String getFileName(){
			String file_base = log_directory + File.separator + "log." +simpleDateFormat.format(new Date()) + "_";
			Integer count = 0;
			String file_complete = file_base + count;
			File target = new File(file_complete);
			while (target.exists() && fileSizeOut(target)) {
				count += 1;
				file_complete = file_base + count;
				target = new File(file_complete);
			}
			return file_complete;
		}

		public Boolean fileSizeOut(File target){
			Long fsize = target.length();
			fsize = fsize / (1024 * 1024);
			if(fsize>=fileSize){
				return true;
			} else {
				return false;
			}
		}


		static class LoggerWriter{
			private String filename_last ;
			private FileOutputStream outputStream;


			public LoggerWriter(final String filename) throws FileNotFoundException {
//				String file_complete = new File(filename)+File.separator+"log."+simpleDateFormat.format(new Date());
				if(filename_last == null){
					filename_last = filename;
				} else if(!filename_last.equals(filename)){
					close();
					filename_last = filename;
				}
				File target = new File(filename_last);
				File parent_directory = target.getParentFile();
				if(!parent_directory.exists()){
					throw new FileNotFoundException(parent_directory.toString());
				}
				this.outputStream = new FileOutputStream(target,true);
			}

			public void close(){
				try {
					if(outputStream!=null){
						outputStream.close();
					}
				} catch (IOException e) {
					throw new RuntimeException("Failed to close OutPutStream",e);
				}
			}

			public boolean isValid(final String filename){
				return this.filename_last.equals(filename);
			}

			public boolean write(final StringBuilder buffer){
				FileLock lock = null;
				try {
					final FileChannel channel = outputStream.getChannel();
					lock = channel.lock(0,Long.MAX_VALUE,false);
					outputStream.write(buffer.toString().getBytes("UTF-8"));
				} catch (IOException e) {
					throw new RuntimeException(e);
				}finally{
					if(lock != null){
						try {
							lock.release();
						} catch (IOException e) {
							throw new RuntimeException(e);
						}
					}
				}
				return true;
			}
		}



	}




	public static class BatchConsumer implements Consumer{
		private final Logger logger = LoggerFactory.getLogger(BatchConsumer.class);
		private final String appid;
		private final String server_uri;
		private Integer batch = 20;
		private Integer timeout = 30000;
		private Integer interval = 3;
		private Long lastFlushTime = System.currentTimeMillis();
		private List<Map<String, Object>> message_channel = new ArrayList<>();
		private HttpConsumer httpConsumer;


		/**
		 * 改接口功能用于实时向tga服务器传输数据，不需要配合其他的传输工具
		 *@param server_uri	传输数据的uri
		 *@param appid		服务器验证的token
		 */
		public BatchConsumer(String server_uri,String appid) {
			this.appid = appid;
			this.server_uri = server_uri;
			JSON.DEFFAULT_DATE_FORMAT = "yyyy-MM-dd HH:mm:ss.SSS";
			this.message_channel = new ArrayList<>();
			httpConsumer = new HttpConsumer(server_uri,appid);
		}
		/**
		 * 改接口功能用于实时向tga服务器传输数据，不需要配合其他的传输工具
		 *@param server_uri	传输数据的uri
		 *@param appid		服务器验证的token
		 *@param batch	    实时传输的数据批次大小
		 *@param timeout    http的timeout时间
		 *@param interval   实际间隔时间
		 * */
		public BatchConsumer(String server_uri,String appid,int batch,int timeout, int interval) {
			this.appid = appid;
			this.server_uri = server_uri;
			JSON.DEFFAULT_DATE_FORMAT = "yyyy-MM-dd HH:mm:ss.SSS";
			this.message_channel = new ArrayList<>();
			this.batch = batch;
			this.timeout = timeout;
			this.interval = interval;
			httpConsumer = new HttpConsumer(server_uri,appid, timeout);
		}

		@Override
		public void add(Map<String, Object> message) {
			message_channel.add(message);
			Long nowTime = System.currentTimeMillis();
			if(message_channel.size() >= batch || (nowTime - lastFlushTime >= interval * 1000 && message_channel.size() >0)){
				flush();
			}
		}

		@Override
		public void flush() {
			try {
				List<Map<String, Object>> messageList = new ArrayList<>();
				if (message_channel.size() > batch){
					messageList = message_channel.subList(0, batch);
				} else {
					messageList = (List<Map<String, Object>>) ((ArrayList<Map<String, Object>>) message_channel).clone();
				}
				String data = JSON.toJSONString(messageList, SerializerFeature.WriteDateUseDateFormat);
				httpConsumer.send(data);
				if (message_channel.size() > batch){
					message_channel = message_channel.subList(batch, message_channel.size());
				} else {
					message_channel.clear();
				}
				lastFlushTime = System.currentTimeMillis();
			} catch (JSONException e) {
				throw new RuntimeException("Failed to become json ",e);
			}catch (Exception e) {
				throw new RuntimeException("Failed to transform with BatchConsumer",e);
			}
		}

		@Override
		public void close() {
			while(message_channel.size() > 0){
				flush();
			}
		}

	}


	private static class HttpConsumer{
		private final Logger logger = LoggerFactory.getLogger(HttpConsumer.class);
		private final String server_uri ;
		private final String appid;
		private Integer connectTimeout = 30000;
		/**
		 * BatchComsumer用于数据传输的接口
		 * 需要的参数
		 * @param server_url 数据网络传输的url，来自tga官网
		 * @parma appid	数据传输时候的验证token
		 * */


		private HttpConsumer(String server_url,String appid, Integer timeout){
			this(server_url,appid);
			this.connectTimeout = timeout;
		}

		private HttpConsumer(String server_url,String appid){
			this.server_uri = server_url;
			this.appid = appid;
		}


		public void send(final String data) throws Exception{
			CloseableHttpClient httpclient = HttpClients.custom().build();
			HttpPost httppost = null;
			try {
				StringEntity params = new StringEntity(this.encodeRecord(data),"UTF-8");
				RequestConfig requestConfig = RequestConfig.custom().setSocketTimeout(connectTimeout + 10000).setConnectTimeout(30000).build();
				httppost = new HttpPost(this.server_uri);
				httppost.setEntity(params);
				httppost.addHeader("appid", this.appid);
				httppost.addHeader("user-agent", "java_sdk_"+libVersion);
				httppost.addHeader("version", libVersion);
				httppost.setConfig(requestConfig);
				CloseableHttpResponse response = httpclient.execute(httppost);
				Integer code = response.getStatusLine().getStatusCode();
				String result = EntityUtils.toString(response.getEntity(),"UTF-8");
				response.close();
				httppost.abort();
				if(code < 200 || code >= 300){
					throw new Exception(String.format("Unexcepted response %d from tga:%s", code,result));
				}
			}catch(Exception e){
				throw new Exception("http transport with error "+e.getCause().getMessage());
			}finally{
				if(httpclient != null){
					try {
						httpclient.close();
					} catch (IOException e) {
						throw new Exception("httpclient with error "+e.getCause().getMessage());
					}
				}
			}
		}

		private String encodeRecord(String data) {
			ByteArrayOutputStream byteArrayBuffer = new ByteArrayOutputStream();
			try {
				GZIPOutputStream var2 = new GZIPOutputStream(byteArrayBuffer);
				var2.write(data.getBytes(StandardCharsets.UTF_8));
				var2.close();
			}catch(IOException var3) {
				logger.error("GZIP compress with exception", var3);
				return null;
			}
			return new String(Base64.encodeBase64(byteArrayBuffer.toByteArray()));
		}
	}


}

