package cn.thinkingdata.tga.javasdk;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.*;
import java.util.regex.Pattern;
import java.util.zip.GZIPOutputStream;

import com.fasterxml.jackson.core.JsonEncoding;
import com.fasterxml.jackson.core.JsonGenerator;
import org.apache.commons.codec.binary.Base64;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;

import cn.thinkingdata.tga.javasdk.exception.InvalidArgumentException;


public class ThinkingDataAnalytics {
	private final Consumer consumer ;
	private final Map<String,Object> publicProperties ;
	private final static String libVversion = "1.1.8";
	private final static Pattern patternTrack = Pattern.compile("^(#[a-z][a-z0-9_]{0,49})|([a-z][a-z0-9_]{0,50})$",Pattern.CASE_INSENSITIVE);
	private final static Pattern patternNonTrack = Pattern.compile("^(#[a-z][a-z0-9_]{0,49})|([a-z][a-z0-9_]{0,50})$",Pattern.CASE_INSENSITIVE);

	
	/**
	 * 实例化tga类，接收一个Consumer的一个实例化对象
	 * @param consumer	BatchConsumer,LoggerConsumer实例
	 * */
	public ThinkingDataAnalytics(final Consumer consumer) {
		this.consumer = consumer;
		this.publicProperties = new ConcurrentHashMap<String, Object>();
		this.clearSuperProperties();
	}
	
	/**
	 * 用户删除
	 * @param 	account_id	账号ID
	 * @param	distinct_id	匿名ID
	 * @throws	InvalidArgumentException	数据错误
	 * */
	public void user_del(String account_id,String distinct_id) throws InvalidArgumentException{
		Map<String,Object> properties = new HashMap<String,Object>();
		properties.put("#time", new Date());
		add(distinct_id, account_id, "user_del", properties);
	}
	
	/**
	 * 用户属性修改，只支持数字属性增加的接口
	 * @param	account_id	账号ID
	 * @param	distinct_id	匿名ID
	 * @param	properties	增加的用户属性
	 * @throws	InvalidArgumentException	数据错误
	 * */
	public void user_add(String account_id,String distinct_id,Map<String,Object> properties) throws InvalidArgumentException{
		add(distinct_id,account_id,"user_add",properties);
	}
	
	
	
	/**
	 * 设置用户属性，首次设置用户的属性,如果该属性已经存在,该操作为无效.
	 * @param	account_id	账号ID
	 * @param	distinct_id	匿名ID
	 * @param	properties	增加的用户属性
	 * @throws	InvalidArgumentException	数据错误
	 * */
	public void user_setOnce(String account_id,String distinct_id,Map<String,Object> properties) throws InvalidArgumentException{
		add(distinct_id,account_id,"user_setOnce",properties);
	}
	
	/**
	 * 设置用户属性，如果已经存在，则覆盖，否则，新创建
	 * @param	account_id	账号ID
	 * @param	distinct_id	匿名ID
	 * @param	properties	增加的用户属性
	 * @throws InvalidArgumentException	数据错误
	 * */
	public void user_set(String account_id,String distinct_id,Map<String,Object> properties) throws InvalidArgumentException{
		add(distinct_id,account_id,"user_set",properties);
	}
	
	/**
	 * 用户事件属性(注册)
	 * @param	account_id	账号ID
	 * @param	distinct_id	匿名ID
	 * @param	event_name	事件名称
	 * @param	properties	事件属性
	 * @throws	InvalidArgumentException	数据错误
	 * */
	public void track(String account_id,String distinct_id,String event_name,Map<String,Object> properties) throws InvalidArgumentException{
			add(distinct_id,account_id,"track",event_name,properties);
	}
	
	//user
	private void add(String distinct_id,String account_id,String type,Map<String,Object> properties) throws InvalidArgumentException{
		add(distinct_id,account_id,type,null,properties);
	}
	
	private void add(String distinct_id,String account_id,String type,String event_name,Map<String, Object> propertiesp) throws InvalidArgumentException{
		HashMap<String,Object> properties = new HashMap<>();
		properties.putAll(propertiesp);
		if(account_id == null && distinct_id ==null){
			throw new InvalidArgumentException("account_id and distinct_id Simultaneously are null ");
		}
		Map<String,Object> event = new HashMap<String,Object>();


		if(properties.containsKey("#time")){
		    if(properties.get("#time") instanceof Date){
                event.put("#time", properties.get("#time"));
                properties.remove("#time");
            }else{
				throw new InvalidArgumentException("type(#time) must be Date.");
			}
		}else{
			event.put("#time",new Date());
		}
		if(properties.containsKey("#ip")){
			event.put("#ip", properties.get("#ip"));
			properties.remove("#ip");
		}
		
		//properties check
		if(properties != null){
			assertProperties(type,properties);
		}
		
		
		Map<String, Object> eventProperties = new HashMap<String, Object>();
	    if (type.equals("track")) {
	      eventProperties.putAll(this.publicProperties);
	    }
	    if (properties != null) {
	      eventProperties.putAll(properties);
	    }
	    if(distinct_id != null){
	    	event.put("#distinct_id", distinct_id);
	    }
	    if(account_id != null){
	    	event.put("#account_id", account_id);
	    }
	    if(event_name != null){
	    	event.put("#event_name", event_name);
	    }
		
		event.put("#type", type);
	    event.put("properties", eventProperties);
	    
		this.consumer.send(event);
		
	}
	
	/**
	 * 公共属性只用于track接口，其他接口无效，且每次都会自动向track事件中添加公共属性
	 * @param properties	公共属性
	 * */
	public void setSuperProperties(Map<String, Object> properties){
		this.publicProperties.putAll(properties);
	}
	
	/**
	 * 清理掉公共属性，之后track属性将不会再加入公共属性
	 * */
	public void clearSuperProperties(){
		this.publicProperties.clear();
		this.publicProperties.put("#lib", "tga_java_sdk");
		this.publicProperties.put("#lib_version", libVversion);
	}
	
	/**
	 * 判断属性是否满足要求
	 * @param	type	事件类型
	 * @param	properties	属性值
	 * */
	private void assertProperties(String type, Map<String, Object> properties) throws InvalidArgumentException {
		assertType("type catgory",type);
		
		Pattern pattern_key = getPatternKey(type);
		
		for(Entry<String, Object> property:properties.entrySet()){
			if(property.getValue() == null)continue;
			if(pattern_key.matcher(property.getKey()).matches()){
				if(!(property.getValue() instanceof Number) && !(property.getValue() instanceof Date) && !(property.getValue() instanceof String) && !(property.getValue() instanceof Boolean)){
					throw new InvalidArgumentException("The property value should be a basic type: Number, String, Date, Boolean.");
				}else{
					if(type.equals("user_add")&& !(property.getValue() instanceof Number)){
						throw new InvalidArgumentException("Type user_add only support Number");
					}
				}
			}else{
				throw new InvalidArgumentException("type "+type+"'key "+property.getKey()+" is invalid");
			}
			
		}
		
	}
	
	private Pattern getPatternKey(String type) {
		if(type.equals("track")){
			return patternTrack;
		}else{
			return patternNonTrack;
		}
	}
	
	private boolean assertType(String description, String type) throws InvalidArgumentException {
		if ("user_set,user_setonce,user_add,user_del,track".contains(type.toLowerCase())){
			return true;
		}
		throw new InvalidArgumentException(description+" without support : "+type);
	}
	
	/**
	 * 立即提交数据到相应的接收器
	 * */
	public void flush(){
		this.consumer.flush();
	}
	/**
	 * 关闭并退出sdk,阻塞式的
	 * */
	public void close(){
		this.consumer.close();
	}
	

	
	public static class BatchConsumer implements Consumer{
		private final Logger logger = LoggerFactory.getLogger(BatchConsumer.class);
		private boolean isClose = false;
        private final String appid;
		private final int batch_size;
		private final String server_uri;
        private final ObjectMapper jsonMapper;
        private final static int InitFlushSize = 500;
		private final ThreadPoolExecutor executorService;
        private ArrayList<Map<String, Object>> message_channel;


        /**
		 * 改接口功能用于实时向tga服务器传输数据，不需要配合其他的传输工具
		 *@param server_uri	传输数据的uri
		 *@param appid		服务器验证的token	
		 */
		public BatchConsumer(String server_uri,String appid) {
			this(server_uri,appid, InitFlushSize,5);
		}
		/**
		 * 改接口功能用于实时向tga服务器传输数据，不需要配合其他的传输工具
		 *@param server_uri	传输数据的uri
		 *@param appid		服务器验证的token	
		 *@param batch_size	实时传输的数据大小
         *@param concurrentNum 异步数据传输端的并发数量
		 * */
		public BatchConsumer(String server_uri,String appid,int batch_size,int concurrentNum) {
			this.appid = appid;
			this.server_uri = server_uri;
            this.batch_size = batch_size;
			this.jsonMapper = new ObjectMapper();
			this.message_channel = new ArrayList<>();
			this.executorService = initExecutorService(concurrentNum);
			this.jsonMapper.setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);
			this.jsonMapper.setDateFormat(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS"));
            this.jsonMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
		}
		/**
         * 线程池初始化操作
         * */
        public ThreadPoolExecutor initExecutorService(int concurrentNum) {
            int mumPoolSize = Math.min(10,concurrentNum);
            int lineSize = Math.min(Math.max(20,1000000/this.batch_size),500);
            ThreadPoolExecutor executor = new ThreadPoolExecutor(mumPoolSize,mumPoolSize,120,TimeUnit.SECONDS,new LinkedBlockingQueue(lineSize),new RejectedExecutionHandler() {
                @Override
                public void rejectedExecution(Runnable r, ThreadPoolExecutor executor) {
                    executor.getQueue().remove(r);
                }
            });
            return executor;
        }
		@Override
		public void send(Map<String, Object> message) {
			synchronized(this){
			    if(this.isClose){
                        logger.error("tga已经关闭,丢弃新增的数据.");
                }else {//if(this.executorService.getQueue().size() <= Math.min(InitFlushSize,this.batch_size))
                    message_channel.add(message);
                    if(message_channel.size() >= batch_size){
                        flush();
                    }
                }
			}
		}
		@Override
		public void flush() {
			synchronized(this){
				try {
                    String data = jsonMapper.writeValueAsString(message_channel);
					message_channel.clear();
					HttpConsumer httpConsumer = new HttpConsumer(this.server_uri,this.appid);
					httpConsumer.setData(data);
					this.executorService.execute(httpConsumer);
                    this.executorService.purge();
                } catch (JsonProcessingException e) {
					throw new RuntimeException("Failed to become json ",e);
				}catch (Exception e) {
					throw new RuntimeException("Failed to transform with BatchConsumer",e);
				}
			}
		}

		@Override
		public void close() {
            this.isClose = true;
            if(message_channel.size() > 0){
                flush();
            }
            while (!this.executorService.isTerminated()){
                this.executorService.shutdown();
                try {
                    Thread.sleep(300);
                } catch (InterruptedException e) {
                }
            }

        }
		
	}

	public static class LoggerConsumer implements Consumer{
        private final Logger logger = LoggerFactory.getLogger(LoggerConsumer.class);
		private final int batch_size ;
		private final String filename_prefix;
		private final ObjectMapper jsonMapper;
		private final StringBuilder message_buffer;
		private final static int BUFFER_LIMITATION = 1 * 1024 * 1024 * 1024;
		private final SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
		private LoggerWriter logger_writer ;
		private boolean isClose = false;
		
		
		
		public LoggerConsumer(String filename_prefix) throws FileNotFoundException {
			this(filename_prefix,8192);
		}
		/**
		 * @param 	filename_prefix	log数据保存的文件目录
		 * @param 	batch_size 		 一次写入数据的数据量
		 * @throws 	FileNotFoundException	如果设置的数据保存目录不存在，将会抛出错误
		 * **/
		public LoggerConsumer(String filename_prefix,int batch_size) throws FileNotFoundException {
			this.jsonMapper = new ObjectMapper();
			this.batch_size = batch_size;
			this.message_buffer = new StringBuilder();
			this.filename_prefix = filename_prefix;
			this.logger_writer = new LoggerWriter(filename_prefix);
			this.jsonMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
			this.jsonMapper.setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);
			this.jsonMapper.setDateFormat(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS"));
		}
		
		@Override
		public synchronized void send(Map<String, Object> message) {
		    if(this.isClose){
                try {
                    logger.error("丢弃的数据"+jsonMapper.writeValueAsString(message));
                } catch (JsonProcessingException e) {
                }
            }else{
                if(message_buffer.length() < BUFFER_LIMITATION){
                    try {
                        message_buffer.append(jsonMapper.writeValueAsString(message));
                        message_buffer.append("\n");
                    } catch (JsonProcessingException e) {
                        throw new RuntimeException("Failed to become json",e);
                    }
                }else{
                    throw new RuntimeException("Logging buffer exceeded the allowed limitation.");
                }
                if(message_buffer.length() >= batch_size){
                    this.flush();
                }
            }
		}

		@Override
		public synchronized void flush() {
			if(message_buffer.length() == 0){
				return ;
			}
			String file = filename_prefix+File.separator+"log."+simpleDateFormat.format(new Date());
			if(!logger_writer.isValid(file)){
				logger_writer.close();
				logger_writer = null;
			}
			if (logger_writer == null){
				try {
					logger_writer = new LoggerWriter(filename_prefix);
				} catch (FileNotFoundException e) {
					throw new RuntimeException(e);
				}
			}
			if(logger_writer.write(message_buffer)){
				message_buffer.setLength(0);
			}
			
		}

		@Override
		public void close() {
		    this.isClose = true;
			this.flush();
			this.logger_writer.close();
		}
		
		static class LoggerWriter{
			private String filename_last ;
			private FileOutputStream outputStream;
			private final SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
			
			public LoggerWriter(final String filename) throws FileNotFoundException {
				String file_complete = new File(filename)+File.separator+"log."+simpleDateFormat.format(new Date());
				if(filename_last == null || !filename_last.equals(file_complete)){
					this.filename_last = file_complete;
				}
				File parent_directory = new File(file_complete).getParentFile();
				if(!parent_directory.exists()){
					throw new FileNotFoundException(parent_directory.toString());
				}
				this.outputStream = new FileOutputStream(new File(file_complete),true);
			}
			
			public void close(){
				try {
					outputStream.close();
				} catch (IOException e) {
					throw new RuntimeException("Failed to close OutPutStream",e);
				}
			}
			
			public boolean isValid(final String filename){
				return this.filename_last.equals(filename);
			}
			
			public boolean write(final StringBuilder buffer){
				FileLock lock = null;
				try {
					final FileChannel channel = outputStream.getChannel();
					lock = channel.lock(0,Long.MAX_VALUE,false);
					outputStream.write(buffer.toString().getBytes("UTF-8"));
				} catch (IOException e) {
					throw new RuntimeException(e); 
				}finally{
					if(lock != null){
						try {
							lock.release();
						} catch (IOException e) {
							throw new RuntimeException(e);
						}
					}
				}
				return true;
			}
		}
		
		
		
	}


	private static class HttpConsumer implements Runnable{
		private final Logger logger = LoggerFactory.getLogger(HttpConsumer.class);
		private final String server_uri ;
		private final String appid;
		private String data;
		/**
		 * BatchComsumer用于数据传输的接口
		 * 需要的参数
		 * @param server_url 数据网络传输的url，来自tga官网
		 * @parma appid	数据传输时候的验证token
		 * */
		public HttpConsumer(String server_url,String appid){
			this(server_url,appid,true);
		}
		
		private HttpConsumer(String server_url,String appidp,boolean compress_data){
			this.server_uri = server_url;
			this.appid = appidp;
		}
		@Override
		public void run() {
			for (int i = 1 ; i< 4 ; i++){
				try{
                    this.consumer(this.data);
					break;
				}catch(Exception e){
                    try {
                        TimeUnit.SECONDS.sleep(1);
                        logger.warn("第"+i+"尝试发送数据"+e.getCause().getMessage());
					} catch (InterruptedException e1) {

					}
				}

			}

		}
		public void setData(String data){
			this.data = data;
		}

		public void consumer(final String data) throws Exception{
            CloseableHttpClient httpclient = HttpClients.custom().build();
			HttpPost httppost = null;
			int exceptionCount = 0;
			for(int i = 0; i<4 ; i++){
				try {
					StringEntity params = new StringEntity(this.encodeRecord(data),"UTF-8");
					RequestConfig requestConfig = RequestConfig.custom().setSocketTimeout(60000).setConnectTimeout(30000).build();
					httppost = new HttpPost(this.server_uri);
					httppost.setEntity(params);
					httppost.addHeader("appid", this.appid);
					httppost.addHeader("user-agent", "java_sdk_"+libVversion);
					httppost.setConfig(requestConfig);
					CloseableHttpResponse response = httpclient.execute(httppost);
					int code = response.getStatusLine().getStatusCode();
					String result = EntityUtils.toString(response.getEntity(),"UTF-8");
					response.close();
					if(code < 200 || code >= 300){
						throw new Exception(String.format("Unexcepted response %d from tga:%s", code,result));
					}
					break;
				}catch(Exception e){
					exceptionCount++;
					if(exceptionCount >= 3){
						throw new Exception("http transport with error "+e.getCause().getMessage());
					}
				}finally{
					if(httpclient != null){
						try {
							httpclient.close();
							httppost.abort();
						} catch (IOException e) {
							// TODO Auto-generated catch block
						}
					}
				}
			}

			
		}

		private String encodeRecord(String data) {
			ByteArrayOutputStream byteArrayBuffer = new ByteArrayOutputStream();
			try {
				GZIPOutputStream var2 = new GZIPOutputStream(byteArrayBuffer);
				var2.write(data.getBytes(StandardCharsets.UTF_8));
				var2.close();
			}catch(IOException var3) {
				logger.error("GZIP compress with exception", var3);
				return null;
			}
			return new String(Base64.encodeBase64(byteArrayBuffer.toByteArray()));
		}
	}

	
}

