/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.strategy.utils;

import android.text.TextUtils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import cn.thinkingdata.core.utils.Base64Coder;
import cn.thinkingdata.core.utils.MessageDigestUtils;
import cn.thinkingdata.core.utils.TimeUtil;
import cn.thinkingdata.remoteconfig.TDRemoteConfig;
import cn.thinkingdata.remoteconfig.sysconfig.SystemConfigManager;
import cn.thinkingdata.strategy.core.period.PeriodScrollInfo;
import cn.thinkingdata.strategy.rules.api.Facts;
import cn.thinkingdata.strategy.rules.filters.contition.EQRule;
import cn.thinkingdata.strategy.rules.filters.factory.RuleFactory;
import cn.thinkingdata.strategy.rules.support.composite.ActivationRuleGroup;
import cn.thinkingdata.strategy.rules.support.composite.CompositeRule;
import cn.thinkingdata.strategy.rules.support.composite.UnitRuleGroup;
import cn.thinkingdata.strategy.task.TDClientValueManager;
import cn.thinkingdata.strategysdk.BuildConfig;

/**
 * @author liulongbing
 * @since 2024/3/13
 */
public class CommonUtils {
    public static Facts jsonToFacts(JSONObject json) {
        Facts facts = new Facts();
        try {
            Iterator<String> keys = json.keys();
            while (keys.hasNext()) {
                String key = keys.next();
                Object value = json.opt(key);
                if (TextUtils.equals(key, StrategyConstants.EVENT_PROPERTIES_KEY)) {
                    JSONObject properties = json.optJSONObject(StrategyConstants.EVENT_PROPERTIES_KEY);
                    if (properties != null) {
                        Iterator<String> proKeys = properties.keys();
                        while (proKeys.hasNext()) {
                            String k = proKeys.next();
                            Object v = properties.opt(k);
                            facts.put(k.toLowerCase(), v);
                        }
                    }
                } else {
                    facts.put(key, value);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return facts;
    }

    public static CompositeRule parseCompound(int relation, JSONArray filters, double zoneOffset) {
        if (filters == null || filters.length() == 0) return null;
        CompositeRule compositeRule = null;
        if (relation == 1) {
            compositeRule = new UnitRuleGroup();
        } else {
            compositeRule = new ActivationRuleGroup();
        }
        for (int j = 0; j < filters.length(); j++) {
            JSONObject fit = filters.optJSONObject(j);
            if (fit != null) {
                if (fit.has("filterType")) {
                    int r = fit.optInt("relation");
                    JSONArray f = fit.optJSONArray("filts");
                    CompositeRule rule = parseCompound(r, f, zoneOffset);
                    if (rule != null) {
                        compositeRule.addRule(rule);
                    }
                } else {
                    Object r = RuleFactory.createFilterRule(fit, zoneOffset);
                    if (r != null) {
                        compositeRule.addRule(r);
                    }
                }
            }
        }
        return compositeRule;
    }

    public static CompositeRule parseCompound(int relation, JSONArray filters, double zoneOffset, List<String> columns) {
        if (filters == null || filters.length() == 0) return null;
        CompositeRule compositeRule = null;
        if (relation == 1) {
            compositeRule = new UnitRuleGroup();
        } else {
            compositeRule = new ActivationRuleGroup();
        }
        for (int j = 0; j < filters.length(); j++) {
            JSONObject fit = filters.optJSONObject(j);
            if (fit != null) {
                if (fit.has("filterType")) {
                    int r = fit.optInt("relation");
                    JSONArray f = fit.optJSONArray("filts");
                    CompositeRule rule = parseCompound(r, f, zoneOffset);
                    if (rule != null) {
                        compositeRule.addRule(rule);
                    }
                } else {
                    String columnName = fit.optString("columnName");
                    columns.add(columnName);
                    Object r = RuleFactory.createFilterRule(fit, zoneOffset);
                    if (r != null) {
                        compositeRule.addRule(r);
                    }
                }
            }
        }
        return compositeRule;
    }

    public static void parseStartTime(String startTime, PeriodScrollInfo info) {
        try {
            if (startTime == null) return;
            String[] times = startTime.split(":");
            if (times.length == 2) {
                info.startHour = Integer.parseInt(times[0]);
                info.startMinute = Integer.parseInt(times[1]);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public static Object buildEventNameRule(String eventName) throws JSONException {
        JSONObject eventNameJson = new JSONObject();
        eventNameJson.put("calcuSymbol", StrategyConstants.C00);
        eventNameJson.put("columnName", StrategyConstants.EVENT_NAME_KEY);
        eventNameJson.put("columnType", StrategyConstants.STR_TYPE);
        JSONArray ftv = new JSONArray();
        ftv.put(eventName);
        eventNameJson.put("ftv", ftv);
        return new EQRule(eventNameJson);
    }

    public static JSONObject buildArrayPositionRule(String symbol, String columnName, JSONArray ftv) {
        JSONObject arrayPositionJson = new JSONObject();
        try {
            arrayPositionJson.put("calcuSymbol", symbol);
            arrayPositionJson.put("columnName", columnName);
            arrayPositionJson.put("columnType", StrategyConstants.STR_TYPE);
            arrayPositionJson.put("ftv", ftv);
        } catch (JSONException e) {
            e.printStackTrace();
            return null;
        }
        return arrayPositionJson;
    }

    public static String getLogFormatTime(long timeStamp) {
        if (TDStrategyLog.mEnableLog) {
            return TimeUtil.formatDate(new Date(timeStamp), StrategyConstants.TIME_PATTERN, null);
        } else {
            return timeStamp + "";
        }
    }

    public static boolean isEmpty(Object obj) {
        return obj == null || obj.toString().length() == 0;
    }

    public static boolean isNull(Object obj) {
        return obj == null || obj.toString().equalsIgnoreCase("null");
    }

    public static long getTimeStampForObj(Object obj, Object zoneOffset) {
        double zoneOffsetValue = 0;
        if (zoneOffset != null) {
            zoneOffsetValue = Double.parseDouble(zoneOffset.toString());
        }
        if (obj instanceof Date) {
            return (( Date ) obj).getTime();
        } else {
            String v = obj.toString();
            return TimeUtil.getFormatDate(v, StrategyConstants.TIME_PATTERN, zoneOffsetValue).getTime();
        }
    }

    public static Object getColumnValue(Facts facts, String[] splits) {
        if (splits == null || splits.length < 2) return null;
        String columnName = splits[1];
        Object value = null;
        if (columnName.startsWith("#")) {
            value = facts.get(columnName);
        }
        if (CommonUtils.isEmpty(value)) {
            value = TDClientValueManager.getInstance().get(columnName);
        }
        return value;
    }

    public static List<Facts> getFactList(Object value) {
        List<Facts> lists = new ArrayList<>();
        if (value instanceof JSONArray) {
            JSONArray jsonArray = ( JSONArray ) value;
            for (int i = 0; i < jsonArray.length(); i++) {
                Object obj = jsonArray.opt(i);
                if (obj instanceof JSONObject) {
                    lists.add(jsonToFacts(( JSONObject ) obj));
                }
            }
        } else if (value instanceof List) {
            List<?> list = ( List<?> ) value;
            for (Object obj : list) {
                if (obj instanceof JSONObject) {
                    lists.add(jsonToFacts(( JSONObject ) obj));
                }
            }
        } else if (value instanceof JSONObject[]) {
            JSONObject[] jsonObjects = ( JSONObject[] ) value;
            for (JSONObject obj : jsonObjects) {
                if (obj != null) {
                    lists.add(jsonToFacts(obj));
                }
            }
        }
        return lists;
    }

    public static List<String> getValueList(Object value) {
        List<String> lists = new ArrayList<>();
        if (value instanceof JSONArray) {
            JSONArray jsonArray = ( JSONArray ) value;
            for (int i = 0; i < jsonArray.length(); i++) {
                Object obj = jsonArray.opt(i);
                if (obj != null) {
                    lists.add(obj.toString());
                }
            }
        } else if (value instanceof List) {
            List<?> list = ( List<?> ) value;
            for (Object obj : list) {
                if (obj != null) {
                    lists.add(obj.toString());
                }
            }
        } else if (value instanceof JSONObject[]) {
            JSONObject[] jsonObjects = ( JSONObject[] ) value;
            for (JSONObject obj : jsonObjects) {
                if (obj != null) {
                    lists.add(obj.toString());
                }
            }
        }
        return lists;
    }

    public static boolean isStrategyStop(String appId) {
        return SystemConfigManager.getInstance().isStrategyStop(appId, BuildConfig.VERSION_NAME);
    }

    public static String MD5(String str) {
        return MessageDigestUtils.calculateMD5(str + "td-strategy");
    }

    public static boolean isTestTask(int debugType) {
        return debugType == StrategyConstants.DEBUG_TYPE_TASK_TEST || debugType == StrategyConstants.DEBUG_TYPE_CHANNEL_TEST;
    }

    public static boolean isChannelTestTask(int debugType) {
        return debugType == StrategyConstants.DEBUG_TYPE_CHANNEL_TEST;
    }

    public static boolean isCommonEvent(JSONObject eventJson) {
        String type = eventJson.optString(StrategyConstants.EVENT_TYPE_KEY);
        String firstCheckId = eventJson.optString(StrategyConstants.FIRST_CHECK_ID_KEY);
        return TextUtils.equals(type, StrategyConstants.EVENT_TYPE_TRACK) && TextUtils.isEmpty(firstCheckId);
    }

}
