/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.strategy.utils;

import android.text.TextUtils;
import android.util.Pair;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;

import cn.thinkingdata.core.preset.TDPresetModel;
import cn.thinkingdata.core.router.TRouter;
import cn.thinkingdata.core.router.TRouterMap;
import cn.thinkingdata.core.router.provider.IAnalyticsProvider;
import cn.thinkingdata.core.utils.TimeUtil;
import cn.thinkingdata.strategy.TDStrategy;
import cn.thinkingdata.strategy.core.trigger.entity.ClientTriggerRecord;
import cn.thinkingdata.strategy.rules.api.Facts;

/**
 * @author liulongbing
 * @since 2024/3/19
 */
public class AnalyticUtils {

    private static final int PLUGIN_FROM = 4;

    private static final String TRIGGER_EVENT_NAME = "te_ops_client_trigger_record";
    private static final String CLIENT_DEBUG_EVENT = "te_ops_client_debug_test";
    private static final String RISK_RECORD_EVENT = "te_ops_client_risk_record";
    public static final String CHANNEL_TEST_EVENT = "ta_channel_test_event";

    public static final int WITHOUT_SPLIT_FLOW_ID = 12;//分流主体不存在时丢弃用户
    public static final int RACE_SKIP_PUSH = 10;//赛马实验等待放量丢失用户
    public static final int FREQUENCY_CONTROL = 2;//频率控制
    public static final int FATIGUE_CONTROL = 3;//疲劳控制
    public static final int SAMPLE = 8;//实验抽样淘汰
    public static final int EXP_SKIP_PUSH = 9;//实验抽样淘汰
    public static final int FAIL = 5;//推送失败
    public static final int SUCCESS = 6;//推送成功
    public static final int PUSH_ID_IS_NULL = 13;//发送id为空
    public static final int TARGET_USER_IS_FALSE = 101;//不在目标用户范围
    public static final int CLIENT_QP_IS_FALSE = 102;//客户端条件淘汰

    public static final String[] eventBlackList = new String[]{TRIGGER_EVENT_NAME, CLIENT_DEBUG_EVENT, RISK_RECORD_EVENT};
    public static final String[] eventWhiteList = new String[]{CHANNEL_TEST_EVENT};

    public static List<String> hasTrackTestTaskIds = new ArrayList<>();

    public static void trackTriggerEvent(ClientTriggerRecord record) {
        try {
            if (!CommonUtils.isTestTask(record.taskMode) && (record.status == TARGET_USER_IS_FALSE || record.status == CLIENT_QP_IS_FALSE))
                return;
            JSONObject properties = new JSONObject();
            properties.put("#client_user_id", record.clientUserId);
            properties.put("#ops_task_id", record.taskId);
            properties.put("#ops_push_id", record.pushId);
            properties.put("#ops_exp_group_id", record.expGroupId);
            properties.put("#ops_trigger_time", record.triggerTime);
            properties.put("#ops_actual_push_time", record.actualPushTime);
            properties.put("#ops_push_status", record.status);
            if (CommonUtils.isTestTask(record.taskMode)) {
                track(TRIGGER_EVENT_NAME, properties, record.appId, 2);
            } else {
                track(TRIGGER_EVENT_NAME, properties, record.appId, 0);
            }
        } catch (Exception ignore) {
        }
    }

    public static void trackRiskEvent(String appId) {
        try {
            JSONObject properties = new JSONObject();
            properties.put("#ops_risk_type", "2");
            track(RISK_RECORD_EVENT, properties, appId, 0);
        } catch (Exception ignore) {
        }
    }

    public static void trackClientDebugEvent(String appId, List<String> taskIdList) {
        if (taskIdList.isEmpty()) return;
        try {
            JSONObject properties = new JSONObject();
            JSONObject resultJson = new JSONObject();
            resultJson.put("business_type", "1");
            resultJson.put("is_pull_success", true);
            JSONArray taskArray = new JSONArray();
            for (String s : taskIdList) {
                if (hasTrackTestTaskIds.contains(s)) continue;
                hasTrackTestTaskIds.add(s);
                taskArray.put(s);
            }
            if (taskArray.length() <= 0) return;
            resultJson.put("ops_task_id_list", taskArray);
            properties.put("#rcc_pull_result", resultJson);
            track(CLIENT_DEBUG_EVENT, properties, appId, 2);
        } catch (Exception ignore) {
        }
    }

    /**
     * @param eventName      事件名称
     * @param properties     属性
     * @param appId          appId
     * @param trackDebugType 0 normal 1 debug 2 debugOnly
     */
    private static void track(String eventName, JSONObject properties, String appId, int trackDebugType) {
        TRouter.getInstance().build(TRouterMap.ANALYTIC_ROUTE_PATH)
                .withAction("track")
                .withString("eventName", eventName)
                .withString("appId", appId)
                .withInt("from", PLUGIN_FROM)
                .withObject("properties", properties)
                .withInt("trackDebugType", trackDebugType)
                .navigation();
    }

    public static long getCurrentTimeStamp() {
        IAnalyticsProvider provider = getAnalyticProvider();
        if (provider != null) {
            return provider.getCurrentTimeStamp().first;
        }
        return System.currentTimeMillis();
    }

    public static Pair<Long, Boolean> getCurrentTimePair() {
        IAnalyticsProvider provider = getAnalyticProvider();
        if (provider != null) {
            return provider.getCurrentTimeStamp();
        }
        return new Pair<>(System.currentTimeMillis(), false);
    }

    public static IAnalyticsProvider getAnalyticProvider() {
        try {
            return ( IAnalyticsProvider ) TRouter.getInstance().build(TRouterMap.ANALYTIC_PROVIDER_ROUTE_PATH).navigation();
        } catch (Exception ignored) {
            return null;
        }
    }

    public static boolean isBlackEvent(String eventName) {
        for (String blackEvent : eventBlackList) {
            if (blackEvent.equals(eventName)) {
                return true;
            }
        }
        return false;
    }

    public static boolean isWhiteEvent(String eventName) {
        for (String whiteEvent : eventWhiteList) {
            if (whiteEvent.equals(eventName)) {
                return true;
            }
        }
        return false;
    }

    public static Facts buildAnalyticPresetFacts(String appId) {
        Facts facts = new Facts();
        if (TDStrategy.mContext != null) {
            Map<String, Object> presetProperties = TDPresetModel.getInstance(TDStrategy.mContext).getPresetProperties();
            for (String key : presetProperties.keySet()) {
                facts.put(key, presetProperties.get(key));
            }
        }
        IAnalyticsProvider provider = getAnalyticProvider();
        if (provider != null) {
            Map<String, Object> analyticsProperties = provider.getAnalyticsProperties(appId);
            for (String key : analyticsProperties.keySet()) {
                facts.put(key, analyticsProperties.get(key));
            }
            String accountId = provider.getLoginId(appId);
            if (!TextUtils.isEmpty(accountId)) {
                facts.put("#account_id", accountId);
            }
            facts.put("#distinct_id", provider.getDistinctId(appId));
        }
        return facts;
    }

    public static TimeZone getAnalyticsTimeZone(String appId) {
        IAnalyticsProvider provider = getAnalyticProvider();
        if (provider != null) {
            Map<String, Object> analyticsProperties = provider.getAnalyticsProperties(appId);
            Object zoneObj = analyticsProperties.get(StrategyConstants.EVENT_ZONE_OFFSET);
            if (zoneObj != null) {
                return TimeUtil.getTimeZone(( double ) zoneObj);
            }
        }
        return TimeZone.getDefault();
    }

    public static String getAccountId(String appId) {
        IAnalyticsProvider provider = getAnalyticProvider();
        if (provider != null) {
            return provider.getLoginId(appId);
        }
        return "";
    }

    public static String getDistinctId(String appId) {
        IAnalyticsProvider provider = getAnalyticProvider();
        if (provider != null) {
            return provider.getDistinctId(appId);
        }
        return "";
    }

}
