/*
 * Copyright (C) 2024 ThinkingData
 */
package cn.thinkingdata.strategy.storage.db;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.text.TextUtils;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import cn.thinkingdata.core.utils.ZipUtils;
import cn.thinkingdata.strategy.TDStrategy;
import cn.thinkingdata.strategy.utils.AnalyticUtils;
import cn.thinkingdata.strategy.utils.CommonUtils;
import cn.thinkingdata.strategy.utils.FileUtils;
import cn.thinkingdata.strategy.utils.SpUtils;
import cn.thinkingdata.strategy.utils.StrategyConstants;

/**
 * @author liulongbing
 * @since 2024/3/8
 */
public class StrategyDataBaseHelper extends SQLiteOpenHelper {

    private static final String DB_NAME = "td_strategy.db";
    private static final int DB_VERSION = 1;

    private static final String KEY_ID = "id";

    private static final String TABLE_NAME_TASK = "t_task";
    private static final String KEY_APP_ID = "app_id";
    private static final String KEY_CLIENT_UID = "client_uid";
    public static final String KEY_TASK_ID = "task_id";
    public static final String KEY_TASK = "raw_data";
    public static final String KEY_TASK_VERSION = "task_version";
    private static final String CREATE_TASK_TABLE = String.format("CREATE TABLE IF NOT EXISTS %s (%s INTEGER PRIMARY KEY AUTOINCREMENT, %s TEXT NOT NULL,%s TEXT NOT NULL, %s TEXT NOT NULL, %s TEXT NOT NULL, %s INTEGER NULL)", TABLE_NAME_TASK, KEY_ID, KEY_APP_ID, KEY_CLIENT_UID, KEY_TASK_ID, KEY_TASK, KEY_TASK_VERSION);

    private static final String TABLE_NAME_EVENT = "t_event";
    private static final String KEY_EVENT = "key";
    private static final String KEY_TRIGGER_ID = "trigger_id";
    private static final String KEY_TRIGGER_TIME = "event_time";
    private static final String KEY_TRIGGER_RESULT = "num";
    private static final String KEY_EVENT_PROPERTIES = "event";
    private static final String CREATE_ANALYTIC_TABLE = String.format("CREATE TABLE IF NOT EXISTS %s (%s INTEGER PRIMARY KEY AUTOINCREMENT, %s TEXT NOT NULL,%s TEXT NOT NULL, %s TEXT NOT NULL, %s INTEGER NOT NULL, %s TEXT NOT NULL, %s TEXT NULL, %s INTEGER NOT NULL, %s INTEGER NULL)", TABLE_NAME_EVENT, KEY_ID, KEY_APP_ID, KEY_CLIENT_UID, KEY_TASK_ID, KEY_TRIGGER_ID, KEY_EVENT, KEY_EVENT_PROPERTIES, KEY_TRIGGER_RESULT, KEY_TRIGGER_TIME);

    private static final String TABLE_NAME_LIMIT = "t_task_trigger";
    private static final String KEY_LIMIT_TIME = "trigger_time";
    private static final String CREATE_LIMIT_TABLE = String.format("CREATE TABLE IF NOT EXISTS %s (%s INTEGER PRIMARY KEY AUTOINCREMENT, %s TEXT NOT NULL,%s TEXT NOT NULL, %s TEXT NOT NULL, %s INTEGER NOT NULL)", TABLE_NAME_LIMIT, KEY_ID, KEY_APP_ID, KEY_CLIENT_UID, KEY_TASK_ID, KEY_LIMIT_TIME);

    private static final String TABLE_NAME_TOUCH_LIMIT = "t_channel_trigger";
    private static final String KEY_PUSH_ID = "push_id";
    private static final String KEY_CHANNEL_ID = "channel_id";
    private static final String CREATE_TOUCH_LIMIT_TABLE = String.format("CREATE TABLE IF NOT EXISTS %s (%s INTEGER PRIMARY KEY AUTOINCREMENT, %s TEXT NOT NULL,%s TEXT NOT NULL, %s TEXT NOT NULL, %s INTEGER NOT NULL)", TABLE_NAME_TOUCH_LIMIT, KEY_ID, KEY_APP_ID, KEY_PUSH_ID, KEY_CHANNEL_ID, KEY_LIMIT_TIME);

    private static File dbFile;
    private static OnDatabaseChangeListener mListener;
    private static StrategyDataBaseHelper mDataBaseHelper;

    public static StrategyDataBaseHelper getHelper() {
        if (mDataBaseHelper == null || !FileUtils.isDatabaseExists(dbFile)) {
            mDataBaseHelper = new StrategyDataBaseHelper(TDStrategy.mContext);
        }
        return mDataBaseHelper;
    }

    public static void addDatabaseChangeListener(OnDatabaseChangeListener listener) {
        mListener = listener;
    }

    public StrategyDataBaseHelper(Context context) {
        super(context, DB_NAME, null, DB_VERSION);
        dbFile = context.getDatabasePath(DB_NAME);
    }


    @Override
    public void onCreate(SQLiteDatabase db) {
        db.execSQL(CREATE_TASK_TABLE);
        db.execSQL(CREATE_ANALYTIC_TABLE);
        db.execSQL(CREATE_LIMIT_TABLE);
        db.execSQL(CREATE_TOUCH_LIMIT_TABLE);
        updateModifyInfo();
    }

    @Override
    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
    }

    public void createNewTask(StrategyTaskInfo info) {
        try {
            SQLiteDatabase db = getWritableDatabase();
            ContentValues values = new ContentValues();
            values.put(KEY_APP_ID, info.appId);
            values.put(KEY_CLIENT_UID, info.userId);
            values.put(KEY_TASK_ID, info.taskId);
            values.put(KEY_TASK, ZipUtils.gzip(info.task));
            values.put(KEY_TASK_VERSION, info.taskVersion);
            long count = db.insert(TABLE_NAME_TASK, null, values);
            if (count > 0) {
                updateModifyInfo();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void updateTask(StrategyTaskInfo info) {
        try {
            SQLiteDatabase db = getWritableDatabase();
            ContentValues values = new ContentValues();
            values.put(KEY_TASK, ZipUtils.gzip(info.task));
            values.put(KEY_TASK_VERSION, info.taskVersion);
            long count = db.update(TABLE_NAME_TASK, values, KEY_APP_ID + "=? AND " + KEY_CLIENT_UID + "=?  AND " + KEY_TASK_ID + "=?", new String[]{info.appId, info.userId, info.taskId});
            if (count > 0) {
                updateModifyInfo();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void deleteTask(String appId, String userId, String taskId) {
        try {
            SQLiteDatabase db = getWritableDatabase();
            long count = db.delete(TABLE_NAME_TASK, KEY_APP_ID + "=? AND " + KEY_CLIENT_UID + "=? AND " + KEY_TASK_ID + "=?", new String[]{appId, userId, taskId});
            if (count > 0) {
                updateModifyInfo();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void deleteTaskByLimit() {
        try {
            SQLiteDatabase dbQuery = getReadableDatabase();
            Cursor cursor = dbQuery.rawQuery("SELECT COUNT(*) FROM " + TABLE_NAME_TASK, null);
            cursor.moveToFirst();
            long total = cursor.getInt(0);
            cursor.close();
            //超出限制
            if (total > StrategyConstants.DATA_BASE_LIMIT) {
                SQLiteDatabase db = getWritableDatabase();
                long deleteCount = total - StrategyConstants.DATA_DELETE_LIMIT;
                String subQuery = "(SELECT " + KEY_ID + " FROM " + TABLE_NAME_TASK + " ORDER BY " + KEY_ID + " ASC LIMIT " + deleteCount + ")";
                long count = db.delete(TABLE_NAME_TASK, KEY_ID + " IN " + subQuery, null);
                if (count > 0) {
                    updateModifyInfo();
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    public Map<String, StrategyTaskInfo> getTasksByUser(String appId, String userId) {
        //文件被修改 删除文件
        if (isDbModified()) return new HashMap<>();
        Cursor cursor = null;
        Map<String, StrategyTaskInfo> maps = new HashMap<>();
        try {
            SQLiteDatabase db = getReadableDatabase();
            cursor = db.query(TABLE_NAME_TASK, null, KEY_APP_ID + "=? AND " + KEY_CLIENT_UID + "=?", new String[]{appId, userId}, null, null, null);
            if (cursor != null) {
                while (cursor.moveToNext()) {
                    StrategyTaskInfo info = new StrategyTaskInfo();
                    int appIdIndex = cursor.getColumnIndex(KEY_APP_ID);
                    info.appId = cursor.getString(appIdIndex);
                    int userIdIndex = cursor.getColumnIndex(KEY_CLIENT_UID);
                    info.userId = cursor.getString(userIdIndex);
                    int taskIdIndex = cursor.getColumnIndex(KEY_TASK_ID);
                    info.taskId = cursor.getString(taskIdIndex);
                    int taskIndex = cursor.getColumnIndex(KEY_TASK);
                    info.task = ZipUtils.unGzip(cursor.getString(taskIndex));
                    int taskVersionIndex = cursor.getColumnIndex(KEY_TASK_VERSION);
                    info.taskVersion = cursor.getLong(taskVersionIndex);
                    maps.put(info.taskId, info);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return maps;
    }

    public List<TriggerResultInfo> getTriggerResultList(TriggerResultInfo info, long start, long end) {
        //文件被修改 删除文件
        if (isDbModified()) return new ArrayList<>();
        Cursor cursor = null;
        List<TriggerResultInfo> list = new ArrayList<>();
        if (null == info) return list;
        try {
            SQLiteDatabase db = getReadableDatabase();
            cursor = db.rawQuery("SELECT * FROM " + TABLE_NAME_EVENT + " WHERE " + KEY_APP_ID + "=? AND " + KEY_CLIENT_UID + "=?  AND " + KEY_TASK_ID + "=?  AND " + KEY_TRIGGER_ID + "=?  AND " + KEY_TRIGGER_TIME + ">=?  AND " + KEY_TRIGGER_TIME + "<=? ORDER BY " + KEY_TRIGGER_TIME + " DESC", new String[]{info.appId, info.userId, info.taskId, info.triggerId + "", start + "", end + ""});
            if (cursor != null) {
                while (cursor.moveToNext()) {
                    TriggerResultInfo triggerInfo = new TriggerResultInfo();
                    int idIndex = cursor.getColumnIndex(KEY_ID);
                    triggerInfo._id = cursor.getLong(idIndex);
                    int appIdIndex = cursor.getColumnIndex(KEY_APP_ID);
                    triggerInfo.appId = cursor.getString(appIdIndex);
                    int userIdIndex = cursor.getColumnIndex(KEY_CLIENT_UID);
                    triggerInfo.userId = cursor.getString(userIdIndex);
                    int taskIdIndex = cursor.getColumnIndex(KEY_TASK_ID);
                    triggerInfo.taskId = cursor.getString(taskIdIndex);
                    int triggerIdIndex = cursor.getColumnIndex(KEY_TRIGGER_ID);
                    triggerInfo.triggerId = cursor.getInt(triggerIdIndex);
                    int keyIndex = cursor.getColumnIndex(KEY_EVENT);
                    triggerInfo.key = cursor.getString(keyIndex);
                    int resultIndex = cursor.getColumnIndex(KEY_TRIGGER_RESULT);
                    triggerInfo.result = cursor.getInt(resultIndex);
                    int relationIndex = cursor.getColumnIndex(KEY_EVENT_PROPERTIES);
                    triggerInfo.relationProperties = cursor.getString(relationIndex);
                    int timeIndex = cursor.getColumnIndex(KEY_TRIGGER_TIME);
                    triggerInfo.triggerTimestamp = cursor.getLong(timeIndex);
                    list.add(triggerInfo);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return list;
    }

    public void addTriggerResult(TriggerResultInfo info) {
        try {
            ContentValues values = new ContentValues();
            values.put(KEY_APP_ID, info.appId);
            values.put(KEY_CLIENT_UID, info.userId);
            values.put(KEY_TASK_ID, info.taskId);
            values.put(KEY_TRIGGER_ID, info.triggerId);
            values.put(KEY_EVENT, info.key);
            values.put(KEY_TRIGGER_RESULT, info.result);
            values.put(KEY_EVENT_PROPERTIES, info.relationProperties);
            values.put(KEY_TRIGGER_TIME, AnalyticUtils.getCurrentTimeStamp());
            SQLiteDatabase db = getWritableDatabase();
            long count = db.insert(TABLE_NAME_EVENT, null, values);
            if (count > 0) {
                updateModifyInfo();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void deleteTriggerResult(String appId, String userId, String taskId) {
        try {
            SQLiteDatabase db = getWritableDatabase();
            long count = db.delete(TABLE_NAME_EVENT, KEY_APP_ID + "=? AND " + KEY_CLIENT_UID + "=? AND " + KEY_TASK_ID + "=?", new String[]{appId, userId, taskId});
            if (count > 0) {
                updateModifyInfo();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void deleteTriggerResultLimit() {
        try {
            SQLiteDatabase dbQuery = getReadableDatabase();
            Cursor cursor = dbQuery.rawQuery("SELECT COUNT(*) FROM " + TABLE_NAME_EVENT, null);
            cursor.moveToFirst();
            long total = cursor.getInt(0);
            cursor.close();
            if (total > StrategyConstants.DATA_BASE_LIMIT) {
                SQLiteDatabase db = getWritableDatabase();
                long deleteCount = total - StrategyConstants.DATA_DELETE_LIMIT;
                String subQuery = "(SELECT " + KEY_ID + " FROM " + TABLE_NAME_EVENT + " ORDER BY " + KEY_TRIGGER_TIME + " ASC LIMIT " + deleteCount + ")";
                long count = db.delete(TABLE_NAME_EVENT, KEY_ID + " IN " + subQuery, null);
                if (count > 0) {
                    updateModifyInfo();
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void deleteTriggerResultByIds(List<String> ids) {
        try {
            String[] placeholders = new String[ids.size()];
            Arrays.fill(placeholders, "?");
            String placeholdersStr = TextUtils.join(",", placeholders);
            SQLiteDatabase db = getWritableDatabase();
            long count = db.delete(TABLE_NAME_EVENT, KEY_ID + " IN (" + placeholdersStr + ")", ids.toArray(new String[0]));
            if (count > 0) {
                updateModifyInfo();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public int getFrequencyLimitCount(FrequencyLimitInfo info, long start, long end) {
        if (isDbModified()) return -1;
        Cursor cursor = null;
        if (null == info) return -1;
        try {
            SQLiteDatabase db = getReadableDatabase();
            cursor = db.rawQuery("SELECT COUNT(*) FROM " + TABLE_NAME_LIMIT + " WHERE " + KEY_APP_ID + "=? AND " + KEY_CLIENT_UID + "=?  AND " + KEY_TASK_ID + "=?  AND " + KEY_LIMIT_TIME + ">=?  AND " + KEY_LIMIT_TIME + "<=?", new String[]{info.appId, info.userId, info.taskId, start + "", end + ""});
            cursor.moveToFirst();
            return cursor.getInt(0);
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return -1;
    }

    public void addFrequencyLimit(FrequencyLimitInfo info) {
        try {
            ContentValues values = new ContentValues();
            values.put(KEY_APP_ID, info.appId);
            values.put(KEY_CLIENT_UID, info.userId);
            values.put(KEY_TASK_ID, info.taskId);
            values.put(KEY_LIMIT_TIME, AnalyticUtils.getCurrentTimeStamp());
            SQLiteDatabase db = getWritableDatabase();
            long count = db.insert(TABLE_NAME_LIMIT, null, values);
            if (count > 0) {
                updateModifyInfo();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void deleteFrequencyLimit(String appId, String userId, String taskId) {
        try {
            SQLiteDatabase db = getWritableDatabase();
            long count = db.delete(TABLE_NAME_LIMIT, KEY_APP_ID + "=? AND " + KEY_CLIENT_UID + "=? AND " + KEY_TASK_ID + "=?", new String[]{appId, userId, taskId});
            if (count > 0) {
                updateModifyInfo();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void deleteFrequencyByLimit() {
        try {
            SQLiteDatabase dbQuery = getReadableDatabase();
            Cursor cursor = dbQuery.rawQuery("SELECT COUNT(*) FROM " + TABLE_NAME_LIMIT, null);
            cursor.moveToFirst();
            long total = cursor.getInt(0);
            cursor.close();
            //超出限制
            if (total > StrategyConstants.DATA_BASE_LIMIT) {
                SQLiteDatabase db = getWritableDatabase();
                long deleteCount = total - StrategyConstants.DATA_DELETE_LIMIT;
                String subQuery = "(SELECT " + KEY_ID + " FROM " + TABLE_NAME_LIMIT + " ORDER BY " + KEY_LIMIT_TIME + " ASC LIMIT " + deleteCount + ")";
                long count = db.delete(TABLE_NAME_LIMIT, KEY_ID + " IN " + subQuery, null);
                if (count > 0) {
                    updateModifyInfo();
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void addTouchLimit(TouchLimitInfo info) {
        try {
            ContentValues values = new ContentValues();
            values.put(KEY_APP_ID, info.appId);
            values.put(KEY_PUSH_ID, info.pushId);
            values.put(KEY_CHANNEL_ID, info.channelId);
            values.put(KEY_LIMIT_TIME, AnalyticUtils.getCurrentTimeStamp());
            SQLiteDatabase db = getWritableDatabase();
            long count = db.insert(TABLE_NAME_TOUCH_LIMIT, null, values);
            if (count > 0) {
                updateModifyInfo();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public int getTouchLimitCount(TouchLimitInfo info, long start, long end) {
        if (isDbModified()) return -1;
        Cursor cursor = null;
        if (null == info) return -1;
        try {
            SQLiteDatabase db = getReadableDatabase();
            cursor = db.rawQuery("SELECT COUNT(*) FROM " + TABLE_NAME_TOUCH_LIMIT + " WHERE " + KEY_APP_ID + "=? AND " + KEY_PUSH_ID + "=?  AND " + KEY_CHANNEL_ID + "=?  AND " + KEY_LIMIT_TIME + ">=?  AND " + KEY_LIMIT_TIME + "<=?", new String[]{info.appId, info.pushId, info.channelId, start + "", end + ""});
            cursor.moveToFirst();
            return cursor.getInt(0);
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return -1;
    }

    public void deleteTouchLimitByTime(long timeStamp) {
        try {
            SQLiteDatabase db = getWritableDatabase();
            long count = db.delete(TABLE_NAME_TOUCH_LIMIT, KEY_LIMIT_TIME + "<=?", new String[]{timeStamp + ""});
            if (count > 0) {
                updateModifyInfo();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void updateModifyInfo() {
        try {
            if (dbFile != null && dbFile.exists()) {
                long lastModified = dbFile.lastModified();
                SpUtils.getInstance().putString(SpUtils.KEY_DB_MODIFY_TIME, CommonUtils.MD5(lastModified + ""));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    private boolean isDbModified() {
        try {
            if (dbFile != null && dbFile.exists()) {
                String lastModified = CommonUtils.MD5(dbFile.lastModified() + "");
                String lastM = SpUtils.getInstance().getString(SpUtils.KEY_DB_MODIFY_TIME, "");
                if (TextUtils.isEmpty(lastM)) {
                    SpUtils.getInstance().putString(SpUtils.KEY_DB_MODIFY_TIME, lastModified);
                    return false;
                }
                boolean isModified = !TextUtils.equals(lastModified + "", lastM);
                if (isModified) {
                    close();
                    //删除数据库文件 同时清除MODIFY信息
                    dbFile.delete();
                    SpUtils.getInstance().putString(SpUtils.KEY_DB_MODIFY_TIME, "");
                    if (mListener != null) {
                        mListener.onDbModified();
                    }
                }
                return isModified;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }


    public interface OnDatabaseChangeListener {
        void onDbModified();
    }

}
